#ifndef lint
static char *sccsid = "@(#)%M%  %I%  Teemu Torma %H%";
#endif

/* Miscancelleus functions, logging, sequence numberic etc.

   @(#)Copyright (c) 1987 by Teemu Torma

   Permission is given to distribute this program and alter this code as
   needed to adapt it to forign systems provided that this header is
   included and that the original author's name is preserved. */

/* LINTLIBRARY */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <varargs.h>
#include <unistd.h>
#include <sys/types.h>
#include <time.h>
#include <sys/utsname.h>
#include <errno.h>
#include "hsu.h"
#include "config.h"
#include "fnet.h"
#include "sysexits.h"
#include "nodelist.h"
#include "shuffle.h"

#define labs(n) (((n) < 0l) ? (-(n)) : (n))

extern void exit(), perror();
extern long atol();
extern time_t time();
extern int errno;
extern char *sys_errlist[];

FILE *logfp = NULL;

/* Lock file descriptor up to the end. If yur system doesn't have lockf()
   (also known as locking()), or other region or file locking function
   or system call, this should be done with lock-files. */

int
lock(fd)
     int fd;
{
#ifdef LOCK_LOCKF
  return lockf(fd, F_LOCK, 0l);
#else
  return locking(fd, F_LOCK, 0l);
#endif
}

/* Unlock file descriptor up to the end. Routine which calls this should
   first seek to original position. */

int
unlock(fd)
     int fd;
{
#ifdef LOCK_LOCKF
  return lockf(fd, F_ULOCK, 0l);
#else
  return locking(fd, F_ULOCK, 0l);
#endif
}

/* Return ascii-date in specified format. If format string is null, return
   date as date(1) returns it. Format is same than date(1) has, in addition
   %z, which means timezone name. Clock is the time to convert, if NULL,
   we'll use current time. */

char *
date(fmt, clock)
     char *fmt;
     time_t *clock;
{
  /* names for weekdays */
  static char *weekdays[] = {
    "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat",
  };

  /* names for months */
  static char *months[] = {
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec",
  };

  static char buffer[80];
  char *bp = buffer;
  time_t _clock;
  struct tm *tm;

  if (!clock)
    _clock = time((long *) 0);
  tm = localtime(clock ? clock : &_clock);

  /* if no format string, this is default */
  if (!fmt)
    fmt = "%a %h %d %T %z 19%y";

  for (*bp = 0; *fmt; fmt++)
    switch (*fmt)
      {
      case '%':
        switch (*++fmt)
          {
            /* newline */
          case 'n':
            *bp++ = '\n';
            break;
            /* tabulator */
          case 't':
            *bp++ = '\t';
            break;
            /* month number 1-12 */
          case 'm':
            (void) sprintf(bp, "%02d", tm->tm_mon + 1);
            while (*bp)
              bp++;
            break;
            /* day of month 1-31 */
          case 'd':
            (void) sprintf(bp, "%2d", tm->tm_mday);
            while (*bp)
              bp++;
            break;
          case 'q':
            (void) sprintf(bp, "%02d", tm->tm_mday);
            while (*bp)
              bp++;
            break;
            /* year 00-99 */
          case 'y':
            (void) sprintf(bp, "%02d", tm->tm_year);
            while (*bp)
              bp++;
            break;
            /* date in format YY/MM/DD */
          case 'D':
            (void) sprintf(bp, "%02d/%02d/%02d", tm->tm_year,
                           tm->tm_mon + 1, tm->tm_mday);
            while (*bp)
              bp++;
            break;
            /* hour 0-23 */
          case 'H':
            (void) sprintf(bp, "%02d", tm->tm_hour);
            while (*bp)
              bp++;
            break;
            /* minutes 0-59 */
          case 'M':
            (void) sprintf(bp, "%02d", tm->tm_min);
            while (*bp)
              bp++;
            break;
            /* seconds 0-59 */
          case 'S':
            (void) sprintf(bp, "%02d", tm->tm_sec);
            while (*bp)
              bp++;
            break;
            /* time in format HH:MM:SS */
          case 'T':
            (void) sprintf(bp, "%02d:%02d:%02d", tm->tm_hour, tm->tm_min,
                           tm->tm_sec);
            while (*bp)
              bp++;
            break;
            /* day of year 1-356 */
          case 'j':
            (void) sprintf(bp, "%03d", tm->tm_yday + 1);
            while (*bp)
              bp++;
            break;
            /* weekday 0-6 */
          case 'w':
            (void) sprintf(bp, "%d", tm->tm_wday);
            while (*bp)
              bp++;
            break;
            /* name of weekday 'Mon', 'Tue', ... , 'Sun' */
          case 'a':
            (void) strcpy(bp, weekdays[tm->tm_wday]);
            while (*bp)
              bp++;
            break;
            /* name of month 'Jan', 'Feb', ... , 'Dec' */
          case 'h':
            (void) strcpy(bp, months[tm->tm_mon]);
            while (*bp)
              bp++;
            break;
            /* name of timezone, e.g. EST */
          case 'z':
            (void) strcpy(bp, *tzname);
            while (*bp)
              bp++;
            break;
            /* numeric time zone, e.g. +0200 */
          case 'o':
            (void) sprintf(bp, "%c%02ld%02ld", (timezone <= 0l) ? '+' : '-',
                           (labs(timezone) / (60l * 60l)),
                           (labs(timezone) % (60l * 60l)));
            while (*bp)
              bp++;
            break;
          case 'l':
            /* military time zone, Z = UT, A = -1, M = -12, (J not used),
               N = +1, Y = +12.. */
            *bp = (timezone == 0l) ? 'Z' : ((int) (labs(timezone) /
                                                   (60l * 60l)) +
                                            ((timezone < 0l) ? 'M' : '@'));
            if (timezone > 0l && *bp >= 'J')
              (*bp)++;
            *++bp = 0;
            break;
          default:
            *bp++ = *fmt;
            break;
          }
        break;
      default:
        *bp++ = *fmt;
        break;
      }

  *bp = 0;
  return buffer;
}

/* Log to logfile. If logfile is not open, open it.

   If first character in format string is '$', print also errno. If external
   variable verbose is set, logging will be done also to stderr. */

/* VARARGS */
void
log(va_alist)
     va_dcl
{
  va_list args;
  char *fmt;
  int eno = errno;

  va_start(args);

  fmt = va_arg(args, char *);

  if (!logfp)
    if ((logfp = fopen(LOGFILE, "a")) == NULL)
      {
    perror("Cannot open log file");
    return;
      }

  (void) fprintf(logfp, "%s: ", date("%a, %d %h %y %T %o", (long *) 0));
  (void) vfprintf(logfp, *fmt == '$' ? fmt + 1 : fmt, args);
  if (*fmt == '$')
    (void) fprintf(logfp, " errno = %d (%s)\n", eno, sys_errlist[eno]);
  else
    (void) fprintf(logfp, "\n");
  (void) fflush(logfp);

  /* if verbose is set, print also to stderr */
  if (verbose)
    {
      (void) vfprintf(stderr, fmt + (*fmt == '$'), args);
      if (*fmt == '$')
        (void) fprintf(stderr, " errno = %d (%s)\n", eno,
                       sys_errlist[eno]);
      else
        (void) fprintf(stderr, "\n");
      (void) fflush(stderr);
    }

  va_end(args);
}

/* Debug output. First argument should be number, rest are used arguments
   for vfprintf(3S). If external variable verbose has equal or greater
   value than first number, vfprintf(3S) will be used to print other
   arguments to stderr. */

/* VARARGS */
void
debug(va_alist)
     va_dcl
{
  va_list args;
  char *fmt;
  int debug_level;

  va_start(args);

  debug_level = va_arg(args, int);
  fmt = va_arg(args, char *);

  if (debug_level <= verbose)
    {
      if (*fmt != '>' && *fmt != '<')
    (void) fprintf(stderr, "\n%ld: ", time( (long *) NULL));
      (void) vfprintf(stderr, fmt, args);
    }

  va_end(args);
}

/* Get next job number. New sequemnt number will be taken from file
   LIBDIR/seq, which is in ascii-format and new number will be saved
   back there. */

long
job_number()
{
  char seqfile[128], buffer[14];
  FILE *fp;
  long seqn = 0;

  (void) sprintf(seqfile, "%s/seq", LIBDIR);
  if (fp = fopen(seqfile, "r+"))
    {
      (void) lock(fileno(fp));
      if (fgets(buffer, 14, fp))
        seqn = atol(buffer);
      seqn++;
      (void) rewind(fp);
      (void) fprintf(fp, "%ld\n", seqn);
      (void) unlock(fileno(fp));
      (void) fclose(fp);
    }
  else
    {
      log("$Can not open seq-file %s", seqfile);
      exit(EX_OSFILE);
    }
  return seqn;
}

/* General sequencer */

long
sequencer(filename)
     char *filename;
{
  char seqfile[128], buffer[14];
  FILE *fp;
  long seqn = 0;

  (void) sprintf(seqfile, "%s", filename);
  if ((fp = fopen(seqfile, "r+")) == NULL)
    {
      if (errno == ENOENT)
    {
      if ((fp = fopen(seqfile, "w+")) == NULL)
        {
          log("$Can not create seq-file %s", seqfile);
          exit(EX_OSFILE);
        }
      fputs("1", fp);
      fclose(fp);
      if ((fp = fopen(seqfile, "r+")) == NULL)
        {
          log("$Can not open new seq-file %s", seqfile);
          exit(EX_OSFILE);
        }
    }
      else
    {
      log("$Can not open seq-file %s", seqfile);
      exit(EX_OSFILE);
    }
    }

  (void) lock(fileno(fp));
  if (fgets(buffer, 14, fp))
    seqn = atol(buffer);
  else
    seqn = 0; /* This can theoretically fail */

  seqn++;
  (void) rewind(fp);
  (void) fprintf(fp, "%ld\n", seqn);
  (void) unlock(fileno(fp));
  (void) fclose(fp);
  return seqn;
}

/* Returns current last sequence number */
long
getsequencer(filename)
     char *filename;
{
  char seqfile[128], buffer[14];
  FILE *fp;
  long seqn = 0;

  (void) sprintf(seqfile, "%s", filename);
  if ((fp = fopen(seqfile, "r+")) == NULL)
    {
      if (errno == ENOENT)
    {
      if ((fp = fopen(seqfile, "w+")) == NULL)
        {
          log("$Can not create seq-file %s", seqfile);
          exit(EX_OSFILE);
        }
      fputs("1", fp);
      fclose(fp);
      if ((fp = fopen(seqfile, "r+")) == NULL)
        {
          log("$Can not open new seq-file %s", seqfile);
          exit(EX_OSFILE);
        }
    }
      else
    {
      log("$Can not open seq-file %s", seqfile);
      exit(EX_OSFILE);
    }
    }

  (void) lock(fileno(fp));
  if (fgets(buffer, 14, fp))
    seqn = atol(buffer);
  else
    seqn = 0; /* This can theoretically fail */

  (void) unlock(fileno(fp));
  (void) fclose(fp);
  return seqn;
}

/* Get full pathname for spoolfile with new job number. File is in
   spool directory and contains prefix followed by four digit
   job number. */

char *
spoolfile(prefix)
     char *prefix;
{
  static char file[BUFLEN];

  (void) sprintf(file, "%s/%s%08ld", SPOOL, prefix, job_number());
  return file;
}

/* Return basename of s */

char *
basename(s)
     register char *s;
{
  register char *p = s;

  while (*s)
    if (*s++ == '/')
      p = s;
  return p;
}

/* Open file with directory name and filename. */

FILE *
pfopen(dir, name, mode)
     char *dir, *name, *mode;
{
  char filename[128];

  (void) strcpy(filename, dir);
  (void) strcat(filename, "/");
  (void) strcat(filename, name);

  return fopen(filename, mode);
}

char *baseit(n)
     long n;
{
  static char tab[] =
    "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
  int count = 0;

  SHUFFLEBUFFERS;

  while (n)
    {
      tcharp[count] = tab[n % strlen(tab)];
      n = n / strlen(tab);
      count++;
    }

  tcharp[count] = 0;
  return tcharp;
}

/* Create packet name for node given */

sprintpacketname(s, node)
     char *s;
     Node node;
{
  sprintf(s, "%s%s.%s.%s", node.point ?
      sprintfs("%s.", baseit( (long) node.point)) : "",
      baseit( (long) node.node), baseit( (long) node.net),
      baseit( (long) node.zone));
}

/* Create packet name for inbound xx.xx.xx.xx.num. If point
   number is 0, don't include it. All numbers are in ~63-base to squeeze
   them to as small space as possible. It could be more sensible solution
   to make them directory trees but I would need more time for that. This
   trick makes finding packets difficult.
   */

sprintipacketname(s, node)
     char *s;
     Node node;
{
  sprintf(s, "%s%s.%s.%s.%s", node.point ?
      sprintfs("%s.", baseit( (long) node.point)) : "",
      baseit( (long) node.node), baseit( (long) node.net),
      baseit( (long) node.zone), baseit(sequencer(IPACKETSEQUENCE)));
}

/* Get line from config file. If *-character is in first column, report
   EOF, and otherwise return line with comments stripped. This causes
   effect, that each section in configuration file looks like it's own
   file. Arguments and return value are the same than with fgets(3S). */

char *
getcl(buffer, len, fp)
     char *buffer;
     int len;
     FILE *fp;
{
  char *cp;

  while (fgets(buffer, len, fp))
    {
      buffer[strlen(buffer) - 1] = 0;
      if (*buffer == '*')
        return (char *) NULL;
      /* everything after #-sign is comment */
      if (cp = strchr(buffer, '#'))
        *cp = 0;
      /* if there's something left, return it */
      if (*buffer)
        return buffer;
    }
  return (char *) NULL;
}

/* Scan config file to specified section. This mechanism is not very
   effective, but otherwise it would get too complicated. */

void
section(number, config)
     int number;
     FILE *config;
{
  char buffer[BUFLEN];

  (void) rewind(config);
  while (--number)
    while (getcl(buffer, BUFLEN, config))
      /* skip section */;
}

/* Get header field from file. */

#define MAX_HEADER_LEN 256

char *mheader(fp, headername)
     FILE *fp;
     char *headername;
{
  static char header[MAX_HEADER_LEN];
  long position;

  position = ftell(fp);

  rewind(fp);

  /* Blank line terminates also, there shouldn't be any headers
     after it any more */

  while (fgets(header, MAX_HEADER_LEN, fp) && *header != '\n')
    if (!strncmp(header, headername, strlen(headername)))
      {
    /* Remove \n at end */
    header[strlen(header) - 1] = 0;
    fseek(fp, position, 0);
    return header + strlen(headername);
      }

  /* Not found, return empty string */

  fseek(fp, position, 0);
  return "";
}



