#ifndef lint
static char *sccsid = "@(#)%M%  %I%  Teemu Torma %H%";
#endif

/* Receive file(s) using Xmodem/TeLink/MODEM7 Batch protocols.

   @(#)Copyright (c) 1987 by Teemu Torma

   Permission is given to distribute this program and alter this code as
   needed to adapt it to forign systems provided that this header is
   included and that the original author's name is preserved. */

#include <stdio.h>
#include <sys/types.h>
#include <string.h>
#include "hsu.h"
#include "fnet.h"
#include "fio.h"
#include "crc.h"

extern time_t time();

/* Macros to check timeouts */

#define Timeout(t)      (time((long *) NULL) - stime > t)
#define SetStart()      (stime = time((long *) NULL))

/* General error states to break the loops. */

#define Error           (-1)
#define Done            (-2)

/* States for XMODEM/TeLink receiver. */

#define RecStart        (0)
#define WaitFirst       (1)
#define WaitBlock       (2)

/* States for BATCH file receiver. */

#define RecvName        (0)
#define CheckFNm        (1)
#define CheckFile       (2)

/* States for MODEM7 filename receiver. */

#define SendNak         (0)
#define WaitAck         (1)
#define WaitChar        (2)
#define WaitOkCk        (3)

static received_block_number = 0;
int receiving_data = FALSE;

bool
xtrecblk(rxbuf, blocknum, crcmode)
     char *rxbuf;
     int blocknum;
     bool crcmode;
{
  register int c, cnt, checksum;
  register unsigned short crc;

  if (blocknum == -1)
    debug(1, "Receiving block ignoring block number");
  else
    debug(1, "Receiving block %d", blocknum);

  received_block_number = blocknum;

  if ((c = readline(10)) == TIMEOUT || c != (blocknum & 0377))
    {
      if (c == TIMEOUT)
        debug(1, "Timeout on block number receive");
      else
    {
      if (blocknum == -1)
        if (c == 0 || c == 1) /* Both telink and xmodem allowed */
          {
        blocknum = c;
        received_block_number = blocknum;
        debug(1, "Setting blocknum to %d", blocknum);
        goto blockok;
          }

      debug(1, "Bad block number %d, expected %d", c, blocknum & 0377);
    }
      goto blidge;
    }

 blockok:
  if ((c = readline(10)) == TIMEOUT || c != (~blocknum & 0377))
    {
      if (c == TIMEOUT)
        debug(1, "Timeout on complement block number receive");
      else
        debug(1, "Bad ~block number %d, expected %d", c, (~blocknum & 0377));
      goto blidge;
    }

  receiving_data = TRUE;
  for (crc = 0, checksum = 0, cnt = 0; cnt < BlockSize; cnt++)
    if ((c = readline(5)) == TIMEOUT)
      {
        debug(1, "Timeout while receiving block");
    receiving_data = FALSE;
        goto blidge;
      }
    else
      {
        checksum += c;
        crc = updcrc(c, crc);
        rxbuf[cnt] = c;
      }
  receiving_data = FALSE;
  debug(5, "Received %d bytes", BlockSize);

  if (crcmode)
    {
      if (readline(10) != (int) ((crc >> 8) & 0377)
          || readline(10) != (int) (crc & 0377))
        {
          debug(1, "Crc error");
          goto blidge;
        }
    }
  else
    {
      if (readline(10) != (checksum & 0377))
        {
          debug(1, "Checksum error");
          goto blidge;
        }
    }
  return True;

 blidge:
  debug(1, "Fail, Skipping rest of the block");
  while (readline(1) != TIMEOUT)
    /* skip rest of the block */;
  return False;
}

bool
xtrec(filename, xmodem)
     char *filename;
     int xmodem;
{
  int state = RecStart;
  time_t stime;
  int tries = 0;
  bool crcmode = True;
  FILE *fp;
  char rxbuf[BlockSize];
  int cnt, block = 1, c;

  if ((fp = fopen(filename, "w")) == NULL)
    {
      log("$Unable to open %s for writing", filename);
      return False;
    }
  log("Receive file %s", filename);
  SetStart();
  while (state >= RecStart && state <= WaitBlock)
    switch (state)
      {
      case RecStart:
    debug(1,"Send %s", crcmode ? "C" : "NAK");
        sendline(crcmode ? 'C' : NAK);
    sendline(0);
    sendline(~0);
        state = WaitFirst;
        break;
      case WaitFirst:
        if (tries > 10 || Timeout(60))
          {
            if (tries > 10)
              log("Too many tries on xmodem receive");
            else
              log("Timeout on xmodem receive start");
            state = Error;
          }
        else if ((c = readline(10)) == EOT)
          {
            sendline(ACK);
            log("No file to receive");
            state = Done;
          }
        else if (c == SYN || c == SOH) /* Probably telink block? */
          {
            debug(1, "Startup in %s mode", crcmode ? "crc" : "checksum");
            if (!xtrecblk(rxbuf, -1 /* c == SYN ? 0 : 1 */ , crcmode))
              {
        debug(1, "Retry, bad block");
                tries++; /* No valid block received */
        state = RecStart;
#ifdef NEEDED
        block = 1; /* Bad block, ignore? */
                state = WaitBlock;
#endif
              }
            else
              {
                if (c == SOH && received_block_number == 1)
                  {
                    for (cnt = 0; cnt < BlockSize; cnt++)
                      (void) putc(c, fp);
                    block = 2;
                    debug(2, "Block written onto disk");
                  }
                else if (c == SYN || received_block_number == 0)
                  {
            sendline(ACK); /* Tell sender we got it */
            sendline(0);
            sendline(~0);
                    block = 1;
                    debug(1, "TeLink block ignored");
                  }
        else
          {
            debug(1, "Retry, bad block");
            tries++;
            state = RecStart;
          }
                state = WaitBlock;
              }
          }
        else if (c == TIMEOUT || c == TSYNCH) /* Tsynch ? Maybe it... */
          {
            debug(1, "Timout on Xmodem rcv start");
            tries++;
            state = RecStart;
          }
        else if (tries > 3 || Timeout(30))
          {
        log("Try checksum mode");
            crcmode = False;
            state = RecStart;
        sleep(1);
          }
        break;
      case WaitBlock:
        SetStart();
        for (tries = 0; state == WaitBlock; tries++)
      if (tries > 10 || Timeout(60))
        {
          if (tries > 10)
        log("Too many retries (%d) on %s", tries, filename);
          else
        log("Timeout on receive %s, tries %d", filename, tries);
          state = Error;
        }
      else if ((c = readline(10)) == EOT)
        {
          log("File %s received ok", filename);
          state = Done;
        }
      else if (c == SOH)
        {
          if (xtrecblk(rxbuf, block, crcmode))
        {
          for (cnt = 0; cnt < BlockSize; cnt++)
            (void) putc(rxbuf[cnt], fp);
          debug(2, "Block written onto disk, tries %d", tries);
          sendline(ACK);
          sendline(block);
          sendline(~block);
          block++;
          tries = 0;
          SetStart();
        }
          else
        {
          debug(1, "Block not received correctly, tries %d", tries);
          sendline(NAK);
          sendline(block);
          sendline(~block);
          tries++;
        }
        }
      else if (c == TIMEOUT)
        {
          debug(1, "Timeout on block %d, tries %d", block, tries);
          sendline(NAK);
          sendline(block);
          sendline(~block);
          tries++;
        }
      else
        tries = 0; /* Trash from send-ahead... skip it */

      }

  (void) fclose(fp);

  return state != Error;
}

int
recmdmfn(filename)
     char *filename;
{
  int state = SendNak;
  time_t stime;
  int tries = 0, c, pos;

  SetStart();
  while (state >= SendNak && state <= WaitOkCk)
    switch (state)
      {
      case SendNak:
        if (tries > 20)
          {
            log("Too many tries to get filename");
            state = Error;
          }
        else if (Timeout(60))
          {
            log("Timeout while getting filename");
            state = Error;
          }
        else
          {
            sendline(NAK);
            state = WaitAck;
            tries++;
          }
        break;
      case WaitAck:
        switch (readline(5))
          {
          case ACK:
            pos = 0;
            state = WaitChar;
            break;
          case EOT:
            pos = 0;
            state = Done;
            break;
          case TIMEOUT:
            debug(2, "Timout while waiting ACK/EOT on MDM7");
            state = SendNak;
            break;
          default:
            state = SendNak;
            debug(2, "Garbage on line while getting filename");
            (void) sleep((unsigned) 1);
            flush();
            break;
          }
        break;
      case WaitChar:
        switch (c = readline(1))
          {
          case EOT:
            pos = 0;
            debug(2, "Got EOT in middle of filename, no files left");
            state = Done;
            break;
          case SUB:
            filename[pos] = 0;
            debug(2, "Got fn %s, sending checksum", filename);
            for (pos = 0, c = SUB; filename[pos]; pos++)
              c += filename[pos];
            sendline(c & 0377);
            state = WaitOkCk;
            break;
          case 'u':
            debug(2, "Got 'u', send NAK again");
            state = SendNak;
            break;
          case TIMEOUT:
            debug(2, "Timeout while waiting char of filename");
            state = SendNak;
            break;
          default:
            filename[pos++] = c;
            debug(3, "Got char '%c' of filename", c);
            break;
          }
        break;
      case WaitOkCk:
        if (readline(1) == ACK)
          {
            debug(1, "Got filename %s ok", filename);
            state = Done;
          }
        else
          {
            debug(1, "Checksum faiure in filename %s", filename);
            state = SendNak;
          }
        break;
      }

  return state == Error ? -1 : pos ? 1 : 0;
}

bool
batchrec(filename)
     char *filename;
{
  int state = RecvName;
  char filen[100];
  int ok;
  if (!filename) filename = filen;

  while (state >= RecvName && state <= CheckFile)
    switch (state)
      {
      case RecvName:
        ok = recmdmfn(filename);
        state = CheckFNm;
        break;
      case CheckFNm:
        switch (ok)
          {
          case -1:
            debug(1, "Abort batch receive");
            state = Error;
            break;
          case 0:
            log("All files received successfully");
            state = Done;
            break;
          case 1:
            ok = xtrec(filename, FALSE);
            state = CheckFile;
          }
        break;
      case CheckFile:
        if (ok)
          {
            debug(1, "%s received successfully", filename);
            state = RecvName;
          }
        else
          {
            log("Batch receive aborted, %s not received", filename);
            state = Error;
          }
        break;
      }

  return state != Error;
}
