#ifndef lint
static char *sccsid = "@(#)%M%  %I%  Teemu Torma %H%";
#endif

/* Routines to get and translate information in nodelist.

   @(#)Copyright (c) 1987 by Teemu Torma

   Permission is given to distribute this program and alter this code as
   needed to adapt it to forign systems provided that this header is
   included and that the original author's name is preserved. */

/*
  Sat Oct  8 17:36:11 1988
  Rewrote nodelist index handling and reading. Now index is kept
  in memory, and has zones also.
  */

/* LINTLIBRARY */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <malloc.h>
#include <values.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "hsu.h"
#include "config.h"
#include "fnet.h"
#include "nodelist.h"

extern long atol();
Node originnode;
NODEINDEX *nodeindex = NULL;
NAMEINDEX *nameindex = NULL;
int nodes, names, compile_zone, compile_region, compile_net;

fine_convert(name)
     char *name;
{
  int wasspace = TRUE;

  for (; *name; name++)
    {
      if (*name == '_') *name = ' ';
      if (isspace(*name))
    wasspace = TRUE;
      else if (wasspace)
    {
      *name = toupper(*name);
      wasspace = FALSE;
    }
      else
    *name = tolower(*name);
    }
}

/* Strip bad chars from names, ie chars not which may be dangerous in
   different networks. Currectly removes all non-alphanumeric chars,
   but leaves whitespace alone. Strips 8bit chars also. */

stripbad(name)
     char *name;
{
  char *d;

  d = alloca(strlen(name) + 1);
  strcpy(d, name);
  for (; *d; d++) if (isalpha(*d) || isspace(*d) || strchr("-._", *d))
    *name++ = *d;

  *name = *d; /* Terminating null, result can be shorter */
}

/* Compare two strings, considering '_' and ' ', and converting
   {|}[\] to aoaAOA instead, and ignoring case.

   ascii_convert(string) does conversion.
   */

char *ascii_convert(s)
     char *s;
{
  char *p;

  p = s;
  while (*s) {
    switch (*s) {
     case '_':
      *s = ' ';
      break;
     case '{':
     case '}':
     case '[':
     case ']':
      *s = 'a';
      break;
     case '|':
     case '\\':
      *s = 'o';
      break;
     default:
      *s = tolower(*s);
    }
    s++;
  }
  return p;
}

stricmp(d, s)
     char *d, *s;
{
  char ds[BUFSIZ], ss[BUFSIZ];

  return strcmp(ascii_convert(strcpy(ds, d)),
        ascii_convert(strcpy(ss, s)));
}

strnicmp(d, s, n)
     char *d, *s;
     int n;
{
  return strncmp(ascii_convert(strcpy(alloca(strlen(d) + 2), d)),
         ascii_convert(strcpy(alloca(strlen(s) + 2), s)), n);
}

/* Compare nodelist index entry (for qsort) */
cmpnodeindex(node1, node2)
     NODEINDEX *node1, *node2;
{
  if (node1->zone < node2->zone) return -1;
  if (node1->zone > node2->zone) return 1;
  if (node1->net < node2->net) return -1;
  if (node1->net > node2->net) return 1;
  if (node1->node < node2->node) return -1;
  if (node1->node > node2->node) return 1;
  return 0; /* Same node */
}

cmpnameindex(name1, name2)
     NAMEINDEX *name1, *name2;
{
  int n;
  n = stricmp(name1->name, name2->name);
  /* debug(9, "name1 %s name2 %s = %d", name1->name, name2->name, n); */
  return n;
}

/* Read file in to a buffer allocating buffer for it */

long read_file(buffer, name)
     char **buffer, *name;
{
  FILE *fp;
  long size = 0;

  if (fp = fopen(name, "r")) {
    fseek(fp, 0L, SEEK_END);
    size = ftell(fp);

    /* Apparently we are on 16-bit? */
    if (size > MAXINT)
      {
    log("Brain damaged CPU architecture reading file?");
    size = 0;
    goto out;
      }

    fseek(fp, 0L, SEEK_SET);

    if (*buffer)
      *buffer = realloc( *buffer, (unsigned) size);
    else
      *buffer = malloc( (unsigned) size);

    debug(2, "Reading %d bytes from %s", size, name);
    if (fread( *buffer, 1, (int) size, fp) != size) size = 0;
    fclose(fp);
  }

 out:
  return size;
}

write_file(buffer, name, size)
     char *buffer, *name;
     int size;
{
  FILE *fp;
  int rvalue = 0;

  if (fp = fopen(name, "w+"))
    {
      debug(2, "Writing %d bytes to %s", size, name);
      if (fwrite(buffer, 1, size, fp) != size) rvalue = -1;
      fclose(fp);
    }

  return rvalue;
}

read_nodeindex()
{
  nodes = read_file( (char **) &nodeindex,
            sprintfs("%s/%s", LIBDIR, INODELIST)) / sizeof(NODEINDEX);
  return nodes == 0L;
}

read_nameindex()
{
  names = read_file( (char **) &nameindex,
            sprintfs("%s/%s", LIBDIR, NAMELIST)) / sizeof(NAMEINDEX);
  return names == 0L;
}

write_nodeindex()
{
  return write_file( (char *) nodeindex,
            sprintfs("%s/%s", LIBDIR, INODELIST),
            (int) (nodes * sizeof(NODEINDEX)));
}

write_nameindex()
{
  return write_file( (char *) nameindex,
            sprintfs("%s/%s", LIBDIR, NAMELIST),
            (int) (names * sizeof(NAMEINDEX)));
}

/* Make nodelist's index-file. That index-file will be used to search
   desired net/region from nodelist.
   Return NULL if everything was fine, otherwise error-message string. */

char *
update_index()
{
  struct stat nlstat, inlstat;
  FILE *nl = NULL, *nlidx = NULL, *namefp = NULL;
  char nodelist[BUFLEN], inodelist[BUFLEN], namelist[BUFLEN];
  char buffer[BUFSIZ];
  long offset;
  char *error;
  Node node;
  NODEINDEX nodei;
  NAMEINDEX namei;
  extern void qsort();

  /* generate nodelist and index-file names */
  (void) sprintf(nodelist, "%s/%s", LIBDIR, NODELIST);
  (void) sprintf(inodelist, "%s/%s", LIBDIR, INODELIST);
  (void) sprintf(namelist, "%s/%s", LIBDIR, NAMELIST);

  /* get statuses of nodelist and index */
  if (stat(nodelist, &nlstat) == -1)
    return "$Error in getting nodelist status";

  errno = 0;
  if (stat(inodelist, &inlstat) == -1 && errno != ENOENT)
    return "$Error in getting status of existing nodelist-index";

  /* If index-file does exists then check modification times and
     first lines. If nodelist is older and first lines are the same,
     no update is needed. If index-file should be rebuild, assume
     also rebuilding namelist. */

  if (errno == 0 && nlstat.st_mtime <= inlstat.st_mtime)
    {
      error = NULL;
      goto done;
    }
  else
    log("Update nodelist-index: nodelist is newer than index");

  /* open index-file for writing */
  if (nlidx != NULL)
    (void) fclose(nlidx);
  if ((nlidx = fopen(inodelist, "w")) == NULL)
    {
      error = "$Unable to open nodelist-index for writing";
      goto done;
    }
  if (namefp)
    (void) fclose(namefp);
  if ((namefp = fopen(namelist, "w")) == NULL)
    {
      error = "$Unable to open namelist-index for writing";
      goto done;
    }

  if (!nl)
    if ((nl = fopen(nodelist, "r")) == NULL)
      return "$Unable to open nodelist";

  (void) rewind(nl);

  compile_zone = MY_ZONE;
  compile_region = MY_REGION;
  compile_net = MY_NET;
  nodes = 0;
  names = 0;

  /* save host/region offsets */
  for (offset = ftell(nl); fgets(buffer, BUFSIZ, nl); offset = ftell(nl))
    {
      if (*buffer == '\n' || *buffer == ';') continue;

      parse_entry(&node, buffer);

      nodei.zone = node.zone;
      nodei.net = node.net;
      nodei.node = node.node;
      nodei.offset = offset;

#ifdef NEEDED
      debug(8, "%s", buffer);
      debug(8, "writing %d:%d/%d", node.zone, node.net, node.node);
#endif
      FWRITE( (char *) &nodei, sizeof(NODEINDEX), 1, nlidx);
      if (ferror(nlidx))
    {
      error = "$Cannot write index, no space?";
      goto done;
    }

      if (node.type != REGION && node.type != HOST &&
      node.type != ZONE && node.type != KENL && node.type != HUB)
    {
      strcpy(namei.name, node.sysop);
      namei.offset = offset;
      namei.zone = compile_zone;
      namei.net = compile_net;
      names++;
      FWRITE( (char *) &namei, sizeof(NAMEINDEX), 1, namefp);
      if (ferror(namefp))
        {
          error = "$Cannot write name index, no space?";
          goto done;
        }
    }

      nodes++;
    }
  error = NULL;

  /* Ok, now get both indices back and qsort them */

  (void) fclose(nl);
  nl = NULL;
  (void) fclose(nlidx);
  nlidx = NULL;
  (void) fclose(namefp);
  namefp = NULL;

  if (read_nodeindex())
    {
      error = "$Cannot read nodelist index";
      goto done;
    }
  if (read_nameindex())
    {
      error = "$Cannot read name index";
      goto done;
    }

  log("Sorting nodelist index");
  (void) qsort( (char *) nodeindex, (unsigned) nodes,
           sizeof(NODEINDEX), cmpnodeindex);
  log("Sorting name index");
  (void) qsort( (char *) nameindex, (unsigned) names,
           sizeof(NAMEINDEX), cmpnameindex);
  log("Sorted indices");

  if (write_nodeindex())
    {
      error = "$Cannot write nodelist index";
      goto done;
    }
  if (write_nameindex())
    {
      error = "$Cannot write name index";
      goto done;
    }

  /* done, close files and return error message */
 done:
  if (nl)
    (void) fclose(nl);
  if (nlidx)
    (void) fclose(nlidx);
  if (namefp)
    (void) fclose(namefp);
  return error;
}

/* Convert underscores in string to spaces. In nodelist there is always
   underscore insted of space (flags is special case). */

void
convert_space(s)
     register char *s;
{
  while (*s)
    {
      if (*s == '_')
        *s = ' ';
      s++;
    }
}

/* Get one comma-terminated field from buffer, retrun pointer to right
   after terminating comma. Convert underscores to spaces in string. */

char *
parse_field(buffer, entry, size)
     char *buffer, *entry;
     int size;
{
  char *np = entry;

  /* copy string */
  while (--size >= 0 && *buffer && *buffer != ',')
    *entry++ = *buffer++;
  *entry = 0;

  switch (*buffer)
    {
    case 0:
      /* end of buffer */
      log("No closing comma in field");
      return (char *) 0;
    case ',':
      /* succesful copy */
      convert_space(np);
      return buffer + 1;
    default:
      /* buffer too long, find comma */
      while (*buffer && *buffer != ',')
        buffer++;
      if (*buffer)
        {
          convert_space(np);
          return buffer + 1;
        }
      else
        {
          log("Missing comma in field");
          return (char *) 0;
        }
    }
  /* NOTREACHED */
}

/* Parse one line of nodelist to node structure. Return NULL is there
   was error's in that line or missing fields. */

Node *
parse_entry(entry, buffer)
     Node *entry;
     char *buffer;
{
  char *cp;
  int n;

  /* strip newline if exists */
  if (cp = strchr(buffer, '\n'))
    *cp = 0;

  /* get type of entry */
  if (!strncmp("Zone,", buffer, 5))
    entry->type = ZONE;
  else if (!strncmp("Region,", buffer, 7))
    entry->type = REGION;
  else if (!strncmp("Host,", buffer, 5))
    entry->type = HOST;
  else if (!strncmp("Hub,", buffer, 4))
    entry->type = HUB;
  else if (!strncmp("Pvt,", buffer, 4))
    entry->type = PVT;
  else if (!strncmp("Hold,", buffer, 5))
    entry->type = HOLD;
  else if (!strncmp("Down,", buffer, 5))
    entry->type = DOWN;
  else if (!strncmp("Kenl,", buffer, 5))
    entry->type = KENL;
  else if (*buffer == ',')
    entry->type = NORMAL;
  else
    {
      log("Unknown type in line '%s'", buffer);
      return (Node *) 0;
    }

  /* get net/region/node number */
  if ((cp = strchr(buffer, ',')) == NULL)
    {
      log("Missing zone/net/node/region in line '%s'", buffer);
      return (Node *) 0;
    }
  if ((n = atoi(++cp)) == 0)
    {
      log("Value of zone/net/node/region is 0 in line '%s'", buffer);
      return (Node *) 0;
    }
  if (entry->type == ZONE)
    {
      entry->zone = n;
      entry->net = 0;
      entry->node = 0;
      entry->point = 0;
      compile_zone = n;
      debug(8, "Zone %d", compile_zone);
    }
  else if (entry->type == REGION)
    {
      entry->zone = compile_zone;
      entry->region = n;
      entry->net = n;    /* For compatibility with old version */
      entry->node = 0;
      entry->point = 0;
      compile_region = n;
      compile_net = n;    /* For compatibility with old version */
      debug(8, "Region %d", compile_region);
    }
  else if (entry->type == HOST)
    {
      entry->zone = compile_zone;
      entry->region = compile_region;
      entry->net = n;
      entry->node = 0;
      entry->point = 0;
      compile_net = n;
      debug(8, "Net %d", compile_net);
    }
  else
    {
      entry->zone = compile_zone;
      entry->region = compile_region;
      entry->net = compile_net;
      entry->node = n;
      entry->point = 0;
    }
  while (*cp && *cp++ != ',');

  /* get node/net/region name */
  if ((cp = parse_field(cp, entry->name, sizeof(entry->name))) == NULL)
    {
      log("Invalid name in line '%s'", buffer);
      return (Node *) 0;
    }

  /* get city */
  if ((cp = parse_field(cp, entry->city, sizeof(entry->city))) == NULL)
    {
      log("Invalid city in line '%s'", buffer);
      return (Node *) 0;
    }

  /* get sysop */
  if ((cp = parse_field(cp, entry->sysop, sizeof(entry->sysop))) == NULL)
    {
      log("Invalid sysop in line '%s'", buffer);
      return (Node *) 0;
    }

  /* get phone number */
  if ((cp = parse_field(cp, entry->phone, sizeof(entry->phone))) == NULL)
    {
      log("Invalid phone-number in line '%s'", buffer);
      return (Node *) 0;
    }

  /* get maximum baud rate */
  if ((n = atoi(cp)) == 0)
    {
      log("Baud rate is 0 in line '%s'", buffer);
      return (Node *) 0;
    }
  entry->speed = n;
  while (*cp && *cp++ != ',');

  /* get flags */
  (void) strncpy(entry->flags, cp, sizeof(entry->flags));
  entry->flags[sizeof(entry->flags) - 1] = 0;

  /* all done */
  return entry;
}

/* Get entry for one node from nodelist. Return NULL, if there is no
   entry for that node. */

Node *
node_entry(node)
     Node node;
{
  static Node entry;
  FILE *fp;
  char buffer[BUFSIZ];
  extern char *bsearch();
  long offset;
  NODEINDEX *nodeip, nodei;

  /* Read index file into memory */

  if (!nodeindex)
    if (read_nodeindex())
      {
    log("$Unable to read nodelist");
    return (Node *) 0;
      }

  nodei.zone = node.zone;
  nodei.net = node.net;
  nodei.node = node.node;

  debug(2, "Searching %s from %d nodes", ascnodei(nodei), nodes);
  nodeip = (NODEINDEX *) bsearch( (char *) &nodei, (char *) nodeindex,
                 (unsigned) nodes, sizeof(NODEINDEX),
                 cmpnodeindex);

  if (nodeip) {
    offset = nodeip->offset;

    /* open nodelist */
    (void) sprintf(buffer, "%s/%s", LIBDIR, NODELIST);
    if ((fp = fopen(buffer, "r")) == NULL)
      {
    log("$Unable to open %s", buffer);
    return (Node *) 0;
      }

    if (fseek(fp, offset, 0))
      {
    log("$Seek error on nodelist");
    (void) fclose(fp);
    return (Node *) 0;
      }

    fgets(buffer, BUFSIZ, fp);
    fclose(fp);

    compile_zone = nodeip->zone;
    compile_net = nodeip->net;
    return parse_entry(&entry, buffer);
  }

  log("Could not find node %s", ascnodei(nodei));

  /* we didn't find net/region */
  (void) fclose(fp);
  return (Node *) 0;
}

char *dialtable[] = { DIALTABLE };

dial_translation(dest, source)
     char *dest, *source;
{
  register int count = 0;

  for (;;) {
    if (!*dialtable[count] ||
    !strncmp(dialtable[count], source, strlen(dialtable[count]))) {

      /* Matched, take prefix, */
      strcpy(dest, dialtable[count + 1]);

      /* Then add phone number */
      strcat(dest, source + strlen(dialtable[count]));
      return;
    }
    count += 2;
  }
}

static char **aliastable = NULL;
static allocated = 0;
static aliases = 0;

expand_aliastable()
{
  if (!aliastable)
    {
      aliastable = (char **) malloc(sizeof(char *) * 20);
      if (aliastable) allocated = 20;
    }

  if (aliases == allocated) /* Need more pointers */
    {
      allocated += 20;
      aliastable = (char **) realloc( (char *) aliastable,
                     sizeof(char *) * allocated);
      if (!aliastable)
    {
      log("Cannot realloc %d bytes", sizeof(char *) * allocated);
      return -1;
    }
    }
  return 0;
}

#define FILENAME_SIZE 256

search_name(name)
     char *name;
{
  Node entry;
  FILE *fp;
  char buffer[BUFSIZ];
  extern char *bsearch();
  char *nlname;
  long offset;
  NAMEINDEX *nameip, namei;

  if (!nameindex)
    if (read_nameindex())
      {
    log("$Unable to read namelist");
    return -1;
      }

  strncpy(namei.name, name, 36);
  namei.name[35] = 0;

  nameip = (NAMEINDEX *) bsearch( (char *) &namei, (char *) nameindex,
                 (unsigned) names, sizeof(NAMEINDEX),
                 cmpnameindex);

  if (nameip)
    {
      offset = nameip->offset;

      /* Open nodelist */
      if ((fp =
       fopen(nlname = sprintfs("%s/%s", LIBDIR, NODELIST), "r")) == NULL)
    {
      log("$Unable to open nodelist %s", nlname);
      return -1;
    }

      if (fseek(fp, offset, 0))
    {
      log("$Seek error on nodelist");
      (void) fclose(fp);
      return -1;
    }

      fgets(buffer, BUFSIZ, fp);
      fclose(fp);

      compile_zone = nameip->zone;
      compile_net = nameip->net;
      if (parse_entry(&entry, buffer) == NULL) return -1;

      debug(1, "Search name %s returns %d:%d/%d.%d", name,
        entry.zone, entry.net, entry.node, entry.point);
      memcpy( (char *) &originnode, (char *) &entry, sizeof(Node));
      return 0;
    }

  debug(1, "Search name %s return no node", name);
  return -1;
}

