/*
 * st -- set tape position on a multi-file dump volume to that of
 *	 the specified filesystem
 *
 * usage: st [-f [host:]/dev/rmtx] [-db] filesystem
 *
 * -f [host:]/dev/rmtx	tape device name.  If the host: prefix
 *	is used, the backup tape will be addressed over the network to
 *	the tape drive on the named host.
 *
 * -db	Write debug information to stderr
 *
 * Written by Paul Pomes, University of Illinois, Computing Services Office
 * Copyright 1985 by Paul Pomes and University of Illinois Board of Trustees
 *
 * This program may be freely reproduced and distributed provided the
 * copyright notice above is included in all copies.  It may not be
 * included in any software product or distribution sold for profit
 * without the permission of the author.
 *
 * UUCP:	{ihnp4,pur-ee,convex}!uiucdcs!uiucuxc!paul
 * ARPANET:	paul%uiucuxc@uiuc.arpa
 * CSNET:	paul%uiucuxc@uiuc.csnet
 * ICBM:	40 07 N / 88 13 W
 * US Mail:	Univ of Illinois, CSO, 1304 W Springfield Ave, Urbana, IL  61801
 *
 * $Log:	st.c,v $
 * Revision 1.2  85/07/11  13:27:51  paul
 * Updated author information.   -pbp
 * 
 * Revision 1.1  85/07/02  13:48:08  root
 * Initial revision
 * 
 */

#ifndef lint
static char	RcsId[] = "$Header: st.c,v 1.2 85/07/11 13:27:51 paul Exp $";
#endif

#include	<stdio.h>
#include	<strings.h>
#include	<sys/file.h>
#include	<sys/types.h>
#include	<sys/mtio.h>
#include	<sys/ioctl.h>
#include	<dumptab.h>

#define	equal(s1, s2)	(strcmp(s1, s2) == 0)

/* default tape devices */
#define		DEF_DEV		"/dev/rmt0"
#define		DEF_NRDEV	"/dev/nrmt0"

/* size of shorter strings, e.g., host[] */
#define		SSTR_SIZE	12

/* size of longer strings, e.g., command[] */
#define		LSTR_SIZE	160

/* commands to read the /etc/dumptab file in tape file #0.
 * argument to DD_CMD is nrdevice; arguments to RDD_CMD are host, nrdevice,
 * and host again.
 */
#define		DD_CMD		"dd if=%s of=/tmp/xyzzy.mdump bs=512"
#define		RDD_CMD		"rsh %s dd if=%s of=/tmp/xyzzy.mdump bs=512; rcp %s:/tmp/xyzzy.mdump /tmp/xyzzy.mdump"

/* copy of argv[0] for error messages */
char		*self;

/* debug messages printed if set */
int		debug;

/* mag tape stuff */
struct mt_cmds {
	char	*c_name;
	int	c_code;
	int	c_ronly;
} com[] = {
	{ "eof",	MTWEOF,	0 },
	{ "fsf",	MTFSF,	1 },
	{ "rew",	MTREW,	1 },
	{ "status",	MTNOP,	1 },
	{ 0 }
};

main(argc, argv)
int	argc;
char	**argv;
{
	/* a useful counter */
	int		i;

	/* tape device */
	char		*device, *nrdevice;

	/* filesystem to position to */
	char		*filesys;

	/* host machine */
	char		host[SSTR_SIZE];

	/* dumptab info */
	struct dumptab	*dt;

	/* library routines */
	char		*malloc();

	/* squirrel a copy of *argv[0] away for use in error messages */
	self = malloc((unsigned) (strlen(*argv) + 1));
	(void) strcpy(self, *argv);

	host[0] = '\0';
	device = DEF_DEV;
	nrdevice = DEF_NRDEV;
	/*
	 * parse arguments
	 */
	i = 1;
	while (i < argc && *argv[i] == '-') {
		if (equal(argv[i]+1, "db")) {
			/* db - set debug level */
			debug++;
			i++;
			fprintf(stderr, "%s: debug option enabled\n", self);
		}
		else if (equal(argv[i]+1, "f"))
		{
			/*
			 * f - read tape device from next argument
			 * turn /dev/rmt names into /dev/nrmt
			 */
			char	temp[40], *p1, *p2;

			i++;
			p2 = (char *) NULL;
			if ((p1 = index(argv[i], ':')) != 0) {
				(void) strncpy(host, argv[i], p1-argv[i]+1);
				*(index(host, ':')) = '\0';
				p1++;
			}
			if (p1 == 0)
				p1 = argv[i];
			device = malloc((unsigned) (strlen(p1) + 1));
			(void) strcpy(device, p1);
			if ((p2 = rindex(argv[i], '/')) == 0) {
				fprintf(stderr, "%s: Device must be a tape drive, e.g., /dev/rmt1, uxc:/dev/rmt2\n", self);
				exit(1);
			}
			p2++;
			(void) strncpy(temp, p1, p2-p1);
			*(temp + (p2-p1)) = '\0';
			nrdevice = malloc((unsigned) (strlen(argv[i])+2));
			if (*p2 == 'n')
				(void) sprintf(nrdevice, "%s%s", temp, p2);
			else
				(void) sprintf(nrdevice, "%sn%s", temp, p2);
			if (debug)
				fprintf(stderr, "host %s, device %s, nrdevice %s\n", host, device, nrdevice);
			i++;
		}
		else
		{
			/* command line errors */
			fprintf(stderr, "%s: %s - bad flag\n", self, argv[i]+1);
		    Usage:
			fprintf(stderr, "Usage: %s [-f [host:]/dev/rmtx] [-db] filesystem\n", self);
			exit(1);
		}
	}
	/* last argument is filesystem name */
	if (argv[i] == NULL) {
		fprintf(stderr, "%s: No filesystem specified\n", self);
		goto Usage;
	}
	else {
		filesys = malloc((unsigned) (strlen(argv[i]) + 1));
		(void) strcpy(filesys, argv[i]);
		if (debug)
			fprintf(stderr, "filesys %s\n", filesys);
	}
	read_dtinfo(host, device);
	setdtfile("/tmp/xyzzy.mdump");
	setdtent();
	if (dt = getdtnam(filesys)) {
		printf("Skipping %d tape files...", dt->dt_position);
		(void) fflush(stdout);
		for (i = 0; i < dt->dt_position; i++) {
			mtio(host, nrdevice, "fsf");
			printf("%d ", i+1);
			(void) fflush(stdout);
		}
		printf("done.\n");
		(void) unlink("/tmp/xyzzy.mdump");
		exit(0);
	}
	fprintf(stderr, "%s: %s not in tape file #0 (and thus not on tape)\n",
		self, filesys);
	fprintf(stderr, "\tCopy of tape file #0 in /tmp/xyzzy.mdump\n");
	exit(1);
}

/*
 * mtio -- perform a local/remote tape operation
 *
 *	for local tapes (host[0] == '\0') use the ioctl calls.
 *	remote tapes use an rsh command.
 *
 *	parameters:
 *		host (IN) -- name of remote host
 *		dev (IN) -- tape drive name
 *		cp (IN) -- tape opcode
 *	returns:
 *		none
 *	side effects:
 *		none
 *	deficiencies:
 *		any error causes an error exit from the program
 */
mtio(host, dev, cp)
char	*host, *dev, *cp;
{
	struct mt_cmds	*comp;

	/* tape file descriptor */
	int		mtfd;

	/* return code from system() call */
	int		status;

	/* from sys/mtio.h */
	struct mtop	mt_com;

	/* buffer for rsh command */
	char		command[LSTR_SIZE];

	if (*host == '\0') {
		for (comp = com; comp->c_name != NULL; comp++)
			if (strncmp(cp, comp->c_name, strlen(cp)) == 0)
				break;
		if (comp->c_name == NULL) {
			fprintf(stderr, "%s: mtio: don't grok \"%s\"\n", self, cp);
			exit(1);
		}
		if ((mtfd = open(dev, comp->c_ronly ? 0 : 2)) < 0) {
			fprintf(stderr, "%s: mtio: open of ", self);
			perror(dev);
			exit(1);
		}
		mt_com.mt_op = comp->c_code;
		mt_com.mt_count = 1;
		if (ioctl(mtfd, MTIOCTOP, &mt_com) < 0) {
			fprintf(stderr, "%s: mtio: %s %s %d ", self, dev,
				comp->c_name, mt_com.mt_count);
			perror("failed");
			exit(1);
		}
		(void) close(mtfd);
	}
	else {
		(void) strcpy(command, "rsh ");
		(void) strcat(command, host);
		(void) strcat(command, " mt -t ");
		(void) strcat(command, dev);
		(void) strcat(command, " ");
		(void) strcat(command, cp);
		if (debug)
			fprintf(stderr, "tape command %s\n", command);
		if (status = (system(command) >> 8)) {
			fprintf(stderr, "%s: %s exitted abnormally (%d).  Restart %s\n", self, command, status, self);
			exit(1);
		}
	}
}

/*
 * read_dtinfo -- copy the dumptab file contained within tape file #0
 *
 *	the /etc/dumptab file used to create the backup tape volume
 *	is contained within the first tape file.  this routine copies
 *	it to the magic name of /tmp/xyzzy.mdump on the current host.
 *
 *	parameters:
 *		host (IN) -- name of remote host
 *		dev (IN) -- tape drive name
 *	returns:
 *		none
 *	side effects:
 *		none
 *	deficiencies:
 *		any error causes an error exit from the program
 */
read_dtinfo(host, dev)
char	*host, *dev;
{
	char	command[LSTR_SIZE];
	int	status;

	mtio(host, dev, "rew");
	if (*host)
		(void) sprintf(command, RDD_CMD, host, dev, host);
	else
		(void) sprintf(command, DD_CMD, dev);
	printf("Reading /etc/dumptab info from first tape file.\n");
	if (status = (system(command) >> 8)) {
		fprintf(stderr, "%s: %s exitted abnormally (%d).  Restart %s\n", self, command, status, self);
		exit(1);
	}
}
