/*
 * makeddtape - read an unlabeled tape, and produce a shell script which
 *		will use "dd" to read the tape in.
 *
 * Usage:	makeddtape [-bblocksize] [-nnfiles] [-sskipfiles] [tape]
 *
 *	Where:
 *		blocksize	is the number of 512-byte blocks in a
 *				record (this is normally determined
 *				on the fly; use -b when blocksize is
 *				greater than 100).
 *		nfiles		is the expected number of files on the
 *				tape, or the maximum number of files to
 *				look at (default 20).
 *		skipfiles	is the number of files from the beginninf
 *				to skip over.
 *		tape		is the name of the tape drive (default
 *				/dev/rmt0).  The no-rewind on close
 *				device is computed internally.
 *
 * A shell script is produced on the standard output which will read the
 * tape and write the data into "file001", "file002", etc.
 *
 * You can edit the following variables in the shell script:
 *
 *	DDFLAGS		Options to the "dd" program, such as "conv=unblock".
 *			DO NOT set the input and output block sizes, this is
 *			done by the program.
 *	IFILE		The input file name (no-rewind-on-close tape drive).
 *	FILEROOT	The root of the file names, the default is "file".
 *	FILEOWNER	The person the file should be chowned to.
 *	FILEGROUP	The person the file should be chgrped to.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 */
#include <stdio.h>

#define BUFSIZE		51200				/* max bufsize	*/
#define min(a, b)	((a) < (b) ? (a) : (b))		/* compute min	*/

int skip = 0;				/* number of tape files to skip	*/
int nfiles = 20;			/* number of tape files to read	*/

char buf[BUFSIZE];
int bufsize = BUFSIZE;
char *tape = "/dev/rmt0";

main(argc, argv)
int argc;
char **argv;
{
	char *mkntape();
	char *pname, *ntape;
	int fd, size, count, lastsize, filenum;

	pname = *argv;

	/*
	 * Process the arguments...
	 */
	while (--argc) {
		/*
		 * Tape drive name.  We accept just a number,
		 * in which case we put it onto the end of
		 * "/dev/rmt", or we accept a full pathname.
		 */
		if (**++argv != '-') {
			if ((**argv >= '0') && (**argv <= '9'))
				tape[strlen(tape)-1] = **argv;
			else
				tape = *argv;

			continue;
		}

		/*
		 * Flags.
		 */
		switch (*++*argv) {
		case 'b':			/* block size	*/
			bufsize = atoi(++*argv);
			bufsize *= 512;
			break;
		case 'n':			/* number of files	*/
			nfiles = atoi(++*argv);
			break;
		case 's':			/* skip this many files	*/
			skip = atoi(++*argv);
			break;
		default:
			fprintf(stderr, "Usage: %s [-bblocksize] [-nnfiles] [-sskip] [tapedrive]\n", pname);
			exit(1);
		}
	}

	/*
	 * Grab the tape drive.
	 */
	if ((fd = open(tape, 0)) < 0) {
		perror(tape);
		exit(1);
	}

	/*
	 * Get the no-rewind-on-close device name.
	 */
	ntape = mkntape(tape);

	/*
	 * Starting stuff.
	 */
	printf("#!/bin/sh\n\n");
	printf("DDFLAGS=\"\"\n");
	printf("IFILE=\"%s\"\n", ntape);
	printf("FILEROOT=\"file\"\n");
	printf("FILEOWNER=\"root\"\n");
	printf("FILEGROUP=\"other\"\n\n");

	count = 0;
	filenum = 1;
	lastsize = -2;

	/*
	 * Forever...
	 */
	for (;;) {
		/*
		 * Count the number of records of this size.
		 */
		while (((size = read(fd, buf, min(bufsize, BUFSIZE))) > 0) && (size == lastsize))
			count++;

		/*
		 * End of this size record, print the count
		 * and I/O redirection.
		 */
		if (count)
			printf("count=%d >> ${FILEROOT}%03d\n", count, filenum);

		/*
		 * Depending on the size of the record...
		 */
		switch (size) {
		case -1:		/* EOT or error	*/
			fprintf(stderr, "%s: read error terminates file %d.\n", pname, filenum);
			exit(1);
		case 0:			/* EOF	*/
			/*
			 * Decide whether we're done (in which case we
			 * rewind the tape) or if we need another file
			 * (in which case we skip over the EOF mark).
			 */
			if ((filenum >= nfiles) && (nfiles > 0)) {
				printf("chown ${FILEOWNER} ${FILEROOT}%03d; chgrp ${FILEGROUP} ${FILEROOT}%03d\n", filenum, filenum);
				printf("\nmt -f ${IFILE} rew\n");
				exit(0);
			}
			else if ((count == 0) && (size == 0) && (lastsize == -2)) {
				printf("\nmt -f ${IFILE} rew\n");
				exit(0);
			}
			else {
				/*
				 * Change owner and group on the file.
				 */
				printf("chown ${FILEOWNER} ${FILEROOT}%03d; chgrp ${FILEGROUP} ${FILEROOT}%03d\n", filenum, filenum);
				printf("mt -f ${IFILE} fsf\n\n");
			}

			lastsize = -2;
			count = 0;
			filenum++;
			break;
		default:
			/*
			 * Start the command...
			 */
			printf("dd ${DDFLAGS} if=${IFILE} ibs=%d obs=%d ", size, 8*BUFSIZ);

			lastsize = size;
			count = 1;
			break;
		}
	}
}

/*
 * makentape - convert tape drive name to no-rewind-on-close name.  This
 *	       routine is naive at best; it assumes the no-rewind device
 *	       of "/dev/foo" is "/dev/nfoo".
 */
char *mkntape(t)
char *t;
{
	char *p, *q;
	char *rindex();
	static char buf[BUFSIZ];

	q = buf;

	if ((p = rindex(t, '/')) != NULL) {
		while (t <= p)
			*q++ = *t++;
		p++;
	}
	else {
		p = t;
	}

	*q++ = 'n';

	while (*p)
		*q++ = *p++;

	*q++ = NULL;
	return(buf);
}
