#! /bin/sh

# beer -- Backup files Easily, Effectively, and Reliably
#
# rewritten from the C version, must be setuid to root
#
# Arnold Robbins, 11/85
# gatech!arnold

# this program makes heavy use of the 'test' command, via its
# alias of '['. On regular BSD, this might slow it down considerably,
# however, BRL modified the shell to make 'test' and '[' built-in
# commands.

PATH=/etc/beer:/bin:/usr/bin:/usr/ucb:/usr/brl/bin:/usr/local/bin

# turn on debugging if any arguments present

if [ $# != 0 ]
then
	debug=yes
	uid
else
	debug=
fi

# get the date, and set verbose versions

set `date`
case $1 in
Mon)	weekday=monday
	yesterday=sunday
	;;
Tue)	weekday=tuesday
	yesterday=monday
	;;
Wed)	weekday=wednesday
	yesterday=tuesday
	;;
Thu)	weekday=thursday
	yesterday=wednesday
	;;
Fri)	weekday=friday
	yesterday=thursday
	;;
Sat)	weekday=saturday
	yesterday=friday
	;;
Sun)	weekday=sunday
	yesterday=saturday
	;;
esac

shortday=$1

case $2 in
Jan)	month=1 ;;
Feb)	month=2 ;;
Mar)	month=3 ;;
Apr)	month=4 ;;
May)	month=5 ;;
Jun)	month=6 ;;
Jul)	month=7 ;;
Aug)	month=8 ;;
Sep)	month=9 ;;
Oct)	month=10 ;;
Nov)	month=11 ;;
Dec)	month=12 ;;
esac

day=$3

if [ "$debug" ]
then
	echo weekday = $weekday
	echo yesterday = $yesterday
	echo shortday = $shortday
	echo month = $month
	echo day = $day
fi

nondaily=monday		# MUST be in lower case

# determine default backup type

if [ "$nondaily" = "$weekday" ]
then
	if [ "$day" -ge 1 -a "$day" -le 7 ]
	then
		backup=monthly
		prompt=m
	else
		backup=weekly
		prompt=w
	fi
else
	backup=daily
	prompt=d
fi

if [ "$debug" ]
then
	echo nondaily = $nondaily
	echo default backup = $backup
	echo prompt = $prompt
fi

while :		# infinite loop
do
	cat << EOF
Please enter type of backup.
Only the first letter is significant.
Case is ignored.
Types of backups are:
	Quarterly
	Monthly
	Weekly
	Daily
EOF
	echo -n "Choice (default: $prompt) ? "
	if read answer
	then
		case "$answer" in
		[Dd]*)	backup=daily
			break
			;;
		[Ww]*)	backup=weekly
			break
			;;
		[Mm]*)	backup=monthly
			break
			;;
		[Qq]*)	backup=quarterly
			break
			;;
		"")	break	# took the default
			;;
		*)	echo "$answer": not a valid response
			;;
		esac
	else
		exit 3	# operator typed an EOF
	fi
done

if [ "$debug" ]
then
	echo backup type is $backup
fi

operator=operator	# MUST be in lower case

if verify $operator
then
	if [ "$debug" ]
	then
		echo operator verified
	fi
else
	echo you are not $operator! >&2
	exit 1
fi

# now set up which file to compare for being newer than

case "$backup" in
daily)		if test -f /etc/beer/touch/weekly
		then
			newest=/etc/beer/touch/weekly
		elif test -f /etc/beer/touch/monthly
		then
			newest=/etc/beer/touch/monthly
		elif test -f /etc/beer/touch/quarterly
		then
			newest=/etc/beer/touch/quarterly
		else
			echo no files in /etc/beer/touch to compare with! >&2
			exit 2
		fi
		;;

weekly)		if test -f /etc/beer/touch/monthly
		then
			newest=/etc/beer/touch/monthly
		elif test -f /etc/beer/touch/quarterly
		then
			newest=/etc/beer/touch/quarterly
		else
			echo no files in /etc/beer/touch to compare with! >&2
			exit 2
		fi
		;;

monthly)	if test -f /etc/beer/touch/quarterly
		then
			newest=/etc/beer/touch/quarterly
		else
			echo no files in /etc/beer/touch to compare with! >&2
			exit 2
		fi
		;;
esac

if [ "$debug" ]
then
	echo newest "(file to compare against)" is $newest
fi

# Now find out, for the given backup type, how many tapesets there are,
# and which one to use next. First, fix the file to all lower case.

tr A-Z a-z < /etc/beer/tapesets | sed 's/#.*//' > /tmp/$$.tapesets
	
# now save all the info

set `grep "^daily" /tmp/$$.tapesets`
dailyamt=$2
dailywhich=$3

set `grep "^weekly" /tmp/$$.tapesets`
weeklyamt=$2
weeklywhich=$3

set `grep "^monthly" /tmp/$$.tapesets`
monthlyamt=$2
monthlywhich=$3

if [ "$backup" != quarterly ]
then
	# doing an incremental backup

	# get which tapeset to use now, and save the info
	
	set `grep "^$backup" /tmp/$$.tapesets`
	
	howmany=$2	# number of tapes
	whichtouse=$3	# which set to use now
	
	if [ "$debug" ]
	then
		echo howmany = $howmany
		echo which to use = $whichtouse
	fi
	
	# first, decide if necessary to rewrite the file
	# we rewrite the file for weekly and monthly backups.
	# if a daily backup, rewrite the file only if weeklies were
	# done yesterday.
	
	changesets=yes
	if [ "$backup" = daily -a "$yesterday" != "$nondaily" ]
	then
		changesets=
	fi
	
	if [ "$changesets" ]
	then
		if [ "$debug" ]
		then
			echo changing tape sets
			cat /etc/beer/tapesets
		fi
	
		# redo the file, if necessary
	
		nextup=`expr $whichtouse + 1`	# only call expr once

		if [ "$nextup" -gt "$howmany" ]
		then
			newwhich=1
		else
			newwhich=$nextup
		fi
	
		# save the values, so they can be restored for later use
		olddaily=dailywhich
		oldweekly=weeklywhich
		oldmonthly=monthlywhich

		case "$backup" in
		daily)		dailywhich=$newwhich ;;
		weekly)		weeklywhich=$newwhich ;;
		monthly)	monthlywhich=$newwhich ;;
		esac
	
		# rewrite the file (kerblam!)
		cat << EOF > /etc/beer/tapesets
# /etc/beer/tapesets
#
# This file describes how many sets of tapes are available for a
# particular type of backup, and which set to use the next time
# beer is called. While it may edited, beer will recreate it
# each time it is run, automatically updating the numbers.
#
# type		how many	which to use next
#
monthly		$monthlyamt		$monthlywhich

weekly		$weeklyamt		$weeklywhich

daily		$dailyamt		$dailywhich
EOF
		if [ "$debug" ]
		then
			cat /etc/beer/tapesets
		fi

		# restore the values, to use later when removing
		# unneeded log files

		dailywhich=olddaily
		weeklywhich=oldweekly
		monthlywhich=oldmonthly
	fi

	# now set up the logfile name

	dir=/etc/beer/log
	case "$backup" in
	daily)		logfile=$dir/$backup.$whichtouse.$month-$day-$shortday ;;
	weekly|monthly)	logfile=$dir/$backup.$whichtouse.$month-$day ;;
	esac

	if [ "$debug" ]
	then
		echo logfile is $logfile
	fi
fi

# now determine which files to backup

case "$backup" in
daily|weekly)	special="/etc /dev /lib /bin"
		regular="/usr/local /usr/spool/mail /u"
		;;
monthly)	special="/etc /dev /lib /bin"
		regular="/usr /u"
		;;
quarterly)	all="/ /usr /usr/man /usr/doc /usr/spool /usr/ingres /u /u/ai /u/staff /u/vlsi /u/graphics /u/graphics/pix /usr/src/csnet"
		;;
esac

if [ "$backup" != quarterly ]
then
	if [ "$debug" ]
	then
		echo finding files
	fi

	echo -n Finding files ....

	find $special -newer $newest -print > /tmp/$$.list
	find $regular -type f -newer $newest -print | grep -v '\.o$' |
		grep -v '/,[^/]*$' | grep -v '/core$' >> /tmp/$$.list

	echo done. Ready to dump to tape.

	if [ "$debug" ]
	then
		echo found `wc -l /tmp/$$.list` files to be dumped
	fi

	bruopts="-cavv"

	echo -n "Enter device (default: /dev/rmt8) "
	if read dev
	then
		if [ "$dev" ]
		then
			bruopts="$bruopts -f $dev"
			echo -n "Enter new block size (default: 20K) "
			if read blocksize
			then
				if [ "$blocksize" ]
				then
					bruopts="$bruopts -b $blocksize"
				fi
			else
				exit 3
			fi
		fi
	else
		exit 3
	fi

	# now put a label in
	bruopts="$bruopts -L '$backup backup - `date`'"

	if [ "$debug" ]
	then
		echo bruopts = $bruopts
	fi

	case "$backup" in
	daily)	echo -n Put "${weekday}'s" tape from daily tape set
		echo `roman $whichtouse` on the drive.
		;;
	weekly)	echo Put tape 1 of $backup tape set `roman $whichtouse`
		echo on the drive.
		;;
	monthly) echo Put tape 1 of $backup tape set `roman $whichtouse`
		echo on the drive.
		;;
	esac
	echo Mount additional tapes when prompted by bru.
	echo -n "hit return when ready (EOF to exit): "

	if read junk
	then
		# use eval to get quoting etc. to work
		eval "bru $bruopts - < /tmp/$$.list > $logfile"
	else
		exit 3
	fi
else
	# quarterly backups

	for fs in $all
	do
		bruopts="-cm"	# don't use -a for quarterlies

		echo -n Current file system is $fs. Skip? "(n) "
		if read ans
		then
			case "$ans" in
			[Yy]*)	continue ;;
			esac
		else
			exit 3
		fi

		echo -n "Enter device (default: /dev/rmt8) "
		if read dev
		then
			if [ "$dev" ]
			then
				bruopts="$bruopts -f $dev"
				echo -n "Enter new block size (default: 20K) "
				if read blocksize
				then
					if [ "$blocksize" ]
					then
						bruopts="$bruopts -b $blocksize"
					fi
				else
					exit 3
				fi
			fi
		else
			exit 3
		fi

		echo -n Send output to printer? "(y) "
		if read ans
		then
			case "$ans" in
			[Nn]*)	output=
				;;
			*)	output=" | pr -h 'quarterly dump of $fs - `date`' | lpr"
				bruopts="-vv $bruopts"
				;;
			esac
		else
			exit 3
		fi

		# now put a label in
		bruopts="$bruopts -L 'quarterly backup - $fs - `date`'"

		if [ "$debug" ]
		then
			echo bruopts = $bruopts
			echo backing up $fs
			echo output = "$output"
		fi

		cd $fs

		echo Put the first tape for $fs on the drive.
		echo Mount additional tapes when prompted by bru.
		echo -n hit return when ready "(EOF to exit): "
		if read junk
		then
			# we must use eval, to get possible pipes to work right
			eval "bru $bruopts . $output"
		else
			exit 3
		fi
	done
fi

# now update the touch files

case "$backup" in
daily)	;;
*)	echo /etc/beer/touch/$backup last updated on `date` > /etc/beer/touch/$backup
	;;
esac

# then delete the old touch files that aren't needed anymore
case "$backup" in
quarterly)	rm -f /etc/beer/touch/monthly /etc/beer/touch/weekly
		;;
monthly)	rm -f /etc/beer/touch/weekly
		;;
esac

# delete all the appropriate log files

case "$backup" in
quarterly)	# all daily and weekly logs, all monthly but last
		rm -f /etc/beer/log/daily* /etc/beer/weekly*
		case "$monthlywhich" in
		1)	save=$monthlyamt ;;
		*)	save=`expr $monthlywhich - 1` ;;
		esac
		for i in `generate 1 $monthlyamt`
		do
			if [ $i -ne $save ]
			then
				if [ "$debug" ]
				then
					echo removing /etc/beer/log/monthly.${i}*
				fi
				rm -f /etc/beer/log/monthly.${i}*
			fi
		done
		;;

monthly)	# all daily logs, all weekly but last
		rm -f /etc/beer/log/daily*
		case "$weeklywhich" in
		1)	save=$weeklyamt ;;
		*)	save=`expr $weeklywhich - 1` ;;
		esac
		for i in `generate 1 $weeklyamt`
		do
			if [ $i -ne $save ]
			then
				if [ "$debug" ]
				then
					echo removing /etc/beer/log/weekly.${i}*
				fi
				rm -f /etc/beer/log/weekly.${i}*
			fi
		done
		;;

weekly)		# all daily logs
		rm -f /etc/beer/log/daily*
		;;
esac

rm -f /tmp/$$.tapesets /tmp/$$.list # /tmp/$$.beercan

exit 0
