#ifndef lint
static char *RCSid = "$Header: /ecn1/src/ecn/backup/RCS/rundump.c,v 1.8 88/04/21 13:25:27 davy Exp $";
#endif

/*
 * rundump.c - run the dump program
 *
 * David A. Curry
 * Purdue Engineering Computer Network
 * November, 1985
 *
 * $Log:	rundump.c,v $
 * Revision 1.8  88/04/21  13:25:27  davy
 * Fixed so the status union is shared between chld_handler and rundump.c.
 * This fix only affects Sun workstations.
 * 
 * Revision 1.7  87/04/05  21:35:54  davy
 * Added message for stupid Sun's rdump.  Why do these assholes insist
 * on changing messages?
 * 
 * Revision 1.6  87/03/23  15:12:43  davy
 * More fixes for Sun's broken ptys.
 * 
 * Revision 1.5  87/03/20  13:08:34  davy
 * Modified to run under Sun/OS 3.0.
 * 
 * Revision 1.4  86/01/07  09:24:17  root
 * Fixed the "yes or no" garbage which broke the inode message
 * matching code.  This happened during the last fix (sigh).
 * 
 * Revision 1.3  86/01/04  04:39:22  root
 * Fixed problem with NEEDS ATTENTION messages from dump.  Even though dump
 * gets stopped, he has an alarm scheduled.  When he gets started after a
 * long wait, he gets the alarm, and reprints the message.  This would cause
 * backup to take action again, even though it's really the "same" message.
 * The big symptom here was on a multi-tape dump, you'd mount the second tape
 * and backup would immediately rewind it.
 * 
 * Revision 1.2  86/01/04  01:53:28  root
 * Fixed bug in inode-continuation printf, also raised sleep time in
 * stop/start of dump from 1 second to 3.
 * 
 * Revision 1.1  86/01/04  01:50:57  root
 * Initial revision
 * 
 */
#ifdef sun
#include <sys/param.h>
#include <sys/time.h>
#include <sys/vnode.h>
#include <ufs/inode.h>
#include <sys/file.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <ufs/fs.h>
#include <dumprestor.h>
#include <signal.h>
#include <stdio.h>
#else
#include <sys/param.h>
#include <sys/inode.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <sys/fs.h>
#include <protocols/dumprestore.h>
#include <signal.h>
#include <stdio.h>
#endif

#include "backup.h"
#include "externs.h"

#define match(a, b)	(strncmp((a), (b), strlen((b))) == 0)

FILE *ifp, *ofp;		/* pty input and output			*/
union wait status;		/* return status from dump processes	*/

static int estblocks;		/* estimated blocks to be dumped	*/
static float esttapes;		/* estimated number of tapes we'll use	*/
static int trueblocks;		/* true number of blocks dumped		*/
static short firstreel;		/* 1 if this is first reel of dump	*/

/*
 * dump_disk - figure out the appropriate tape, etc. and run the dump.
 */
dump_disk(b)
struct backup *b;
{
	register int density, tenthsperirg;

	/*
	 * If we need a new tape, get one.  Otherwise, we just
	 * increment the file number.
	 */
	if (Tapemounted == STOP)
		get_new_tape(b);
	else
		Tapefile++;

	/*
	 * Mark down where in the dump this file system starts.
	 */
	b->d_sreel = Tapereel;

	/*
	 * If we're dumping many things to the same tape,
	 * we need to see if there is room for this dump
	 * on the remainder of the tape.
	 */
	if (ison(Sametapes)) {
		/*
		 * Estimate the number of blocks to be dumped.  This
		 * estimate will be somewhat high (dump's is somewhat
		 * low), but we don't really care.
		 */
		estimate(b, &estblocks, &esttapes);

#ifdef DEBUG
		debug("estimated %ld blocks on %.2f tapes.\n", estblocks, esttapes, 0);
		debug("Tapefile: %d Tapeleft: %d Blocksleft: %d\n", Tapefile, Tapeleft, Blocksleft, 0);
#endif

		/*
		 * If there's not enough room on the tape, and
		 * this isn't a new tape, get a new one.  We
		 * don't split dumps across tapes unless they
		 * start on the first file of the first tape.
		 * This is because the error recovery would be
		 * a major pain if we did.
		 */
		if ((estblocks > Blocksleft) && (Tapefile > Firstdumprec)) {
			if (ison(Catalog)) {
				tape_rewind();
				catalog();
			}

			get_new_tape(b);
		}
	}

	fprintf(logfp, "%-15s level %c dump begins on reel %d, file %d.\n", b->d_device, b->d_levels[Weekday], Tapereel, Tapefile);

	/*
	 * Really run dump.
	 */
	if (exec_dump(b) == STOP)
		return(STOP);

	/*
	 * If we're not putting more than one dump on a tape,
	 * then take the tape drive off line.  It was already
	 * rewound by dump.
	 */
	if (!ison(Sametapes)) {
		if (ison(Catalog))
			catalog();

		tape_offline();
	}

	/*
	 * Calculate how many blocks and how much tape
	 * we have left.
	 */
	density = atoi(Density);
	tenthsperirg = density == 6250 ? 3 : 7;

	trueblocks = roundup(trueblocks, 10);
	trueblocks += (((trueblocks * tenthsperirg) / 10) * density) / (TP_BSIZE * atoi(Blocksize));

	Blocksleft -= trueblocks;
	Tapeleft -= (trueblocks * TP_BSIZE) / (12 * density);

	return(GO);
}

/*
 * exec_dump - execute the dump program.
 */
exec_dump(b)
struct backup *b;
{
	char c;
	register int i;
	char buf[1024];

	/*
	 * Construct dump command.
	 */
	if (ison(Sametapes))
		(void) sprintf(buf, "%s %c%s%c %s %s %d %s %s", Dumpcommand, b->d_levels[Weekday],
			DUMPFLAGS, (ison(Update) ? DUMPUPDATEFLAG : ' '), Norewtape,
			Density, Tapeleft, Blocksize, b->d_device);
	else
		(void) sprintf(buf, "%s %c%s%c %s %s %d %s %s", Dumpcommand, b->d_levels[Weekday],
			DUMPFLAGS, (ison(Update) ? DUMPUPDATEFLAG : ' '), Rewtape,
			Density, Tapeleft, Blocksize, b->d_device);

	/*
	 * Tell them what's going on.
	 */
	message("running %s\n", buf, 0);

	/*
	 * Open a pty and run the command.
	 */
	ptyopen(buf, &ifp, &ofp);

	if ((ifp == NULL) || (ofp == NULL))
		fatal("cannot start up %s.\n", Dumpcommand, 0);

	/*
	 * Read from the command...
	 */
	while (fscanf(ofp, "%[^\n?]%c", buf, &c) != EOF) {
		i = strlen(buf);
		buf[i++] = c;
		buf[i] = NULL;

		/*
		 * Handle the line.
		 */
		if (handle(b, buf, ifp, ofp) == STOP)
			return(STOP);

		buf[0] = NULL;

		/*
		 * See if it's done.
		 */
		if (wait3(&status, WNOHANG, (struct rusage *) 0) == Processid)
			break;
	}

	(void) fclose(ifp);
	(void) fclose(ofp);

	/*
	 * Did the program exit okay?
	 */
	if (WIFEXITED(status)) {
		if (status.w_retcode != 0)
			message("WARNING: %s exited abnormally.\n", Dumpcommand, 0);

		Processid = -1;
		return(GO);
	}

	return(STOP);
}

/*
 * handle - deal with a line from dump.
 */
handle(b, buf, ifp, ofp)
struct backup *b;
FILE *ifp, *ofp;
char *buf;
{
	int n, m;
	static int last_was_retry = 0;
	static int last_was_restart = 0;
	static int last_was_chg_tape = 0;
	char *yesorno = ": (\"yes\" or \"no\")";

loop:
	while ((*buf == ' ') || (*buf == '\t') || (*buf == '\n'))
		buf++;

	/*
	 * Blank line.
	 */
	if (*buf == NULL)
		return(GO);

	/*
	 * Eat the "yes or no" message which comes on NEEDS ATTENTION
	 * lines.  This is ignored in the match, and gets a newline
	 * added after timeout, etc. and gets real ugly.  See dump's
	 * dumpoptr.c.
	 */
	if (match(buf, yesorno)) {
		buf += strlen(yesorno);
		goto loop;
	}

	/*
	 * The message immediately preceding this question is
	 * "unrecoverable error", so we don't want to continue.
	 * Beats me why they even ask.
	 */
	if (match(buf, "DUMP: NEEDS ATTENTION: Do you want to attempt to continue?")) {
		message("dump had an unrecoverable error; aborting the dump.\n", 0);
		fprintf(logfp, "\n\t================ DUMP ABORTED ================\n");
		answer_dump("no\n", ifp, ofp);
		last_was_chg_tape = 0;
		last_was_restart = 0;
		last_was_retry = 0;
		return(STOP);
	}

	/*
	 * This question is asked when dump got a write error on the
	 * tape.  It gives the operator a chance to switch tapes.
	 */
	if (match(buf, "DUMP: NEEDS ATTENTION: Do you want to restart?")) {
		if (last_was_restart)
			return(GO);

		last_was_retry = 0;
		last_was_restart = 1;
		last_was_chg_tape = 0;

		stop_dump();

		message("the tape you have mounted is no good for some reason.\n", 0);
		message("you need to get a new tape to replace the bad one.\n", 0);

		Tapereel--;
		get_new_tape(b);
		fprintf(logfp, "\n\t================ REEL %d RESTARTED ================\n", Tapereel);

		/*
		 * If this is the first reel of this dump, and
		 * we're on the same tape as another dump, we
		 * have to re-run that dump too, so bag the
		 * dump right now.
		 */
		if ((firstreel == 1) && ison(Sametapes) && (Tapefile > Firstdumprec)) {
			answer_dump("no\n", ifp, ofp);
			return(STOP);
		}

		answer_dump("yes\n", ifp, ofp);
		return(GO);
	}

	/*
	 * This is asked when dump is ready for the next tape.
	 */
	if (match(buf, "DUMP: NEEDS ATTENTION: Is the new tape mounted and ready to go?") ||
	    match(buf, "DUMP: NEEDS ATTENTION: Next tape ready?")) {
		if (last_was_chg_tape)
			return(GO);

		last_was_retry = 0;
		last_was_restart = 0;
		last_was_chg_tape = 1;

		stop_dump();

		if (ison(Catalog)) {
			tape_rewind();
			catalog();
		}

		get_new_tape(b);
		answer_dump("yes\n", ifp, ofp);

		/*
		 * This message gets continued when we see the inode
		 * line, below.
		 */
		fprintf(logfp, "\t\t    continues onto reel %d ", Tapereel);
		return(GO);
	}

	/*
	 * This one comes if the open on the tape drive fails.
	 */
	if (match(buf, "DUMP: NEEDS ATTENTION: Cannot open tape.  Do you want to retry the open?")) {
		if (last_was_retry)
			return(GO);

		last_was_retry = 1;
		last_was_restart = 0;
		last_was_chg_tape = 0;

		stop_dump();

		message("tape drive is off line.\n", 0);
		message("put it on line and then come back.\n", 0);

		while (query("is the tape drive now on line?", 0) == STOP)
			abort_backup();

		answer_dump("yes\n", ifp, ofp);
		return(GO);
	}

	/*
	 * We know it wasn't one of these now.
	 */
	last_was_retry = 0;
	last_was_restart = 0;
	last_was_chg_tape = 0;

	/*
	 * This is the final message from dump before "DUMP IS DONE",
	 * etc.  Yes, the "DUMP:" really is there twice.  We want the
	 * number of blocks.
	 */
	if (sscanf(buf, "DUMP: DUMP: %d tape blocks", &n) == 1)
		trueblocks = n;

	/*
	 * If we get the inode number message, then we're on a multi-tape
	 * dump.  We want to save this information in the log file.
	 */
	if (sscanf(buf, "DUMP: Tape %d begins with blocks from ino %d", &n, &m) == 2)
		fprintf(logfp, "with blocks from inode %d.\n", m);

	/*
	 * We want to swallow these messages, since we handled them
	 * differently above.
	 */
	if (match(buf, "DUMP: This tape will rewind.  After it is rewound,") ||
	    match(buf, "DUMP: replace the faulty tape with a new one;") ||
	    match(buf, "DUMP: this dump volume will be rewritten.") ||
	    match(buf, "DUMP: Tape rewinding") ||
	    match(buf, "DUMP: Tape write error on ") ||
	    match(buf, "DUMP: The ENTIRE dump is aborted."))
	    	return(GO);

	(void) printf("    %s", buf);
	return(GO);
}

/*
 * stop_dump - make dump stop, so we can read from the terminal without
 *	       him harping at us.
 */
stop_dump()
{
	if (Processid != -1) {
		(void) killpg(Processid, SIGSTOP);
		(void) sleep(1);
	}
}

/*
 * start_dump - make dump start up again, so he can go about his business.
 */
start_dump()
{
	if (Processid != -1) {
		(void) killpg(Processid, SIGCONT);
		(void) sleep(1);
	}
}

/*
 * answer_dump - give dump an answer.
 */
answer_dump(s, ifp, ofp)
char *s;
FILE *ifp, ofp;
{
	(void) fseek(ofp, 0L, L_XTND);
	start_dump();

	(void) fwrite(s, 1, strlen(s), ifp);
	(void) fflush(ifp);
}
