#ifndef lint
static char *RCSid = "$Header: /ecn1/src/ecn/backup/RCS/controlfile.c,v 1.4 87/09/22 09:58:04 davy Exp $";
#endif

/*
 * controlfile.c - routines to read the control file, handle variables
 *
 * David A. Curry
 * Purdue Engineering Computer Network
 * November, 1985
 *
 * $Log:	controlfile.c,v $
 * Revision 1.4  87/09/22  09:58:04  davy
 * Installed a new variable "nodumpmsgs" to turn off "no dump today"
 * messages.
 * 
 * Revision 1.3  87/03/23  15:08:12  davy
 * Removed restrictions on labels and verification when remote dump
 * is on.
 * 
 * Revision 1.2  87/03/23  08:49:35  davy
 * Changed so that Weeknum is now the week of the year mod 13.
 * 
 * Revision 1.1  87/03/23  08:36:35  davy
 * Initial revision
 * 
 */
#include <strings.h>
#include <ctype.h>
#include <stdio.h>

#include "backup.h"
#include "externs.h"
#include "defaults.h"

/*
 * read_control_file - read in the control file and build the
 *		       backup list
 */
read_control_file(backup_list)
struct backup **backup_list;
{
	FILE *control;
	char line[BUFSIZ];
	register int lineno;
	struct backup *malloc();
	register struct backup *b, *lb;

	if ((control = fopen(Controlfile, "r")) == NULL)
		fatal("cannot open control file \"%s\".\n", Controlfile, 0);

	if ((*backup_list = malloc(sizeof(struct backup))) == NULL)
		fatal("not enough memory.\n", 0);

	lineno = 0;
	b = lb = *backup_list;

	/*
	 * For each line in the file...
	 */
	while (fgets(line, sizeof(line), control) != NULL) {
		b->b_lineno = ++lineno;

		/*
		 * Skip comments and blank lines.
		 */
		if ((*line == COMMENTCHAR) || (*line == '\n'))
			continue;

		/*
		 * See what kind of command this is.
		 */
		if (sscanf(line, "newtape = %s", b->n_newreelnums) == 1)
			b->b_type = B_NEWTAPE;
		else if (sscanf(line, "%s = %s", b->v_name, b->v_value) == 2)
			b->b_type = B_VARIABLE;
		else if (sscanf(line, "%s %s %s", b->d_device, b->d_levels, b->d_weeks) == 3)
			b->b_type = B_DUMPCMD;
		else
			fatal("\"%s\": line %d: garbled command line.\n", Controlfile, lineno, 0);

		/*
		 * If it's a dump command, check some stuff and
		 * set some stuff.  We initially say everything
		 * has to be dumped; this can get changed later
		 * when we read in the dumpdates file.
		 */
		if (b->b_type == B_DUMPCMD) {
			if (!alldigitsorsdl(b->d_levels))
				fatal("\"%s\": line %d: levels field must be all digits and/or '%c'es.\n", Controlfile, b->b_lineno, SKIPDUMPLEVEL, 0);

			if (!alladorsdl(b->d_weeks))
				fatal("\"%s\": line %d: weeks field must be all digits and/or letters.\n", Controlfile, b->b_lineno, 0);

			b->d_sreel = 0;
			b->d_needed = 1;
			b->d_lastdump = 0;
		}

		/*
		 * Check the new tape arg.
		 */
		if (b->b_type == B_NEWTAPE) {
			if (!alladorsdl(b->n_newreelnums))
				fatal("\"%s\": line %d: new reel field must be all alpha-numeric or '%c'es.\n", Controlfile, b->b_lineno, SKIPDUMPLEVEL, 0);
		}

		if ((b->b_next = malloc(sizeof(struct backup))) == NULL)
			fatal("not enough memory.\n", 0);

		lb = b;
		b = b->b_next;
	}

	free(b);
	lb->b_next = NULL;

	(void) fclose(control);

	/*
	 * Check the variables for correctness.
	 */
	check_vars();
}

/*
 * set_variable - set variable to a new value.
 */
set_variable(b)
register struct backup *b;
{
	register char *s;

	for (s = b->v_name; *s; s++) {
		if (isupper(*s))
			*s = tolower(*s);
	}

	if (!strcmp(b->v_name, "label"))
		(void) strcpy(Label, b->v_value);
	else if (!strcmp(b->v_name, "update"))
		(void) strcpy(Update, b->v_value);
	else if (!strcmp(b->v_name, "verify"))
		(void) strcpy(Verify, b->v_value);
	else if (!strcmp(b->v_name, "density"))
		(void) strcpy(Density, b->v_value);
	else if (!strcmp(b->v_name, "catalog"))
		(void) strcpy(Catalog, b->v_value);
	else if (!strcmp(b->v_name, "hostname"))
		(void) strcpy(Hostname, b->v_value);
	else if (!strcmp(b->v_name, "blocksize"))
		(void) strcpy(Blocksize, b->v_value);
	else if (!strcmp(b->v_name, "sametapes"))
		(void) strcpy(Sametapes, b->v_value);
	else if (!strcmp(b->v_name, "tapelength"))
		(void) strcpy(Tapelength, b->v_value);
	else if (!strcmp(b->v_name, "remotedump"))
		(void) strcpy(Remotedump, b->v_value);
	else if (!strcmp(b->v_name, "nodumpmsgs"))
		(void) strcpy(Nodumpmsgs, b->v_value);
	else if (!strcmp(b->v_name, "rewtape"))
		(void) strcpy(Rewtape, b->v_value);
	else if (!strcmp(b->v_name, "norewtape"))
		(void) strcpy(Norewtape, b->v_value);
	else if (!strcmp(b->v_name, "dumpcommand"))
		(void) strcpy(Dumpcommand, b->v_value);
	else if (!strcmp(b->v_name, "oninterrupt"))
		(void) strcpy(Oninterrupt, b->v_value);
	else
		fatal("\"%s\": line %d: unknown variable name: \"%s\".\n", Controlfile, b->b_lineno, b->v_name, 0);
}

/*
 * set_defaults - set default values of variables.
 */
set_defaults()
{
	(void) strcpy(Label, D_LABEL);
	(void) strcpy(Update, D_UPDATE);
	(void) strcpy(Verify, D_VERIFY);
	(void) strcpy(Density, D_DENSITY);
	(void) strcpy(Rewtape, D_REWTAPE);
	(void) strcpy(Catalog, D_CATALOG);
	(void) strcpy(Hostname, D_HOSTNAME);
	(void) strcpy(Blocksize, D_BLOCKSIZE);
	(void) strcpy(Norewtape, D_NOREWTAPE);
	(void) strcpy(Sametapes, D_SAMETAPES);
	(void) strcpy(Tapelength, D_TAPELENGTH);
	(void) strcpy(Remotedump, D_REMOTEDUMP);
	(void) strcpy(Nodumpmsgs, D_NODUMPMSGS);
	(void) strcpy(Dumpcommand, D_DUMPCOMMAND);
	(void) strcpy(Oninterrupt, D_ONINTERRUPT);
	(void) strcpy(Controlfile, D_CONTROLFILE);

	if (gethostname(Hostname, 16) < 0)
		fatal("who am i?", 0);
}

/*
 * check_vars - some variables should only have certain values.
 */
check_vars()
{
	if ((strcmp(Label, "on") != 0) && (strcmp(Label, "off") != 0))
		fatal("\"%s\": \"label\" may only have the values \"on\" and \"off\".\n", Controlfile, 0);
	if ((strcmp(Update, "on") != 0) && (strcmp(Update, "off") != 0))
		fatal("\"%s\": \"update\" may only have the values \"on\" and \"off\".\n", Controlfile, 0);
	if ((strcmp(Verify, "on") != 0) && (strcmp(Verify, "off") != 0))
		fatal("\"%s\": \"verify\" may only have the values \"on\" and \"off\".\n", Controlfile, 0);
	if ((strcmp(Catalog, "on") != 0) && (strcmp(Catalog, "off") != 0))
		fatal("\"%s\": \"catalog\" may only have the values \"on\" and \"off\".\n", Controlfile, 0);
	if ((strcmp(Sametapes, "on") != 0) && (strcmp(Sametapes, "off") != 0))
		fatal("\"%s\": \"sametapes\" may only have the values \"on\" and \"off\".\n", Controlfile, 0);
	if ((strcmp(Remotedump, "on") != 0) && (strcmp(Remotedump, "off") != 0))
		fatal("\"%s\": \"remotedump\" may only have the values \"on\" and \"off\".\n", Controlfile, 0);
	if ((strcmp(Nodumpmsgs, "on") != 0) && (strcmp(Nodumpmsgs, "off") != 0))
		fatal("\"%s\": \"nodumpmsgs\" may only have the values \"on\" and \"off\".\n", Controlfile, 0);
	if ((strcmp(Oninterrupt, "handle") != 0) && (strcmp(Oninterrupt, "default") != 0))
		fatal("\"%s\": \"oninterrupt\" may only have the values \"handle\" and \"default\".\n", Controlfile, 0);
}

/*
 * alldigits - returns 1 if string is all digits.
 */
alldigits(s)
register char *s;
{
	while (*s) {
		if (!isdigit(*s))
			return(0);
		s++;
	}

	return(1);
}

/*
 * alldigitsorsdl - returns 1 if string is all digits or SKIPDUMPLEVEL
 *		    characters.
 */
alldigitsorsdl(s)
register char *s;
{
	while (*s) {
		if (!isdigit(*s) && (*s != SKIPDUMPLEVEL))
			return(0);
		s++;
	}

	return(1);
}

/*
 * alladorsdl - all alpha, digits, or SKIPDUMPLEVEL chars.
 */
alladorsdl(s)
register char *s;
{
	while (*s) {
		if (!isdigit(*s) && !isalpha(*s) && (*s != SKIPDUMPLEVEL))
			return(0);
		s++;
	}

	return(1);
}
