#ifndef lint
static char *RCSid = "$Header: /ecn1/src/ecn/backup/RCS/backup.c,v 1.13 88/04/21 13:24:14 davy Exp $";
#endif

/*
 * backup.c - main driving point for the backup program
 *
 * David A. Curry
 * Purdue Engineering Computer Network
 * November, 1985
 *
 * $Log:	backup.c,v $
 * Revision 1.13  88/04/21  13:24:14  davy
 * Fixed so the status union is shared between chld_handler and rundump.c.
 * This fix only affects Sun workstations.
 * 
 * Revision 1.12  87/09/22  09:57:13  davy
 * Installed a new variable "nodumpmsgs" to turn off "no dump today"
 * messages.
 * 
 * Revision 1.11  87/09/19  09:27:48  davy
 * Ooops.  Print chars with %c, not %d.
 * 
 * Revision 1.10  87/09/18  07:57:28  davy
 * Now prints level number on "no dump needed" messages.
 * 
 * Revision 1.9  87/04/06  14:32:18  davy
 * Fixed the SIGCHLD handler to only make the process look like it exited
 * if it really did exit.  Sigh, all this just because of Sun's bugs.
 * What fun.
 * 
 * Revision 1.8  87/03/23  15:06:41  davy
 * More fixes for Sun's broken ptys... the previous code was not enough.
 * 
 * Revision 1.7  87/03/23  08:49:13  davy
 * Changed so that Weeknum is now the week of the year mod 13.
 * 
 * Revision 1.6  87/03/20  13:08:05  davy
 * Modified to run under Sun/OS 3.0.
 * 
 * Revision 1.5  86/02/08  17:09:31  davy
 * Added a call to setlinebuf on logfp... found a case where you need
 * to flush buffers that often (when the system crashes).
 * 
 * Revision 1.4  86/02/03  07:17:50  davy
 * Added calls to console() to log start and end of dumps.
 * 
 * Revision 1.3  86/01/04  04:38:43  root
 * Added setpriority() call to set BACKUPPRIORITY... this is like DUMPPRIORITY
 * used to be, except dump still gets that priority.
 * 
 * Revision 1.2  86/01/04  02:53:40  root
 * Moved setpriority() call from here to pty.c, so that backup itself
 * runs at normal priority.
 * 
 * Revision 1.1  86/01/04  02:47:13  root
 * Initial revision
 * 
 * Revision 1.1  86/01/04  02:46:15  root
 * Initial revision
 * 
 */
#include <sys/param.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <strings.h>
#include <signal.h>
#include <ctype.h>
#include <stdio.h>

#include "backup.h"

FILE *logfp;			/* log file file pointer		*/
int Weekday;			/* Current day of week (Sun = 0)	*/
int Weeknum;			/* Current week of the year, mod 13	*/
int Firstdumprec;		/* number of first file of dump		*/
int Tapereel = 0;		/* Current reel of tape we're on	*/
int Tapefile = 1;		/* Current file on tape (1 is label)	*/
int Tapeleft = 0;		/* Amount of tape left on reel (feet)	*/
int Processid = -1;		/* Number of current child process	*/
int Blocksleft = 0;		/* Number of tape blocks left on reel	*/
int Tapemounted = STOP;		/* GO if tape is mounted		*/

char Label[8];			/* "on" if we should write tape labels	*/
char Update[8];			/* "on" if we should call dump with -u	*/
char Verify[8];			/* "on" if we should verify tape labels	*/
char Density[8];		/* Density we're writing at		*/
char Catalog[8];		/* "on" if we should catalog tapes	*/
char Hostname[16];		/* Hostname we're dumping		*/
char Blocksize[8];		/* Tape block size (1024-byte units)	*/
char Sametapes[8];		/* "on" if multiple dumps on one tape	*/
char Tapelength[8];		/* Length of tape (feet)		*/
char Remotedump[8];		/* "on" if this is a remote dump	*/
char Nodumpmsgs[8];		/* "on" toprint "no dump needed today"	*/
char Date[DATESIZE];		/* Current date - ctime format		*/
char Rewtape[DEVSIZE];		/* Rewind-on-close tape device		*/
char Norewtape[DEVSIZE];	/* No rewind-on-close tape device	*/
char Dumpcommand[VARSIZE];	/* Path to dump command			*/
char Oninterrupt[VARSIZE];	/* What to do on interrupt		*/
char Controlfile[VARSIZE];	/* Name of control file			*/

main(argc, argv)
int argc;
char **argv;
{
	int i;
	char *ctime();
	struct sigvec sv;
	struct timeval tv;
	struct rlimit rlim;
	register struct backup *b;
	struct backup *backup_list;
	extern int sig_handler(), stop_handler();
#ifdef sun
	extern int chld_handler();
#endif

#ifndef DEBUG
	/*
	 * Kick our priority.
	 */
	(void) setpriority(PRIO_PROCESS, getpid(), BACKUPPRIORITY);
#endif

	/*
	 * Kick our limits.
	 */
	rlim.rlim_cur = rlim.rlim_max = RLIM_INFINITY;

	(void) setrlimit(RLIMIT_CPU, &rlim);
	(void) setrlimit(RLIMIT_RSS, &rlim);
	(void) setrlimit(RLIMIT_DATA, &rlim);
	(void) setrlimit(RLIMIT_STACK, &rlim);

	/*
	 * Set the default values of the control variables.
	 */
	set_defaults();

	/*
	 * They may have given us an alternate control file
	 * on the command line.
	 */
	if (argc > 1) {
		if (argc > 2)
			fatal("usage: backup [control file]\n", 0);

		(void) strcpy(Controlfile, *++argv);
	}

	/*
	 * Read in the control file.
	 */
	read_control_file(&backup_list);

	/*
	 * Read in the dates of previous dumps.
	 */
	read_dump_dates(backup_list);

#ifdef DEBUG
	debug_dump(backup_list);
#endif

	console("dumps started.\n", 0);

	/*
	 * Set first dump record.
	 */
	if (ison(Label))
		Firstdumprec = 2;
	else
		Firstdumprec = 1;

	/*
	 * Handle interrupts, if we're supposed to.
	 */
	if (!strcmp(Oninterrupt, "handle")) {
		sv.sv_mask = 0;
#ifdef SV_ONSTACK
		sv.sv_flags = 0;
#else
		sv.sv_onstack = 0;
#endif
		sv.sv_handler = sig_handler;
		(void) sigvec(SIGINT, &sv, (struct sigvec *) 0);
		(void) sigvec(SIGHUP, &sv, (struct sigvec *) 0);
		(void) sigvec(SIGQUIT, &sv, (struct sigvec *) 0);
		(void) sigvec(SIGTERM, &sv, (struct sigvec *) 0);
	}

	/*
	 * Handle stops, regardless.
	 */
	sv.sv_mask = 0;
#ifdef SV_ONSTACK
	sv.sv_flags = 0;
#else
	sv.sv_onstack = 0;
#endif
	sv.sv_handler = stop_handler;
	(void) sigvec(SIGTSTP, &sv, (struct sigvec *) 0);

#ifdef sun
	/*
	 * Sun 3.0 ptys don't close properly, so we have to
	 * catch SIGCHLD.
	 */
	sv.sv_mask = 0;
#ifdef SV_ONSTACK
	sv.sv_flags = 0;
#else
	sv.sv_onstack = 0;
#endif
	sv.sv_handler = chld_handler;
	(void) sigvec(SIGCHLD, &sv, (struct sigvec *) 0);
#endif

	/*
	 * Start the log file of what we're doing.
	 */
	if ((logfp = fopen(INFOFILE, "a")) == NULL)
		fatal("cannot open log file \"%s\"\n", INFOFILE, 0);

	(void) setlinebuf(logfp);
	(void) gettimeofday(&tv, (struct timezone *) 0);
	(void) fprintf(logfp, "---- %.24s ------------------------------------------------\n", ctime(&tv.tv_sec));

	/*
	 * Process the lines.
	 */
	b = backup_list;

	while (b != NULL) {
		/*
		 * If it's a newtape command, take the current tape off
		 * line, and set Tapereel up.  Then the next dump command
		 * will ask for the new tape.
		 */
		if (b->b_type == B_NEWTAPE) {
			if (b->n_newreelnums[Weekday] != SKIPDUMPLEVEL) {
				i = tonum(b->n_newreelnums[Weekday]);

				if (i > Tapereel) {
					if (Tapemounted == GO) {
						tape_rewind();

						if (ison(Catalog))
							catalog();

						tape_offline();
					}

					Tapereel = i - 1;
				}
			}

			b = b->b_next;
			continue;
		}

		/*
		 * If it's a variable command, set the
		 * variable.
		 */
		if (b->b_type == B_VARIABLE) {
			set_variable(b);
			b = b->b_next;
			continue;
		}

		/*
		 * It's a dump command.
		 */

		/*
		 * If this one doesn't need dumping,
		 * skip it.
		 */
		if (b->d_needed == 0) {
			if (ison(Nodumpmsgs))
				fprintf(logfp, "%-15s no dump needed today.\n", b->d_device);
			b = b->b_next;
			continue;
		}

		/*
		 * If we get STOP from dump_disk, indicating an
		 * aborted dump,then we rewind our list until the
		 * start of this reel and go again.
		 */
		if (dump_disk(b) == STOP) {
			message("re-running all dumps that were on this tape.\n");

			for (b = backup_list; b != NULL; b = b->b_next) {
				if (b->b_type != B_DUMPCMD)
					continue;

				if (b->d_needed == 0)
					continue;

				if (b->d_sreel == Tapereel)
					break;
			}

			continue;
		}

		b = b->b_next;
	}

	/*
	 * Take off the last tape, if necessary.
	 */
	if (Tapemounted == GO) {
		tape_rewind();

		if (ison(Catalog))
			catalog();

		tape_offline();
	}

	message("all dumps completed.\n", 0);
	console("all dumps completed.\n", 0);
	exit(0);
}

/*
 * tonum - convert a letter to a number - sort of like hexadecimal,
 *	   except we go to 'z'.
 */
tonum(c)
char c;
{
	if (isdigit(c))
		return(c - '0');

	if (isalpha(c)) {
		if (isupper(c))
			c = tolower(c);

		return(c - 'a' + 10);
	}

	return(-1);
}

/*
 * sig_handler - handle signals.
 */
sig_handler(sig)
int sig;
{
	extern char *sys_siglist[];

	message("signal received (%s).\n", sys_siglist[sig], 0);
	abort_backup();
}

/*
 * stop_handler - handle stops.
 */
stop_handler()
{
	struct sigvec sv;

	/*
	 * Stop dump.
	 */
	stop_dump();

	/*
	 * Stop ourselves.
	 */
	sv.sv_mask = 0;
#ifdef SV_ONSTACK
	sv.sv_flags = 0;
#else
	sv.sv_onstack = 0;
#endif
	sv.sv_handler = SIG_DFL;
	(void) sigvec(SIGTSTP, &sv, (struct sigvec *) 0);

	(void) sigsetmask(0);
	(void) kill(0, SIGTSTP);

	/*
	 * Process stops here.  We continue
	 * with the statement below.
	 */

	/*
	 * Catch the signal again.
	 */
	sv.sv_handler = stop_handler;
	(void) sigvec(SIGTSTP, &sv, (struct sigvec *) 0);

	/*
	 * Start dump again.
	 */
	start_dump();
}

#ifdef sun
/*
 * chld_handler - handle SIGCHLD.
 */
chld_handler()
{
#include <sys/wait.h>
	register int i;
	union wait tmp;
	extern FILE *ifp, *ofp;
	extern union wait status;	/* in rundump.c */

	/*
	 * We need to find out what happened to dump.
	 */
	while ((i = wait3(&tmp, WNOHANG, (struct rusage *) 0)) != Processid)
		if (i <= 0)
			return;

	status = tmp;

	/*
	 * If dump exited, make it look that way.  We cannot blindly
	 * do this, since we get SIGCHLD from stop_dump and start_dump
	 * also.
	 */
	if (WIFEXITED(status)) {
		fclose(ifp);
		fclose(ofp);
	}

	return;
}
#endif

#ifdef DEBUG
debug_dump(backup_list)
struct backup *backup_list;
{
	char *ctime();
	register struct backup *b;

	for (b = backup_list; b != NULL; b = b->b_next) {
		if (b->b_type == B_VARIABLE)
			set_variable(b);
	}

	debug("Date = %s", Date, 0);
	debug("Weekday = %d\n", Weekday, 0);
	debug("Weeknum = %d\n", Weeknum, 0);
	debug("Density = %s\n", Density, 0);
	debug("Rewtape = %s\n", Rewtape, 0);
	debug("Catalog = %s\n", Catalog, 0);
	debug("Hostname = %s\n", Hostname, 0);
	debug("Blocksize = %s\n", Blocksize, 0);
	debug("Norewtape = %s\n", Norewtape, 0);
	debug("Sametapes = %s\n", Sametapes, 0);
	debug("Tapelength = %s\n", Tapelength, 0);
	debug("Remotedump = %s\n", Remotedump, 0);
	debug("Dumpcommand = %s\n", Dumpcommand, 0);
	debug("Oninterrupt = %s\n", Oninterrupt, 0);
	debug("Controlfile = %s\n", Controlfile, 0);
	debug("\n", 0);

	for (b = backup_list; b != NULL; b = b->b_next) {
		if (b->b_type == B_DUMPCMD)
			debug("%-12s %-8s %-8s %-8s %s", b->d_device, b->d_levels, b->d_weeks, b->d_needed ? "dump" : "no dump", ctime(&b->d_lastdump), 0);

		if (b->b_type == B_NEWTAPE)
			if (b->n_newreelnums[Weekday] != SKIPDUMPLEVEL)
				debug("---- change tapes to reel %d ----\n", tonum(b->n_newreelnums[Weekday]), 0);
	}

	debug("\n", 0);
}
#endif

