#ifndef lint
static char *RCSid = "$Header: /ecn1/src/ecn/backup/RCS/dumpdates.c,v 1.5 87/09/18 07:57:50 davy Exp $";
#endif

/*
 * dumpdates.c - read in the DUMPDATES file to find the last date
 *		 each file system was dumped.
 *
 * David A. Curry
 * Purdue Engineering Computer Network
 * November, 1985
 *
 * $Log:	dumpdates.c,v $
 * Revision 1.5  87/09/18  07:57:50  davy
 * Now prints out current date, and what week of the cycle it is.
 * 
 * Revision 1.4  87/03/23  15:10:09  davy
 * Ooops.  The call to tonum() should've been tolet()...
 * 
 * Revision 1.3  87/03/23  08:49:41  davy
 * Changed so that Weeknum is now the week of the year mod 13.
 * 
 * Revision 1.2  86/07/30  06:56:42  davy
 * Corrected the week number calculation.  Shouldn't affect too much,
 * only moves days which are multiples of 7 (the 7th was week 2, now
 * it's week 1).
 * 
 * Revision 1.1  86/07/30  06:52:35  davy
 * Initial revision
 * 
 */
#include <sys/time.h>
#include <strings.h>
#include <stdio.h>

#include "backup.h"
#include "externs.h"

/*
 * Positions of things in a ctime string.
 */
#define CT_DAY		0
#define CT_MONTH	4
#define CT_MDAY		8
#define CT_HOUR		11
#define CT_MINUTE	14
#define CT_SECOND	17
#define CT_YEAR		20

/*
 * Seconds in other units.
 */
#define DAY_SECS	86400
#define HOUR_SECS	3600
#define MIN_SECS	60

static int days[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
static char months[] = "JanFebMarAprMayJunJulAugSepOctNovDec";
static char weekdays[] = "SunMonTueWedThuFriSat";

/*
 * read_dump_dates - read the dumpdates file and find the last time
 *		     each file system was dumped.
 */
read_dump_dates(backup_list)
struct backup *backup_list;
{
	char tolet();
	char *index();
	long unctime();
	FILE *dumpdates;
	struct timeval tv;
	register long clock;
	char c, week, level;
	register struct backup *b;
	char dev[DEVSIZE], date[DATESIZE];

	/*
	 * Find out what day of the week and what week of
	 * the year this is.
	 */
	(void) gettimeofday(&tv, (struct timezone *) 0);
	(void) strcpy(Date, ctime(&tv.tv_sec));
	Weeknum = weeknum(&tv.tv_sec);
	Weekday = weekday(Date);
	week = tolet(Weeknum);

	/*
	 * Explain what week it is.
	 */
	message("date is %.6s, %.4s; week %d of the cycle.\n", Date+4, Date+20, Weeknum);

	if ((dumpdates = fopen(DUMPDATES, "r")) == NULL)
		fatal("cannot open dump dates file \"%s\".\n", DUMPDATES, 0);

	/*
	 * For each line in the file...
	 */
	while (fscanf(dumpdates, DUMPDATESFMT, dev, &level, date, &c) != EOF) {
		/*
		 * See if we can find this file system.
		 */
		for (b = backup_list; b != NULL; b = b->b_next) {
			if (b->b_type != B_DUMPCMD)
				continue;

			if (strcmp(b->d_device, dev) != 0)
				continue;

			/*
			 * Is this file system supposed to be dumped
			 * this week?
			 */
			if (index(b->d_weeks, week) == NULL) {
				b->d_needed = 0;
				continue;
			}

			/*
			 * Are we dumping anything today?
			 */
			if (b->d_levels[Weekday] == SKIPDUMPLEVEL) {
				b->d_needed = 0;
				continue;
			}

			/*
			 * See if this file system has
			 * been dumped already.
			 */
			if (!strncmp(date, Date, 11)) {
				message("\"%s\" has already been dumped today.\n", b->d_device, 0);

				if (query("do you wish to dump it again?", 0) == STOP)
					b->d_needed = 0;
			}

			clock = unctime(date);

			/*
			 * If this level is less than today's level,
			 * and the time is more recent than we've
			 * seen so far, save this time.
			 */
			if (level < b->d_levels[Weekday]) {
				if (clock > b->d_lastdump)
					b->d_lastdump = clock;
			}
		}
	}

	(void) fclose(dumpdates);

	/*
	 * Hard-set full dumps to run from the epoch.
	 */
	for (b = backup_list; b != NULL; b = b->b_next) {
		if (b->b_type != B_DUMPCMD)
			continue;

		if (b->d_levels[Weekday] == FULLDUMPLEVEL)
			b->d_lastdump = 0;
	}
}

/*
 * unctime - inverse of ctime.
 */
long unctime(date)
char *date;
{
	register int i;
	struct timeval tv;
	struct timezone tz;
	register long clock;
	int mon, day, year, hour, minute, second;

	/*
	 * Strip leading white space and trailing
	 * garbage.
	 */
	while ((*date == ' ') || (*date == '\t'))
		date++;
	date[25] = NULL;

	clock = 0;

	/*
	 * Break the date out.
	 */
	mon = month(&date[CT_MONTH]);
	(void) sscanf(&date[CT_MDAY], "%d", &day);
	(void) sscanf(&date[CT_YEAR], "%d", &year);
	(void) sscanf(&date[CT_HOUR], "%d", &hour);
	(void) sscanf(&date[CT_MINUTE], "%d", &minute);
	(void) sscanf(&date[CT_SECOND], "%d", &second);

	/*
	 * Start counting...
	 */
	for (i=1970; i < year; i++)
		clock += DAY_SECS * ydays(i);
	for (i=0; i < mon; i++)
		clock += DAY_SECS * days[i];

	/*
	 * Add a day for leap year.
	 */
	if ((ydays(year) == 366) && (mon > 1))
		clock += DAY_SECS;

	clock += DAY_SECS * (day - 1);
	clock += HOUR_SECS * hour;
	clock += MIN_SECS * minute;
	clock += second;

	/*
	 * Adjust from GMT to local time zone.
	 */
	(void) gettimeofday(&tv, &tz);
	clock += tz.tz_minuteswest * MIN_SECS;

	return(clock);
}

/*
 * ydays - return the number of days in a year.
 */
ydays(year)
register int year;
{
	if ((year % 4) == 0)
		return(366);

	return(365);
}

/*
 * month - return the number of the month (Jan = 0)
 */
month(mname)
register char *mname;
{
	register int i;
	register char *cp;

	for (i=0, cp = months; cp != NULL; i++, cp += 3) {
		if (!strncmp(mname, cp, 3))
			return(i);
	}

	/*
	 * Should never get here.
	 */
	return(0);
}

/*
 * weekday - return the number of the day (Sun = 0)
 */
weekday(dname)
register char *dname;
{
	register int i;
	register char *cp;

	for (i=0, cp = weekdays; cp != NULL; i++, cp += 3) {
		if (!strncmp(dname, cp, 3))
			return(i);
	}

	/*
	 * Should never get here.
	 */
	return(0);
}

/*
 * weeknum - return the week of the year, mod 13.
 */
weeknum(clock)
register long *clock;
{
	register int x;
	register struct tm *t;
	struct tm *localtime();

	t = localtime(clock);

	x = (t->tm_yday+ 6) / 7;	/* week of year, 1-53	*/
	x = x % 13;			/* "quarter" week	*/

	return(x == 0 ? 13 : x);
}

char tolet(digit)
register int digit;
{
	if ((digit >= 0) && (digit <= 9))
		return('0' + digit);

	return('a' + digit - 10);
}
