#ifndef lint
static char *RCSid = "$Header: /ecn1/src/ecn/backup/RCS/cleanlog.c,v 1.4 87/09/18 07:46:40 davy Exp $";
#endif

/*
 * cleanlog - clean backup's log file
 *
 * Usage: cleanlog [-s Day #] [logfile]
 *
 * where "Day" is one of "Sun", "Mon", etc. and # is the number of days
 * of this output to save.  Default is 1 day for each day.
 *
 * David A. Curry
 * January, 1985
 *
 * $Log:	cleanlog.c,v $
 * Revision 1.4  87/09/18  07:46:40  davy
 * Raised array sizes.
 * 
 * Revision 1.3  86/07/30  07:08:01  davy
 * Made entries static for stupid Gould base registers.
 * 
 * Revision 1.2  86/03/22  07:47:24  davy
 * Fixed a bug in saving several entries for a day... they'd all migrate
 * to the beginning of the file and it kept them all.  Had to lengthen
 * the strcmp fro day to day plus month plus date.
 * 
 */
#include <stdio.h>

#define LOGFILE		"/etc/backup.tapes"
#define MAXDAYS		7
#define MAXLINES	128
#define MAXENTRIES	128

/*
 * One file entry looks like this.
 */
static struct entry {
	char day[8];			/* day of the entry	*/
	char keepit;			/* 1 if keeping it	*/
	char lines[MAXLINES][128];	/* lines of the entry	*/
} entries[MAXENTRIES];

/*
 * Info about days.
 */
struct saveday {
	char day[8];			/* name of day		*/
	int keepamt;			/* no. of days to save	*/
} savedays[MAXDAYS];

int nentries = 0;
char *progname = NULL;
char *logfile = LOGFILE;
char *days = "SunMonTueWedThuFriSat";

main(argc, argv)
int argc;
char **argv;
{
	register int i;
	register char *s;

	progname = *argv;

	/*
	 * Initialize number of entries to save for each day to one.
	 */
	for (i=0, s=days; (i < MAXDAYS) && (*s != NULL); i++, s+=3) {
		strncpy(savedays[i].day, s, 3);
		savedays[i].keepamt = 1;
	}

	while (--argc) {
		if (**++argv != '-') {
			logfile = *argv;
			continue;
		}

		switch (*++*argv) {
		case 's':
			if ((--argc <= 0) || ((i = isday(*++argv)) < 0))
				usage();

			if ((--argc <= 0) || (!isnum(*++argv)))
				usage();

			savedays[i].keepamt = atoi(*argv);
			break;
		default:
			usage();
		}
	}

	loadup();

	decide();

	dumpout();

	exit(0);
}

/*
 * loadup - load in the lines of the file.
 */
loadup()
{
	FILE *fp;
	register int i;
	char buf[BUFSIZ];

	if ((fp = fopen(logfile, "r")) == NULL) {
		fprintf(stderr, "%s: cannot open %s for reading.\n", progname, logfile);
		exit(1);
	}

	if (fgets(buf, BUFSIZ, fp) == NULL)
		exit(0);

	/*
	 * Skip any crud before the first start-of-entry line.
	 */
	while (strncmp(buf, "---- ", 5) != 0) {
		if (fgets(buf, BUFSIZ, fp) == NULL)
			exit(0);
	}

	/*
	 * For each entry...
	 */
	do {
		/*
		 * Grab the day.
		 */
		sscanf(buf, "---- %s ", entries[nentries].day);

		i = 0;

		/*
		 * Read in lines until we hit end of file or another
		 * start-of-entry line.
		 */
		do {
			strcpy(entries[nentries].lines[i++], buf);

			if (fgets(buf, BUFSIZ, fp) == NULL)
				break;
		} while (strncmp(buf, "---- ", 5) != 0);

		entries[nentries].lines[i][0] = NULL;
		nentries++;
	} while (!feof(fp));

	/*
	 * No entries.
	 */
	if ((nentries == 0) && (i == 1))
		exit(0);

	fclose(fp);
}

/*
 * decide - decide which ones to keep.
 */
decide()
{
	register int i, j;

	/*
	 * For each day...
	 */
	for (i=0; i < MAXDAYS; i++) {
		/*
		 * Move backwards through entries, marking N of this
		 * day.
		 */
		for (j=nentries-1; j >= 0; j--) {
			/*
			 * Not the right day.
			 */
			if (strcmp(entries[j].day, savedays[i].day) != 0)
				continue;

			/*
			 * Don't need to save any more.
			 */
			if (savedays[i].keepamt <= 0) 
				continue;

			/*
			 * Decrement the number to keep, mark it as keepable.
			 */
			savedays[i].keepamt--;
			entries[j].keepit = 1;

			/*
			 * Mark the rest of the entries for today too;
			 * so they don't count as other days.
			 */
			while ((j > 0) && (strcmp(entries[j-1].day, savedays[i].day) == 0) &&
			       (strncmp(entries[j-1].lines[0], entries[j].lines[0], 16) == 0))
				entries[--j].keepit = 1;
		}
	}
}

/*
 * dumpout - dump out the ones we want.
 */
dumpout()
{
	FILE *fp;
	register int i, j;

	if ((fp = fopen(logfile, "w")) == NULL) {
		fprintf(stderr, "%s: cannot create %s.\n", progname, logfile);
		exit(1);
	}

	for (i=0; i < nentries; i++) {
		if (entries[i].keepit) {
			for (j=0; entries[i].lines[j][0] != NULL; j++)
				fprintf(fp, "%s", entries[i].lines[j]);
		}
	}

	fclose(fp);
}

isday(s)
char *s;
{
	register char *t;

	t = days;

	while (*t) {
		if (!strncmp(s, t, 3))
			return((t - days) / 3);
		t += 3;
	}

	return(-1);
}

isnum(s)
char *s;
{
	while (*s) {
		if ((*s < '0') || (*s > '9'))
			return(0);
		s++;
	}

	return(1);
}

usage()
{
	fprintf(stderr, "Usage: %s [-s Day #] [logfile]\n", progname);
	exit(1);
}
