static char *RCSid = "$Header: /ecn1/src/ecn/ansitape/RCS/ansitape.c,v 1.4 88/05/04 11:06:46 davy Exp $";

/*
 * ansitape - read and write ANSI label tapes
 *
 * This program is essentially a souped-up version of "ansitar", whose
 * original author is unknown.  "Ansitar" was modified and extended by
 * David R. Brown and Pierre A. MacKay.  Almost all the code which actually
 * implements the ANSI standard (reading and writing the headers, etc.)
 * was taken verbatim from "ansitar".
 *
 * The main extensions to "ansitar" contained in this program are:
 *
 *	-	directories are followed recursively, like the
 *		real "tar" does.
 *	-	unique filenames on the tape are guaranteed (only
 *		within the same writing session).
 *	-	an interactive mode was added
 *	-	writing the tape to a file was added - note that
 *		reading from the file doesn't really work.
 *
 * The meanings of the flags are:
 *	-	- ignored - for dummies who can't read the doc.
 *	a	- convert all filenames to alphanumeric characters,
 *		  by deleting all punctuation.
 *	b	- use the next argument as a tape block size.
 *	c	- create a new multi-file tape, destroying all data
 *		  previously on the tape.
 *	f	- use next argument as a filename to put output into
 *		  instead of a tape drive.  If the name is "-", use
 *		  stdin or stdout as appropriate.
 *	h	- follow symbolic links.
 *	i	- interactive mode - prompt before doing anything.
 *	k	- next argument is number of files to skip on extract.
 *	l	- use next argument as non-standard label size.
 *	m	- check for magic numbers in files, and if found,
 *		  prompt before putting the file on tape.
 *	n	- use next argument as a file-set identifer (save-set
 *		  name).
 *	r	- append named files to end of tape.
 *	s	- swap bytes on input and output.
 *	t	- show table of contents for tape.
 *	u	- convert filenames to upper case when writing to
 *		  tape, convert them to lower case when reading.
 *	v	- verbose mode.
 *	x	- extract the named files from tape.
 *	B	- use next two arguments as block size and line size,
 *		  respectively for unblocking fixed-length records.
 *	C	- convert LF to CRLF on writing, convert CRLF to LF on
 *		  reading.
 *	D	- write variable-length records (System level 3, D format)
 *		  with blocksize 2048 and max. line size 2048.
 *	F	- write HDR2/EOF2 labels for automatic unblocking.
 *	N	- use next argument as tape volume name.
 *	P	- Use "pip" (FILES-11) modified format. 
 *	R	- convert filenames to RT-11 format, XXXXXX.XXX, and set
 *		  "a" and "u" flags.
 *	S	- same as R, except use RSTS 80-character labels.
 *	U	- write UHLn labels.  The file's pathname is put into
 *		  UHL1 and UHL2.  The owner, group, mode, size, and inode
 *		  number are put into UHL3.  The creation, modification,
 *		  and access times are put into UHL4, UHL5, and UHL6
 *		  respectively.
 *	V	- convert filenames to VMS format, XXXXXXXXX.XXX, and set
 *		  "a" and "u" flags.
 *	Z	- dump all labels onto standard output as they are read or
 *		  written.
 *
 * David A. Curry
 * Engineering Computer Network
 * Purdue University
 * August 22, 1984
 *
 * $Log:	ansitape.c,v $
 * Revision 1.4  88/05/04  11:06:46  davy
 * Fixed a typo which broke the "r" option.
 * 
 * Revision 1.3  87/12/14  09:17:43  davy
 * Fixed to use return value from realloc instead of ignoring it.
 * 
 * Revision 1.2  87/09/23  14:15:58  davy
 * Fixed to check writes to the tape drive.
 * 
 * Revision 1.1  87/09/23  11:20:56  davy
 * Initial revision
 * 
 */
#include <sys/param.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/mtio.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <stdio.h>
#include <ctype.h>
#include <pwd.h>

#define TRUE		1
#define FALSE		0
#define DBYTES		4
#define MAXLINE		2048		/* THIS IS 255 IN ANSITAR.C !!!!! */
#define ISOBLOCK	2048
#define WILDCARD	'*'
#define HASHSIZE	256

#define usage()		fatal("Usage: ansitape [abcfhilmnrstuvxBCDFNPRSUVZ] [blocksize] [linesize] [labelsize] [volumename] [savesetname] files\n");
#define max(a, b)	(a > b ? a : b)
#define skiparg()	argc--; argv++;

/*
 * Lengths of VOL1 header fields.
 */
#define VOL1_LABID	3	/* Label Identifier (VOL)		*/
#define VOL1_LABNO	1	/* Label Number (1)			*/
#define VOL1_VOLID	6	/* Volume Identifer ("a" characters)	*/
#define VOL1_ACCES	1	/* Accessibility ("a" chars, " " = all)	*/
#define VOL1_RSVD1	26	/* Reserved for Future Standardization	*/
#define VOL1_OWNID	14	/* Owner Identifer ("a" characters)	*/
#define VOL1_RSVD2	28	/* Reserved for Future Standardization	*/
#define VOL1_LABSV	1	/* Label-Standard Version (1, 2, 3)	*/

/*
 * Lengths of HDR1, EOF1, and EOV1 header fields.
 */
#define HDR1_LABID	3	/* Label Identifier (HDR, EOF, EOV)	*/
#define HDR1_LABNO	1	/* Label Number (1)			*/
#define HDR1_FILID	17	/* File Identifier ("a" characters)	*/
#define HDR1_FSTID	6	/* File-Set Identifier ("a" chars)	*/
#define HDR1_FSCNO	4	/* File Section Number ("n" chars)	*/
#define HDR1_FSQNO	4	/* File Sequence Number ("n" chars)	*/
#define HDR1_GENNO	4	/* Generation Number ("n" characters)	*/
#define HDR1_GNVNO	2	/* Generation Version No. ("n" chars)	*/
#define HDR1_CRDAT	6	/* Creation Date (" yyddd") ("n" chars)	*/
#define HDR1_EXDAT	6	/* Expiration Date (" yyddd")		*/
#define HDR1_ACCES	1	/* Accessibility ("a" char, " " = all)	*/
#define HDR1_BLKCT	6	/* Block Count ("n" chars)		*/
#define HDR1_SYSCD	13	/* System Code ("a" characters)		*/
#define HDR1_RSVD1	7	/* Reserved for Future Standardization	*/

/*
 * Lengths of HDR2, EOF2, and EOV2 header fields.
 */
#define HDR2_LABID	3	/* Label Identifier (HDR, EOF, EOV)	*/
#define HDR2_LABNO	1	/* Label Number (2)			*/
#define HDR2_RCFMT	1	/* Record Format (F, D)			*/
#define HDR2_BLKLN	5	/* Block Length ("n" characters)	*/
#define HDR2_RECLN	5	/* Rcord Length ("n" characters)	*/
#define HDR2_RSVD1	35	/* Reserved for System Use		*/
#define HDR2_BOFFL	2	/* Buffer-Offset Length ("n" chars)	*/
#define HDR2_RSVD2	28	/* Reserved for Future Standardization	*/

/*
 * Lengths of UHLa, UTLa, and UVLn header fields.
 */
#define UHLA_LABID	3	/* Label Identifier (UHL, UTL, UVL)	*/
#define UHLA_LABNO	1	/* Label No. (1-9 UVL,1-9a-zA-Z UHL/UTL)*/
#define UHLA_USRAP	76	/* Reserved for User Applications	*/

/*
 * VOL1 label structure.
 */
struct vol1_label {
	char labid[VOL1_LABID];
	char labno;
	char volid[VOL1_VOLID];
	char acces;
	char rsvd1[VOL1_RSVD1];
	char ownid[VOL1_OWNID];
	char rsvd2[VOL1_RSVD2];
	char labsv;
};

/*
 * HDR1, EOF1, EOV1 label structure.
 */
struct hdr1_label {
	char labid[HDR1_LABID];
	char labno;
	char filid[HDR1_FILID];
	char fstid[HDR1_FSTID];
	char fscno[HDR1_FSCNO];
	char fsqno[HDR1_FSQNO];
	char genno[HDR1_GENNO];
	char gnvno[HDR1_GNVNO];
	char crdat[HDR1_CRDAT];
	char exdat[HDR1_EXDAT];
	char acces;
	char blkct[HDR1_BLKCT];
	char syscd[HDR1_SYSCD];
	char rsvd1[HDR1_RSVD1];
};

/*
 * HDR2, EOF2, EOV2 label structure.
 */
struct hdr2_label {
	char labid[HDR2_LABID];
	char labno;
	char rcfmt;
	char blkln[HDR2_BLKLN];
	char recln[HDR2_RECLN];
	char rsvd1[HDR2_RSVD1];
	char boffl[HDR2_BOFFL];
	char rsvd2[HDR2_RSVD2];
};

/*
 * UHLa, UTLa, UVLn label structure.
 */
struct uhla_label {
	char labid[UHLA_LABID];
	char labno;
	char usrap[UHLA_USRAP];
};

struct hashent {
	int munged;			/* 1 if we munged the name	*/
	struct stat sbuf;		/* stat structure for file	*/
	char path[MAXPATHLEN];		/* full pathname to file	*/
	char name[HDR1_FILID+1];	/* name of file on tape		*/
	struct hashent *next;		/* ptr. to next hash tab entry	*/
};

/*
 * Global variables.
 */
int vms = 0;			/*  'V'	- do VMS filename munging	*/
int rt11 = 0;			/*  'R'	- do RT-11 filename munging	*/
int rsts = 0;			/*  'S'	- do RSTS filename munging	*/
int crlf = 0;			/*  'C' - LF->CRLF and CRLF->LF		*/
int tfile = 0;			/*  'f'	- name of archive is next arg	*/
int alpha = 0;			/*  'a'	- make all names alpha-numeric	*/
int magic = 0;			/*  'm'	- check for magic numbers	*/
int table = 0;			/*  't'	- print table of contents	*/
int upper = 0;			/*  'R', 'S', 'V', 'u' - upper-case	*/
int create = 0;			/*  'c'	- create a new tape		*/
int confirm = 0;		/*  'i'	- interactive mode		*/
int extract = 0;		/*  'x'	- extract files from tape	*/
int replace = 0;		/*  'r'	- replace files on tape		*/
int pipfile = 0;		/*  'P'	- use PIP (FILES-11) format	*/
int verbose = 0;		/*  'v'	- verbose			*/
int blocking = 0;		/*  'b', 'B', 'D' - do blocking		*/
int symlinks = 0;		/*  'h'	- follow symbolic links		*/
int skipfiles = 0;		/*  'k' - skip files on extract		*/
int swapbytes = 0;		/*  's'	- swap even and odd bytes	*/
int userlabels = 0;		/*  'U'	- use UHLa labels on tape	*/
int dumplabels = 0;		/*  'Z' - dump all labels to stdout	*/

int tf;				/* file descriptor for tape archive	*/
int varlen;			/* use variable length records		*/
int fixlen;			/* use fixed length records		*/
int didlog = 0;			/* non-zero if used NMCHGS file		*/
int nfiles = 0;			/* how many files to put on tape	*/
int filenum = 0;		/* number of file we're on		*/
int bfactor = 0;		/* blocking factor			*/
int tapeunit = 0;		/* tape unit to use			*/
unsigned linesize = 0;		/* length of a line			*/
unsigned blocksize = 512;	/* block size				*/
int labelsize = sizeof(struct vol1_label);	/* label size		*/

char *buffer;
char **filetab;			/* pointer to files in argv		*/
char *linebuffer;
char *deffsi = "FILES";		/* file set identifier			*/
char *defvsn = "VOLUME";	/* volume serial number			*/
char *namefile = "NMCHGS";	/* list of name changes			*/
char *alongtime = " 99364";	/* dates are given as <sp> year dayofyr	*/
char *tapefile = "/dev/rmt0";	/* default tape drive to use		*/
char curdate[HDR1_CRDAT+1];	/* current date in ANSI format		*/
char Namefile[64];		/* holds modified namefile name		*/

struct vol1_label vol1;
struct uhla_label uhla;
struct hdr1_label hdr1, eof1;
struct hdr2_label hdr2, eof2;

char startdir[MAXPATHLEN];		/* directory we're called from	*/
char presentdir[MAXPATHLEN];		/* directory we're in right now	*/
struct hashent *hashtab[HASHSIZE];	/* filename hash table		*/
struct hashent *loghp, *addhashtab();

FILE *logfp;

FILE *popen();
char *calloc();
char *realloc();
struct hashent *malloc();

main(argc, argv)
int argc;
char **argv;
{
	register char *ap;
	struct rlimit rlim;
	int bufsize, openmode;

	/*
	 * All output is done to stderr, since we may be
	 * writing to stdout.  So, we buffer stderr.
	 */
	setlinebuf(stderr);

	if (argc < 2)
		usage();

	skiparg();

	/*
	 * Jack up our memory limits.  This is because of
	 * the filename hashing we do - each hash table
	 * entry is 1100 bytes.
	 */
	rlim.rlim_cur = rlim.rlim_max = RLIM_INFINITY;
	setrlimit(RLIMIT_RSS, &rlim);
	setrlimit(RLIMIT_DATA, &rlim);
	setrlimit(RLIMIT_STACK, &rlim);
	
	openmode = O_RDONLY;
	ap = *argv;
	skiparg();

	/*
	 * Process the arguments.
	 */
	while (*ap) {
		switch (*ap) {
		case '-':		/* freebie			*/
			break;
		case 'a':		/* alphanumeric filenames	*/
			alpha = 1;
			break;
		case 'b':		/* set blocksize		*/
		case 'B':		/* set blocksize and linesize	*/
		case 'F':		/* write fixed length records	*/
			blocking = 1;
			blocksize = atoi(*argv);

			if (blocksize <= 0)
				fatal("bad block size: %s\n", *argv);

			skiparg();

			if ((*ap == 'B') || (*ap == 'F')) {
				linesize = atoi(*argv);

				if ((linesize <= 0) || ((blocksize % linesize) != 0))
					fatal("bad line size: %s\n", *argv);

				bfactor = blocksize / linesize;
				skiparg();
			}

			if (*ap == 'F')
				fixlen = 1;

			break;
		case 'c':		/* create a new tape		*/
			create = 1;
			openmode = O_RDWR;
			break;
		case 'f':		/* next arg is filename to use	*/
			tfile = 1;
			tapefile = *argv;
			skiparg();
			break;
		case 'h':		/* follow symbolic links	*/
			symlinks = 1;
			break;
		case 'i':		/* interactive mode		*/
			confirm = 1;
			break;
		case 'k':		/* skip first n files on x	*/
			skipfiles = atoi(*argv);
			skiparg();

			if (skipfiles < 0)
				skipfiles = -skipfiles;
			break;
		case 'l':		/* next arg is nonstd labelsize	*/
			labelsize = atoi(*argv);

			if (labelsize < sizeof(struct hdr1_label))
				fatal("label size must be >= %d.\n", sizeof(struct hdr1_label));

			skiparg();
			break;
		case 'm':		/* check for magic numbers	*/
			magic = 1;
			break;
		case 'n':		/* next arg is save set name	*/
			deffsi = *argv;

			if (strlen(deffsi) > HDR1_FSTID)
				defvsn[HDR1_FSTID] = NULL;

			skiparg();
			break;
		case 'r':		/* append files to end of tape	*/
			replace = 1;
			openmode = O_RDWR;
			break;
		case 's':		/* swap bytes on input/output	*/
			swapbytes = 1;
			break;
		case 't':		/* table of contents		*/
			table = 1;
			break;
		case 'u':		/* upper/lower case conversion	*/
			upper = 1;
			break;
		case 'v':		/* verbose mode			*/
			verbose = 1;
			break;
		case 'x':		/* extract files from tape	*/
			extract = 1;
			break;
		case 'C':		/* LF->CRLF and CRLF->LF	*/
			crlf = 1;
			break;
		case 'D':		/* variable length records	*/
			varlen = 1;
			break;
		case 'N':		/* next arg is tape volume name	*/
			defvsn = *argv;

			if (strlen(defvsn) > VOL1_VOLID)
				defvsn[VOL1_VOLID] = NULL;

			skiparg();
			break;
		case 'P':		/* "pip" (FILES-11) format	*/
			pipfile = 1;
			break;
		case 'R':		/* RT-11 name conversions	*/
			rt11 = 1;
			upper = 1;
			labelsize = 512;
			break;
		case 'S':		/* RSTS name conversion		*/
			rsts = 1;
			upper = 1;
			break;
		case 'U':		/* write UHLn labels		*/
			userlabels = 1;
			break;
		case 'V':		/* VMS filename conversion	*/
			vms = 1;
			upper = 1;
			break;
		case 'Z':		/* dump all headers to stdout	*/
			dumplabels = 1;
			break;
		default:		/* tape drive number given	*/
			if (isdigit(*ap)) {
				tapeunit = *ap;
				break;
			}

			fatal("bad option: %c\n", *ap);
		}

		ap++;
	}

	/*
	 * Gotta have 1 and only 1 of crtx.
	 */
	if ((create + replace + table + extract) != 1)
		fatal("must specify one and only one of c, r, t, x\n");

	/*
	 * In case they misunderstand that k only works when
	 * extracting, don't let them use it otherwise.
	 */
	if (skipfiles && !extract)
		fatal("may only skip files when extracting from tape\n");

	/*
	 * Can't have more than one of RSV.
	 */
	if ((rt11 + rsts + vms) > 1)
		fatal("may only specify one of R, S, V\n");

	/*
	 * Mark start of filename list.
	 */
	filetab = argv;
	filetab[argc] = NULL;

	if (tapeunit)
		tapefile[strlen(tapefile)-1] = tapeunit;

	/*
	 * Open the tape drive (file) for the appropriate mode.
	 */
	if (!strcmp(tapefile, "-")) {
		tf = (openmode == O_RDONLY ? 0 : 1);
	}
	else {
		if ((tf = open(tapefile, openmode)) < 0) {
			if (tfile && (openmode == O_RDWR))
				tf = open(tapefile, O_RDWR | O_CREAT, 0644);

			if (tf < 0)
				fatal("cannot open %s%s\n", tapefile, (openmode == O_RDWR) ? " for writing." : ".");
		}
	}

#ifdef DEBUG
	fprintf(stderr, "DEBUG-> Tapefile: %s  Openmode: %d  Tf: %d\n", tapefile, openmode, tf);
#endif

	bufsize = max(blocksize, labelsize);

#ifdef DEBUG
	fprintf(stderr, "DEBUG-> Bufsize: %d  Blocksize: %d  Linesize: %d\n", bufsize, blocksize, linesize);
#endif

	/*
	 * Allocate buffers.
	 */
	if ((buffer = calloc(1, bufsize + 10)) == NULL)
		fatal("cannot allocate buffer of %d bytes.\n", bufsize + 10);

	if (linesize) {
		if ((linebuffer = calloc(1, linesize + 10)) == NULL)
			fatal("cannot allocate buffer of %d bytes.\n", linesize + 10);
	}

	/*
	 * Get the ANSI format date.
	 */
	getansidate(curdate);

	if (verbose && blocking) {
		fprintf(stderr, "blocksize: %d", blocksize);

		if (linesize)
			fprintf(stderr, " linesize: %d", linesize);

		fputc('\n', stderr);
	}

	/*
	 * Decide what to do.
	 */
	if (create || replace)
		doupdate(tf);
	else if (extract)
		doextract(tf);
	else if (table)
		dotable(tf);
	else
		usage();

	/*
	 * If we didn't write anything to the NMCHGS
	 * file, zap the file.
	 */
	if (!didlog)
		unlink(Namefile);

	unlink("#tmp.tmp");
	exit(0);
}

/*
 * dotable - print a table of contents.
 */
dotable(tf)
int tf;
{
	long bytes;
	char fileid[HDR1_FILID+1];
	int n, files, lbseq, blocks;

	/*
	 * Get and print the volume header.
	 */
	getvol(tf, &vol1);
	prtvol(&vol1);
	fputc('\n', stderr);

	if (dumplabels)
		dumpvol1(&vol1);

	files = 0;
	lbseq = '1';

	/*
	 * Start reading headers and printing them.
	 */
	while (gethdr(tf, &hdr1)) {
		if (dumplabels)
			dumphdr1(&hdr1);

		files++;

		if (verbose)
			prthdr(&hdr1);

		getmark(tf, &hdr2, lbseq);
		sncpy(fileid, hdr1.filid, HDR1_FILID);

		if (dumplabels)
			dumphdr2(&hdr2);

		blocks = 0;
		bytes = 0L;

		/*
		 * Skip over the contents of the file.
		 */
		while ((n = getrec(tf, buffer, blocksize)) > 0) {
			blocks++;
			bytes += n;
		}

		lbseq = '1';
		geteof(tf, &eof1);

		if (dumplabels)
			dumphdr1(&eof1);

		if (verbose)
			prthdr(&eof1);

		getmark(tf, &hdr2, lbseq);
		cmphdreof(&hdr1, &eof1);

		if (dumplabels)
			dumphdr2(&hdr2);

		if (verbose)
			fputc('\n', stderr);

		trimsp(fileid);
		fprintf(stderr, "t   %s, %d blocks, %ld bytes\n", fileid, blocks, bytes);

		if (linesize)
			fprintf(stderr, "\tF-type: blocksize: %d linesize: %d\n", blocksize, linesize);
		else
			fprintf(stderr, "\tD-type: blocksize: %d\n", blocksize);

		if (verbose)
			fputc('\n', stderr);
	}

	fprintf(stderr, "\n%d files\n", files);
	close(tf);
}

/*
 * doextract - extract files from tape.
 */
doextract(tf)
int tf;
{
	FILE *fp;
	long bytes;
	char *ans, *prompt();
	char fileid[HDR1_FILID+1];
	int n, newn, xall, lbseq, blocks;

	lbseq = '1';
	xall = (*filetab == NULL);

	/*
	 * Get the volume header.
	 */
	getvol(tf, &vol1);

	if (dumplabels)
		dumpvol1(&vol1);

	if (verbose)
		prtvol(&vol1);

	while (gethdr(tf, &hdr1)) {
		if (dumplabels)
			dumphdr1(&hdr1);

		/*
		 * Move to a tape mark, grab the filename,
		 * and trim the trailing spaces off it.
		 */
		getmark(tf, &hdr2, lbseq);
		sncpy(fileid, hdr1.filid, HDR1_FILID);
		trimsp(fileid);

		if (dumplabels)
			dumphdr2(&hdr2);

		/*
		 * Convert filenames if necessary.
		 */
		if (rt11)
			fromrt11(fileid);
		if (rsts)
			fromrsts(fileid);
		if (vms)
			fromvms(fileid);

		/*
		 * If they asked us to skip files, see
		 * if we're done skipping yet.  If not,
		 * skip the file and decrement the number
		 * of files to skip.
		 */
		if (skipfiles) {
			skipfile(tf);
			skipfiles--;
			continue;
		}

		/*
		 * If we're doing all the files or this file matches
		 * one of the names in the arg list, copy it.  Otherwise,
		 * skip this file.
		 */
		if (xall || lookup(filetab, fileid, upper)) {
			if (upper)
				makelower(fileid);
			if (alpha)
				makealpha(fileid);

			/*
			 * If interactive, ask what to do.
			 */
			if (confirm) {
				ans = prompt("Extract %s from tape? ", fileid);

				if ((*ans != 'y') && (*ans != NULL)) {
					skipfile(tf);
					goto bottom;
				}
			}

			if ((fp = fopen(fileid, "w")) == NULL) {
				fprintf(stderr, "ansitape: cannot create %s, skipping.\n", fileid);
				skipfile(tf);
			}
			else {
				blocks = 0;
				bytes = 0L;

				if (crlf) {
					char buf[1024];

					fclose(fp);
					sprintf(buf, "sed -e 's/\r//' > %s", fileid);

					if ((fp = popen(buf, "w")) == NULL) {
						fprintf(stderr, "ansitape: cannot create pipe to sed.\n");
						skipfile(tf);
					}
				}

				/*
				 * Copy the file from tape.
				 */
				while ((n = getrec(tf, buffer, blocksize)) > 0) {
					if (linesize) {
						lunblock(fp, buffer, n, linesize);
					}
					else if (varlen) {
						newn = varunblock(fp, buffer, n);
						n = newn;
					}
					else if (pipfile) {
						pipunblock(fp, buffer, n);
					}
					else {
						if (swapbytes)
							doswap(buffer, n);

						fwrite(buffer, n, 1, fp);
					}

					blocks++;
					bytes += n;
				}

				if (crlf)
					pclose(fp);
				else
					fclose(fp);

				if (verbose)
					fprintf(stderr, "x   %s, %d blocks, %ld bytes\n", fileid, blocks, bytes);
			}
		}
		else {
			skipfile(tf);
		}

bottom:
		lbseq = '1';
		geteof(tf, &eof1);

		if (dumplabels)
			dumphdr1(&eof1);

		cmphdreof(&hdr1, &eof1);
		getmark(tf, &hdr2, lbseq);

		if (dumplabels)
			dumphdr2(&hdr2);
	}
}

/*
 * doupdate - add files to tape.
 */
doupdate(tf)
int tf;
{
	FILE *fp;
	long bytes;
	register int i;
	char line[MAXLINE];
	char recln[HDR2_RECLN+1];
	char fileid[HDR1_FILID+1];
	register struct hashent *hp;
	int n, lbseq, blocks, sequence;

	/*
	 * Gather all the filenames to be done into a hashtable,
	 * and convert duplicate names to unique ones, etc.
	 */
	gatherfiles();

	/*
	 * Decrement so we aren't counting NMCHGS.
	 */
	nfiles--;

	lbseq = '1';
	filenum = 1;
	sequence = 0;

	/*
	 * If creating a tape, put a new volume header on,
	 * otherwise, skip to the end of the
	 * tape.
	 */
	if (create) {
		initvol(&vol1);
		putvol(tf, &vol1);

		if (dumplabels)
			dumpvol1(&vol1);
	}
	else {
		getvol(tf, &vol1);

		if (dumplabels)
			dumpvol1(&vol1);

		while (gethdr(tf, &hdr1)) {
			if (dumplabels)
				dumphdr1(&hdr1);

			sncpy(line, hdr1.fsqno, HDR1_FSQNO);

			sequence = atoi(line);

			getmark(tf, &hdr2, lbseq);

			if (dumplabels)
				dumphdr2(&hdr2);

			skipfile(tf);

			lbseq = '1';

			geteof(tf, &eof1);

			if (dumplabels)
				dumphdr1(&eof1);

			getmark(tf, &hdr2, lbseq);

			if (dumplabels)
				dumphdr2(&hdr2);
		}

		backspace(tf);
	}

	/*
	 * Run through the hashtable, putting each file
	 * there onto the tape.
	 */
	for (i=0; i < HASHSIZE; i++) {
#ifdef DEBUG
		fprintf(stderr, "DEBUG-> Now updating files with hashcode %d\n", i);
#endif

		if ((hp = hashtab[i]) == NULL)
			continue;

		while (hp != NULL) {
#ifdef DEBUG
			fprintf(stderr, "DEBUG-> now reading '%s', hc = %d\n", hp->path, i);
#endif

			/*
			 * Save the logfile for last.
			 */
			if (hp == loghp) {
				hp = hp->next;
				continue;
			}

addit:			if ((fp = fopen(hp->path, "r")) == NULL)
				fatal("cannot open %s\n", hp->path);

			if (crlf) {
				char buf[1024];

				fclose(fp);
				sprintf(buf, "sed -e 's/$/\r/' < %s", hp->path);

				if ((fp = popen(buf, "r")) == NULL)
					fatal("cannot create pipe to sed.\n");
			}

			sequence++;

			/*
			 * Put out the initial file header.
			 */
			inithdr(&hdr1, hp->name, sequence);
			puthdr(tf, &hdr1);

			if (dumplabels)
				dumphdr1(&hdr1);

			/*
			 * If doing variable length records
			 * do it this way.
			 */
			if (varlen) {
				blocksize = ISOBLOCK;

				if ((buffer = realloc(buffer, blocksize + 10)) == NULL)
					fatal("cannot reallocate buffer of %d bytes.\n", blocksize+10);

				linesize = 0;

				inithdr2(&hdr2, 'D', blocksize, 0);

				n = dblock(fp, buffer, blocksize);

				utoaz(n, recln, HDR2_RECLN);
				blcopy(hdr2.recln, recln, HDR2_RECLN);
				puthdr(tf, &hdr2);

				if (dumplabels)
					dumphdr2(&hdr2);

				if (crlf)
					pclose(fp);
				else
					fclose(fp);

				if ((fp = fopen("#tmp.tmp", "r")) == NULL)
					fatal("cannot re-read #tmp.tmp\n");
			}

			/*
			 * If doing fixed lenth records do it
			 * this way.
			 */
			if (fixlen) {
				inithdr2(&hdr2, 'F', blocksize, linesize);
				puthdr(tf, &hdr2);

				if (dumplabels)
					dumphdr2(&hdr2);
			}

			/*
			 * Stick on the UHLn labels.
			 */
			if (userlabels)
				douserlabels(tf, hp);

			/*
			 * Put out a tape mark.
			 */
			tapemark(tf);

			blocks = 0;
			bytes = 0L;

			/*
			 * Decide how to write out the file.
			 */
			if (linesize) {
				while ((n = lblock(fp, buffer, linesize, bfactor)) > 0) {
					if (swapbytes)
						doswap(buffer, n);

					if (n % 2)
						buffer[n++] = NULL;

					if ((n = write(tf, buffer, n)) < 0)
						fatal("tape write error.\n");

					blocks++;
					bytes += n;
				}
			}
			else if (pipfile) {
				while ((n = pipblock(fp, buffer, blocksize)) > 0) {
					if (swapbytes)
						doswap(buffer, n);

					if ((n = write(tf, buffer, n)) < 0)
						fatal("tape write error.\n");

					blocks++;
					bytes += n;
				}
			}
			else {
				while ((n = fread(buffer, sizeof(char), blocksize, fp)) > 0) {
					if (swapbytes)
						doswap(buffer, n);

					if (n % 2)
						buffer[n++] = NULL;

					if ((n = write(tf, buffer, n)) < 0)
						fatal("tape write error.\n");

					blocks++;
					bytes += n;
				}
			}

			/*
			 * Mark the tape and out out the EOF headers.
			 */
			if (crlf && !varlen)
				pclose(fp);
			else
				fclose(fp);

			tapemark(tf);
			blcopy(hdr1.labid, "EOF", HDR1_LABID);
			utoaz(blocks, line, HDR1_BLKCT);
			blcopy(hdr1.blkct, line, HDR1_BLKCT);
			puthdr(tf, &hdr1);

			if (dumplabels)
				dumphdr1(&hdr1);

			if (varlen || fixlen) {
				blcopy(hdr2.labid, "EOF", HDR2_LABID);
				puthdr(tf, &hdr2);

				if (dumplabels)
					dumphdr2(&hdr2);
			}

			tapemark(tf);

			if (verbose && (hp != loghp))
				fprintf(stderr, "a   %s, %d blocks, %ld bytes, file %d of %d\n", hp->path, blocks, bytes, filenum++, nfiles);
			else
				fprintf(stderr, "a   %s\n", hp->path);

			/*
			 * If we hacked the filename, log that information.
			 */
			if (hp->munged && logfp) {
				fprintf(logfp, "File '%s' renamed to '%s'\n", hp->path, hp->name);
				didlog++;
			}

			hp = hp->next;
		}
	}

	/*
	 * If we wrote something in the log file, put
	 * the logfile on the tape.
	 */
	if (didlog && logfp) {
		hp = loghp;
		hp->munged = 0;
		hp->next = NULL;

		fclose(logfp);
		logfp = NULL;

		goto addit;
	}
	
	tapemark(tf);
}

/*
 * gatherfiles - gathers the names of all the files we are going to put
 *		 onto the tape, makes them unique, mungs them if necessary,
 *		 and stores them in the hashtable.
 */
gatherfiles()
{
	register int i;
	struct stat sbuf;
	register char *s;
	char dir[MAXPATHLEN];
	char *ans, *getwd(), *prompt(), *rindex();

	/*
	 * Get our current directory.
	 */
	if (getwd(startdir) == NULL)
		fatal("getwd error: %s\n", startdir);

	/*
	 * Initialize the hash table.
	 */
	inithash(startdir);

	/*
	 * For each file on the command line....
	 */
	for (i=0; filetab[i] != NULL; i++) {
		/*
		 * See what it is.
		 */
		if (lstat(filetab[i], &sbuf) < 0)
			fatal("cannot stat %s\n", filetab[i]);

		/*
		 * Skip or process a symbolic link.
		 */
		if ((sbuf.st_mode & S_IFMT) == S_IFLNK) {
			if (!symlinks) {
				fprintf(stderr, "ansitape: symbolic link: %s (not followed)\n", filetab[i]);
				continue;
			}

			/*
			 * Find out about what the link points
			 * to.
			 */
			if (stat(filetab[i], &sbuf) < 0) {
				fprintf(stderr, "ansitape: symbolic link to nowhere: %s\n", filetab[i]);
				continue;
			}
		}

		/*
		 * If it's a directory, follow it.  Otherwise,
		 * check for magic numbers, do the interactive
		 * mode stuff, and stick the file in the hashtable.
		 */
		if ((sbuf.st_mode & S_IFMT) == S_IFDIR) {
			if (chdir(filetab[i]) < 0)
				fatal("cannot change into %s\n", filetab[i]);

			if (getwd(presentdir) == NULL)
				fatal("getwd error: %s\n", presentdir);

			followdir();
		}
		else if ((sbuf.st_mode & S_IFMT) == S_IFREG) {
			if (magic) {
				if (!checkmagic(filetab[i], (*filetab[i] == '/' ? "" : startdir)))
					continue;
			}

			if (confirm) {
				ans = prompt("Copy %s onto tape? ", filetab[i]);

				if ((*ans != 'y') && (*ans != NULL))
					continue;
			}

			if ((s = rindex(filetab[i], '/')) != NULL) {
				*s++ = NULL;

				if (*filetab[i] == '/')
					strcpy(dir, filetab[i]);
				else
					sprintf(dir, "%s/%s", startdir, filetab[i]);
			}
			else {
				s = filetab[i];
				strcpy(dir, startdir);
			}
			
			hashfile(s, dir, sbuf);
		}
		else {
			fprintf(stderr, "ansitape: special file: %s (not copied)\n", filetab[i]);
		}

		/*
		 * Go back where we started.
		 */
		chdir(startdir);
	}
}

/*
 * followdir - recurses through a directory adding files to the hashtable.
 */
followdir()
{
	struct stat sbuf;
	register DIR *dirp;
	char *ans, *prompt();
	register struct direct *entry;

	if ((dirp = opendir(presentdir)) == NULL)
		fatal("cannot open directory %s\n", presentdir);

	/*
	 * Start reading the directory...
	 */
	while ((entry = readdir(dirp)) != NULL) {
		/*
		 * Skip "." and ".."
		 */
		if (!strcmp(entry->d_name, ".") || !strcmp(entry->d_name, ".."))
			continue;

		/*
		 * Find out about the file.
		 */
		if (lstat(entry->d_name, &sbuf) < 0)
			fatal("cannot stat %s/%s\n", presentdir, entry->d_name);

		/*
		 * Process symbolic links.
		 */
		if ((sbuf.st_mode & S_IFMT) == S_IFLNK) {
			if (!symlinks) {
				fprintf(stderr, "ansitape: symbolic link: %s/%s (not followed)\n", presentdir, entry->d_name);
				continue;
			}

			if (stat(entry->d_name, &sbuf) < 0) {
				fprintf(stderr, "ansitape: symbolic link to nowhere: %s/%s\n", presentdir, entry->d_name);
				continue;
			}
		}

		/*
		 * Handle directories and files.
		 */
		if ((sbuf.st_mode & S_IFMT) == S_IFDIR) {
			down(entry->d_name);
			followdir();
		}
		else if ((sbuf.st_mode & S_IFMT) == S_IFREG) {
			if (magic) {
				if (!checkmagic(entry->d_name, presentdir))
					continue;
			}

			if (confirm) {
				ans = prompt("Copy %s/%s onto tape? ", presentdir, entry->d_name);

				if ((*ans != 'y') && (*ans != NULL))
					continue;
			}

			hashfile(entry->d_name, presentdir, sbuf);
		}
		else {
			fprintf(stderr, "ansitape: special file: %s/%s (not copied)\n", presentdir, entry->d_name);
		}
	}

	closedir(dirp);
	up();
}

/*
 * down - move down a directory level, keeping track of it in presentdir.
 */
down(dir)
char *dir;
{
	strcat(presentdir, "/");
	strcat(presentdir, dir);

	if (chdir(presentdir) < 0)
		fatal("cannot change into directory %s\n", presentdir);
}

/*
 * up - move up a level, keeping track of it in presentdir.
 */
up()
{
	char *rindex();
	register char *s;

	s = rindex(presentdir, '/');

	if (s != presentdir)
		*s = NULL;
	else
		*++s = NULL;

	if (chdir(presentdir) < 0)
		fatal("up: fatal internal error.\n");
}

/*
 * hashfile - compute a filename's hashcode, make sure it's a unique name,
 *	      and add it to the hashtable.
 */
hashfile(file, dir, sbuf)
char *file, *dir;
struct stat sbuf;
{
	int munged;
	register char *s;
	register int hashcode;
	char fileid[HDR1_FILID+1];

	munged = 0;
	sncpy(fileid, file, HDR1_FILID);

	/*
	 * Mung the pathname if necessary.
	 */
	if (rt11)
		munged += tort11(fileid);
	if (rsts)
		munged += torsts(fileid);
	if (vms)
		munged += tovms(fileid);
	if (upper)
		munged += makeupper(fileid);
	if (alpha)
		munged += makealpha(fileid);

	/*
	 * Compute the hashcode.
	 */
	s = fileid;
	hashcode = 0;

	while (*s)
		hashcode += *s++;

	hashcode %= HASHSIZE;

	/*
	 * If this name is already in the hashtable, get
	 * a unique name.
	 */
	if (inhashtab(hashcode, fileid)) {
		hashcode = unique(fileid);
		munged = 1;
	}

	/*
	 * Add the filename to the table.
	 */
	addhashtab(hashcode, fileid, dir, file, munged, sbuf);
}

/*
 * inhashtab - returns non-zero if name is already in the hashtable.
 */
inhashtab(hc, name)
int hc;
char *name;
{
	register struct hashent *hp;

	if ((hp = hashtab[hc]) == NULL)
		return(0);

	while (hp != NULL) {
		if (!strcmp(hp->name, name))
			return(1);

		hp = hp->next;
	}

	return(0);
}

/*
 * addhashtab - add an entry into the hashtable.
 */
struct hashent *addhashtab(hc, name, dir, file, munged, sbuf)
int hc;
int munged;
struct stat sbuf;
char *name, *dir, *file;
{
	register struct hashent *hp;

#ifdef DEBUG
	fprintf(stderr, "DEBUG-> Adding %s/%s to hashtable as %s, hc = %d\n", dir, file, name, hc);
#endif

	/*
	 * First entry.
	 */
	if ((hp = hashtab[hc]) == NULL) {
		if ((hashtab[hc] = malloc(sizeof(struct hashent))) == NULL)
			fatal("out of hashtable memory.\n");

		hp = hashtab[hc];

		hp->next = NULL;
		hp->sbuf = sbuf;
		hp->munged = munged;
		strcpy(hp->name, name);
		sprintf(hp->path, "%s/%s", dir, file);

		nfiles++;
		return(hp);
	}

	/*
	 * Go to the end of the linked list.
	 */
	while (hp->next != NULL)
		hp = hp->next;

	/*
	 * Add the new entry.
	 */
	if ((hp->next = malloc(sizeof(struct hashent))) == NULL)
		fatal("out of hashtable memory.\n");

	hp = hp->next;

	hp->next = NULL;
	hp->sbuf = sbuf;
	hp->munged = munged;
	strcpy(hp->name, name);
	sprintf(hp->path, "%s/%s", dir, file);
	nfiles++;
	return(hp);
}

/*
 * inithash - adds the filename in Namefile (our log file) to the
 * 	      hashtable so that it won't get converted to some other
 *	      name later.
 */
inithash(dir)
char *dir;
{
	char path[1024];
	register char *s;
	struct stat sbuf;
	register int hashcode;

	strcpy(Namefile, namefile);

	/*
	 * Mung filename if needed.
	 */
	if (rt11)
		tort11(Namefile);
	if (rsts)
		torsts(Namefile);
	if (vms)
		tovms(Namefile);
	if (upper)
		makeupper(Namefile);
	if (alpha)
		makealpha(Namefile);

	sprintf(path, "%s/%s", dir, Namefile);

	/*
	 * Make sure we aren't gonna overwrite the file.
	 */
	if (stat(path, &sbuf) >= 0)
		fatal("need filename %s for program's use, please rename.\n", path);

	if ((logfp = fopen(path, "w")) == NULL)
		fatal("cannot create %s\n", path);

	/*
	 * Compute the hash code.
	 */
	s = Namefile;
	hashcode = 0;
	
	while (*s)
		hashcode += *s++;
	hashcode %= HASHSIZE;

	if (stat(path, &sbuf) < 0)
		fatal("%s disappeared!\n");

	loghp = addhashtab(hashcode, Namefile, dir, Namefile, 0, sbuf);
}

/*
 * unique - convert name to a unique name.  Names look like AAAAA, BAAAA, etc.
 */
unique(name)
char *name;
{
	static int len;
	register char *s;
	register int hashcode;
	static char *np = NULL;
	static char *newname = "AAAA@";

	if (np == NULL)
		len = strlen(newname);

	/*
	 * Increment the name, mung it if necessary,
	 * and see if it's in the hashtable.  Do this
	 * until name isn't in the hashtable.
	 */
	do {
		np = &newname[len];
		*np-- = NULL;

		while ((*np == 'Z') && (np != newname))
			np--;

		*np += 1;

		while (*++np)
			*np = 'A';

		if (rt11)
			tort11(newname);
		if (rsts)
			torsts(newname);
		if (vms)
			tovms(newname);
		if (upper)
			makeupper(newname);
		if (alpha)
			makealpha(newname);

		s = newname;
		hashcode = 0;

		while (*s)
			hashcode += *s++;

		hashcode %= HASHSIZE;
	} while (inhashtab(hashcode, newname));

#ifdef DEBUG
	fprintf(stderr, "DEBUG-> returning unique name '%s'\n", newname);
#endif

	/*
	 * Replace old name with new.
	 */
	strcpy(name, newname);
	return(hashcode);
}

/*
 * getvol - reads a volume header off the tape.
 */
getvol(tf, volp)
int tf;
struct vol1_label *volp;
{
	register int n;

	if (labelsize == sizeof(struct vol1_label)) {
		if ((n = read(tf, (char *) volp, sizeof(struct vol1_label))) < 0)
			fatal("tape read error.\n");
	}
	else {
		if ((n = read(tf, buffer, labelsize)) < 0)
			fatal("tape read error.\n");

		mybcopy((char *) volp, buffer, sizeof(struct vol1_label));
	}

	if ((n != labelsize) || (strncmp(volp->labid, "VOL", VOL1_LABID) != 0) ||
	    (volp->labno != '1')) {
		fprintf(stderr, "ansitape: warning: volume label (VOL1) missing.\n");
		backspace(tf);
		return;
	}

	if (rt11 || rsts) {
		if ((n = read(tf, buffer, labelsize)) < 0)
			fatal("tape read error.\n");

		mybcopy((char *) &hdr1, buffer, sizeof(struct hdr1_label));

		if ((n == labelsize) && (strncmp(hdr1.labid, "HDR", HDR1_LABID) == 0))
			backspace(tf);
		else
			fprintf(stderr, "ansitape: possible RT-11/RSTS bootstrap block.\n");
	}
}

/*
 * gethdr - reads a HDR1/EOF1 label off the tape.
 */
gethdr(tf, hdrp)
int tf;
struct hdr1_label *hdrp;
{
	register int n;

	if (labelsize == sizeof(struct hdr1_label)) {
		if ((n = read(tf, (char *) hdrp, sizeof(struct hdr1_label))) < 0)
			fatal("tape read error.\n");
	}
	else {
		if ((n = read(tf, buffer, labelsize)) < 0)
			fatal("tape read error.\n");

		mybcopy((char *) hdrp, buffer, sizeof(struct hdr1_label));
	}

	if (n == 0)
		return(FALSE);

	if ((n != labelsize) || (strncmp(hdrp->labid, "HDR", HDR1_LABID) != 0) ||
	    (hdrp->labno != '1'))
	    	hdrerr(tf, hdrp);

	return(TRUE);
}

/*
 * hdrerr - print a header error.
 */
hdrerr(tf, hdrp)
int tf;
struct hdr1_label *hdrp;
{
	register int n, found;

	found = FALSE;
	fprintf(stderr, "ansitape: warning: file label (HDR1) error - skipping.\n");

	/*
	 * Try to skip over the mess.
	 */
	while (!found) {
		skipfile(tf);

		if (labelsize == sizeof(struct hdr1_label)) {
			if ((n = read(tf, (char *) hdrp, sizeof(struct hdr1_label))) < 0)
				fatal("tape read error.\n");
		}
		else {
			if ((n = read(tf, buffer, labelsize)) < 0)
				fatal("tape write error.\n");

			mybcopy((char *) hdrp, buffer, sizeof(struct hdr1_label));
		}

		if ((n == labelsize) && (strncmp(hdrp->labid, "HDR", HDR1_LABID) == 0))
			found = TRUE;
	}
}

/*
 * geteof - reads an EOF1 header.
 */
geteof(tf, eofp)
int tf;
struct hdr1_label *eofp;
{
	register int n;

	if (labelsize == sizeof(struct hdr1_label)) {
		if ((n = read(tf, (char *) eofp, sizeof(struct hdr1_label))) < 0)
			fatal("tape read error.\n");
	}
	else {
		if ((n = read(tf, buffer, labelsize)) < 0)
			fatal("tape read error.\n");

		mybcopy((char *) eofp, buffer, sizeof(struct hdr1_label));
	}

	if ((n != labelsize) || (strncmp(eofp->labid, "EOF", HDR1_LABID) != 0) ||
	    (eofp->labno != '1'))
	    	fprintf(stderr, "ansitape: warning: file label (EOF1) error.\n");
}

/*
 * getrec - read a record from tape.
 */
getrec(f, buf, size)
char *buf;
int f, size;
{
	register int n;

	n = read(f, buf, size);

	if (n < 0)
		fatal("read error (record may be larger than %d bytes).\n", size);

	return(n);
}

/*
 * getmark - move forward to the next tape mark, grab a HDR2/EOF2 label.
 */
getmark(tf, hdr2p, labseq)
int tf, labseq;
struct hdr2_label *hdr2p;
{
	register int n, s;
	char rec[sizeof(struct hdr1_label)];
	char lstring[sizeof(struct hdr1_label) + 1];
	char blkln[HDR2_BLKLN + 1], recln[HDR2_RECLN + 1];

	s = labseq + 1;

	if ((n = read(tf, rec, sizeof(rec))) < 0)
		fatal("tape read error.\n");

	if (n == 0)
		return;

	/*
	 * Process HDR2 labels.
	 */
	if ((n == sizeof(struct hdr1_label)) && (strncmp(rec, "HDR", 3) == 0)) {
		mybcopy((char *) hdr2p, rec, sizeof(struct hdr1_label));

		if (hdr2p->labno == s) {
			sncpy(blkln, hdr2p->blkln, HDR2_BLKLN);
			sncpy(recln, hdr2p->recln, HDR2_RECLN);

			if (verbose && table)
				prthdr2(&hdr2);

			blocksize = max(blocksize, atoi(blkln));

			if (hdr2p->rcfmt == 'D') {
				linesize = 0;
				varlen = TRUE;
			}
			else {
				linesize = max(linesize, atoi(recln));
				varlen = FALSE;
			}

			if ((buffer = realloc(buffer, blocksize + 10)) == NULL)
				fatal("cannot reallocate buffer of %d bytes.\n", blocksize+10);
		}
	}

	/*
	 * Skip over HDR3-9, UHLn, UVLa, UTLn labels.
	 */
	while ((n == sizeof(struct hdr1_label)) &&
	       ((strncmp(rec, "HDR", 3) == 0) || (strncmp(rec, "EOF", 3) == 0) ||
	        (strncmp(rec, "UHL", 3) == 0) || (strncmp(rec, "UVL", 3) == 0) ||
	        (strncmp(rec, "UTL", 3) == 0))) {

	        if (verbose && table) {
	        	sncpy(lstring, rec, sizeof(struct hdr1_label));
	        	trimsp(lstring);
	        	fprintf(stderr, "%s\n", lstring);
	        }

		if ((n = read(tf, rec, sizeof(rec))) < 0)
			fatal("tape read error.\n");

	        if (n == 0)
	        	return;
	}

	fprintf(stderr, "ansitape: warning: tape mark missing.\n");
}

/*
 * cmphdreof - make sure the HDR and EOF labels agree with each other.
 */
cmphdreof(hdrp, eofp)
struct hdr1_label *hdrp, *eofp;
{
	char line[MAXLINE];
	static int len = HDR1_FILID + HDR1_FSTID + HDR1_FSCNO + HDR1_FSQNO +
			 HDR1_GENNO + HDR1_GNVNO + HDR1_CRDAT + HDR1_EXDAT + 1;

	if ((strncmp(hdrp->filid, eofp->filid, len) != 0) ||
	    (strncmp(hdrp->syscd, eofp->syscd, HDR1_SYSCD) != 0)) {
	    	sncpy(line, hdrp->filid, HDR1_FILID);
	    	trimsp(line);
	    	fprintf(stderr, "ansitape: warning: HDR and EOF labels for %s disagree.\n", line);
	}
}

/*
 * putvol - write a volume header on tape.
 */
putvol(tf, volp)
int tf;
struct vol1_label *volp;
{
	register int len;

	if (labelsize == sizeof(struct vol1_label)) {
		if (write(tf, (char *) volp, sizeof(struct vol1_label)) < 0)
			fatal("tape write error.\n");
		return;
	}

	mybcopy(buffer, (char *) volp, sizeof(struct vol1_label));

	len = labelsize - sizeof(struct vol1_label);

	blcopy(&buffer[sizeof(struct vol1_label)], "", len);

	if (write(tf, buffer, labelsize) < 0)
		fatal("tape write error.\n");
}

/*
 * puthdr - put a HDR1/EOF1 label on tape.
 */
puthdr(tf, hdrp)
int tf;
struct hdr1_label *hdrp;
{
	register int len;

	if (labelsize == sizeof(struct hdr1_label)) {
		if (write(tf, (char *) hdrp, sizeof(struct hdr1_label)) < 0)
			fatal("tape write error.\n");
		return;
	}

	mybcopy(buffer, (char *) hdrp, sizeof(struct hdr1_label));

	len = labelsize - sizeof(struct hdr1_label);

	mybcopy(&buffer[sizeof(struct hdr1_label)], "", len);

	if (write(tf, buffer, labelsize) < 0)
		fatal("tape write error.\n");
}

/*
 * prtvol - pretty-print the volume header.
 */
prtvol(volp)
struct vol1_label *volp;
{
	char labid[VOL1_LABID + 1], volid[VOL1_VOLID + 1], ownid[VOL1_OWNID + 1];

	sncpy(labid, volp->labid, VOL1_LABID);
	sncpy(volid, volp->volid, VOL1_VOLID);
	sncpy(ownid, volp->ownid, VOL1_OWNID);

	fprintf(stderr, "Volume label:\n");
	fprintf(stderr, "\tLabel: %s%c  Serial: %s  Access: %c\n", labid, volp->labno, volid, volp->acces);
	fprintf(stderr, "\tOwner: %s  Standard: %c\n", ownid, volp->labsv);
}

/*
 * prthdr - pretty-print a HDR1/EOF1 label.
 */
prthdr(hdrp)
struct hdr1_label *hdrp;
{
	char labid[HDR1_LABID + 1], filid[HDR1_FILID + 1], fstid[HDR1_FSTID + 1];
	char fscno[HDR1_FSCNO + 1], fsqno[HDR1_FSQNO + 1], genno[HDR1_GENNO + 1];
	char gnvno[HDR1_GNVNO + 1], crdat[HDR1_CRDAT + 1], exdat[HDR1_EXDAT + 1];
	char blkct[HDR1_BLKCT + 1], syscd[HDR1_SYSCD + 1];

	sncpy(labid, hdrp->labid, HDR1_LABID);
	sncpy(filid, hdrp->filid, HDR1_FILID);
	sncpy(fstid, hdrp->fstid, HDR1_FSTID);
	sncpy(fscno, hdrp->fscno, HDR1_FSCNO);
	sncpy(fsqno, hdrp->fsqno, HDR1_FSQNO);
	sncpy(genno, hdrp->genno, HDR1_GENNO);
	sncpy(gnvno, hdrp->gnvno, HDR1_GNVNO);
	sncpy(crdat, hdrp->crdat, HDR1_CRDAT);
	sncpy(exdat, hdrp->exdat, HDR1_EXDAT);
	sncpy(blkct, hdrp->blkct, HDR1_BLKCT);
	sncpy(syscd, hdrp->syscd, HDR1_SYSCD);

	fprintf(stderr, "File label:\n");
	fprintf(stderr, "\tLabel: %s%c  File: %s\n", labid, hdrp->labno, filid);
	fprintf(stderr, "\tSet: %s  Section: %s  Sequence: %s\n", fstid, fscno, fsqno);
	fprintf(stderr, "\tGeneration: %s  Generation Version: %s\n", genno, gnvno);
	fprintf(stderr, "\tCreated: %s  Expires: %s  Access: %c\n", crdat, exdat, hdrp->acces);
	fprintf(stderr, "\tBlocks: %s  System: %s\n", blkct, syscd);
}

/*
 * prthdr2 - pretty-print a HDR2/EOF2 label.
 */
prthdr2(hdr2p)
struct hdr2_label *hdr2p;
{
	char labid[HDR2_LABID + 1], blkln[HDR2_BLKLN + 1];
	char recln[HDR2_RECLN + 1], boffl[HDR2_BOFFL + 1];

	sncpy(labid, hdr2p->labid, HDR2_LABID);
	sncpy(blkln, hdr2p->blkln, HDR2_BLKLN);
	sncpy(recln, hdr2p->recln, HDR2_RECLN);
	sncpy(boffl, hdr2p->boffl, HDR2_BOFFL);

	fprintf(stderr, "File label:\n");
	fprintf(stderr, "\tLabel: %s%c  Datatype: %c\n", labid, hdr2p->labno, hdr2p->rcfmt);
	fprintf(stderr, "\tMaximum block size: %s  Maximum records size: %s\n", blkln, recln);
	fprintf(stderr, "\tByte offset at head of block: %s\n", boffl);
}

/*
 * initvol - initialize the volume header structure.
 */
initvol(volp)
struct vol1_label *volp;
{
	struct passwd *getpwuid();
	register struct passwd *p;

	blcopy((char *) volp, "", sizeof(struct vol1_label));

	blcopy(volp->labid, "VOL", VOL1_LABID);
	blcopy(volp->volid, defvsn, VOL1_VOLID);

	volp->labno = '1';
	volp->acces = ' ';

	if ((p = getpwuid(getuid())) == NULL)
		blcopy(volp->ownid, "(unknown)", VOL1_OWNID);
	else
		blcopy(volp->ownid, p->pw_name, VOL1_OWNID);

	if (varlen || fixlen)
		volp->labsv = '3';
	else
		volp->labsv = '1';
}

/*
 * inithdr - initialize a header struture.
 */
inithdr(hdrp, filename, seq)
int seq;
char *filename;
struct hdr1_label *hdrp;
{
	char fsqno[HDR1_FSQNO + 1];

	blcopy((char *) hdrp, "", sizeof(struct hdr1_label));

	blcopy(hdrp->labid, "HDR", HDR1_LABID);
	blcopy(hdrp->filid, filename, HDR1_FILID);
	blcopy(hdrp->fstid, deffsi, HDR1_FSTID);
	blcopy(hdrp->fscno, "0001", HDR1_FSCNO);

	utoaz(seq, fsqno, HDR1_FSQNO);

	blcopy(hdrp->fsqno, fsqno, HDR1_FSQNO);
	blcopy(hdrp->genno, "0001", HDR1_GENNO);
	blcopy(hdrp->gnvno, "00", HDR1_GNVNO);
	blcopy(hdrp->crdat, curdate, HDR1_CRDAT);
	blcopy(hdrp->exdat, alongtime, HDR1_EXDAT);
	blcopy(hdrp->blkct, "000000", HDR1_BLKCT);
	blcopy(hdrp->syscd, "ANSI/UNIX 4.2", HDR1_SYSCD);

	hdrp->labno = '1';
}

/*
 * inithdr2 - initialize a HDR2 label.
 */
inithdr2(hdr2p, rectype, b, r)
int b, r, rectype;
struct hdr2_label *hdr2p;
{
	char blkln[HDR2_BLKLN + 1], recln[HDR2_RECLN + 1];

	blcopy(hdr2p, "", sizeof(struct hdr2_label));

	blcopy(hdr2p->labid, "HDR", HDR2_LABID);

	utoaz(b, blkln, HDR2_BLKLN);
	utoaz(r, recln, HDR2_RECLN);

	blcopy(hdr2p->blkln, blkln, HDR2_BLKLN);
	blcopy(hdr2p->recln, recln, HDR2_RECLN);
	blcopy(hdr2p->boffl, "00", HDR2_BOFFL);

	hdr2p->labno = '2';
	hdr2p->rcfmt = rectype;
}

/*
 * dblock - block records for D format (variable length).
 */
dblock(fp, buffer, blocksize)
FILE *fp;
char *buffer;
int blocksize;
{
	FILE *fs;
	register char *bb, *db;
	char dbuffer[MAXLINE+DBYTES+1];
	register int n, nl, nf, nlm, brem;

	if ((fs = fopen("#tmp.tmp", "w")) == NULL)
		fatal("cannot create #tmp.tmp\n");

	n = 1;
	nlm = 0;
	bb = buffer;
	brem = blocksize + 1;

	while (n) {
		n = drecord(fp, dbuffer);
		nlm = max(nlm, n);

		if (n && ((brem -= n) > 0)) {
			for (nl=0; nl < n; ++nl)
				*bb++ = dbuffer[nl];
		}
		else {
			brem += n;

			while (brem-- > 0)
				*bb++ = '^';

			nl = n;
			bb = buffer;

			if ((nf = fwrite(buffer, 1, blocksize, fs)) != blocksize) {
				perror("fwrite");
				exit(1);
			}

			if (n) {
				for (nl=0; nl < n; ++nl)
					*bb++ = dbuffer[nl];

				brem = blocksize - n;
			}
		}
	}

	fclose(fs);
	return(nlm);
}

/*
 * drecord - part of D blocking.
 */
drecord(fp, lbuffer)
FILE *fp;
char *lbuffer;
{
	register int n;
	char count[DBYTES + 1];

	n = filldln(fp, lbuffer);

	utoaz(n, count, DBYTES);
	mybcopy(lbuffer, count, DBYTES);

	return(n);
}

/*
 * filldln - fill a D format line buffer with characters.
 */
filldln(fp, lbuf)
FILE *fp;
char *lbuf;
{
	register char *lb;
	register int c, i, linelim;

	lb = lbuf;
	i = DBYTES;
	lb += DBYTES;
	linelim = MAXLINE + DBYTES;

/****************
 * This is stupid.  If you read a null from the file, the whole thing
 * pukes and you end up not writing nulls to the tape, and converting
 * them to newlines when you read the tape back in.
 *	while ((c = fgetc(fp)) && (c != EOF) && (c != '\n')) {
 ****************/
	while (((c = fgetc(fp)) != EOF) && (c != '\n')) {
		*lb++ = c;

/****************
 * This code causes a fencepost error if you hit a long line (like in
 * a binary file.  You end up one past the end of the buffer.
 *		if (i++ > (linelim - 1))
 *			break;
 * Below is the way it should be.
 ****************/
		if (++i > (linelim - 1))
			break;
	}

	*lb = NULL;

	if (c == '\n')
		return(i);
	if (c == EOF)
		return(0);

/****************
 * This message removed because people found it confusing.
 *	fprintf(stderr, "ansitape: warning: %d byte line too long, broken into parts.\n", i);
 ****************/

	return(i);
}

/*
 * lblock - block records for weird line sizes.
 */
lblock(fp, buffer, lsize, bfactor)
FILE *fp;
char *buffer;
int lsize, bfactor;
{
	register int c, i;
	register char *lp, *bp, *linelim;

	bp = buffer;

	for (i=0; i < bfactor; i++) {
		lp = bp;
		linelim = &lp[lsize];

		while (lp < linelim) {
			c = fgetc(fp);

			if ((c == '\n') || (c == EOF))
				break;

			*lp++ = c;
		}

		if ((c == EOF) && (lp == bp))
			break;

		while ((c != '\n') && (c != EOF))
			c = fgetc(fp);

		while (lp < linelim)
			*lp++ = ' ';

		bp += lsize;
	}

	return(bp - buffer);
}

/*
 * lunblock - unblock records with weird line sizes.
 */
lunblock(fp, buffer, blen, lsize)
FILE *fp;
char *buffer;
int blen, lsize;
{
	register int i;
	char swapbuf[2];
	register char *bp, *bp1, *lastp, *buflim;

	bp = buffer;
	buflim = &buffer[blen];

	i = 1;
	while (bp < buflim) {
		lastp = &bp[lsize - 1];

		while ((lastp >= bp) && isspace(*lastp))
			lastp--;

		for (bp1=bp; bp1 <= lastp; bp1++) {
			if (swapbytes) {
				swapbuf[i] = *bp1;
				i = (i+1) % 2;
				
				if (i == 1)
					fwrite(swapbuf, sizeof(char), 2, fp);
			}
			else {
				fputc(*bp1, fp);
			}
		}

		if (swapbytes) {
			if (i == 0)
				fputc(swapbuf[1], fp);

			i = 1;
		}

		fputc('\n', fp);

		bp += lsize;
	}
}

/*
 * blcopy - copy n characters or src into dest, pad with blanks.
 */
blcopy(dest, src, n)
char *dest, *src;
int n;
{
	register int i;

	i = 0;

	while ((i < n) && *src) {
		*dest++ = *src++;
		i++;
	}

	while (i++ < n)
		*dest++ = ' ';
}

/*
 * sncpy - copy n characters of src into dest, null terminate.
 */
sncpy(dest, src, n)
char *dest, *src;
int n;
{
	register int i;

	i = 0;

	while ((i < n) && *src) {
		*dest++ = *src++;
		i++;
	}

	*dest = NULL;
}

/*
 * utoaz - put integer n into buf as character string.
 */
utoaz(n, buf, size)
char *buf;
int n, size;
{
	register char *p;

	p = &buf[size - 1];

	while ((p >= buf) && (n != 0)) {
		*p = (n % 10) + '0';
		n /= 10;
		p--;
	}

	while (p >= buf) {
		*p = '0';
		p--;
	}
}

/*
 * tapemark - write a tape mark.
 */
tapemark(tf)
int tf;
{
	struct mtop mtop;

	mtop.mt_count = 1;
	mtop.mt_op = MTWEOF;

	if (ioctl(tf, MTIOCTOP, &mtop) < 0)
		fatal("tape ioctl error.\n");
}

/*
 * skipfile - skip over a file on tape.
 */
skipfile(tf)
int tf;
{
	struct mtop mtop;

	mtop.mt_count = 1;
	mtop.mt_op = MTFSF;

	if (ioctl(tf, MTIOCTOP, &mtop) < 0)
		fatal("tape ioctl error.\n");
}

/*
 * backspace - back up to previous tape mark on tape.
 */
backspace(tf)
int tf;
{
	struct mtop mtop;

	mtop.mt_count = 1;
	mtop.mt_op = MTBSF;

	if (ioctl(tf, MTIOCTOP, &mtop) < 0)
		fatal("tape ioctl error.\n");
}

/*
 * getansidate - convert the current date to ANSI format, which is
 *
 *			<sp>YYDDD
 *
 * 		 where <sp> is a required space, YY is the year, and
 *		 DDD is the day of the year (1-365).
 */
getansidate(curdate)
char *curdate;
{
	time_t now;
	struct tm *timep, *localtime();

	time(&now);
	timep = localtime(&now);

	curdate[0] = ' ';
	utoaz(timep->tm_year, &curdate[1], 2);
	utoaz(timep->tm_yday, &curdate[3], 3);
	curdate[6] = NULL;
}

/*
 * lookup - see if name is in table tab.
 */
lookup(tab, name, ucase)
char **tab;
char *name;
int ucase;
{
	register int i;
	char lower[MAXLINE];

	/*
	 * Do case conversion if needed.
	 */
	if (ucase) {
		strcpy(lower, name);
		makelower(lower);
	}

	for (i=0; tab[i] != NULL; i++) {
		if ((wildcmp(tab[i], name, WILDCARD) == 0) ||
		    (ucase && (wildcmp(tab[i], lower, WILDCARD) == 0)))
		    	return(TRUE);
	}

	return(FALSE);
}

/*
 * makelower - convert s to lower case.
 */
makelower(s)
char *s;
{
	register char *p;
	register int munged;

	p = s;
	munged = 0;

	while (*p) {
		if (isupper(*p)) {
			*p = tolower(*p);
			munged = 1;
		}
		
		p++;
	}

	return(munged);
}

/*
 * makeupper - convert s to upper case.
 */
makeupper(s)
char *s;
{
	register char *p;
	register int munged;

	p = s;
	munged = 0;

	while (*p) {
		if (islower(*p)) {
			*p = toupper(*p);
			munged = 1;
		}
		
		p++;
	}

	return(munged);
}

/*
 * makealpha - make name alpha-numeric only.
 */
makealpha(name)
char *name;
{
	register int munged;
	register char *op, *np;

	munged = 0;
	op = np = name;

	while (*op) {
		if (isalpha(*op) || isdigit(*op))
			*np++ = *op;
		else
			munged = 1;

		op++;
	}

	*np = NULL;
	return(munged);
}

/*
 * haslower - check for lower case.
 */
haslower(s)
char *s;
{
	register char *p;

	p = s;

	while (*p) {
		if (islower(*p))
			return(TRUE);
		p++;
	}

	return(FALSE);
}

/*
 * tort11 - convert a pathname to RT-11 format.
 */
tort11(name)
char *name;
{
	char buf[32];
	char *index();
	register char *extp;

	extp = index(name, '.');

	if (extp != NULL)
		*extp++ = NULL;

	makealpha(name);
	strncpy(buf, name, 6);
	buf[6] = NULL;			/* strncpy sucks */

	strcat(buf, ".");

	if (extp != NULL) {
		makealpha(extp);
		strncat(buf, extp, 3);
	}
	else {
		strcat(buf, "ext");
	}

	strcpy(name, buf);
	return(1);
}

/*
 * torsts - same as tort11.
 */
torsts(name)
char *name;
{
	return(tort11(name));
}

/*
 * tovms - convert a pathname to VMS format.
 */
tovms(name)
char *name;
{
	char buf[32];
	char *index();
	register char *extp;

	extp = index(name, '.');

	if (extp != NULL)
		*extp++ = NULL;

	makealpha(name);
	strncpy(buf, name, 9);
	buf[9] = NULL;			/* strncpy sucks */

	strcat(buf, ".");

	if (extp != NULL) {
		makealpha(extp);
		strncat(buf, extp, 3);
	}
	else {
		strcat(buf, "ext");
	}

	strcpy(name, buf);
	return(1);
}

/*
 * fromrt11 - convert a filename from RT-11 format.
 */
fromrt11(name)
char *name;
{
	register char *op, *np;

	op = np = name;

	while (*op) {
		if (!isspace(*op))
			*np++ = *op;
		op++;
	}

	*np = NULL;
}

/*
 * fromrsts - same as fromrt11.
 */
fromrsts(name)
char *name;
{
	fromrt11(name);
}

/*
 * fromvms - same as fromrt11.
 */
fromvms(name)
char *name;
{
	fromrt11(name);
}

/*
 * trimsp - remove trailing spaces from s.
 */
trimsp(s)
char *s;
{
	register char *p;

	p = &s[strlen(s)-1];

	while ((p >= s) && isspace(*p))
		p--;

	*++p = NULL;
}

/*
 * bcopy - copy size bytes from src to dest, ignore nulls.
 */
mybcopy(dest, src, size)
char *dest, *src;
int size;
{
	while (size-- > 0)
		*dest++ = *src++;
}

/*
 * prompt - prompt a user for input, return the input.
 */
/* VARARGS */
char *prompt(s, a1, a2)
char *s;
{
	register int i;
	static int fd = -1;
	static char buf[16];

	if (fd < 0) {
		if ((fd = open("/dev/tty", 0)) < 0)
			fatal("cannot open /dev/tty.\n");
	}

	fprintf(stderr, s, a1, a2);
	fflush(stderr);

	i = read(fd, buf, sizeof(buf));
	buf[i-1] = NULL;

	for (i=0; buf[i] != NULL; i++) {
		if (isupper(buf[i]))
			buf[i] = tolower(buf[i]);
	}

	return(buf);
}

/*
 * fatal - print diagnostic and exit.
 */
/* VARARGS */
fatal(s, a1, a2, a3, a4)
char *s;
{
	fprintf(stderr, "ansitape: ");
	fprintf(stderr, s, a1, a2, a3, a4);
	exit(1);
}

/*
 * varunblock - unblock variable length records.
 */
varunblock(fp, buffer, blen)
FILE *fp;
int blen;
char *buffer;
{
	char swapbuf[2];
	int i, count, newblen;
	register char *bp, *buflim;
	register char *bp1, *lastp;

	newblen = 0;
	bp = buffer;
	buflim = &buffer[blen];

	i = 1;
	while (bp < buflim) {
		sscanf(bp, "%4d", &count);
		count -= 4;

		if (*bp == 0x5e)	/* '^' */
			break;

		bp += 4;
		lastp = bp + count;

		while (bp < lastp) {
			if (swapbytes) {
				swapbuf[i] = *bp++;
				i = (i+1) % 2;
				
				if (i == 1)
					fwrite(swapbuf, sizeof(char), 2, fp);
			}
			else {
				fputc(*bp++, fp);
			}

			newblen++;
		}

		if (swapbytes) {
			if (i == 0)
				fputc(swapbuf[1], fp);
			i = 1;
		}

		fputc('\n', fp);
		newblen++;
	}

	return(newblen);
}

/*
 * wildcmp - search for filenames by matching wildcards.
 */
wildcmp(wilds, str, wc)
char *wilds, *str;
char wc;
{
	char wcrd;
	register int flag;
	register char *l, *s;

	if (strlen(str) == 0)
		return(-1);

	wcrd = wc & 127;
	flag = 0;

	for (l=wilds,s=str; *l != NULL; ) {
		if (*l == wcrd) {
			if (flag)
				return(0);

			l++;
			flag = 1;
			s = str + strlen(str) - (strlen(wilds) - (l - wilds));
		}
		else {
			if (*l != *s)
				return(*l - *s);

			l++;
			s++;
		}
	}

	if (*s != NULL)
		return(*l - *s);

	return(0);
}

/*
 * pipunblock - unblock PIP format records.
 */
pipunblock(fp, buffer, n)
int n;
FILE *fp;
char *buffer;
{
	char swapbuf[2];
	register char *cp;
	register int i, j, len;

	j = 1;
	for (cp=buffer; cp < (buffer + n); ) {
		for (i=0,len=0; i < 4; i++)
			len = len * 10 + *cp++ - '0';

		for (len -= 4; len > 0; len--) {
			if (swapbytes) {
				swapbuf[j] = *cp++;
				j = (j+1) % 2;
				
				if (j == 1)
					fwrite(swapbuf, sizeof(char), 2, fp);
			}
			else {
				fputc(*cp++, fp);
			}
		}

		if (swapbytes) {
			if (j == 0)
					fputc(swapbuf[1], fp);
			j = 1;
		}

		fputc('\n', fp);

		while ((*cp == '^') && (cp < (buffer + n)))
			cp++;
	}
}

/*
 * pipblock - block PIP format records.
 */
pipblock(fp, buffer, blocksize)
FILE *fp;
char *buffer;
int blocksize;
{
	register int n;
	register char *cp;
	static int nline = 0;
	static char linebuf[512];

	if (blocksize % 512)
		fatal("block size must be a multiple of 512 for PIP tapes.\n");

	for (cp=buffer; cp < (buffer + blocksize); ) {
		if (nline <= 0) {
			if (fgets(linebuf, 508, fp) == NULL) {
				nline = -1;
			}
			else {
				linebuf[508] = NULL;
				nline = strlen(linebuf);
			}
		}

		if ((nline > (508 - (cp - buffer) % 512)) ||
		    ((nline < 0) && ((cp - buffer) % 512 != 0)))
			while (((cp - buffer) % 512) != 0)
			    	*cp++ = '^';

		if (((cp - buffer) >= blocksize) || (nline < 0))
			return(cp - buffer);

		sprintf(cp, "%04d", nline+4);
		cp += 4;

		strncpy(cp, linebuf, 508);
		cp[508] = NULL;			/* strncpy sucks */

		cp += nline;
		nline = 0;
	}

	return(cp - buffer);
}

/*
 * checkmagic - check files for magic numbers and ask what to
 *		do with them.
 */
checkmagic(file, path)
char *file, *path;
{
	int fd;
	char *rindex();
	char *ans, *prompt();
	char buf[1024], fname[1024];

	sprintf(fname, "%s%s%s", path, (*path ? "/" : ""), file);

	if ((fd = open(fname, 0)) < 0)
		fatal("cannot open %s for reading.\n", fname);

	read(fd, buf, sizeof(buf));
	close(fd);

	switch (*(int *) buf) {
	case 0413:			/* "a.out" files. */
	case 0410:
	case 0411:
	case 0407:
		ans = prompt("%s is an executable (a.out) file - copy? ", fname);

		if ((*ans != 'y') && (*ans != NULL))
			return(FALSE);

		return(TRUE);
	case 0177555:				/* archives */
	case 0177545:
		ans = prompt("%s is an old-format archive - copy? ", fname);

		if ((*ans != 'y') && (*ans != NULL))
			return(FALSE);

		return(TRUE);
	case 070707:				/* cpio data */
		ans = prompt("%s is cpio data - copy? ", fname);

		if ((*ans != 'y') && (*ans != NULL))
			return(FALSE);

		return(TRUE);
	case 052525:				/* ECN's PDS stuff */
		ans = prompt("%s is a PDS picture file - copy? ", fname);

		if ((*ans != 'y') && (*ans != NULL))
			return(FALSE);

		return(TRUE);
	case 017437:				/* packed file	*/
		ans = prompt("%s is a packed file -\n    'n' to skip, 'y' to copy, 'u' to unpack and copy? ", fname);

		if ((*ans == 'y') || (*ans == NULL))
			return(TRUE);

		if (*ans == 'u') {
			sprintf(buf, "unpack %s", fname);
			system(buf);

			ans = rindex(file, '.');
			*ans = NULL;
			return(TRUE);
		}

		return(FALSE);
	case 017777:				/* compacted file */
		ans = prompt("%s is a compacted file -\n    'n' to skip, 'y' to copy, 'u' to uncompact and copy? ", fname);

		if ((*ans == 'y') || (*ans == NULL))
			return(TRUE);

		if (*ans == 'u') {
			sprintf(buf, "uncompact %s", fname);
			system(buf);

			ans = rindex(file, '.');
			*ans = NULL;
			return(TRUE);
		}

		return(FALSE);
	}

	/*
	 * Check shorts as well as ints, since ya never know
	 * about these things...
	 */
	switch (*(short *) buf) {
	case 0413:
	case 0410:
	case 0411:
	case 0407:
		ans = prompt("%s is an executable (a.out) file - copy? ", fname);

		if ((*ans != 'y') && (*ans != NULL))
			return(FALSE);

		return(TRUE);
	case 0177555:
	case 0177545:
		ans = prompt("%s is an old-format archive - copy? ", fname);

		if ((*ans != 'y') && (*ans != NULL))
			return(FALSE);

		return(TRUE);
	case 070707:
		ans = prompt("%s is cpio data - copy? ", fname);

		if ((*ans != 'y') && (*ans != NULL))
			return(FALSE);

		return(TRUE);
	case 052525:
		ans = prompt("%s is a PDS picture file - copy? ", fname);

		if ((*ans != 'y') && (*ans != NULL))
			return(FALSE);

		return(TRUE);
	case 017437:
		ans = prompt("%s is a packed file -\n    'n' to skip, 'y' to copy, 'u' to unpack and copy? ", fname);

		if ((*ans == 'y') || (*ans == NULL))
			return(TRUE);

		if (*ans == 'u') {
			sprintf(buf, "unpack %s", fname);
			system(buf);

			ans = rindex(file, '.');
			*ans = NULL;
			return(TRUE);
		}

		return(FALSE);
	case 017777:
		ans = prompt("%s is a compacted file -\n    'n' to skip, 'y' to copy, 'u' to uncompact and copy? ", fname);

		if ((*ans == 'y') || (*ans == NULL))
			return(TRUE);

		if (*ans == 'u') {
			sprintf(buf, "uncompact %s", fname);
			system(buf);

			ans = rindex(file, '.');
			*ans = NULL;
			return(TRUE);
		}

		return(FALSE);
	}

	return(TRUE);
}

/*
 * doswap - swap bytes of buf.
 */
doswap(buf, n)
char *buf;
int n;
{
	char *tmp;

#ifdef DEBUG
	fprintf(stderr, "DEBUG-> doswap()\n");
#endif

	if ((tmp = calloc(1, n+5)) == NULL)
		fatal("not enough memory to swap %d bytes.\n", n);

	swab(buf, tmp, (n % 2 ? n - 1 : n));
	mybcopy(buf, tmp, n);
	free(tmp);
}

addcr(buffer, n)
char *buffer;
register int n;
{
	char *buf;
	register char *s, *t;

	if ((buf = calloc(1, 2 * n + 1)) == NULL)
		fatal("not enough memory for %d byte buffer.\n", 2 * n +1);

	s = buf;
	t = buffer;

	while (*t && (n > 0)) {
		if (*t == '\n')
			*s++ = '\r';
		*s++ = *t++;
		n--;
	}

	*s++ = NULL;
	strcpy(buffer, buf);
	free(buf);

	return(strlen(buffer));
}

delcr(buffer, n)
char *buffer;
register int n;
{
	char *buf;
	register char *s, *t;

	if ((buf = calloc(1, n + 1)) == NULL)
		fatal("not enough memory for %d byte buffer.\n", n+1);

	s = buf;
	t = buffer;

	while (n--) {
		if (*t == '\r') {
			t++;
			continue;
		}

		*s++ = *t++;
	}

	*s = NULL;
	strcpy(buffer, buf);
	free(buf);

	return(strlen(buffer));
}

/*
 * douserlabels - make the UHL1-6 labels for the file in hp.
 */
douserlabels(tf, hp)
int tf;
struct hashent *hp;
{
	char *ctime();
	char buf[UHLA_USRAP + 1];
	register struct uhla_label *uhlap;

	uhlap = &uhla;

	blcopy(uhlap, "", sizeof(struct uhla_label));
	blcopy(uhlap->labid, "UHL", 3);
	uhlap->labno = '1';
	
	blcopy(uhlap->usrap, hp->path, UHLA_USRAP);
	putuhla(tf, uhlap);

	if (strlen(hp->path) > UHLA_USRAP) {
		blcopy(uhlap, "", sizeof(struct uhla_label));
		blcopy(uhlap->labid, "UHL", 3);
		uhlap->labno = '2';

		blcopy(uhlap->usrap, hp->path + UHLA_USRAP, UHLA_USRAP);
		putuhla(tf, uhlap);
	}

	blcopy(uhlap, "", sizeof(struct uhla_label));
	blcopy(uhlap->labid, "UHL", 3);
	uhlap->labno = '3';

	sprintf(buf, "Uid: %d  Gid: %d  Mode: %o  Size: %ld  Inode: %u",
		hp->sbuf.st_uid, hp->sbuf.st_gid, hp->sbuf.st_mode,
		hp->sbuf.st_size, hp->sbuf.st_ino);
	blcopy(uhlap->usrap, buf, UHLA_USRAP);
	putuhla(tf, uhlap);

	blcopy(uhlap, "", sizeof(struct uhla_label));
	blcopy(uhlap->labid, "UHL", 3);
	uhlap->labno = '4';

	sprintf(buf, "Created: %.24s", ctime(&(hp->sbuf.st_ctime)));
	blcopy(uhlap->usrap, buf, UHLA_USRAP);
	putuhla(tf, uhlap);

	blcopy(uhlap, "", sizeof(struct uhla_label));
	blcopy(uhlap->labid, "UHL", 3);
	uhlap->labno = '5';

	sprintf(buf, "Last Modified: %.24s", ctime(&(hp->sbuf.st_mtime)));
	blcopy(uhlap->usrap, buf, UHLA_USRAP);
	putuhla(tf, uhlap);

	blcopy(uhlap, "", sizeof(struct uhla_label));
	blcopy(uhlap->labid, "UHL", 3);
	uhlap->labno = '6';

	sprintf(buf, "Last Accessed: %.24s", ctime(&(hp->sbuf.st_atime)));
	blcopy(uhlap->usrap, buf, UHLA_USRAP);
	putuhla(tf, uhlap);
}

/*
 * putuhla - write a UHL label onto tape.
 */
putuhla(tf, uhlap)
int tf;
struct uhla_label *uhlap;
{
	register int len;

	if (labelsize == sizeof(struct uhla_label)) {
		if (write(tf, (char *) uhlap, sizeof(struct uhla_label)) < 0)
			fatal("tape write error.\n");
		return;
	}

	mybcopy(buffer, (char *) uhlap, sizeof(struct uhla_label));

	len = labelsize - sizeof(struct uhla_label);

	blcopy(&buffer[sizeof(struct uhla_label)], "", len);

	if (write(tf, buffer, labelsize) < 0)
		fatal("tape write error.\n");
}

/*
 * dumpvol1 - dump a VOL1 label
 */
dumpvol1(v)
register struct vol1_label *v;
{
	fprintf(stderr, "----------------------------------------------------------------\n");
	fprintf(stderr, "VOL1 LABEL:\n");
	putlc("Label Identifier: ", v->labid, VOL1_LABID);
	putlc("Label Number: ", v->labno, VOL1_LABNO);
	putlc("Volume Identifier: ", v->volid, VOL1_VOLID);
	putlc("Accessibility: ", v->acces, VOL1_ACCES);
	putlc("Reserved Field: ", v->rsvd1, VOL1_RSVD1);
	putlc("Owner Identifier: ", v->ownid, VOL1_OWNID);
	putlc("Reserved Field: ", v->rsvd2, VOL1_RSVD2);
	putlc("Label-Standard Version: ", v->labsv, VOL1_LABSV);
	fprintf(stderr, "----------------------------------------------------------------\n");
}

/*
 * dumphdr1 - dump a HDR1/EOF1/EOV1 label.
 */
dumphdr1(h)
register struct hdr1_label *h;
{
	fprintf(stderr, "----------------------------------------------------------------\n");
	fprintf(stderr, "HDR1/EOF1/EOV1 LABEL:\n");
	putlc("Label Identifier: ", h->labid, HDR1_LABID);
	putlc("Label Number: ", h->labno, HDR1_LABNO);
	putlc("File Identifier: ", h->filid, HDR1_FILID);
	putlc("File-Set Identifier: ", h->fstid, HDR1_FSTID);
	putlc("File Section Number: ", h->fscno, HDR1_FSCNO);
	putlc("File Sequence Number: ", h->fsqno, HDR1_FSQNO);
	putlc("Generation Number: ", h->genno, HDR1_GENNO);
	putlc("Generation Version Number: ", h->gnvno, HDR1_GNVNO);
	putlc("Creation Date: ", h->crdat, HDR1_CRDAT);
	putlc("Expiration Date: ", h->exdat, HDR1_EXDAT);
	putlc("Accessibility: ", h->acces, HDR1_ACCES);
	putlc("Block Count: ", h->blkct, HDR1_BLKCT);
	putlc("System Code: ", h->syscd, HDR1_SYSCD);
	putlc("Reserved Field: ", h->rsvd1, HDR1_RSVD1);
	fprintf(stderr, "----------------------------------------------------------------\n");
}

/*
 * dumphdr2 - dump a HDR2/EOF2/EOV2 label.
 */
dumphdr2(h)
register struct hdr2_label *h;
{
	fprintf(stderr, "----------------------------------------------------------------\n");
	fprintf(stderr, "HDR2/EOF2/EOV2 LABEL:\n");
	putlc("Label Identifier: ", h->labid, HDR2_LABID);
	putlc("Label Number: ", h->labno, HDR2_LABNO);
	putlc("Record Format: ", h->rcfmt, HDR2_RCFMT);
	putlc("Block Length: ", h->blkln, HDR2_BLKLN);
	putlc("Record Length: ", h->recln, HDR2_RECLN);
	putlc("Reserved Field: ", h->rsvd1, HDR2_RSVD1);
	putlc("Buffer Offset Length: ", h->boffl, HDR2_BOFFL);
	putlc("Reserved Field: ", h->rsvd2, HDR2_RSVD2);
	fprintf(stderr, "----------------------------------------------------------------\n");
}

/*
 * putlc - put a label component on output
 */
putlc(name, comp, n)
register char *name, *comp;
register int n;
{
	fprintf(stderr, "\t%s", name);

	if (n == 1)
		outch((char) comp);
	else
		while (n--)
			outch(*comp++);

	putc('\n', stderr);
}

/*
 * outch - print a character as itself, as "^X", or in octal.
 */
outch(c)
char c;
{
	if ((c >= ' ') && (c <= '~'))
		fprintf(stderr, "'%c' ", c);
	else if ((c < ' ') || (c == 0177))
		fprintf(stderr, "'^%c' ", (c | 0100));
	else
		fprintf(stderr, "'%04o' ", c);
}
