/*++
/* NAME
/*	su-someone 1
/* SUMMARY
/*	use your own password to switch to another user ID
/* SYNOPSIS
/*	su-\fIsomeone\fR [arguments]
/* DESCRIPTION
/*	The su-\fIsomeone\fR command allows a limited group of users to 
/*	work under a given user id (e.g. \fIsomeone\fR) without having know 
/*	its password. Users have to enter \fItheir own\fR password in
/*	order to get the privileges of the \fIsomeone\fR account.
/*
/*	After successful validation the user is given a shell process
/*	(default: /bin/sh) with the appropriate uid, gid and with a sane
/*	environment and umask. Where possible, the shell prompt will reflect
/*	the uid of that process.
/*
/*	Any arguments given to su-\fIsomeone\fR are passed as arguments to the
/*	shell process.
/*
/*	Only authorized users are allowed to use this command; in
/*	order to make cheating a bit more difficult, the table of authorized
/*	users is compiled into the program.
/*
/*	With BSD-like UNIX implementations, the executable file should be
/*	installed set-uid and set-gid to the \fIsomeone\fR account.
/*
/*	With System-V Release 2, the executable should be set-uid root and
/*	the \fIsomeone\fR account name must be wired into the program.
/*
/*	You can give the executable any name you want, but be sure that
/*	only the owner can overwrite it.
/* COMMANDS
/*	/bin/sh, other login shells.
/* FILES
/*	/dev/tty, to read the password.
/*
/*	Shell startup files, but usually no login procedures.
/* ENVIRONMENT VARIABLES
/*	SHELL, PATH, IFS, HOME and PS1 are overwritten.
/* SEE ALSO
/*	su(1)
/* DIAGNOSTICS
/*	The program prints a diagnostic and terminates with a non-zero
/*	exit status if there are problems (authorization, environment
/*	update, shell startup).
/* AUTHOR(S)
/*	Wietse Venema
/*	Eindhoven University of Technology
/*	Department of Mathematics and Computer Science
/*	Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* LAST MODIFICATION
/*	90/05/27 20:54:03
/* VERSION/RELEASE
/*	1.7
/*--*/

 /*
  * This program is in the public domain. You can do anything with it as long
  * as you do not remove the references to the original author.
  */

#ifndef lint
static char sccsid[] = "@(#) su-someone.c 1.7 90/05/27 20:54:03";
#endif

#include <stdio.h>
#include <pwd.h>
#include <varargs.h>

 /*
  * The first purpose of the program is to avoid the security problems that
  * arise when a password has to be known to several people.
  * 
  * The second purpose of the program is to set up a decent environment: shell,
  * umask, path, home and field separators.
  * 
  * The default is to invoke the Bourne shell. If you want to use the shell as
  * specified in the password data base, define USE_HER_SHELL.
  */

#ifdef	USE_HER_SHELL
#define SHELL	(*pwd->pw_shell ? pwd->pw_shell : "/bin/sh")
#else
#define SHELL	"/bin/sh"
#endif

#define	UMASK	022
#define	PATH	"/usr/ucb:/bin:/usr/bin:/usr/local/bin:."
#define	IFS	" \t\n"

 /* Library functions... */

extern struct passwd *getpwuid();
extern struct passwd *getpwnam();
extern char *getpass();
extern char *crypt();
extern void exit();
extern void perror();
extern char *rindex();
extern char *malloc();

 /* Local stuff... */

static char *progname;
static int islegal();
static void newenv();
static void giveup();

int     main(argc, argv)
int     argc;
char  **argv;
{
    struct passwd *pwd;
    char    ps1[BUFSIZ];
    char   *pass;

    progname = *argv;

    /* Validate the user who invoked us. */

    if ((pwd = getpwuid(getuid())) == NULL)
	giveup("I don't know you");
    if (!islegal(pwd->pw_name))
	giveup("You are not authorized to use this program");
    if ((pass = getpass("Enter YOUR OWN password:")) == 0)
	giveup("Can't read your password");
    if (strcmp(pwd->pw_passwd, crypt(pass, pwd->pw_passwd)) != 0)
	giveup("Sorry");

    /*
     * Erase the password now that we do not need it anymore. This avoids a
     * security problem should we ever dump core.
     */

    while (*pass)
	*pass++ = 0;

    /*
     * With BSD-like UNIX, install the program set-uid, set-gid. The program
     * will deduce the desired uid, gid from the effective uid, gid.
     * 
     * With SYSV-like UNIX, install the program set-uid root. The program will
     * deduce the desired uid, gid from a hard-wired name.
     */

#ifndef	NAME					/* use effective uid, gid */
    if ((pwd = getpwuid(geteuid())) == NULL)
	giveup("I don't know her");
#else						/* use hard-wired name */
    if ((pwd = getpwnam(NAME)) == 0)
	giveup("I don't know '%s'", NAME);
#endif

    /* Switch to the desired uid, gid. */

    if (setgid(pwd->pw_gid))
	giveup("setgid(%d) failed", pwd->pw_gid);
    if (setuid(pwd->pw_uid))
	giveup("setuid(%d) failed", pwd->pw_uid);

    /*
     * Set up a sane environment and umask. Abort if we cannot update
     * some critical environment variables.
     */

    (void) umask(UMASK);
    newenv("PATH", PATH);
    newenv("SHELL", SHELL);
    newenv("HOME", pwd->pw_dir);
    newenv("IFS", IFS);

    /*
     * The following is relevant only for /bin/sh and relatives. Attempt to
     * change the prompt to reflect the switched uid.
     */

    (void) sprintf(ps1, "PS1=SU-%s> ", pwd->pw_name);
    (void) putenv(ps1);

    /*
     * Ready to invoke a shell. All arguments given to us are passed on to
     * that shell. 
     */

    if ((argv[0] = rindex(SHELL, '/')) == 0)
	argv[0] = SHELL;
    else
	argv[0]++;				/* skip '/' character */
    (void) execv(SHELL, argv);
    giveup("Can't invoke %s", SHELL);
    /* NOTREACHED */
}

/* islegal - check user is in table of legal names */

static int islegal(logname)
char   *logname;
{
    char  **cpp;
    static char *legalname[] = {
	"wietse",
	0,				/* TERMINATOR */
    };

    for (cpp = legalname; *cpp; cpp++) {
	if (0 == strcmp(logname, *cpp))
	    return (1);
    }
    return (0);
}

/* newenv - update environment variable; abort in case of problems */

static void newenv(name, value)
char   *name;
char   *value;
{
    char   *cp;

    if ((cp = malloc((unsigned) strlen(name) + strlen(value) + 2)) == 0)
	giveup("Out of memory");
    (void) sprintf(cp, "%s=%s", name, value);	/* yes this is gross. so what */
    if (putenv(cp))
	giveup("Can't update %s environment variable", name);
}

/* giveup - print diagnostic on the standard error output and terminate */

/* VARARGS */

static void giveup(va_alist) 
va_dcl
{
    va_list ap;
    char   *fmt;

    (void) fprintf(stderr, "%s: ", progname);
    va_start(ap);
    fmt = va_arg(ap, char *);
    (void) vfprintf(stderr, fmt, ap);
    va_end(ap);
    (void) putc('\n', stderr);
    exit(1);
}
