/*
 * parser.y - parser for the config file
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * October, 1986
 *
 * $Log:	parser.y,v $
 * Revision 1.2  86/12/01  14:35:09  davy
 * Added "timespec" nonterminal to support the "between" command.
 * 
 * Revision 1.1  86/12/01  10:23:32  davy
 * Initial revision
 * 
 */
%{

#ifndef lint
static char *RCSid = "$Header: /usr/harbor/davy/system/spot/RCS/parser.y,v 1.2 86/12/01 14:35:09 davy Exp $";
#endif

#include <syslog.h>
#include <stdio.h>
#include <ctype.h>
#include "defs.h"

extern char *configfile;
static int yylineno = 1;
static FILE *yyfp = NULL;

extern char *deny_msg;
extern char *freeup_msg;
extern char *killidle_msg;
extern char *killmulti_msg;
extern char *killsession_msg;

extern struct denial *deny;
extern struct listhead *commands;
extern struct listhead *namelists;
extern struct listhead *exemptions;

extern int maxusers, cycletime, graceperiod;

static int tstart, tfinish;
static struct listhead *lh;
static struct namelist *last_nl = NULL;

struct listhead *lookuplh();
struct namelist *makenl(), *copynl(), *dofunction();

%}

%union {
	int	ival;
	char	*sval;
	int	*ipval;
	char	**spval;
	struct namelist *nval;
}

%type <nval> namelist names name

%token <ival> NUMBER
%token <ipval> IVARIABLE
%token <spval> SVARIABLE
%token <sval> STRING QSTRING
%token <ival> AND BETWEEN COMMAND DENY EXEMPT FROM FUNCTION SET TERMINALTO

%start file

%%

file	:	/* empty */
	|	file command
	;

command	:	STRING '=' namelist
			{ addlist(&namelists, 0, 0, $1, $3, 0, 0); }
	|	timespec COMMAND NUMBER FROM namelist
			{ addlist(&commands, $2, $3, NULL, $5, tstart, tfinish); }
	|	timespec EXEMPT namelist FROM COMMAND
			{ addlist(&exemptions, $5, 0, NULL, $3, tstart, tfinish); }
	|	timespec DENY namelist FROM namelist
			{ adddeny(&deny, $3, $5, tstart, tfinish); }
	|	timespec TERMINALTO namelist
			{ addlist(&exemptions, CMD_TERMINAL, 0, NULL, $3, tstart, tfinish); }
	|	SET IVARIABLE '=' NUMBER
			{ *$2 = $4; }
	|	SET SVARIABLE '=' QSTRING
			{ *$2 = $4; }
	|	error
			{ exit(1); }
	;

timespec:	/* empty */
		{ tstart = 0000; tfinish = 2359; }
	|	BETWEEN NUMBER AND NUMBER
		{ tstart = $2;
		  tfinish = $4;

		  if (tstart < 0)
		  	tstart = 0;
		  if (tfinish > 2359)
		  	tfinish = 2359;
		}
	;

namelist:	name
			{ $$ = $1; }
	|	'(' names ')'
			{ $$ = $2; }
	;

names	:	/* empty */
			{ $$ = last_nl = NULL; }
	|	names name
			{ if (last_nl == NULL) {
				last_nl = $2;

				while (last_nl->nl_next != NULL)
					last_nl = last_nl->nl_next;

				$$ = $2;
			  }
			  else {
				last_nl->nl_next = $2;

				while (last_nl->nl_next != NULL)
					last_nl = last_nl->nl_next;

			  	$$ = $1;
			  }
			}
	;

name	:	FUNCTION '(' STRING ')'
			{ $$ = dofunction($1, $3); }
	|	'$' STRING
			{ if ((lh = lookuplh(namelists, $2)) == NULL)
			  	$$ = NULL;
			  else
			  	$$ = copynl(lh->lh_namelist);
			}
	|	STRING
			{ $$ = makenl($1); }
	|	NUMBER
			{ $$ = makenl(sprintf(malloc(16), "%d", $1)); }
	;

%%

/*
 * yylex - lexical analayzer
 */
yylex()
{
	char *savestr();
	register char *s;
	register int c, n;
	char buf[4 * BUFSIZ];

	/*
	 * Open the file if needed.
	 */
	if (yyfp == NULL) {
		if ((yyfp = fopen(configfile, "r")) == NULL) {
			message("cannot open \"%s\" for reading - exiting.\n", configfile);
			syslog(LOG_ERR, "cannot open \"%s\" for reading.", configfile);
			exit(1);
		}
	}

	/*
	 * See what we got.
	 */
again:
	switch (c = getc(yyfp)) {
	case EOF:			/* end of file			*/
		return(0);
	case '\"':			/* string in quotes		*/
		s = buf;
		while (((c = getc(yyfp)) != EOF) && (c != '\"'))
			*s++ = c;
		*s = NULL;

		if (c == EOF)
			return(0);

		yylval.sval = savestr(buf);
		return(QSTRING);
	case '#':			/* comment			*/
		while (((c = getc(yyfp)) != EOF) && (c != '\n'))
			;
		if (c == EOF)
			return(0);
		/* fall through */
	case '\n':
		yylineno++;
		/* fall through */
	case '\t':
	case ' ':
		goto again;
	case '=':
	case '$':
	case '(':
	case ')':
		return(c);
	}

	/*
	 * Number.
	 */
	if (isdigit(c)) {
		n = 0;

		do {
			n = n * 10 + c - '0';
			c = getc(yyfp);
		} while (isdigit(c));

		ungetc(c, yyfp);
		yylval.ival = n;

		return(NUMBER);
	}

	/*
	 * Variable or string.
	 */
	if (isalpha(c) || (c == '_')) {
		s = buf;

		/*
		 * Gobble up the string.
		 */
		do {
			*s++ = c;
			c = getc(yyfp);
		} while (isalpha(c) || isdigit(c) || (c == '_'));

		*s = NULL;
		ungetc(c, yyfp);

		/*
		 * See if it's some kind of keyword.
		 */
		if (!strcmp(buf, "freeup")) {
			yylval.ival = CMD_FREEUP; return(COMMAND);
		}
		if (!strcmp(buf, "killidle")) {
			yylval.ival = CMD_KILLIDLE; return(COMMAND);
		}
		if (!strcmp(buf, "killmulti")) {
			yylval.ival = CMD_KILLMULTI; return(COMMAND);
		}
		if (!strcmp(buf, "killsession")) {
			yylval.ival = CMD_KILLSESSION; return(COMMAND);
		}

		if (!strcmp(buf, "group")) {
			yylval.ival = FUN_GROUP; return(FUNCTION);
		}
		if (!strcmp(buf, "class")) {
			yylval.ival = FUN_CLASS; return(FUNCTION);
		}
		if (!strcmp(buf, "school")) {
			yylval.ival = FUN_SCHOOL; return(FUNCTION);
		}

		if (!strcmp(buf, "cycle")) {
			yylval.ipval = &cycletime; return(IVARIABLE);
		}
		if (!strcmp(buf, "grace")) {
			yylval.ipval = &graceperiod; return(IVARIABLE);
		}
		if (!strcmp(buf, "maxusers")) {
			yylval.ipval = &maxusers; return(IVARIABLE);
		}
		if (!strcmp(buf, "deny_msg")) {
			yylval.spval = &deny_msg; return(SVARIABLE);
		}
		if (!strcmp(buf, "freeup_msg")) {
			yylval.spval = &freeup_msg; return(SVARIABLE);
		}
		if (!strcmp(buf, "killidle_msg")) {
			yylval.spval = &killidle_msg; return(SVARIABLE);
		}
		if (!strcmp(buf, "killmulti_msg")) {
			yylval.spval = &killmulti_msg; return(SVARIABLE);
		}
		if (!strcmp(buf, "killsession_msg")) {
			yylval.spval = &killsession_msg; return(SVARIABLE);
		}

		if (!strcmp(buf, "and"))
			return(AND);
		if (!strcmp(buf, "set"))
			return(SET);
		if (!strcmp(buf, "deny"))
			return(DENY);
		if (!strcmp(buf, "from"))
			return(FROM);
		if (!strcmp(buf, "exempt"))
			return(EXEMPT);
		if (!strcmp(buf, "between"))
			return(BETWEEN);
		if (!strcmp(buf, "terminalto"))
			return(TERMINALTO);

		/*
		 * Nope, just return it.
		 */
		yylval.sval = savestr(buf);
		return(STRING);
	}

	return(-1);
}

/*
 * yyerror - print parser errors.
 */
yyerror(s)
char *s;
{
	char buf[BUFSIZ];

	sprintf(buf, "%s: line %d: %s - exiting.\n", configfile, yylineno, s);
	message(buf, 0);

	syslog(LOG_ERR, "%s: line %d: %s", configfile, yylineno, s);
}
