#ifndef lint
static char *RCSid = "$Header: /usr/harbor/davy/system/spot/RCS/watchdog.c,v 1.2 86/12/01 14:34:58 davy Exp $";
#endif

/*
 * watchdog.c - the watchdog functions
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * October, 1986
 *
 * $Log:	watchdog.c,v $
 * Revision 1.2  86/12/01  14:34:58  davy
 * Added code to support "between" command.
 * 
 * Revision 1.1  86/12/01  10:23:11  davy
 * Initial revision
 * 
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <syslog.h>
#include <stdio.h>
#include <utmp.h>
#include "defs.h"

int curtime;

extern int nvictims;
extern struct victim *victims;

int nutmp;
extern int ud;
struct utmp **putmp;
extern struct utmp *utmp;

extern struct denial *deny;
extern struct listhead *commands;
extern struct listhead *exemptions;
extern int maxusers, cycletime, graceperiod;

struct utmp *findutmptty();

/*
 * watchdog - do our watchdog stuff
 */
watchdog()
{
	union wait status;

#ifdef DEBUG
	printlists();
#endif

	trace("watchdog()\n", 0, 0);

	/*
	 * Make sure these are right.
	 */
	if (cycletime < (2 * graceperiod)) {
		message("cycle must be greater than grace... setting to %d.\n", 2 * graceperiod, 0);
		cycletime = 2 * graceperiod;
	}

	/*
	 * Allocate victims array.
	 */
	if ((victims = (struct victim *) malloc(maxusers * sizeof(struct victim))) == NULL) {
		message("ran out of memory allocating victims array - exiting.\n", 0, 0);
		syslog(LOG_ERR, "ran out of memory allocating victims array.");
		exit(1);
	}

	/*
	 * Allocate pointers for the utmp structures.
	 */
	if ((putmp = (struct utmp **) malloc(maxusers * sizeof(struct utmp *))) == NULL) {
		message("ran out of memory allocating putmp - exiting.\n", 0, 0);
		syslog(LOG_ERR, "ran out of memory allocating putmp.");
		exit(1);
	}

	/*
	 * Forever...
	 */
	for (;;) {
		/*
		 * Read in the utmp file.
		 */
		lseek(ud, 0L, 0);
		trace("reading in utmp\n", 0, 0);
		bzero(utmp, maxusers * sizeof(struct utmp));

		if ((nutmp = read(ud, utmp, maxusers * sizeof(struct utmp))) < 0) {
			message("read error on \"%s\" - exiting.\n", UTMPFILE, 0);
			syslog(LOG_ERR, "read error on \"%s\" (%m).", UTMPFILE);
			exit(1);
		}

		nutmp /= sizeof(struct utmp);
		nvictims = 0;

		curtime = gettime();

		/*
		 * Do each command.
		 */
		dodeny();
		freeup();
		killidle();
		killmulti();
		killsession();

		/*
		 * Warn the victims, sleep for the grace period,
		 * log off the victims, sleep for the remainder
		 * of our cycle time.
		 */
		if (nvictims) {
			warnvictims();
			sleep(graceperiod * 60);

			killvictims();
			sleep((cycletime - graceperiod) * 60);

			/*
			 * Pick up our children.
			 */
			while (wait3(&status, WNOHANG, 0) > 0)
				;
		}
		else {
			sleep(cycletime * 60);
		}
	}
}

/*
 * dodeny - deny people access to terminals
 */
dodeny()
{
	struct denial *dn;
	register int i, nttys;
	register struct namelist *nl;

	trace("dodeny()\n", 0, 0);

	/*
	 * For each deny command...
	 */
	for (dn = deny; dn != NULL; dn = dn->dn_next) {
		/*
		 * If not time for this command, skip it.
		 */
		if (!between(curtime, dn->dn_start, dn->dn_finish))
			continue;

		nttys = 0;

		/*
		 * Get the utmp structures for the listed terminals.
		 */
		for (nl = dn->dn_terminals; nl != NULL; nl = nl->nl_next) {
			if ((putmp[nttys] = findutmptty(nl->nl_name)) != NULL)
				nttys++;
		}

		/*
		 * For each of those terminals...
		 */
		for (i=0; i < nttys; i++) {
			/*
			 * If one of the listed users is logged in on
			 * it, he's gonna get wasted.
			 */
			for (nl = dn->dn_users; nl != NULL; nl = nl->nl_next) {
				if (strncmp(putmp[i]->ut_name, nl->nl_name, UTNAMESIZE) != 0)
					continue;

				message("deny: logging out user \"%.8s\" from \"%.8s\".\n", putmp[i]->ut_name, putmp[i]->ut_line);

				strncpy(victims[nvictims].vi_name, putmp[i]->ut_name, UTNAMESIZE);
				victims[nvictims].vi_command = CMD_DENY;
				victims[nvictims].vi_utmp = putmp[i];
				victims[nvictims].vi_parameter = 0;
				putmp[i]->ut_name[0] = '\0';
				nvictims++;
			}
		}
	}
}
		
/*
 * freeup - free up terminals in use by other departments' users
 */
freeup()
{
	struct listhead *lh;
	extern int timecmp();
	register struct namelist *nl;
	register int i, nttys, nfree;

	trace("freeup()\n", 0, 0);

	/*
	 * For each freeup command...
	 */
	for (lh = commands; lh != NULL; lh = lh->lh_next) {
		if (lh->lh_command != CMD_FREEUP)
			continue;

		/*
		 * If not time for this command, skip it.
		 */
		if (!between(curtime, lh->lh_start, lh->lh_finish))
			continue;

		nttys = 0;
		nfree = 0;

		/*
		 * Find the utmp structures for all the ttys listed.
		 */
		for (nl = lh->lh_namelist; nl != NULL; nl = nl->nl_next) {
			if ((putmp[nttys] = findutmptty(nl->nl_name)) != NULL)
				nttys++;
		}

		/*
		 * Count how many aren't logged in.
		 */
		for (i=0; i < nttys; i++) {
			if (putmp[i]->ut_name[0] == '\0')
				nfree++;
		}

		/*
		 * If that's enough, we don't have to log anyone
		 * off.
		 */
		if (nfree >= lh->lh_parameter)
			continue;

		/*
		 * Sort the ttys by login time, longest logins will
		 * be first.
		 */
		qsort(putmp, nttys, sizeof(struct utmp *), timecmp);

		/*
		 * Until we've freed up enough terminals or run out
		 * of logged in terminals...
		 */
		for (i=0; (i < nttys) && (nfree < lh->lh_parameter); i++) {
			/*
			 * This one's free already.
			 */
			if (putmp[i]->ut_name[0] == '\0')
				continue;

			/*
			 * If this is the TERMINAL login, see what machine
			 * it's going to and check that for grounds to log
			 * them off.
			 */
			if (!strncmp(putmp[i]->ut_name, TERMINAL_LOGIN, UTNAMESIZE)) {
				if (!exempt2(putmp[i])) {
					message("freeup: logging out \"terminal\" to \"%.16s\" from \"%.8s\".\n", putmp[i]->ut_host, putmp[i]->ut_line);

					strncpy(victims[nvictims].vi_name, putmp[i]->ut_name, UTNAMESIZE);
					victims[nvictims].vi_parameter = lh->lh_parameter;
					victims[nvictims].vi_command = CMD_FREEUP;
					victims[nvictims].vi_utmp = putmp[i];
					putmp[i]->ut_name[0] = '\0';
					nvictims++;
					nfree++;

					continue;
				}
			}

			/*
			 * Otherwise, check their exemptions.
			 */
			if (!exempt(CMD_FREEUP, putmp[i]->ut_name)) {
				message("freeup: logging out user \"%.8s\" from \"%.8s\".\n", putmp[i]->ut_name, putmp[i]->ut_line);

				strncpy(victims[nvictims].vi_name, putmp[i]->ut_name, UTNAMESIZE);
				victims[nvictims].vi_parameter = lh->lh_parameter;
				victims[nvictims].vi_command = CMD_FREEUP;
				victims[nvictims].vi_utmp = putmp[i];
				putmp[i]->ut_name[0] = '\0';
				nvictims++;
				nfree++;
			}
		}
	}
}

/*
 * killidle - log off users idle for more than a given amount of time
 */
killidle()
{
	long now;
	char buf[64];
	struct stat sbuf;
	struct listhead *lh;
	register int i, nttys;
	register struct namelist *nl;

	trace("killidle()\n", 0, 0);

	time(&now);

	/*
	 * For each killidle command...
	 */
	for (lh = commands; lh != NULL; lh = lh->lh_next) {
		if (lh->lh_command != CMD_KILLIDLE)
			continue;

		/*
		 * If not time for this command, skip it.
		 */
		if (!between(curtime, lh->lh_start, lh->lh_finish))
			continue;

		nttys = 0;

		/*
		 * Find the utmp structures for this list of ttys.
		 */
		for (nl = lh->lh_namelist; nl != NULL; nl = nl->nl_next) {
			if ((putmp[nttys] = findutmptty(nl->nl_name)) != NULL)
				nttys++;
		}

		/*
		 * For each of those ttys...
		 */
		for (i=0; i < nttys; i++) {
			/*
			 * Not logged in, skip it.
			 */
			if (putmp[i]->ut_name[0] == '\0')
				continue;

			/*
			 * Construct the path to the device and stat it.
			 */
			sprintf(buf, "/dev/%.*s", UTLINESIZE, putmp[i]->ut_line);

			if (stat(buf, &sbuf) < 0)
				continue;

#ifdef DEBUG
			trace("%s idle for %d minutes.\n", buf, (now - sbuf.st_atime) / 60);
#endif

			/*
			 * If he's been idle for less than the parameter,
			 * skip him.
			 */
			if (((now - sbuf.st_atime) / 60) < lh->lh_parameter)
				continue;

			/*
			 * If he's not exempt from this command, log
			 * him off.
			 */
			if (!exempt(CMD_KILLIDLE, putmp[i]->ut_name)) {
				message("killidle: logging out user \"%.8s\" from \"%.8s\".\n", putmp[i]->ut_name, putmp[i]->ut_line);

				strncpy(victims[nvictims].vi_name, putmp[i]->ut_name, UTNAMESIZE);
				victims[nvictims].vi_parameter = lh->lh_parameter;
				victims[nvictims].vi_command = CMD_KILLIDLE;
				victims[nvictims].vi_utmp = putmp[i];
				putmp[i]->ut_name[0] = '\0';
				nvictims++;
			}
		}
	}
}

/*
 * killsession - kill users logged in for more than a given amount of time.
 */
killsession()
{
	long now;
	struct listhead *lh;
	register int i, nttys;
	register struct namelist *nl;

	trace("killsession()\n", 0, 0);

	time(&now);

	/*
	 * For each killsession command...
	 */
	for (lh = commands; lh != NULL; lh = lh->lh_next) {
		if (lh->lh_command != CMD_KILLSESSION)
			continue;

		/*
		 * If not time for this command, skip it.
		 */
		if (!between(curtime, lh->lh_start, lh->lh_finish))
			continue;

		nttys = 0;

		/*
		 * Find the utmp structures for the listed ttys.
		 */
		for (nl = lh->lh_namelist; nl != NULL; nl = nl->nl_next) {
			if ((putmp[nttys] = findutmptty(nl->nl_name)) != NULL)
				nttys++;
		}

		/*
		 * For each of those ttys...
		 */
		for (i=0; i < nttys; i++) {
			/*
			 * Not logged in, never mind...
			 */
			if (putmp[i]->ut_name[0] == '\0')
				continue;

#ifdef DEBUG
			trace("%.8s logged on for %d minutes.\n", putmp[i]->ut_line, (now - putmp[i]->ut_time) / 60);
#endif

			/*
			 * If not logged in long enough yet, skip him.
			 */
			if (((now - putmp[i]->ut_time) / 60) < lh->lh_parameter)
				continue;

			/*
			 * If he's not exempt from killsessions, then
			 * log him off.
			 */
			if (!exempt(CMD_KILLSESSION, putmp[i]->ut_name)) {
				message("killsession: logging out user \"%.8s\" from \"%.8s\".\n", putmp[i]->ut_name, putmp[i]->ut_line);

				strncpy(victims[nvictims].vi_name, putmp[i]->ut_name, UTNAMESIZE);
				victims[nvictims].vi_parameter = lh->lh_parameter;
				victims[nvictims].vi_command = CMD_KILLSESSION;
				victims[nvictims].vi_utmp = putmp[i];
				putmp[i]->ut_name[0] = '\0';
				nvictims++;
			}
		}
	}
}

/*
 * killmulti - kill off users logged in on multiple terminals.
 */
killmulti()
{
	struct listhead *lh;
	extern int namecmp();
	char name[UTNAMESIZE+1];
	register int i, nttys, nlog;
	register struct namelist *nl;

	trace("killmulti()\n", 0, 0);

	/*
	 * For each killmulti command...
	 */
	for (lh = commands; lh != NULL; lh = lh->lh_next) {
		if (lh->lh_command != CMD_KILLMULTI)
			continue;

		/*
		 * If not time for this command, skip it.
		 */
		if (!between(curtime, lh->lh_start, lh->lh_finish))
			continue;

		nttys = 0;

		/*
		 * Find the utmp structures for these ttys.
		 */
		for (nl = lh->lh_namelist; nl != NULL; nl = nl->nl_next) {
			if ((putmp[nttys] = findutmptty(nl->nl_name)) != NULL) {
				if (putmp[nttys]->ut_name[0] != '\0')
					nttys++;
			}
		}

		if (nttys == 0)
			continue;

		/*
		 * Sort by login name.
		 */
		qsort(putmp, nttys, sizeof(struct utmp *), namecmp);

		/*
		 * Save first name.
		 */
		strncpy(name, putmp[0]->ut_name, UTNAMESIZE);
		name[UTNAMESIZE] = '\0';
		nlog = 1;

		/*
		 * Count how many times each guy is logged in.
		 */
		for (i=1; i < nttys; i++) {
			if (!strncmp(name, putmp[i]->ut_name, UTNAMESIZE)) {
				/*
				 * If too many and he's not exempt, log
				 * him off.
				 */
				if (++nlog > lh->lh_parameter) {
					if (!exempt(CMD_KILLMULTI, putmp[i]->ut_name)) {
						message("killmulti: logging out user \"%.8s\" from \"%.8s\".\n", putmp[i]->ut_name, putmp[i]->ut_line);

						strncpy(victims[nvictims].vi_name, putmp[i]->ut_name, UTNAMESIZE);
						victims[nvictims].vi_parameter = lh->lh_parameter;
						victims[nvictims].vi_command = CMD_KILLMULTI;
						victims[nvictims].vi_utmp = putmp[i];
						putmp[i]->ut_name[0] = '\0';
						nvictims++;
					}
				}
			}
			else {
				strncpy(name, putmp[i]->ut_name, UTNAMESIZE);
				name[UTNAMESIZE] = '\0';
				nlog = 1;
			}
		}
	}
}

/*
 * exempt - check the exemptions lists and see if a user is exempt from
 *	    the given command.
 */
exempt(cmd, name)
register int cmd;
register char *name;
{
	register struct listhead *lh;
	register struct namelist *nl;

	for (lh = exemptions; lh != NULL; lh = lh->lh_next) {
		if (lh->lh_command != cmd)
			continue;

		/*
		 * If not time for this exemption, skip it.
		 */
		if (!between(curtime, lh->lh_start, lh->lh_finish))
			continue;

		for (nl = lh->lh_namelist; nl != NULL; nl = nl->nl_next) {
			if (!strncmp(nl->nl_name, name, UTNAMESIZE))
				return(1);
		}
	}

	return(0);
}

/*
 * exempt2 - check if the TERMINAL login is going to an allowable machine.
 */
exempt2(ut)
struct utmp *ut;
{
	register char *s, *t;
	char host[UTHOSTSIZE+1];
	register struct listhead *lh;
	register struct namelist *nl;

	s = ut->ut_host;
	t = host;

	/*
	 * TERMINAL puts "user->host" in the ut_host field.  Get
	 * the host part.
	 */
	while ((*s != '>') && (*s != NULL) && (s < &(ut->ut_host[UTHOSTSIZE])))
		s++;

	if (*s == '>') {
		while ((++s < &(ut->ut_host[UTHOSTSIZE])) && (*s != NULL))
			*t++ = *s;
	}

	*t = NULL;

	/*
	 * See if host is allowed.
	 */
	for (lh = exemptions; lh != NULL; lh = lh->lh_next) {
		if (lh->lh_command != CMD_TERMINAL)
			continue;

		/*
		 * If not time for this exemption, skip it.
		 */
		if (!between(curtime, lh->lh_start, lh->lh_finish))
			continue;

		for (nl = lh->lh_namelist; nl != NULL; nl = nl->nl_next) {
			if (!strcmp(nl->nl_name, host))
				return(1);
		}
	}

	return(0);
}

/*
 * findutmptty - return pointer to utmp structure for tty.
 */
struct utmp *findutmptty(tty)
register char *tty;
{
	register struct utmp *ut;

	for (ut = utmp; ut < &utmp[nutmp]; ut++) {
		if (!strncmp(tty, ut->ut_line, UTLINESIZE))
			return(ut);
	}

	return(NULL);
}

