#define BUFSIZE 1024
#define LINESIZ 133
#define MAXARGS 256
#define FALSE 0
#define TRUE 1
#define	eprintf(s)	fprintf(stderr, s, args[0], sys_errlist[errno])

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>

int	skt, errskt;
FILE	*inskt, *outskt;
extern	int errno;
char	*hostname;
int	catdvi = FALSE;
int	troff = FALSE;
char	**args;
char	*strcpy(), *strcat();
int	pid1, pid2;
extern	char *sys_errlist[];

main(argc, argv)
	int argc;
	char *argv[];
{
	int	uid, needstdin = FALSE;
	char	tcmd[BUFSIZE];
	char 	*fargv[MAXARGS];
	int	reapchild();
	struct	servent	*serv, rserv;

	(void) signal(SIGCHLD, reapchild);

	args = argv;
	needstdin = parseargs(argc, argv, fargv, tcmd);
	if ((serv = getservbyname("shell", "tcp")) == NULL) {
		eprintf("%s: getservbyname: %s\n");
		exit(1);
	}
	rserv = *serv;	/* copy data to avoid loss on next getservent() */
	if (hostname != NULL) {
		if ((skt = rcmd(&hostname, rserv.s_port, "rtroff",
		    "rtroff", tcmd, &errskt)) < 0) {
			eprintf("%s: rcmd: %s\n");
			exit(1);
		}
	}
	else  {
		/*
		 * If the user has not specified a host name, then
		 * we use gethost() to get the host with the least
		 * load.  Ideally, we should traverse the whole
		 * list in order if the first machine doesn't respond,
		 * but it should be pretty safe not to bother (after
		 * all, it just responded to our load request a second
		 * ago...) 
		 */
		struct	hostent *fhent, sfhent, *gethost();
		char	hname[512];

		fhent = gethost("/etc/rtrhosts");
		if (fhent == NULL) {
			fprintf(stderr,"%s: remote host unknown\n", argv[0]);
			exit (1);
		}
		/*
		 * Now we look at the host entry for the ideal
		 * host, and compare it to the host entry for the
		 * current host.  If the addresses are the same,
		 * then we make life easy and execv() things off
		 * locally.
		 */
		(void) strcpy(hname, fhent->h_name); /* static storage */
		sfhent = *fhent;
		sfhent.h_name = hname;
		if (islocalhost(&sfhent))
			dolocalexec();
#ifdef DEBUG
		fprintf(stderr,"host is %s, port is %d\n", sfhent.h_name,
		    ntohs(rserv.s_port));
#endif
		if ((skt = rcmd(&sfhent.h_name, rserv.s_port, "rtroff",
		    "rtroff", tcmd, &errskt)) < 0) {
			eprintf("%s: rcmd: %s\n");
			fprintf(stderr, "Host %s down - try again later\n",
			    sfhent.h_name);
			exit(1);
		}
	}
	uid = getuid();
	(void) setreuid(uid, uid);	/* become the real user */
	/*
	 * Now, life gets interesting.  We have to fork off
	 * some processes to handle sending and receiving
	 * info off all these sockets.  The parent will read
	 * the stderr channel; one child will be responsible
	 * for reading data from the input files and sending
	 * it to the remote troff, and another will be responsible
	 * for reading the results of the troff off the other
	 * side and sending them to stdout.
	 */

	if ((pid1 = fork()) < 0) {
		eprintf("%s: fork: %s\n");
		exit(1);
	}
	if (pid1 == 0) {
		/*
		 * Then we are child #1 and will be sending
		 * input to the other side.
		 */
		if ((outskt = fdopen(skt, "w")) == NULL) {
			eprintf("%s: fdopen for write: %s\n");
			exit(9);
		}
		sendfiles(fargv);
		if (needstdin)
			sendonefile(stdin);
		(void) fflush(outskt);
		(void) shutdown(skt, 1); /* ain't gonna send data no more */
		(void) fclose(outskt);
#ifdef DEBUG
	fprintf(stderr, "rtroff: send process finished normally\n");
#endif
		exit(0);
	}
	/*
	 * Parent.
	 */
	if ((pid2 = fork()) < 0) {
		eprintf("%s: fork: %s\n");
		(void) kill(pid1, SIGTERM);
		exit(1);
	}
	if (pid2 == 0) {
		/*
		 * Then we are the second child and will be reading
		 * results from the other side.
		 */
		if ((inskt = fdopen(skt, "r")) == NULL) {
			eprintf("%s: fdopen for read: %s\n");
			(void) kill(pid1, SIGTERM);
			exit(9);
		}
		getresults(inskt);
		(void) fclose(inskt);
#ifdef DEBUG
	fprintf(stderr, "rtroff: recv process finished normally\n");
#endif
		exit(0);
	}
	/*
	 * Parent again -- read errors and wait for kids to exit.
	 */
	geterrsandwait(errskt);
	(void) close(errskt);
	exit(0);
}

/*
 * Is string "small" contained somewhere within "big"?  If so, return
 * a pointer to the first occurrence.
 */
char *
sindex(big, small)
	char *big;
	register char *small;
{
	register char *cp;
	int len = strlen(small);

	for (cp = big; *cp; cp++)
		if (*cp == *small && strncmp(cp, small, len) == 0)
			return (cp);
	return (NULL);
}

dolocalexec()
{
	int uid;

	if (catdvi)	/* I don't want to set up all these pipes... */
		return;
#ifdef DEBUG
	fprintf(stderr, "Doing local exec\n");
#endif
	uid = getuid();
	(void) setreuid(uid, uid);
	if (troff)
		execv("/usr/bin/troff", args);
	else
		execv("/usr/bin/nroff", args);
	eprintf("%s: exec: %s\n");
	exit(1);
}

int
parseargs(argc, argv, fargv, tcmd)
	int argc;
	char *argv[];
	char *fargv[];
	char *tcmd;
{
	int cnt = 1;
	int tostdout = FALSE;
	int needstdin = FALSE;
	int fcnt = 0;
	char *index();

	if (sindex(argv[0], "troff") != NULL) {
		(void) strcpy(tcmd, "/usr/bin/troff ");
		troff++;
	}
	else
		(void) strcpy(tcmd, "/usr/bin/nroff ");
	hostname = NULL;
	cnt = 1;
	while (cnt < argc) {
		char *arg;

		if (*argv[cnt] == '-') {
			arg = argv[cnt];
			arg++;
			switch (*arg) {
			case '\0':
				needstdin = TRUE;
				break; /* don't copy this arg, since troff
					  doesn't handle options after '-'
					  properly.  It will be put on the
					  end of the argument list later. */
			case 's':
			case 'i':
				needstdin = TRUE;
				(void) strcat(tcmd, argv[cnt]);
				(void) strcat(tcmd, " ");
				break;
			case 't':
			case 'a':
				tostdout = TRUE;
				(void) strcat(tcmd, argv[cnt]);
				(void) strcat(tcmd, " ");
				break;
			case 'h':
				hostname = argv[cnt+1];
				cnt++;
				break;
			case 'c':
				catdvi++;
				break;
			default:
				(void) strcat(tcmd, argv[cnt]);
				(void) strcat(tcmd, " ");
			}
			cnt++;
		}
		else { /* must be a file name */
			fargv[fcnt++] = argv[cnt];
			cnt++;
		}
	}
	if (tostdout == FALSE && sindex(tcmd, "nroff") == 0) /* not nroff */
		(void) strcat(tcmd, "-t "); /* force to stdout */
	(void) strcat(tcmd, "-");  /* must act as if reading only from stdin */
	fargv[fcnt] = 0; /* terminate lists */
	if (catdvi && troff)	/* invoke catdvi and queue it */
		(void) strcat(tcmd, " | /usr/local/bin/catdvi -a -b stdin | qpr -q imagen-imp");

	return (needstdin);
}


sendfiles(fargv)
	char *fargv[];
{
	int cnt = 0;

	if (fargv[0] == 0) {
		sendonefile(stdin);
		return;
	}
	while (fargv[cnt] != 0) {
		FILE *ftosend;
		
		if ((ftosend = fopen(fargv[cnt], "r")) == NULL) {
			fprintf(stderr, "%s: could not open file %s: %s\n",
				args[0], fargv[cnt], sys_errlist[errno]);
			(void) shutdown(skt, 1); /* will eventually SIGPIPE */
			exit(-1);
		}
		else {
			sendonefile(ftosend);
			cnt++;
		}
	}
}

/*
 * Send a single stdio file to the server, handling any .so-type
 * indirections.  N.B.: when sendonefile returns, fd has been
 * fclose()ed.
 */
sendonefile(fd)
	register FILE *fd;
{
	char buf[BUFSIZ];
#define TCMD(s) \
	((buf[0] == '.' || buf[0] == '\'') && strncmp(&buf[1], s, 2) == 0)

	while(fgets(buf, BUFSIZ, fd) != NULL) {
		if (TCMD("so"))	/* include .so file in stream */
			getso(buf);
		else if (TCMD("nx")) {	/* send .nx file */
			getso(buf);
			(void) fclose(fd);
			return;
		}
		else if (TCMD("rd"))	/* handle .rd instruction */
			getrd(buf);
		else
			fputs(buf, outskt);
	}
	(void) fclose(fd);
	return;
}

/*
 * Extracts an nroff/troff style name from a .so, .nx, or .rd command.
 */
getfname(cmd, name)
	register char *cmd, *name;
{

	cmd += 3;		/* skip .so, .nx, whatever */
	while (*cmd == ' ' || *cmd == '\t')
		cmd++;
	while (*cmd && *cmd != ' ' && *cmd != '\t' && *cmd != '\n')
		*name++ = *cmd++;
	*name = '\0';
}

getso(buf)
	char *buf;
{
	char sobuf[LINESIZ];
	FILE *sofd;

	getfname(buf, sobuf);
	if ((sofd = fopen(sobuf, "r")) == NULL) {
		fprintf(stderr, "%s: cannot open %s: %s\n", args[0], sobuf,
		    sys_errlist[errno]);
		(void) shutdown(skt, 1); /* eventually cause SIGPIPE */
		exit(-1);
	}
	sendonefile(sofd);
}

getrd(buf)
	char *buf;
{
	char prompt[LINESIZ];
	register int ichr;
	char chr;

	getfname(buf, prompt);
	fprintf(stderr, "%s", prompt);
	while ((ichr = getchar()) != EOF) {
		if (ichr == '\n') {
			fprintf(stderr, "%s", prompt);
			ichr = getchar();
			if (ichr == '\n' || ichr == EOF)
				break;
			(void) write(skt, "\n", 1);
		}
		chr = ichr;
		(void) write(skt, &chr, 1);
	}
	/* should really pass the end of the line on here */
}

getresults(fd)
	register FILE *fd;
{
	register int nread;
	char buf[BUFSIZE];

	while ((nread = fread(buf, sizeof(char), sizeof(buf), fd)) > 0) {
#ifdef DEBUG
	fprintf(stderr, "getresults: read %d\n", nread);
#endif
		if (fwrite(buf, sizeof(char), nread, stdout) != nread) {
			eprintf("%s: write of results: %s\n");
			(void) kill(pid1,SIGTERM);
			exit (23);
		}
	}
}

geterrsandwait(eskt)
	register int eskt;
{
	register int nread;
	char buf[BUFSIZE];
	union wait status;

	while ((nread = read(eskt, buf, sizeof(buf))) != 0) {
		if (nread < 0)
			if (errno == EINTR) /* SIGCHLD got us */
				continue;
			else
				break;
		(void) write(2, buf, nread);
	}
	/*
	 * Now we wait for the kids to finish up.
	 */
	while (wait(&status) >=0)
#ifdef DEBUG
	fprintf(stderr, "rtroff: wait: got one kid\n");
#else
		;
#endif
}

reapchild()
{
	struct rusage rusage;
	union wait status;

	while (wait3(&status, WNOHANG, &rusage) > 0)
#ifdef DEBUG
	{
		fprintf(stderr, "rtroff: wait3: got one kid\n");
		fprintf(stderr, "resource usage:\n");
		fprintf(stderr, "  sys time: %d sec, %d usec\n",
		    rusage.ru_stime.tv_sec, rusage.ru_stime.tv_usec);
		fprintf(stderr, "  user time: %d sec, %d usec\n",
		    rusage.ru_utime.tv_sec, rusage.ru_utime.tv_usec);
	}
#else
		;
#endif
}
