#ifndef lint
static char *RCSid = "$Header$";
#endif

/*
 * resh - restricted shell
 *
 * Most of the code here was ripped off from the Version 6 UNIX shell.
 * It has been recoded for Berkeley UNIX, and to reorganize a little
 * it, but the essential workings are the same as in the original.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * davy@intrepid.ecn.purdue.edu
 * August, 1987
 *
 * $Log$
 */
#include <sys/param.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <setjmp.h>
#include <signal.h>
#include <errno.h>
#include <stdio.h>
#include "defs.h"

jmp_buf env;

char	*dolp;
char	**argp;				/* argument list pointer	*/
char	**dolv;				/* for doing $x arguments	*/
char	*linep;				/* current position in cmd line	*/
char	*promp;				/* prompt string		*/
char	**eargp;			/* end of argument list vector	*/
char	*arginp;			/* head of command in "sh -c"	*/
char	*elinep;			/* end of cmd line buffer	*/

char	gflg;				/* non-zero if globbing needed	*/
char	error;				/* non-zero if had a parse error*/
char	peekc;				/* input peek character		*/
char	setintr;			/* non-zero if interrupts set	*/
char	interactive;			/* non-zero if interactive shell*/

int	*treep;				/* current position in tree	*/
int	*treeend;			/* end of tree			*/

int	uid;				/* user id of caller		*/
int	dolc;				/* for doing $x arguments	*/
int	onelflg;			/* one line only ("sh -t")	*/

char	pidp[8];			/* ascii version of process id	*/
char	line[LINSIZ];			/* current command line		*/
char	*args[ARGSIZ];			/* argument list vector		*/

int	trebuf[TRESIZ];			/* parse tree			*/

extern int	errno;
extern char	*sys_siglist[];

main(argc, argv, envp)
int argc;
char **argv;
char **envp;
{
	register int f;
	struct sigvec sv;

#ifdef RESTRICT
	setup_restrict(envp);
#endif

	/*
	 * Set up file descriptors.
	 */
	for (f = 2; f < NOFILE; f++)
		close(f);

	if ((f = dup(1)) != 2)
		close(f);

	/*
	 * Get process id.
	 */
	sprintf(pidp, "%d", getpid());

	/*
	 * Set prompt.
	 */
	if ((uid = getuid()) == 0)
		promp = "# ";
	else
		promp = "$ ";

	if (argc > 1) {
		promp = NULL;

		/*
		 * If the argument starts with a '-' it's a flag,
		 * otherwise the name of a file to read commands
		 * from.
		 */
		if (*argv[1] == '-') {
			**argv = '-';

			if ((argv[1][1] == 'c') && (argc > 2))
				arginp = argv[2];
			else if (argv[1][1] == 't')
				onelflg = 2;	/* see readc() */
			else if (argv[1][1] == 'i')
				interactive = 1;
		}
		else {
			close(0);

			if ((f = open(argv[1], O_RDONLY)) < 0) {
				perror(argv[1]);
				errexit();
			}
		}
	}

	if (isatty(0))
		interactive = 1;

	if (interactive) {
		setintr++;

		sv.sv_handler = SIG_IGN;
		sv.sv_flags = SV_INTERRUPT;

		sigvec(SIGINT, &sv, (struct sigvec *) 0);
		sigvec(SIGQUIT, &sv, (struct sigvec *) 0);
	}

	dolc = argc - 1;
	dolv = argv + 1;

	/*
	 * Print a prompt, read a command.
	 */
	for (;;) {
		if (promp != NULL)
			printf("%s", promp);

		peekc = getch();
		mainl();
	}
}

/*
 * mainl - main read-execute loop
 */
mainl()
{
	int *syntax();
	register int *t;
	register char c;
	register char *cp;

	gflg = 0;
	error = 0;
	argp = args;
	linep = line;
	eargp = args + ARGSIZ - SLOP;
	elinep = line + LINSIZ - SLOP;

	/*
	 * Read in a line.
	 */
	do {
		cp = linep;
		word();
	} while (*cp != '\n');

	treep = trebuf;
	treeend = &trebuf[TRESIZ];

	if (gflg == 0) {
		/*
		 * Parse the command.
		 */
		if (error == 0) {
			setjmp(env);

			if (error)
				return;
			t = syntax(args, argp);
		}

		/*
		 * Print an error or execute it.
		 */
		if (error != 0)
			fprintf(stderr, "syntax error.\n");
		else
			execute(t, NULL, NULL);
	}
}

/*
 * word - read in a token and store it in line.
 */
word()
{
	register char c, c1;

	*argp++ = linep;

loop:
	switch (c = getch()) {
	case ' ':			/* skip whitespace		*/
	case '\t':
		goto loop;
	case '"':			/* quoted item			*/
	case '\'':
		c1 = c;

		/*
		 * Read up to closing quote.
		 */
		while ((c = readc()) != c1) {
			/*
			 * End of line reached with no close,
			 * error.
			 */
			if (c == '\n') {
				peekc = c;
				error++;
				return;
			}

			*linep++ = c | QUOTE;
		}

		goto pack;
	case '&':			/* all these are delimiters	*/
	case ';':
	case '<':
	case '>':
	case '(':
	case ')':
	case '|':
	case '\n':
		*linep++ = c;
		*linep++ = '\0';
		return;
	}

	peekc = c;

pack:
	for (;;) {
		c = getch();

		if (any(c, " '\"\t;&<>()|\n")) {
			peekc = c;

			if (any(c, "\"'"))
				goto loop;

			*linep++ = '\0';
			return;
		}

		*linep++ = c;
	}
}

/*
 * tree - return a location in the parse tree.
 */
int *tree(n)
int n;
{
	register int *t;

	t = treep;
	treep += n;

	if (treep > treeend) {
		fprintf(stderr, "command line overflow.\n");
		error++;

		longjmp(env, 0);
	}

	return(t);
}

/*
 * getch - get a character, handle lookahead and "$x" arguments.
 */
getch()
{
	register char c;

	/*
	 * If we have a lookahead character, return it.
	 */
	if (peekc) {
		c = peekc;
		peekc = 0;
		return(c);
	}

	if (argp > eargp) {
		argp -= 10;

		/*
		 * Eat the line.
		 */
		while ((c = getch()) != '\n')
			;

		argp += 10;

		fprintf(stderr, "argument list too long.\n");
		errexit();
		gflg++;

		return(c);
	}

	if (linep > elinep) {
		linep -= 10;

		/*
		 * Eat the line.
		 */
		while ((c = getch()) != '\n')
			;

		linep += 10;

		fprintf(stderr, "line too long.\n");
		errexit();
		gflg++;

		return(c);
	}

	/*
	 * Handle "$x" arguments.
	 */
getd:
	if (dolp) {
		c = *dolp++;

		if (c != '\0')
			return(c);

		dolp = NULL;
	}

	/*
	 * Read a character.
	 */
	c = readc();

	/*
	 * Handle backslash escapes.
	 */
	if (c == '\\') {
		c = readc();

		/*
		 * Handle escapd newlines.
		 */
		if (c == '\n')
			return(' ');

		return(c | QUOTE);
	}

	/*
	 * Handle "$x" escapes.
	 */
	if (c == '$') {
		c = readc();

		/*
		 * "$1" ... "$9"
		 */
		if ((c >= '0') && (c <= '9')) {
			if ((c - '0') < dolc)
				dolp = dolv[c - '0'];

			goto getd;
		}

		/*
		 * "$$" (process id)
		 */
		if (c == '$') {
			dolp = pidp;
			goto getd;
		}
	}

	return(c & 0177);
}

/*
 * readc - read a character from standard input or from the command line
 *	   arguments.
 */
readc()
{
	register int c;
	register char cc;

	/*
	 * "Read" from command line.
	 */
	if (arginp) {
		if (arginp == (char *) 1)
			exit(0);

		if ((c = *arginp++) == '\0') {
			arginp = (char *) 1;
			c = '\n';
		}

		return(c);
	}

	/*
	 * onelflg started as 2, if it's 1, we're done.
	 */
	if (onelflg == 1)
		exit(0);

	/*
	 * Read a character.
	 */
	if ((cc = getchar()) == EOF) {
		putchar('\n');
		exit(0);
	}

	/*
	 * Handle onelflg.
	 */
	if ((cc == '\n') && (onelflg != 0))
		onelflg--;

	return(cc);
}

/*
 * syntax -
 *	empty
 *	syn1
 */
int *syntax(p1, p2)
char **p1, **p2;
{
	int *syn1();

	while (p1 != p2) {
		if (any(**p1, ";&\n"))
			p1++;
		else
			return(syn1(p1, p2));
	}

	return(0);
}

/*
 * syn1 -
 *	syn2
 *	syn2 & syntax
 *	syn2 ; syntax
 */
int *syn1(p1, p2)
char **p1, **p2;
{
	int l;
	int *syn2();
	register char **p;
	register int *t, *t1;

	l = 0;

	for (p=p1; p != p2; p++) {
		switch (**p) {
		case '(':			/* ( cmd )		*/
			l++;
			continue;
		case ')':
			if (--l < 0)
				error++;
			continue;
		case '&':			/* command separators	*/
		case ';':
		case '\n':
			/*
			 * Handle left side of command.
			 */
			if (l == 0) {
				l = **p;
				t = tree(4);

				t[DTYP] = TLST;
				t[DLEF] = (int) syn2(p1, p);
				t[DFLG] = 0;

				if (l == '&') {
					t1 = (int *) t[DLEF];
					t1[DFLG] |= FAND | FPRS | FINT;
				}

				/*
				 * Handle right side of command.
				 */
				t[DRIT] = (int) syntax(p+1, p2);
				return(t);
			}

			break;
		}
	}

	if (l == 0)
		return(syn2(p1, p2));

	/*
	 * Parentheses not closed.
	 */
	error++;
}

/*
 * syn2 -
 *	syn3
 *	syn2 | syn3
 */
int *syn2(p1, p2)
char **p1, **p2;
{
	int *syn3();
	register int l;
	register int *t;
	register char **p;

	l = 0;

	for (p=p1; p != p2; p++) {
		switch (**p) {
		case '(':			/* ( cmd )		*/
			l++;
			continue;
		case ')':
			l--;
			continue;
		case '|':
			/*
			 * Split command at pipe.
			 */
			if (l == 0) {
				t = tree(4);

				t[DTYP] = TFIL;
				t[DLEF] = (int) syn3(p1, p);
				t[DRIT] = (int) syn2(p+1, p2);
				t[DFLG] = 0;

				return(t);
			}

			break;
		}
	}

	return(syn3(p1, p2));
}

/*
 * syn3 -
 *	( syn1 ) [ < in ] [ > out ]
 *	word word* [ < in ] [ > out ]
 */
int *syn3(p1, p2)
char **p1, **p2;
{
	register int *t;
	char **lp, **rp;
	register char **p;
	int n, l, i, o, c, flg;

	flg = 0;

	/*
	 * ( cmd )
	 */
	if (**p2 == ')')
		flg |= FPAR;

	lp = rp = 0;
	i = o = n = l = 0;

	for (p=p1; p != p2; p++) {
		switch (c = **p) {
		case '(':			/* ( cmd )		*/
			if (l == 0) {
				if (lp != 0)
					error++;
				lp = p+1;
			}

			l++;
			continue;
		case ')':
			if (--l == 0)
				rp = p;
			continue;
		case '>':			/* > file		*/
			p++;

			if ((p != p2) && (**p == '>'))
				flg |= FCAT;
			else
				p--;

			/* fall through */
		case '<':			/* < file		*/
			if (l == 0) {
				p++;

				if (p == p2) {
					error++;
					p--;
				}

				if (any(**p, "<>("))
					error++;

				if (c == '<') {
					if (i != 0)
						error++;

					i = (int) *p;
					continue;
				}

				if (o != 0)
					error++;

				o = (int) *p;
			}

			continue;
		default:
			if (l == 0)
				p1[n++] = *p;
		}
	}

	if (lp != 0) {
		if (n != 0)
			error++;

		t = tree(5);

		t[DTYP] = TPAR;
		t[DSPR] = (int) syn1(lp, rp);

		goto out;
	}

	if (n == 0)
		error++;

	p1[n++] = 0;
	t = tree(n+5);
	t[DTYP] = TCOM;

	for (l=0; l < n; l++)
		t[l+DCOM] = (int) p1[l];

out:
	t[DFLG] = flg;
	t[DLEF] = i;
	t[DRIT] = o;

	return(t);
}

/*
 * execute - actually execute the command.  All file opening and stuff is
 *	     done here.
 */
execute(t, pf1, pf2)
int *t, *pf1, *pf2;
{
	int i, f, pv[2];
	register int *t1;
	struct sigvec sv;
	int trim(), tglob();
	register char *cp1, *cp2;

	if (t == NULL)
		return;

	/*
	 * See what kind of command we're dealing with.
	 */
	switch (t[DTYP]) {
	case TCOM:			/* command			*/
		cp1 = (char *) t[DCOM];

		/*
		 * Check for builtins.
		 */
		if (!strcmp(cp1, "cd") || !strcmp(cp1, "chdir")) {
			dochdir(t[DCOM+1]);
			return;
		}

		if (!strcmp(cp1, "shift")) {
			if (dolc < 1) {
				fprintf(stderr, "shift: no arguments.\n");
				return;
			}

			dolv[1] = dolv[0];
			dolv++;
			dolc--;
			return;
		}

		if (!strcmp(cp1, "wait")) {
			pwait(-1, 0);
			return;
		}

		/*
		 * Comment.
		 */
		if (!strcmp(cp1, ":") || !strcmp(cp1, "#"))
			return;

		/* fall through */
	case TPAR:			/* parenthesized command	*/
		f = t[DFLG];
		i = 0;

		if ((f & FPAR) == 0) {
			if ((i = fork()) < 0) {
				fprintf(stderr, "can't fork - try again.\n");
				return;
			}
		}

		/*
		 * Parent.
		 */
		if (i != 0) {
			if ((f & FPIN) != 0) {
				close(pf1[0]);
				close(pf1[1]);
			}

			if ((f & FPRS) != 0)
				printf("%d\n", i);

			if ((f & FAND) != 0)
				return;

			if ((f & FPOU) == 0)
				pwait(i, t);

			return;
		}

		/*
		 * Child.
		 */

		/*
		 * Input redirection.
		 */
		if (t[DLEF] != 0) {
			close(0);

			if ((i = doopen(t[DLEF], O_RDONLY)) < 0)
				exit(1);
		}

		/*
		 * Output redirection.
		 */
		if (t[DRIT] != 0) {
			i = O_WRONLY | O_CREAT;

			/*
			 * If not appending, we want to truncate.
			 */
			if ((f & FCAT) == 0)
				i |= O_TRUNC;

			i = doopen(t[DRIT], i, 0666);

			/*
			 * Error message is printed by doopen.
			 */
			if (i < 0)
				exit(1);

			/*
			 * Append.
			 */
			if ((f & FCAT) != 0) {
				if (i >= 0)
					lseek(i, 0L, L_XTND);
			}

			close(1);
			dup(i);
			close(i);
		}

		/*
		 * Handle a pipe.
		 */
		if ((f & FPIN) != 0) {
			close(0);
			dup(pf1[0]);
			close(pf1[0]);
			close(pf1[1]);
		}

		if ((f & FPOU) != 0) {
			close(1);
			dup(pf2[1]);
			close(pf2[0]);
			close(pf2[1]);
		}

		if (((f & FINT) != 0) && (t[DLEF] == 0) && ((f & FPIN) == 0)) {
			close(0);
			open("/dev/null", O_RDONLY);
		}

		if (((f & FINT) == 0) && (setintr != 0)) {
			sv.sv_flags = 0;
			sv.sv_handler = SIG_DFL;

			sigvec(SIGINT, &sv, (struct sigvec *) 0);
			sigvec(SIGQUIT, &sv, (struct sigvec *) 0);
		}

		if (t[DTYP] == TPAR) {
			if (t1 = (int *) t[DSPR])
				t1[DFLG] |= f & FINT;
			execute(t1, NULL, NULL);
			exit(0);
		}

		/*
		 * Check for globbing characters.
		 */
		gflg = 0;
		scan(t, tglob);

		if (gflg)
			doglob(t+DSPR+1);

		scan(t, trim);
		*linep = '\0';
		texec(t[DCOM], t);
	case TFIL:			/* do a pipe			*/
		f = t[DFLG];
		pipe(pv);

		t1 = (int *) t[DLEF];
		t1[DFLG] |= FPOU | (f & (FPIN | FINT | FPRS));
		execute(t1, pf1, pv);

		t1 = (int *) t[DRIT];
		t1[DFLG] |= FPIN | (f & (FPOU | FINT | FAND | FPRS));
		execute(t1, pv, pf2);

		return;
	case TLST:			/* parentheses			*/
		f = t[DFLG] & FINT;

		if (t1 = (int *) t[DLEF])
			t1[DFLG] |= f;

		execute(t1, NULL, NULL);

		if (t1 = (int *) t[DRIT])
			t1[DFLG] |= f;

		execute(t1, NULL, NULL);
		return;
	}
}

/*
 * texec - really execute a program.
 */
texec(f, at)
char *f;
int *at;
{
	register int *t;

	t = at;
	doexec(f, t+DCOM);

	/*
	 * Can't execute it, must be a shell script.
	 */
	if (errno == ENOEXEC) {
		if (*linep)
			t[DCOM] = (int) linep;

		execv(SHELLNAME, t+DSPR);

		fprintf(stderr, "cannot execute \"%s\".\n", SHELLNAME);
		exit(1);
	}

	perror(f);
	exit(1);
}

/*
 * pwait - wait for processes
 */
pwait(i, t)
int *t;
int i;
{
	register int p, e;
	int s;

	/*
	 * Wait for all children.
	 */
	if (i != 0) {
		for (;;) {
			if ((p = wait(&s)) < 0)
				break;

			e = s & 0177;

			if (p != i) {
				if (e > 0) {
					fprintf(stderr, "%d: ", p);

					if ((e < NSIG) && (sys_siglist[e] != NULL))
						fprintf(stderr, "%s", sys_siglist[e]);
				}

				if (s & 0200)
					fprintf(stderr, " -- Core dumped");
			}

			if (e != 0)
				putc('\n', stderr);
		}
	}
}

/*
 * scan - scan at for characters using function f.
 */
scan(at, f)
int *at;
int (*f)();
{
	register int *t;
	register char c;
	register char *p;

	t = at+DCOM;

	while (p = (char *) *t++) {
		while (c = *p)
			*p++ = (*f)(c);
	}
}

/*
 * tglob - search for globbing characters.
 */
tglob(c)
int c;
{
	if (any(c, "[?*"))
		gflg = 1;

	return(c);
}

/*
 * trim - strip eighth bit
 */
trim(c)
int c;
{
	return(c & 0177);
}

/*
 * any - return non-zero if c is in s.
 */
any(c, as)
int c;
char *as;
{
	register char *s;

	s = as;

	while (*s) {
		if (*s++ == c)
			return(1);
	}

	return(0);
}

/*
 * errexit - handle exiting from shell files.
 */
errexit()
{
	if (promp == NULL) {
		lseek(0, 0L, L_XTND);
		exit(1);
	}
}

static int ncoll;

static char ab[ABSIZ];			/* generated characters		*/
static char *ava[MAXARGS];		/* generated arguments		*/

static char **av = ava;
static char *string = ab;

/*
 * doglob - handle globbing.
 */
doglob(argv)
char **argv;
{
	register char *cp;
	
	*av++ = *argv;
	
	while (*++argv)
		expand(*argv);
	
	if (ncoll == 0) {
		fprintf(stderr, "No match.\n");
		exit(1);
	}

	globexec(*ava, ava);

	perror(*ava);
	exit(1);
}

expand(as)
char *as;
{
	char **oav;
	char *cat();
	DIR *dirp, *opendir();
	register char *s, *cs;
	char *index(), *homedir();
	struct direct *d, *readdir();

	s = cs = as;

	while ((*cs != '*') && (*cs != '?') && (*cs != '[')) {
		if (*cs++ == NULL) {
			*av++ = cat(s, "");
			return;
		}
	}
	
	for (;;) {
		if (cs == s) {
			dirp = opendir(".");
			s = "";
			break;
		}
		
		if (*--cs == '/') {
			*cs = NULL;
			dirp = opendir(s == cs ? "/" : s);
			*cs++ = 0200;
			break;
		}
	}
	
	if (dirp == NULL) {
		fprintf(stderr, "No directory.\n");
		exit(1);
	}
	
	oav = av;
	
	while ((d = readdir(dirp)) != 0) {
		if (d->d_ino == 0)
			continue;
		
		if (match(d->d_name, cs)) {
			*av++ = cat(s, d->d_name);
			ncoll++;
		}

		if (ncoll > MAXARGS)
			toolong();
	}
	
	closedir(dirp);
	sort(oav);
}

match(s, p)
char *s, *p;
{
	if ((*s == '.') && (*p != '.'))
		return(0);
	
	return(amatch(s, p));
}

amatch(as, ap)
char *as, *ap;
{
	register short scc;
	short c, cc, ok, lc;
	register char *s, *p;
	
	s = as;
	p = ap;
	
	if (scc = *s++)
		if ((scc &= 0177) == 0)
			scc = 0200;
	
	switch (c = *p++) {
	case '[':
		ok = 0;
		lc = 077777;
		
		while (cc = *p++) {
			if (cc == ']') {
				if (ok)
					return(amatch(s, p));
				else
					return(0);
			}
			else if (cc == '-') {
				if ((lc <= scc) && (scc <= (c = *p++)))
					ok++;
			}
			else {
				if (scc == (lc = cc))
					ok++;
			}
		}

		return(0);
	default:
		if (c != scc)
			return(0);

		/* fall through */
	case '?':
		if (scc)
			return(amatch(s, p));
		return(0);
	case '*':
		return(umatch(--s, p));
	case '\0':
		return(!scc);
	}
}

umatch(s, p)
char *s, *p;
{
	if (*p == 0)
		return(1);
	
	while (*s) {
		if (amatch(s++, p))
			return(1);
	}
	
	return(0);
}

sort(oav)
char **oav;
{
	register char **p1, **p2, *c;

	p1 = oav;

	while (p1 < av-1) {
		p2 = p1;

		while (++p2 < av) {
			if (compar(*p1, *p2) > 0) {
				c = *p1;
				*p1 = *p2;
				*p2 = c;
			}
		}

		p1++;
	}
}

compar(as1, as2)
char *as1, *as2;
{
	register char *s1, *s2;
	
	s1 = as1;
	s2 = as2;
	
	while (*s1++ == *s2) {
		if (*s2++ == 0)
			return(0);
	}

	return(*--s1 - *s2);
}

char *cat(as1, as2)
char *as1, *as2;
{
	register int c;
	register char *s1, *s2;
	
	s1 = as1;
	s2 = string;

	while (c = *s1++) {
		if (s2 > &ab[ABSIZ])
			toolong();
		
		c &= 0177;
		
		if (c == 0) {
			*s2++ = '/';
			break;
		}
		
		*s2++ = c;
	}
	
	s1 = as2;
	
	do {
		if (s2 > &ab[ABSIZ])
			toolong();

		*s2++ = c = *s1++;
	} while (c);
	
	s1 = string;
	string = s2;
	
	return(s1);
}

globexec(afile, aarg)
char *afile;
char **aarg;
{
	doexec(afile, aarg);
	
	/*
	 * Can't execute it, must be a shell script.
	 */
	if (errno == ENOEXEC) {
		execv(SHELLNAME, aarg);

		fprintf(stderr, "cannot execute \"%s\".\n", SHELLNAME);
		exit(1);
	}

	perror(afile);
	exit(1);
}

toolong()
{
	fprintf(stderr, "argument list too long.\n");
	exit(1);
}
