#ifndef lint
static char *RCSid = "$Header$";
#endif

/*
 * restrict - routines which make the shell restricted.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * davy@intrepid.ecn.purdue.edu
 * August, 1987
 *
 * $Log$
 */
#include <sys/param.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <errno.h>
#include <stdio.h>
#include <pwd.h>

#include "defs.h"

#ifdef RESTRICT
static int	root_len;		/* length of root directory	*/
static char	cur_dir[MAXPATHLEN];	/* current directory		*/
static char	root_dir[MAXPATHLEN];	/* root directory		*/

static char	path[BUFSIZ];		/* search path for execs	*/
static char	*cmdlist[MAXCMDS];	/* list of allowed commands	*/
static char	*environ[MAXENVIRON];	/* environment variables	*/

static int	nowrite = 0;		/* 1 means no file writing	*/
static int	nocreate = 0;		/* 1 means no file creation	*/
static int	nooverwrite = 0;	/* 1 means no file overwrites	*/

static char	*envvars[] = {		/* environ vars we want		*/
	"USER=", "TERM=", "HOSTNAME=", NULL
};
#endif

#ifdef RESTRICT
/*
 * setup_restrict - initialize variables, parse configuration file, read
 *		    list of permitted commands.
 */
setup_restrict(envp)
char **envp;
{
	FILE *fp;
	int setpath = 0;
	char buf[BUFSIZ];
	struct passwd *pw;
	register char *s, *t;
	struct passwd *getpwuid();
	char *savestr(), *getenv();
	register char **env, **cmd, **cp;

	/*
	 * Get current and root directories.
	 */
	getwd(cur_dir);
	strcpy(root_dir, cur_dir);
	root_len = strlen(root_dir);

	env = environ;
	cmd = cmdlist;

	/*
	 * Open configuration file.
	 */
	pw = getpwuid(getuid());
	sprintf(buf, "%s/%s", CONFDIR, pw->pw_name);

	if ((fp = fopen(buf, "r")) == NULL) {
		fprintf(stderr, "cannot open configuration file.\n");
		exit(1);
	}

	/*
	 * Read configuration file.
	 */
	while (fgets(buf, BUFSIZ, fp) != NULL) {
		/*
		 * Eat leading whitespace.
		 */
		s = buf;
		while ((*s == ' ') || (*s == '\t'))
			s++;

		/*
		 * Skip blank lines and comments.
		 */
		if ((*s == '\n') || (*s == '#'))
			continue;

		/*
		 * Eat trailing whitespace.
		 */
		t = s;
		while ((*t != ' ') && (*t != '\t') && (*t != '\n'))
			t++;
		*t++ = NULL;

		/*
		 * "\path" sets the path.
		 */
		if (!strcmp(s, "\\path")) {
			/*
			 * Skip to start of path.
			 */
			while ((*t == ' ') || (*t == '\t'))
				t++;
			s = t;

			/*
			 * Eat trailing whitespace.
			 */
			while ((*t != ' ') && (*t != '\t') && (*t != '\n'))
				t++;
			*t = NULL;

			/*
			 * Save the path.
			 */
			setpath++;
			strcpy(path, s);

			/*
			 * Save it in the environment.
			 */
			sprintf(buf, "PATH=%s", path);
			*env++ = savestr(buf);
			continue;
		}

		/*
		 * "\nooverwrite" prohibits them from overwriting files.
		 */
		if (!strcmp(s, "\\nooverwrite")) {
			nooverwrite = 1;
			continue;
		}

		/*
		 * "\nocreate" prohibits them from creating files.
		 */
		if (!strcmp(s, "\\nocreate")) {
			nocreate = 1;
			continue;
		}

		/*
		 * "\nowrite" means no writing, period.
		 */
		if (!strcmp(s, "\\nowrite")) {
			nowrite = 1;
			continue;
		}

		/*
		 * Save the command in the list of restricted commands.
		 */
		*cmd++ = savestr(s);
	}

	fclose(fp);

	/*
	 * Create SHELL and HOME environment variables.
	 */
	sprintf(buf, "SHELL=%s", SHELLNAME);
	*env++ = savestr(buf);

	sprintf(buf, "HOME=%s", root_dir);
	*env++ = savestr(buf);

	/*
	 * If we didn't make a path already, get one now.
	 */
	if (!setpath) {
		strcpy(path, getenv("PATH"));
		sprintf(buf, "PATH=%s", path);
		*env++ = savestr(buf);
	}

	/*
	 * Copy "interesting" environment variables.
	 */
	while (*envp) {
		for (cp = envvars; *cp; cp++) {
			if (!strncmp(*envp, *cp, strlen(*cp))) {
				*env++ = *envp;
				break;
			}
		}

		envp++;
	}

	*cmd = NULL;
	*env = NULL;
}

/*
 * restricted - returns non-zero if cmd is a restricted command.
 */
restricted(cmd)
char *cmd;
{
	register char **cmds;

	for (cmds = cmdlist; *cmds; cmds++) {
		if (!strcmp(cmd, *cmds))
			return(0);
	}

	return(1);
}
#endif

/*
 * doexec - execute a command, searching through our internal path
 *	    for the command.  Basically, this is a modified execvp.
 */
doexec(cmd, argv)
char *cmd;
char **argv;
{
#ifndef RESTRICT
	execvp(cmd, argv);
#else
	int i;
	char *execat();
	extern int errno;
	register char *cp;
	char pathstr[BUFSIZ];
	char fname[MAXPATHLEN];
	
	if (restricted(cmd)) {
		fprintf(stderr, "%s: restricted command.\n", cmd);
		exit(1);
	}

	errno = ENOENT;
	strcpy(pathstr, path);
	cp = index(cmd, '/') ? "" : pathstr;

	do {
		cp = execat(cp, cmd, fname);

		if (access(fname, F_OK) == 0)
			execve(fname, argv, environ);
	} while(cp);

	return(-1);
#endif
}

/*
 * dochdir - change directories, restricting the user to the tree rooted
 *	     at his home directory.
 */
dochdir(dir)
char *dir;
{
	struct passwd *getpwuid();
	static struct passwd *pw = NULL;

#ifdef RESTRICT
	struct stat st;
	char dest_dir[MAXPATHLEN];

	/*
	 * Null directory means "home".
	 */
	if (dir == NULL)
		dir = root_dir;

	/*
	 * If the directory is not an absolute path,
	 * figure out the path.
	 */
	if (*dir == '/')
		strcpy(dest_dir, dir);
	else
		pathcopy(dest_dir, cur_dir, dir);

	/*
	 * If the end point of the directory is a symbolic link,
	 * and it's not root_dir, forget it.
	 */
	if ((strcmp(dest_dir, root_dir) != 0) && (lstat(dest_dir, &st) >= 0)) {
		if ((st.st_mode & S_IFMT) == S_IFLNK) {
			fprintf(stderr, "%s: permission denied.\n", dest_dir);
			return;
		}
	}

	/*
	 * If the directory is not rooted at root_dir,
	 * forget it.
	 */
	if ((strncmp(dest_dir, root_dir, root_len) != 0) || 
	    ((dest_dir[root_len] != '/') && (dest_dir[root_len] != NULL))) {
		fprintf(stderr, "%s: permission denied.\n", dest_dir);
		return;
	}

	dir = dest_dir;
#else
	if (dir == NULL) {
		if (pw == NULL)
			pw = getpwnam(getuid());

		dir = pw->pw_dir;
	}
#endif

	if (chdir(dir) < 0)
		perror(dir);
#ifdef RESTRICT
	else
		strcpy(cur_dir, dir);
#endif
}

/*
 * doopen - open a file, performing various security checks first.
 */
doopen(file, how, mode)
char *file;
int how, mode;
{
	register int fd;

#ifdef RESTRICT
	struct stat st;
	static int uid = -1;

	if (uid < 0)
		uid = getuid();

	/*
	 * If the file is to be written, make sure it is a regular
	 * file and that we own it.
	 */
	if ((how & (O_WRONLY | O_RDWR)) != 0) {
		/*
		 * Special case.
		 */
		if (!strcmp(file, "/dev/null"))
			goto out;

		/*
		 * If nowrite is set, forget it.
		 */
		if (nowrite) {
			fprintf(stderr, "%s: permission denied.\n", file);
			return(-1);
		}

		/*
		 * If nooverwrite is set, and they're trying to truncate,
		 * add O_EXCL, which makes the open fail if the file
		 * exists.
		 */
		if ((nooverwrite != 0) && ((how & O_TRUNC) != 0))
			how |= O_EXCL;

		/*
		 * Find out what kind of file it is.
		 */
		if (lstat(file, &st) >= 0) {
			if ((st.st_mode & S_IFMT) != S_IFREG) {
				fprintf(stderr, "%s: not a regular file.\n", file);
				return(-1);
			}

			if (st.st_uid != uid) {
				fprintf(stderr, "%s: not owner.\n", file);
				return(-1);
			}
		}
		else {
			/*
			 * If it doesn't exist and they can't create files,
			 * forget it.
			 */
			if (nocreate) {
				fprintf(stderr, "%s: cannote create.\n", file);
				exit(1);
			}
		}
	}

out:
#endif

	if ((fd = open(file, how, mode)) < 0)
		perror(file);

	return(fd);
}

#ifdef RESTRICT
/*
 * pathcopy - copies s1 and s2 into dest, resolving things like "./"
 *	      and "../".
 */
pathcopy(dest, s1, s2)
char *dest, *s1, *s2;
{
	char *rindex();
	char buf[MAXPATHLEN];
	register char *s, *t;

	strcpy(buf, s1);
	s1 = buf;
	s = s2;

	/*
	 * Until we run out of things like "./" and "../"...
	 */
	while (!strncmp(s, "./", 2) || !strncmp(s, "../", 3)) {
		/*
		 * If it was "./", just skip over it.
		 */
		if (*(s+1) == '/') {
			s += 2;
			continue;
		}

		/*
		 * If it was "../", we back up to the previous
		 * '/' in the s1 string.
		 */
		t = rindex(s1, '/');

		/*
		 * Should never be NULL since s1[0] == '/'.
		 * If it's the beginning slash, then leave the
		 * slash there.
		 * If it's a different slash, just set it to null.
		 */
		if (t == NULL)
			break;
		else if (t == s1)
			*(t+1) = NULL;
		else
			*t = NULL;

		/*
		 * Skip over "../".
		 */
		s += 3;
	}

	/*
	 * Now put together the pieces.
	 */
	strcpy(dest, s1);

	/*
	 * If all that remains of s is "..", skip
	 * backwards in dest.
	 */
	if (!strcmp(s, "..")) {
		t = rindex(dest, '/');

		if (t == dest)
			*(t+1) = NULL;
		else
			*t = NULL;

		*s = NULL;
	}

	/*
	 * Only add s if it is non-null and not ".".
	 */
	if ((*s != NULL) && (strcmp(s, ".") != 0)) {
		strcat(dest, "/");
		strcat(dest, s);
	}
}

/*
 * savestr - save a string in dynamic memory.
 */
char *savestr(str)
char *str;
{
	char *malloc();
	register char *s;

	if ((s = malloc(strlen(str) + 1)) == NULL) {
		fprintf(stderr, "out of memory.\n");
		exit(1);
	}

	strcpy(s, str);
	return(s);
}

char *execat(s1, s2, si)
register char *s1, *s2;
char *si;
{
	char *end;
	register char *s;

	s = si;
	end = s + MAXPATHLEN;

	while (*s1 && *s1 != ':' && s < end)
		*s++ = *s1++;

	if (si != s && s < end)
		*s++ = '/';

	while (*s2 && s < end)
		*s++ = *s2++;

	*s = '\0';
	return(*s1 ? ++s1 : 0);
}
#endif
