
/*
 * This program may be freely redistributed but this entire comment 
 * MUST remain intact.
 *
 * Copyright (c) 1989 Clifford Spencer
 *
 * usage: cwd [usernames]
 *
 * for each process on the system print full path of current directory
 * as found in u.u_cdir. Note that this program requires read permission
 * on the raw disks
 * 
 *
 * This program has been compiled successfully on Ultrix 3.0, SunOS 3.5, 
 * and BSD4.3. It won't work for NFS mounted filesystems because of lack 
 * of access to remote devices
 * 
 * author: Cliff Spencer (cspencer@lemming.uucp)
 * Tue Oct 17 21:53:30 EDT 1989
 *
 */
#if !defined(ultrix) && !defined(bsd) && !defined(sun)
#define sun
#endif
#include <sys/param.h>
#if !defined(bsd)       /* user.h pulls this in on BSD */
#include <sys/time.h>
#endif
#if defined(sun)
#include <sys/vnode.h>
#endif
#if defined(bsd)
#include <sys/fs.h>
#else
#include <ufs/fs.h>
#endif
#include <sys/dir.h>
#include <sys/file.h>
#if defined(ultrix) || defined(bsd)
#include <sys/inode.h>
#include <sys/mount.h>
#endif
#if defined(ultrix)
#include <sys/gnode.h>
#endif
#if defined(sun)
#include <ufs/mount.h>
#define KERNEL  /* this is to pick up define for VTOI */
#include <ufs/inode.h>
#undef KERNEL
#endif
#include <machine/pte.h>
#include <sys/buf.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/stat.h>
#include <pwd.h>
#include <utmp.h>
#include <stdio.h>
#include <fstab.h>
#include <nlist.h>
#include <errno.h>

/* 
 * handy info to keep around on each mounted filesystem
 */
struct mount_stuff {
        dev_t   dev;                    /* major/minor of fs */
        char devname[MAXPATHLEN + 1];   /* name of dev */
        char path[MAXPATHLEN + 1];      /* name of mount point */
        int fd;
        struct fs *fs;
};


struct nlist nlst[] = {
        { "_proc" },
#define X_PROC          0
        { "_nproc" },
#define X_NPROC         1
#if defined(sun)
        { "_mounttab" },
#else
        { "_mount" },
#endif
#define X_MOUNT         2
#if defined(ultrix)
        { "_nmount"},
#define X_NMOUNT        3
#endif
        {       0    },
};

daddr_t bmap();
char *emalloc();
void mread();
int kmem;
int mem;
int swap;


#define ANY_UID -1
#define NOSUCH_USER -2

#if defined(ultrix)
#define GTOI(g)         (&(g)->g_req)
#define inode           gnode_req
#define i_number        gr_number
#define i_dev           gr_dev
#endif



main(argc, argv)
int argc;
char **argv;
{
        unsigned nproc, i;
        struct proc *p;

        struct mount_stuff *mount_stuff, *getmountstuff();
        
        (void)nlist("/vmunix", nlst);
        
        for (i = 0; nlst[i].n_name; i++)
                if (nlst[i].n_type == 0)
                        ecmderr(0, "vmunix: can' get symbol <%s>\n", 
                                nlst[i].n_name);
        
        mem = eopen("/dev/mem", O_RDONLY);
        kmem = eopen("/dev/kmem", O_RDONLY);
        swap = eopen("/dev/drum", O_RDONLY);
        
        
        mread(kmem,(daddr_t)nlst[X_NPROC].n_value, 
              (char *)&nproc, sizeof(nproc));
        mread(kmem, (daddr_t)nlst[X_PROC].n_value, (char *)&p, sizeof(p));
        mount_stuff = getmountstuff(nlst);
        
	printf("PID    USER       COMMAND          PATH\n");
        if (argc > 1) {
                while (--argc)
                        cwd(mount_stuff, p, nproc, name2uid(*++argv));
        }
        else
                cwd(mount_stuff, p, nproc, ANY_UID);
}

cwd(mount_stuff, procaddr, nproc, uid)
struct mount_stuff *mount_stuff;
struct proc *procaddr;
unsigned nproc;
int uid;
{
	struct inode *cdir, *getcdir();
        struct proc *p, proc;
        struct mount_stuff *m, *devtostuff();
        struct user *getu();
        char *path;
        struct user *u; 
        char *findpath();
	int proci;
        
	if (uid == NOSUCH_USER)
		return;
	for (proci = 0, p = &proc; proci < nproc; proci++) {
		mread(kmem, (daddr_t)&procaddr[proci], (char *)&proc, 
			sizeof(struct proc));
                if (p->p_stat && p->p_stat != SZOMB) {
                        if (uid != ANY_UID && p->p_uid != uid)
                                continue;
                        if (u = getu(p, mem, kmem, swap)) {
                                cdir = getcdir(kmem,(daddr_t)u->u_cdir);
				printpid(p->p_pid);
                                (void)printuser(u->u_uid);
                                printcomm(u, p->p_pid);
                                m = devtostuff(cdir->i_dev, mount_stuff);
                                if (!m)
                                        printf("<no mount info>");
                                else {
                                        path = findpath(m, cdir->i_number,
                                                        (ino_t)0);
                                        printf("%s", path);
                                }
                                printf("\n");
                        }
                }
        }
}

/*
 * recursively lookup and return the path to the inode described by "thenum"
 * append path components on return
 * args:
 *      mount_stuff: mount info for the appropriate device
 *      dirinum: i number of the current dir containing the
 *              element being looked up
 *      thenum: i number to look up in this dir (0 means none))
 *
 * returns: pathname corresponding to dirinum
 */
char *findpath(mount_stuff, dirinum, thenum)
struct mount_stuff *mount_stuff;
ino_t dirinum;
ino_t thenum;
{
        DIR *open_dir(), *dir;
        struct direct *read_dir(), *dirp;
        struct dinode *ip, *readinode();
        ino_t dotdot,dot;
        int fd = mount_stuff->fd;
        struct fs *fs = mount_stuff->fs;
        static char path[MAXPATHLEN+1];
        
        
        /*
         * hit the root, prepend the mount point and return
         */
        if (dirinum == thenum) {
                strcpy(path, mount_stuff->path);
                return path;
        }
        
        
	ip = readinode(fd, fs, dirinum);
	dir = open_dir(fd, fs, ip);
	if (dir == NULL)
		return "<bogus dir>";
	
	while (dirp = read_dir(fs, ip, dir)) {
		if (strcmp(dirp->d_name, ".") == 0) {
			dot = dirp->d_ino;/* next inode # to lookup */
			continue;
		}
		/* save number of parent for next time */
		else if (strcmp(dirp->d_name, "..") == 0)
			dotdot = dirp->d_ino;
		else if (dirp->d_ino == thenum || thenum == 0) {
			char name[MAXNAMLEN+1];
			
			if (thenum) 
				sprintf(name, "/%s", dirp->d_name);
		       
			(void)findpath(mount_stuff, dotdot, dot);
			if (thenum) {
				/* 
				 * remove redundant leading / for
				 * things mounted on root
				 */
				int len = strlen(path);
				if (path[len-1] == '/') 
					len--;
				strcpy(&path[len],name);
			}
			break;
		}
	}
	close_dir(dir);
        return path;
}


/*
 * read from a block from the disk
 */  
bread(fi, bno, buf, cnt)
int fi;
daddr_t bno;
char *buf;
long cnt;
{
        mread(fi, bno * DEV_BSIZE, buf, cnt);
}


/*
 * map logical to physical block numbers
 */
daddr_t bmap(b, ip, fi, fs)
daddr_t b;
struct dinode *ip;
int fi;
struct fs *fs;
{
        daddr_t ibuf[MAXBSIZE / sizeof (daddr_t)];
	daddr_t iblock;
	int indir;
	int nblocks;
        
        if(b < NDADDR)
                return(ip->di_db[b]);
        b -= NDADDR;
	for (nblocks = 1, indir = 0; indir < NIADDR; indir++) {
		nblocks *= NINDIR(fs);
		if (nblocks > b)
			break;
		else
			b -= nblocks;
	}
	if (indir == NIADDR)
                return((daddr_t)0);
       	iblock = ip->di_ib[indir];
	for (; indir >= 0; indir--) {
		bread(fi, fsbtodb(fs, iblock), (char *)ibuf, sizeof(ibuf));
		nblocks /= NINDIR(fs);
		iblock = ibuf[(b/nblocks) % NINDIR(fs)];
	}
        return iblock;
}

DIR *open_dir(file, fs, ip)
int file;
struct fs *fs;
struct dinode *ip;
{
        static DIR dir;
        
        if ((ip->di_mode & IFMT) != IFDIR) {
                fprintf(stderr, "open_dir:  not a directory\n");
                return NULL;
        }
        dir.dd_loc = 0;
        dir.dd_size = ip->di_size;
#if !defined(bsd)
	if (dir.dd_buf)
		free(dir.dd_buf);
        dir.dd_buf = (char *)emalloc((unsigned)fs->fs_bsize);
#endif
        dir.dd_fd = file;
        return &dir;
}

close_dir(dirp)
DIR *dirp;
{
#if !defined(bsd)
        if (dirp->dd_buf) {
                free(dirp->dd_buf);
                dirp->dd_buf = NULL;
        }
#endif
}



struct direct *read_dir(fs,ip, dir)
struct fs *fs;
struct dinode *ip;
DIR *dir;
{
        daddr_t bn;
        static struct direct *dp;
        
        if (dir->dd_loc >= dir->dd_size)
                return NULL;
        if (blkoff(fs, dir->dd_loc) == 0) {     /* need to read a block */
                bn = lblkno(fs, dir->dd_loc);
                bn = bmap(bn, ip, dir->dd_fd, fs);
                if (bn == 0)
                        return NULL;
                bn = fsbtodb(fs, bn);
                bread(dir->dd_fd, bn, dir->dd_buf, fs->fs_bsize);
        }
        dp = (struct direct *)&dir->dd_buf[dir->dd_loc];
        dir->dd_loc += dp->d_reclen;
        return dp;
}

/*
 * error printing 
 * modelled after routines written by Dan Franklin@bbn
 * if e is -1 print system error message from errno
 * if e is 0 don't print error message
 * else use e as errno
 * 
 * note: using _doprnt is less than portable
 */
cmderr(e, format, args)
int e;
char *format;
char *args;
{
        register int nl = 0;
        register int i = strlen(format) - 1;
        char fmt[BUFSIZ];
        extern int sys_nerr;
        extern char *sys_errlist[];
        extern int errno;
        
        strcpy(fmt, format);
        if ( e == -1)
                e = errno;
        if (fmt[i] == '\n') {
                fmt[i] = NULL;
                nl = 1;
        }
        _doprnt(fmt, &args, stderr);
        if (e > sys_nerr)
                fprintf(stderr," Error %d", e);
        else if (e)
                fprintf(stderr, " %s", sys_errlist[e]);
        if(nl)
                fprintf(stderr,"\n");
}


/*
 * print error message and exit
 */
ecmderr(e, fmt, args)
int e;
char *fmt;
unsigned args;
{
        cmderr(e,fmt,args);
        exit(e);
}

/*
 * read from the specified address 
 */
void mread(fd, addr, buf, cnt)
int fd;
daddr_t addr;
char *buf;
int cnt;
{
        int nread;
        extern daddr_t lseek();
        if (lseek(fd, addr, L_SET) != addr)
                ecmderr(-1, "Seek failed\n");
        if ((nread = read(fd, buf, cnt)) == 0)
                ecmderr(-1, "Read failed (wanted:%d)\n", cnt);
        if (nread > cnt) cmderr(-1,"read for %d returned %d\n", cnt, nread);
}

/* 
 * "safe" open
 */
eopen(name, mode)
char *name;
unsigned mode;
{
        int fd;
        fd = open(name, mode);
        if (fd < 0)
                ecmderr(-1, "cwd: Can't open %s\n", name);
        return fd;
}

/*
 * "safe" malloc 
 */
char *emalloc(n)
unsigned n;
{
        char *p;
        extern char *malloc();
        p = malloc(n);
        if (p == NULL)
                ecmderr(-1, "Malloc failed\n");
        return p;
}


/*
 * read kernel mount table and save the stuff we care about
 */
struct mount_stuff *getmountstuff(nlst)
struct nlist nlst[];
{
        int  nmount, i;
        struct mount *mounttable;
        struct mount_stuff *mount_stuff;
        struct fstab *fstab, *getfsent();
        int m;
        
#if defined(ultrix)
        mread(kmem, (daddr_t)nlst[X_NMOUNT].n_value, &nmount, sizeof(nmount));
#else
        nmount = NMOUNT;
#endif
        
        mounttable = (struct mount *)emalloc(
		(unsigned)nmount * sizeof(struct mount));
        mread(kmem, (daddr_t)nlst[X_MOUNT].n_value, (char *)mounttable, 
              nmount * sizeof(struct mount));
        mount_stuff = (struct mount_stuff *)
                emalloc((nmount + 1)* sizeof(struct mount_stuff));
        
        for (m = 0; fstab = getfsent();) {
                struct stat s;
                struct buf buf;
                char *slash;
                extern char *rindex(), *index();
                
                if (*fstab->fs_file == NULL)
                        continue;
                if (index(fstab->fs_spec,'@')) {
                        fprintf(stderr,"%s: Can't handle NFS yet\n",
                                fstab->fs_spec);
                        continue;
                }
                if (stat(fstab->fs_spec, &s) != 0)
                        ecmderr(-1, "cwd: Can't stat %s\n", fstab->fs_spec);
                strcpy(mount_stuff[m].devname, fstab->fs_spec);
                slash = rindex(mount_stuff[m].devname, '/');
                if (slash) {
                        slash++;
                        bcopy(slash, slash+1, strlen(slash)+1);
                        *slash = 'r';
                }
                strcpy(mount_stuff[m].path, fstab->fs_file);
                
                mount_stuff[m].fd = -1;
                mount_stuff[m].fs = (struct fs *)emalloc(sizeof(struct fs));
                for (i = 0; i < nmount; i++) {
                        if (mounttable[i].m_dev != s.st_rdev)
                                continue;
                        mount_stuff[m].dev = s.st_rdev;
                        mread(kmem, (daddr_t)mounttable[i].m_bufp,
                              (char *)&buf,sizeof(struct buf));
                        mread(kmem, (daddr_t)buf.b_un.b_fs,(char *)mount_stuff[m].fs, 
                              sizeof(struct fs));
                        break;
                }
                m++;
                if (i >= nmount) 
                        cmderr(0, "cwd: Can't find %s in kernel mount table\n",
                               fstab->fs_spec);
        }       
        mount_stuff[m].dev = -1;
        free(mounttable);
        return mount_stuff;
}

/*
 * return a pointer to some mount information using
 * device as the key
 */
struct mount_stuff *devtostuff(dev, mount_stuff)
dev_t dev;
struct mount_stuff mount_stuff[];
{
        int i;
        
        for (i = 0; mount_stuff[i].dev != -1; i++) {
                if (mount_stuff[i].dev == dev) {
                        if (mount_stuff[i].fd == -1)
                                mount_stuff[i].fd = 
                                        eopen(mount_stuff[i].devname,
                                              O_RDONLY);
                        return &mount_stuff[i];
                }
        }
        return NULL;
}

/*
 * massage the u_comm field of a user page
 */
printcomm(u, pid)
struct user *u;
int pid;
{
        char *name;
        switch (pid) {
              case 0:
                name = "swapper";
                break;
              case 2:
                name = "pagedaemon";
                break;
              default:
                name = u->u_comm;
        }
        printf("%-*s ", MAXCOMLEN, name );
}

printpid(pid)
int pid;
{
	printf("%-6d ", pid);
}

/*
 * print a username based on uid
 */ 
printuser(uid)
int uid;
{
        struct passwd *pw;
        pw = getpwuid(uid);
        if (pw)
                printf("%-8.8s   ", pw->pw_name);
        return (pw != NULL);
}

struct dinode *readinode(fd, fs, inum)
int fd;
struct fs *fs;
ino_t inum;
{
        static struct dinode itab[MAXIPG];
        
        bread(fd, fsbtodb(fs, cgimin(fs, inum/fs->fs_ipg)), (char *)itab,
              fs->fs_ipg * sizeof (struct dinode));
        
        return &itab[inum%fs->fs_ipg];
}

name2uid(name)
char *name;
{
        struct passwd *pw;
        pw = getpwnam(name);
        if (pw)
                return pw->pw_uid;
        else {
                fprintf(stderr,"%s: no such user\n",name);
                return NOSUCH_USER;
        }
}


/*
 * return a pointer to a user page
 * this routine was liberally cribbed from the "top" program
 * by William Lefebvre
 */
struct user *getu(p, mem, kmem, swap)
struct proc *p;
{
        /* pad reads from swap to a page */
        static union {
                struct  user u;
                char    upages[UPAGES][NBPG];
        } user;
#if !defined(bsd)
        static struct ucred ucred;
#endif
        char *up;
        struct pte *pte, uptes[UPAGES];
        int n;
#define min(a,b)        (a < b ? a : b)
        /* try in core */
        if (p->p_flag & SLOAD) {
                mread(kmem, (daddr_t)p->p_addr,(char *)uptes, 
                      UPAGES * sizeof(struct pte));
                pte = uptes;
                for(up = (char *)&user, n = sizeof(struct user); n > 0; 
                    n -= NBPG) {
                        mread(mem, (daddr_t)(pte++->pg_pfnum * NBPG),
                              up,min(n,NBPG));
                        up += NBPG;
                }
        }
        else 
                mread(swap, (daddr_t)dtob(p->p_swaddr),
                      (char *)&user,sizeof(user));
#if !defined(bsd)       
        mread(kmem, (daddr_t)user.u.u_cred, &ucred, sizeof(ucred));
        user.u.u_cred = &ucred;
#endif
        return &user.u;
}


/*
 * get the inode corresponding to current working directory
 */
struct inode *getcdir(kmem, addr)
int kmem;
daddr_t addr;
{
        static struct inode in;
        struct inode *cdir;
        
#if defined(ultrix)
        struct gnode g;
        mread(kmem, addr, &g, sizeof(g));
        cdir = GTOI(&g);
#endif
#if defined(bsd)
        mread(kmem, addr, &in, sizeof(in));
        cdir = &in;
#endif
#if defined(sun)
        struct vnode v;
        mread(kmem, addr, &v, sizeof(v));
        mread(kmem, (daddr_t)VTOI(&v), &in, sizeof(in));
        cdir = &in;
#endif
        return cdir;
}
