/*
 * Pracct.c -- print out the system accounting file with a minimum of
 * processing.  This is sort of a RISC version of /etc/sa.  As the man page
 * says, sa has a near google of options; unfortunately, it is rare that
 * it has exactly the right one.  With pracct, you can simply get an ascii
 * dump of the fields of interest and let awk do the the rest.
 */

# ifndef LINT
static char rcsid[] = "$Header: pracct.c,v 1.7 88/01/25 22:57:16 roy Exp $";
# endif

/*
 * $Log:	pracct.c,v $
 * Revision 1.7  88/01/25  22:57:16  roy
 * Removed copyright notice; code placed in public domain.
 * 
 * Revision 1.6  87/11/17  15:10:55  roy
 * Added uid/gid name caches.
 * 
 * Revision 1.5  86/12/02  17:31:43  roy
 * Fixed but with printing ac_comm -- if command name is full
 * length of ac_comm[], it isn't null terminated; we now are
 * careful not to run past this.  Changed symbolic flag format
 * to print '-' instead of ' ' for missing flags to keep the number
 * of fields on an output line constant.
 * 
 * Revision 1.4  86/12/02  14:34:52  roy
 * Added support for -v (verbose) and -a
 * (alternate accounting file) options.
 * 
 * Revision 1.3  86/12/02  00:33:21  roy
 * Added command-line option processing using getopt.
 * 
 * Revision 1.2  86/11/29  23:29:16  roy
 * Fixed rcsid string (capitalized $ H e a d e r : $).
 * Added type casts to keep lint happy.
 * 
 * Revision 1.1  86/11/29  23:07:16  roy
 * Initial revision
 * 
 */

# include <sys/types.h>
# include <sys/acct.h>
# include <sys/file.h>
# include <sys/time.h>
# include <pwd.h>
# include <grp.h>
# include <stdio.h>

# define ACCT_FILE	"/usr/adm/acct"	/* raw accounting data file */
# define DFLT_FLDS	"UcXB"		/* default fields to print */
# define FIELD_SEP	' '		/* how to delimit fields on output */
# define GETOPT_FMT	"va:f:"		/* permissible command-line flags */
# define USAGE_MSG	"pracct [-v] [-a file] [-f [cvVsSxXeEbBuUgGmitTfF]]"

/*
 * Symbolic names and flags for fields in acct(5) structure
 */
# define COMM		'c'	/* command name */
# define UTIME		'v'	/* numeric user CPU time (seconds) */
# define SYM_UTIME	'V'	/* symbolic user time (DD+HH:MM:SS) */
# define STIME		's'	/* numeric system time (seconds) */
# define SYM_STIME	'S'	/* symbolic system time (DD+HH:MM:SS) */
# define SUMTIME	'x'	/* user+system time (seconds) */
# define SYM_SUMTIME	'X'	/* symbolic user+system time (DD+HH:MM:SS) */
# define ETIME		'e'	/* numeric elapsed time (seconds) */
# define SYM_ETIME	'E'	/* symbolic elapsed time (DD+HH:MM:SS) */
# define BTIME		'b'	/* numeric beginning time (seconds) */
# define SYM_BTIME	'B'	/* symbolic beginning time (ctime format) */
# define UID		'u'	/* numeric user I.D. */
# define SYM_UID	'U'	/* login name */
# define GID		'g'	/* numeric group I.D. */
# define SYM_GID	'G'	/* group name */
# define MEM		'm'	/* average memory usage */
# define IO		'i'	/* disk I/O blocks */
# define TTY		't'	/* major/minor device for control tty */
# define SYM_TTY	'T'	/* name of control tty /dev/tty?? */
# define FLAG		'f'	/* flags in octal */
# define SYM_FLAG	'F'	/* flags as FSCDX */

/*
 * Name cache stuff.
 */
# define NC_NUID	256	/* size of uid cache; power of 2 */
# define NC_UIDMASK	0xff	/* log base 2 (NC_NUID) - 1 */
# define NC_NGID	64	/* size of gid cache;  power of 2 */
# define NC_GIDMASK	0x3f	/* log base 2 (NC_NGID) - 1 */

struct nc
{
	int nc_n;		/* numeric uid or gid */
	char *nc_name;		/* user or group name; NULL if unused */
} unc[NC_NUID], gnc[NC_NGID];

main (argc, argv)
int argc;
char *argv[];
{
	struct acct ac;
	int fd, acsize, first, nbytes, verbose, n, maxcom;
	long utime, stime, etime, sumtime, btime;
	char *fields, *f, *ct, *ctime(), *dhms(), *uid2name(), *gid2name();
	char c, *afile;
	extern int optind;
	extern char *optarg;

	fields = DFLT_FLDS;
	verbose = 0;
	afile = ACCT_FILE;

	/*
	 * figure out the maximum number of characters
	 * stored for each command name.
	 */
	maxcom = sizeof (ac.ac_comm) / sizeof (*ac.ac_comm);

	while ((c = getopt (argc, argv, GETOPT_FMT)) != EOF)
	{
		switch (c)
		{
		case 'f':
			fields = optarg;
			break;

		case 'v':
			verbose = 1;
			break;

		case 'a':
			afile = optarg;
			break;

		case '?':
			fprintf (stderr, "pracct: illegal option (-%c)\n", c);
			fprintf (stderr, "pracct: usage: %s\n", USAGE_MSG);
			exit (1);

		default:
			fprintf (stderr, "bad return from getopt (%o)!\n", c);
			abort ();
		}
	}

	/*
	 * Open accounting file for reading.
	 */
	if ((fd = open (afile, O_RDONLY, 0)) < 0)
	{
		perror ("pracct: can't open accounting file");
		exit (1);
	}

	/*
	 * Flush name cache.
	 */
	for (n = 0; n < NC_NUID; n++)
		unc[n].nc_name = NULL;

	for (n = 0; n < NC_NGID; n++)
		gnc[n].nc_name = NULL;

	/*
	 * Read the accounting file, one accounting-structure-worth
	 * at a time, until EOF or error.
	 */
	acsize = sizeof (ac) / sizeof (char);
	while ((nbytes = read (fd, (char *) &ac, acsize)) == acsize)
	{
		/*
		 * Convert from snazzy compressed floating point format to
		 * plain old integers.  It's easier to just do all the
		 * conversions than to check to see which ones we actually
		 * need.  We often want sumtime, so we rarely lose big.
		 * This may get changed later if it proves to be important.
		 */
		utime = expand (ac.ac_utime);
		stime = expand (ac.ac_stime);
		sumtime = utime + stime;
		etime = expand (ac.ac_etime);

		/*
		 * Step though the desired fields in the order specified.
		 * Insert a field delimiter in front of every field except
		 * for the first one.  Warning: fields may have internal
		 * blanks, i.e. those in ctime format.
		 */
		first = 1;
		for (f = fields; *f != NULL; f++)
		{
			if (first)
				first = 0;
			else
				putchar (FIELD_SEP);

			switch (*f)
			{
			case COMM:
				/*
				 * We have to be careful not to print past
				 * the end of the string because the command
				 * name is only null terminated if it is
				 * less than maxcom characters long.
				 */
				printf ("%-*.*s", maxcom, maxcom, ac.ac_comm);
				continue;

			case UTIME:
				printf ("%8ld", utime);
				if (verbose)
					printf ("user");
				continue;

			case SYM_UTIME:
				printf ("%11s", dhms (utime));
				if (verbose)
					printf ("user");
				continue;

			case STIME:
				printf ("%8ld", stime);
				if (verbose)
					printf ("sys");
				continue;

			case SYM_STIME:
				printf ("%11s", dhms (stime));
				if (verbose)
					printf ("sys");
				continue;

			case SUMTIME:
				printf ("%8ld", sumtime);
				if (verbose)
					printf ("u+s");
				continue;

			case SYM_SUMTIME:
				printf ("%11s", dhms (sumtime));
				if (verbose)
					printf ("u+s");
				continue;

			case ETIME:
				printf ("%8ld", etime);
				if (verbose)
					printf ("real");
				continue;

			case SYM_ETIME:
				printf ("%11s", dhms (etime));
				if (verbose)
					printf ("real");
				continue;

			case BTIME:
				printf ("%10ld", (long) ac.ac_btime);
				if (verbose)
					printf ("begin");
				continue;

			case SYM_BTIME:
				/*
				 * Beginning time in ctime format.  Ctime
				 * does us a favor and sticks a newline at
				 * the end of the string, which we then have
				 * to be careful not to print.
				 */
				btime = ac.ac_btime;
				ct = ctime (&btime);
				while (*ct != '\n'  && *ct != NULL)
					putchar (*(ct++));
				if (verbose)
					printf (" begin");
				continue;

			case UID:
				printf ("%5d", ac.ac_uid);
				if (verbose)
					printf ("uid");
				continue;

			case SYM_UID:
				printf ("%-8s", uid2name (ac.ac_uid));
				continue;

			case GID:
				printf ("%5d", ac.ac_gid);
				if (verbose)
					printf ("gid");
				continue;

			case SYM_GID:
				printf ("%-8s", gid2name (ac.ac_gid));
				continue;

			case MEM:
				printf ("%5d", ac.ac_mem);
				if (verbose)
					printf ("mem");
				continue;

			case IO:
				printf ("%10ld", (long) expand (ac.ac_io));
				if (verbose)
					printf ("io");
				continue;

			case TTY:
				if (verbose)
				{
					printf ("%3d", major (ac.ac_tty));
					printf ("/%3d", minor (ac.ac_tty));
				}
				else
				{
					printf ("%3d", major (ac.ac_tty));
					printf (" %3d", minor (ac.ac_tty));
				}
				continue;

			case SYM_TTY:
				printf ("no SYM_TTY yet");
				continue;

			case FLAG:
				printf ("%03o", ac.ac_flag);
				continue;

			case SYM_FLAG:
				putchar (ac.ac_flag & AFORK   ? 'F' : '-');
				putchar (ac.ac_flag & ASU     ? 'S' : '-');
				putchar (ac.ac_flag & ACOMPAT ? 'C' : '-');
				putchar (ac.ac_flag & ACORE   ? 'D' : '-');
				putchar (ac.ac_flag & AXSIG   ? 'X' : '-');
				continue;
			}
		}
		/*
		 * After all the required fields, end the line.
		 */
		putchar ('\n');
	}

	if (nbytes > 0)
	{
		fprintf (stderr, "pracct: short read on accounting file\n");
		exit (1);
	}
	if (nbytes < 0)
	{
		perror ("pracct: read error on accounting file");
		exit (1);
	}
	exit (0);
}

/*
 * Dhms -- convert time in seconds to DD+HH:MM:SS format.  If
 * time is negative or greater than 99+23:59:59, return all ?'s.
 * Result is returned as a fixed-width, null-terminated, static
 * string which is overwritten on each call.
 */
char *dhms (time)
register long time;	/* , no see :-) */
{
	int sec, min, hour, day;
	static char buf[(sizeof ("DD+HH:MM:SS") / sizeof (char)) + 1];

	sec = time % 60;
	time = time / 60;
	min = time % 60;
	time = time / 60;
	hour = time % 24;
	time = time / 24;
	day = time;

	if (time < 0 || day > 99)
		sprintf (buf, "??+??:??:??");
	else
		sprintf (buf, "%02d+%02d:%02d:%02d", day, hour, min, sec);

	return (buf);
}

/*
 * Expand -- convert time from compressed floating point format to normal
 * integer.  This routine is based on one in the 4.2BSD /etc/sa.c source
 * file.  Warning: this depends on the format of t being exactly as in
 * 4.2BSD -- 3 bit power-of-8 exponent in high-order bits and 13 bit
 * mantissa in low-order bits.
 */
time_t expand (t)
comp_t t;
{
	register time_t xtime;
	register exp;

	/*
	 * Get mantissa and exponent.
	 */
	xtime = t & 017777;
	exp = (t >> 13) & 07;

	/*
	 * Compute time = mantissa * 8^exp.
	 */
	while (exp != 0)
	{
		exp--;
		xtime <<= 3;
	}

	return(xtime);
}

/*
 * Uid2name -- convert a numeric user i.d. to a login name by looking it
 * up in /etc/passwd.  The login name is returned as a null-terminated,
 * static string which is overwritten on each call.  A name cache speeds
 * this up many fold (without it, over two-thirds of the user cpu time can
 * be spent here).
 */
char *uid2name (uid)
register int uid;
{
	struct passwd *getpwuid(), *pw;
	register struct nc *entry;
	char *malloc();
	register char *name, *temp;
	register int l;

	entry = &unc[uid & NC_UIDMASK];
	if (entry->nc_name == NULL || entry->nc_n != uid)
	{
		entry->nc_n = uid;
		if (entry->nc_name != NULL)
			free (entry->nc_name);

		if ((pw = getpwuid (uid)) == NULL)
			name = "???";
		else
			name = pw->pw_name;

		l = 0;
		for (temp = name; *temp != NULL; temp++)
			l++;
		if ((entry->nc_name = malloc (l+1)) == NULL)
		{
			fprintf (stderr, "out of memory in uid2name\n");
			exit (1);
		}
		temp = entry->nc_name;
		while ((*(temp++) = *(name++)) != NULL)
			;
	}
	return (entry->nc_name);
}

/*
 * Gid2name -- convert a numeric group i.d. to a group name by
 * looking it up in /etc/group.  Similar to uid2name, q.v.
 */
char *gid2name (gid)
register int gid;
{
	struct group *getgrgid(), *gr;
	register struct nc *entry;
	char *malloc();
	register char *name, *temp;
	register int l;

	entry = &gnc[gid & NC_GIDMASK];
	if (entry->nc_name == NULL || entry->nc_n != gid)
	{
		entry->nc_n = gid;
		if (entry->nc_name != NULL)
			free (entry->nc_name);

		if ((gr = getgrgid (gid)) == NULL)
			name = "???";
		else
			name = gr->gr_name;

		l = 0;
		for (temp = name; *temp != NULL; temp++)
			l++;
		if ((entry->nc_name = malloc (l+1)) == NULL)
		{
			fprintf (stderr, "out of memory in gid2name\n");
			exit (1);
		}
		temp = entry->nc_name;
		while ((*(temp++) = *(name++)) != NULL)
			;
	}
	return (entry->nc_name);
}
