/*
 * pages - show process paging information
 *
 * Written by:	DJ Molny
 *		ccom consultants, inc.
 *		5963 Oakwood Dr.
 *		Lisle, IL 60532
 *		ihnp4!chinet!djmolny
 *
 * This program is placed in the public domain.  It may be distributed
 * and modified freely providing this notice is reproduced in its entirety.
 * This program may not be sold.
 *
 * The "pages" program is highly dependent on AT&T's implementation
 * of the 3B2 UNIX System V Release 3 operating system.  It might be
 * modified to other members of the AT&T 3B family, but porting to
 * other paging implementations of UNIX is probably impossible.
 *
 * Refer to the accompanying manual page for usage instructions and
 * the inevitable disclaimers.
 *
 */

#include <stdio.h>
#include <sys/types.h>
#include <nlist.h>
#include <fcntl.h>
#include <pwd.h>
#include <sys/stat.h>
#include <sys/immu.h>
#include <sys/psw.h>
#include <sys/pcb.h>
#include <sys/signal.h>
#include <sys/fs/s5dir.h>
#include <sys/user.h>
#include <sys/region.h>
#include <sys/param.h>
#include <sys/proc.h>
#include <sys/var.h>
#include <sys/sysmacros.h>
#include <sys/sys3b.h>

static char sccsid[] = "@(#)pages.c	1.11 2/19/88";

#define	PNAME	"pages"
#define DATAFILE "/etc/pg_data"

#define	MAXREAD	(64 * 1024)		/* max bytes to read in one shot */

struct nlist ksymbs[] = {
	"proc",     (long) 0, (short) 0, (unsigned short) 0, (char) 0, (char) 0,
	"v",        (long) 0, (short) 0, (unsigned short) 0, (char) 0, (char) 0,
	"freemem",  (long) 0, (short) 0, (unsigned short) 0, (char) 0, (char) 0,
	(char *) 0, (long) 0, (short) 0, (unsigned short) 0, (char) 0, (char) 0,
};

int quick = 0;		/* quick mode (off by default)		*/
int ttys;		/* tty # to show (defaults to current tty)	*/
int tflag = 0;		/* tty # specified (off by default)		*/
int aflag = 0;		/* show all logged in users (off by default)	*/
int eflag = 0;		/* show all procs (off by default)		*/
int uids = -1;		/* user id # (defaults to unused)		*/

int memfd;
struct var v;

static struct regblk {		/* pertinent info. for the regions we select */
	unsigned long pid;
	char cmd[DIRSIZ];
	short size;
	short incore;
};

struct regblk *reg_tab, *reg_cur;	/* ptrs to region table */

#define Error(s)	{ perror(s); }


main(argc, argv)
int argc;
char **argv;
{
	extern char *optarg;
	struct passwd *pwd, *getpwnam();
	struct stat sbuf, sbuf2;
	char ttynm[128];
	char *malloc();
	int datafd;
	int i;

	while ((i = getopt(argc, argv, "qaeu:t:")) != -1)
	{
		switch(i) {
			case 'q':
				quick++;
				eflag++;
				break;
			case 'e':
				eflag++;
				break;
			case 'a':
				aflag++;
				break;
			case 'u':
				if (!optarg) usage();
				if (*optarg >= '0' && *optarg <= '9')
					uids = atoi(optarg);
				else {
					pwd = getpwnam(optarg);
					if (!pwd)
					{
					    fprintf(stderr,
						"No such user: %s\n", optarg);
					    usage();
					}
					uids = pwd->pw_uid;
				}
				break;
			case 't':
				if (!optarg) usage();
				tflag++;
				strcpy(ttynm, "/dev/");
				if (*optarg >= '0' && *optarg <= '9')
					strcat(ttynm, "tty");
				strcat(ttynm, optarg);
				if (stat(ttynm, &sbuf) == -1)
				{
					perror(ttynm);
					exit(4);
				}

				ttys = sbuf.st_rdev;
				break;
			default:
				usage();
		}
	}


	/* for default case (no arguments), act as if
	   we're invoked with our own tty specified */

	if (!aflag && !eflag && !tflag && uids == -1)
	{
		tflag++;

		/* check stdin, stdout, stderr for a file
		   descriptor connected to a tty */

		for (i = 0; i < 3; i++)
			if (isatty(i))
				break;

		if (i == 3)
		{
			fprintf(stderr, "%s: can't find tty\n", PNAME);
			exit(5);
		}

		/* stat the tty file */
		if (fstat(i, &sbuf) == -1)
		{
			perror(PNAME);
			exit(6);
		}

		/* and extract the internal device number */
		ttys = sbuf.st_rdev;
	}

	/* open kernel memory */

	if ((memfd = open("/dev/kmem", O_RDONLY)) == -1)
	{
		Error("/dev/kmem");
		exit(7);
	}

	/* stat /unix to see if it's changed since
	   we last wrote the data cache file */

	if (stat("/unix", &sbuf) == -1)
	{
		Error("/unix");
		exit(10);
	}

	/* if our cache file is gone, or if unix is newer than
	   the cache file, do the lengthy nlist() call and re-write
	   the cache */

	if (stat(DATAFILE, &sbuf2) == -1 || sbuf.st_mtime > sbuf2.st_mtime)
	{
		if (nlist("/unix", ksymbs) == -1)	/* LONG library call */
		{
			Error("nlist");
			exit(11);
		}

		/* create/truncate the cache file */

		datafd = open(DATAFILE, O_CREAT | O_WRONLY | O_TRUNC, 0664);

		if (datafd == -1)
		{
			Error(DATAFILE);
			exit(12);
		}

		/* save the whole name list structure */
		if (write(datafd, ksymbs, sizeof(ksymbs)) != sizeof(ksymbs))
		{
			Error(DATAFILE);
			exit(13);
		}

		close(datafd);

	} else {	/* cache is current, so read ksymbs from there */

		datafd = open(DATAFILE, O_RDONLY);

		if (datafd == -1)
		{
			Error(DATAFILE);
			exit(14);
		}

		/* snarf in the whole array */
		if (read(datafd, ksymbs, sizeof(ksymbs)) != sizeof(ksymbs))
		{
			Error(DATAFILE);
			exit(16);
		}

		close(datafd);
	}


	/* seek to the "var" kernel structure (contains tunables) */
	if (lseek(memfd, ksymbs[1].n_value, 0) == -1)
	{
		Error("memory lseek");
		exit(17);
	}

	/* and read it */
	if (read(memfd, &v, sizeof(v)) != sizeof(v))
	{
		Error("memory read");
		exit(18);
	}

	/* allocate one regblk per region */
	reg_tab = (struct regblk *) malloc(sizeof(struct regblk) * v.v_region);

	if (reg_tab == NULL)	/* make sure malloc succeeded */
	{
		fprintf(stderr, "can't grab %d bytes\n",
			sizeof(struct regblk) * v.v_region);
		exit(19);
	}

	procloop();			/* loop through proc table */
	showpage();			/* display the paging info */

	return(0);
}


procloop()
{
	struct user u;
	struct region r;
	struct pregion pr;
	struct proc p;
	register struct proc *pp;
	int incore, npages;
	int i;

	pp = &p;
	reg_cur = reg_tab;	/* initialize region table ptr */

	/* for each process... */
	for (i=0; i < v.v_proc; i++)
	{
		/* seek to process table */
		if (lseek(memfd, ksymbs[0].n_value + i * sizeof(p), 0) == -1)
		{
			Error("memory lseek");
			exit(21);
		}

		if (read(memfd, &p, sizeof(p)) != sizeof(p))
		{
			Error("memory read");
			exit(22);
		}

		if (!pp->p_stat)	/* if p_stat == 0, no proc in slot */
			continue;

		/* use handy sys3b call to read the u block */
		/* (see what I mean about non-portable!?) */
		if (sys3b(RDUBLK, pp->p_pid, &u, sizeof(u)) == -1)
			continue;

		/* if we're not showing just any process... */
		if (!eflag)
		{
			if (aflag)	/* show all procs connected to ttys */
			{
				if (u.u_ttyp == 0) /* if no tty, ignore */
					continue;
			} else {	/* all procs, subject to tty # & uid */

				/* skip proc if not the right tty */
				if (tflag && (!u.u_ttyp || ttys != u.u_ttyd))
					continue;

				/* skip proc if not the right user id */
				if (uids != -1 && uids != pp->p_uid)
					continue;
			}
		}

		incore = npages = 0;	/* init counters */

		while(1)		/* zoom thru region table */
		{
			/* lseek may fail if conditions change during our
			   run; therefore this is a non-fatal error */
			if (lseek(memfd, pp->p_region, 0) != (int) pp->p_region)
				break;

			/* reads may also fail non-fatally */
			if (read(memfd, &pr, sizeof(pr)) != sizeof(pr))
				break;

			/* if region ptr = 0, it's part of the chain */
			if (!pr.p_reg)
			{
				/* log it */
				logr((unsigned long) pp->p_pid, npages,
							incore, u.u_comm);
				break;
			}

			/* point to next spot in the chain */
			pp->p_region++;

			/* go to the region itself */
			if (lseek(memfd, pr.p_reg, 0) != (int) pr.p_reg)
				break;

			/* and read it */
			if (read(memfd, &r, sizeof(r)) != sizeof(r))
				break;

			/* if it's PRIVATE or only used in one place,
			   count it against the current process */
			if (r.r_type == RT_PRIVATE || r.r_refcnt == 1)
			{
				npages += r.r_pgsz;
				incore += r.r_nvalid;
			} else		/* it's shared, so just log it */
				logs((unsigned long) pr.p_reg, r.r_pgsz,
					r.r_nvalid, u.u_comm, pr.p_type);
		}
	}
}


/* log a shared region in the table */

logs(addr, size, incore, cmd, ptype)
char *addr;
int size, incore;
char *cmd;
int ptype;
{
	register struct regblk *r;
	char *cmdtype;

	switch(ptype) {
		case PT_TEXT:	/* for shared text, save the cmd name */
			cmdtype = cmd;
			break;
		case PT_DATA:
			cmdtype = "<DATA>";
			break;
		case PT_STACK:	
			cmdtype = "<STACK>";
			break;
		case PT_SHMEM:
			cmdtype = "<SHMEM>";
			break;
		case PT_DMM:
			cmdtype = "<DMM>";
			break;
		case PT_LIBTXT:
			cmdtype = "<LIBTXT>";
			break;
		case PT_LIBDAT:
			cmdtype = "<LIBDAT>";
			break;
		default:
			cmdtype = "<OTHER>";	/* punt */
			break;
		}

	/* loop thru the region table, looking to see if this one
	   has been entered before (to avoid dups) */

	for (r = reg_tab; r < reg_cur; r++)
		if (r->pid == (unsigned long) addr)
			return;

	/* a new region; enter it now */
	r->pid = (unsigned long) addr;
	strncpy(r->cmd, cmdtype, DIRSIZ);
	r->size = size;
	r->incore = incore;

	reg_cur++;		/* bump table ptr */
}



/* log a region in the table */

logr(pid, size, incore, cmd)
unsigned long pid;
int size, incore;
char *cmd;
{
	register struct regblk *r = reg_cur;

	r->pid = (unsigned long) pid;	/* save the process id */
	strncpy(r->cmd, cmd, DIRSIZ);	/* save the cmd name */
	r->size = size;			/* save the region size */
	r->incore = incore;		/* save the # of pages in core */

	reg_cur++;			/* bump table ptr */
}


showpage()
{
	if (quick)
		show_q();
	else
		show_n();
}


/* quick mode */
show_q()
{
	register struct regblk *r;
	int swapped = 0;
	int incore = 0;
	int freemem;

	/* zip thru region table, adding up pages in core & pages swapped */
	for (r = reg_tab; r < reg_cur; r++)
	{
		if (!r->size)
			continue;
		incore += r->incore;
		swapped += r->size - r->incore;
	}

	/* seek to freemem's address */
	if (lseek(memfd, ksymbs[2].n_value, 0) == -1)
	{
		Error("memory lseek");
		exit(23);
	}

	/* read freemem */
	if (read(memfd, &freemem, sizeof(freemem)) != sizeof(freemem))
	{
		Error("memory read");
		exit(24);
	}

	/* print statitistics */
	printf("%4d pages free in core\n", freemem);
	printf("%4d pages active in core\n", incore);
	printf("%4d pages swapped on disk\n", swapped);
}


/* per-process display */
show_n()
{
	register struct regblk *r;

	printf("  PID  NSWAP/TOTL   COMMAND\n");

	/* zip thru region table, printing processes & shared regions */
	for (r = reg_tab; r < reg_cur; r++)
	{
		if (r->pid < 64*1024)
			printf("%5d", r->pid);
		else
			printf("SHARE");

		printf("   %4d %4d   %-.*s\n", r->size - r->incore,
						r->size, DIRSIZ, r->cmd);
	}
}


/* print canned error message */
usage()
{
	fprintf(stderr,
		"usage: %s [-e] [-a] [-u user] [-t tty]\n", PNAME);
	exit(99);
}
