/* CHK=0x71AC */
/* vi: set tabstop=4 shiftwidth=4: */
/*+-------------------------------------------------------------------------
	gettyfe.c - front end to SCO XENIX getty
	by Warren Tucker ...!gatech!{emory,kd4nc}!wht

As exec'd by init, getty gets:
argv[0] = "-"
argv[1] = "ttyxx"
argv[2] = "?" where ? is gettydefs character from /etc/ttys

This program becomes /etc/getty, the supplied getty being renamed
/etc/real-getty.  My sole dial-in line is tty2D (your mileage may vary).
Thus, if gettyfe sees that it is invoked for a tty other than tty2D,
it immediately chains to "real-getty".

gettyfe counts on the following two gettydefs entries:

2 # B1200 HUPCL OPOST CR0 ECHOE NL1 #
	B1200 CS8 SANE HUPCL TAB3 ECHOE IXANY #@\r\nlogin: # 3

3 # B2400  HUPCL OPOST CR0 ECHOE NL1 #
	B2400 CS8 SANE HUPCL TAB3 ECHOE IXANY #@\r\nlogin: # 2

The port is opened and the file descriptor is dup(2)'d three times.
This means that fd 0 (stdin), 1 (stdout) and 2 (stderr) is established
as well as fd 3, the use of which is described below.

The port is set up for 2400 baud, NL mapping to CR and otherwise very
raw.  The caller is prompted to set to 8-N-1 and to press ENTER.  If the
caller is calling at 2400 baud, if he/she/it presses either CR or NL
(via INLCR), then gettyfe executes "real-getty" such that it will prompt
for a login at 2400 baud.  Otherwise, gettyfe changes the baud rate to
1200, repeats the prompt and so on until CR or NL is found at the proper
baud rate.

When a chain to "real-getty" happens after the tty has been opened, fds
0-2 are closed since getty expects to be able to open them.  fd 3 is
left open so that the tty will not be closed.  I'm not sure if this
is necessary or not, or whether or not the problem can be fixed by
omitting HUPCL, but I didn't find out.

Finally, an alarm(30) call gives the caller 30 seconds to start a getty.

Notes for variations:
with port set at 2400 baud, a 1200 baud 8-N CR appears as 0xE6
with port set at 1200 baud, a 2400 baud 8-N CR appears as 0xFE

To compile for XENIX 286 or 386:
% cc -O -i -s gettyfe.c -o gettyfe

It probably will work under BSD4 if you hacked the code to replace
nap calls with select() calls of equivalent (I have put my copy in here)

--------------------------------------------------------------------------*/
/*+:EDITS:*/
/*:01-30-1989-20:08-wht-creation */

#include <fcntl.h>
#include <signal.h>
#include <termio.h>

int fdtty;
struct termio tty_termio;

#if defined(DEBUG)
#include <stdio.h>
/*+-------------------------------------------------------------------------
	debugit(cbaud,user_char)
--------------------------------------------------------------------------*/
debugit(cbaud,user_char)
int cbaud;
unsigned char user_char;
{
FILE *fp = fopen("/gettyfe.log","a");
	fprintf(fp,"baud %d user_char %02x\n",(cbaud == B2400) ? 2400 : 1200,
		user_char);
	fclose(fp);
}	/* end of debugit */
#endif

#if defined(BSD4)
/*+-------------------------------------------------------------------------
	nap(msec) - BSD4 nap(); needs <time.h>
--------------------------------------------------------------------------*/
long
nap(msec)
long	msec;
{
struct timeval tp;
struct timezone tzp;
u_long	start_sec;
long	start_usec;
long	target_usec;
long	now_usec;

	gettimeofday(&tp,&tzp);
	start_sec = tp.tv_sec;
	start_usec = tp.tv_usec;
	target_usec = start_usec + (msec * 1000L);
	now_usec = start_usec;
	while(now_usec < target_usec)
	{
		gettimeofday(&tp,&tzp);
		now_usec = ((tp.tv_sec - start_sec) * 1000000L) + tp.tv_usec;
	}
	return((now_usec - start_usec) / 1000L);
}	/* end of nap */
#endif

/*+-------------------------------------------------------------------------
	bye(code)
--------------------------------------------------------------------------*/
void
bye(code)
int code;
{
#if defined(DEBUG)
FILE *fp = fopen("/gettyfe.log","a");
	fprintf(fp,"exit(%d)\n",code);
	fclose(fp);
#endif
	exit(code);
}	/* end of bye */

/*+-------------------------------------------------------------------------
	login_timeout()
--------------------------------------------------------------------------*/
void
login_timeout()
{
	bye(13);
}	/* end of login_timeout */

/*+-------------------------------------------------------------------------
	setline(cbaud)
--------------------------------------------------------------------------*/
setline(cbaud)
int cbaud;
{
	tty_termio.c_cflag &= ~CBAUD;
	tty_termio.c_cflag |= cbaud;
	ioctl(fdtty,TCSETA,(char *)&tty_termio);
	nap(100L);
	ioctl(fdtty,TCFLSH,0);
}	/* end of setline */

/*+-------------------------------------------------------------------------
	main(argc,argv,envp)
--------------------------------------------------------------------------*/
main(argc,argv,envp)
int		argc;
char	**argv;
char	**envp;
{
int cbaud;
unsigned char user_char;
char ttyname[64];
char *prompt = "\r\nUse 8-N-1 and press ENTER: ";
int promptlen = strlen(prompt);

	if(strcmp(argv[1],"tty2D"))
	{
		execv("/etc/real-getty",argv);
		bye(1);
	}

	strcpy(ttyname,"/dev/");
	strcat(ttyname,argv[1]);

	if((fdtty = open(ttyname,O_RDWR)) < 0)		/* open stdin */
		bye(2);

	dup(0);			/* create stdout */
	dup(0);			/* create stderr */
	dup(0);			/* create an extra we will not close on exec */

	tty_termio.c_iflag = IGNPAR | IGNBRK | INLCR;
	tty_termio.c_oflag = 0;
	tty_termio.c_cflag = CREAD | HUPCL | CS8 | CSTOPB | B2400;
	tty_termio.c_lflag = 0;
	tty_termio.c_cc[VMIN]  = 1;
	tty_termio.c_cc[VTIME] = 0;
	ioctl(fdtty,TCSETA,(char *)&tty_termio);

	nap(1750L);		/* allow for other end carrier detect */

	signal(SIGALRM,login_timeout);
	alarm(30);		/* 30 secs to finish this */

	cbaud = B2400;

	while(1)
	{
		write(1,prompt,promptlen);
		read(0,&user_char,1);
#if defined(DEBUG)
		debugit(cbaud,user_char);
#endif
		switch(user_char)
		{
			case 0x0D:
				close(0);	/* getty will re-open */
				close(1);
				close(2);	/* keep fd 3 open */
				alarm(0);
				execl("/etc/real-getty","-",argv[1],
					(cbaud == B2400) ? "3" : "2",(char *)0);
				bye(3);

			default:
				cbaud = (cbaud == B2400) ? B1200 : B2400;
				setline(cbaud);
		}
	}
	/*NOTREACHED*/

}	/* end of main */

