static char sccsid[] = "@(#)$Id: chkfile.c, V4.1 88/11/16 17:29:30 $";

/*
 * chkfile.c - analyze file
 * Version  : 4.1 - 88/11/16 17:29:30
 *
 * Author   : Michael J. Young
 * USmail   : Software Development Technologies, Inc.
 *            375 Dutton Rd
 *            Sudbury MA 01776
 * UUCP     : harvard!sdti!mjy
 * Internet : mjy@sdti.SDTI.COM
 *
 * =========================================================================
 * Note : This program has been placed in the public domain to permit
 * unrestricted distribution and use.  I have placed no copyright on it, but
 * I request that you keep me informed about any enhancements and bug fixes
 * you make so I can keep an up-to-date copy for further distribution.
 *
 * This program is being provided "as is", with no warrantee as to safety or
 * accuracy of results.  Use at your own risk.
 * =========================================================================
 */

/*
 * Modification History:
 *
 * Thu Jul 28 15:00:44 EDT 1988 - M. Young (mjy@sdti.SDTI.COM),
 *    Extracted from fsanalyze.c
 *
 * Mon Aug 08 11:27:39 EDT 1988 - M. Young (mjy@sdti.SDTI.COM),
 *    Revised OS_TYPE and FS_TYPE macros to avoid name-space conflicts
 *
 * Fri Nov 11 16:25:44 EST 1988 - M. Young (mjy@sdti.SDTI.COM),
 *    Fixed boundary condition error in scan().
 *
 * Wed Nov 16 11:31:32 EST 1988 - M. Young (mjy@sdti.SDTI.COM),
 *    Placed under SCCS
 */

#include "fsconfig.h"
#include "fsanalyze.h"

/*
 * chk_indirects : scans a block containing data block numbers, looking
 * for block numbers that are not sequential.
 */
daddr_t chk_indirects (block, cur_pos, data, inode, lbn)
daddr_t block;                    /* indirect block to check */
daddr_t cur_pos;                  /* current block offset */
struct file_data *data;           /* current file statistics */
struct dinode *inode;             /* inode being processed */
daddr_t lbn;                      /* logical blk_no of 1st data block */
{
    daddr_t ind_blk[MAXINDIR];    /* holds an indirect block */
    int num_blocks;               /* number of data blocks in
                                   * an indirect block */
    daddr_t pos;                  /* current data block */
    daddr_t new_pos;              /* next data block */
    int i;                        /* loop counter */
    boolean sparse;               /* potential sparse 
                                   * file detected */

    num_blocks = nindir(fil_sys);
    data->total_blocks += (bsize(fil_sys) / fsize(fil_sys));

    /*
     * the indirect block itself should be in the correct sequence
     * with the data blocks
     */
    test_fragmentation (block, cur_pos, data);

    /*
     * get the indirect block
     */
    if (fseek (fsys, block * block_size, 0)){
        error (errno, "\nerror seeking indirect block %ld\n", block);
        /* NOTREACHED */
        }
    if (fread (ind_blk, sizeof (daddr_t), num_blocks, fsys) != num_blocks){
        error (errno, "\nerror reading indirect block %ld\n", block);
        /* NOTREACHED */
        }
    pos = block;

    /*
     * scan the data blocks looking for numbers out of sequence
     */
    sparse = FALSE;
    for (i = 0; i < num_blocks; i++){
        new_pos = ind_blk [i];
        if (new_pos == 0){
            sparse = TRUE;
            continue;
            }
        if (sparse){
            data->sparse++;
            }
        data->data_blocks += (blk_size (fil_sys, inode, lbn+i) / fsize(fil_sys));
        data->total_blocks+= (blk_size (fil_sys, inode, lbn+i) / fsize(fil_sys));
        test_fragmentation (new_pos, pos, data);
        pos = new_pos;
        }
    return pos;
    }

/*
 * chk_double_indirects : scans a block containing a list of indirect
 * blocks, checking for data block numbers that are out of sequence.
 */
daddr_t chk_double_indirects (block, cur_pos, data, inode, lbn)
daddr_t block;                            /* indirect block to check */
daddr_t cur_pos;                          /* current block offset */
struct file_data *data;                   /* current file statistics */
struct dinode *inode;                     /* inode being processed */
daddr_t lbn;                              /* logical blk_no of 1st data block */
{
    daddr_t dindirect_blk[MAXINDIR];      /* holds a double-indirect
                                           * block */
    int i;                                /* loop counter */
    int num_blocks;                       /* number of indirect blocks
                                           * in a d-i block */
    boolean sparse;                       /* potential sparse file
                                           * detected */

    num_blocks = nindir(fil_sys);
    data->total_blocks += (bsize(fil_sys) / fsize(fil_sys));

    /*
     * the double-indirect block itself should be in sequence with the
     * data blocks
     */
    test_fragmentation (block, cur_pos, data);

    /*
     * get the d-i block
     */
    if (fseek (fsys, block * block_size, 0)){
        error (errno, "\nerror seeking double indirect block %ld\n",
             block);
        /* NOTREACHED */
        }
    if (fread (dindirect_blk, sizeof (daddr_t), num_blocks, fsys) != num_blocks){
        error (errno, "\nerror reading double indirect block %ld\n",
             block);
        /* NOTREACHED */
        }

    /*
     * scan through the d-i block
     */
    cur_pos = block;
    sparse = FALSE;
    for (i = 0; i < num_blocks; i++){
        if (dindirect_blk[i] == 0){
            sparse = TRUE;
            continue;
            }
        if (sparse){
            data->sparse++;
            }
        cur_pos = chk_indirects (dindirect_blk[i],
                     cur_pos,
                     data,
		     inode,
                     lbn + (i * nindir(fil_sys)));
        }
    return cur_pos;
    }

/*
 * chk_triple_indirects : scans a block containing a list of double
 * indirect blocks, looking for data block numbers that are out of sequence.
 */
daddr_t chk_triple_indirects (block, cur_pos, data, inode, lbn)
daddr_t block;                            /* indirect block to check */
daddr_t cur_pos;                          /* current block offset */
struct file_data *data;                   /* current file statistics */
struct dinode *inode;                     /* inode being processed */
daddr_t lbn;                              /* log. blk_no of 1st data block */
{
    daddr_t tindirect_blk[MAXINDIR];      /* holds a triple-indirect
                                           * block */
    int i;                                /* loop counter */
    int num_blocks;                       /* number of double-indirect
                                           * blocks in a triple-i blk */
    boolean sparse;                       /* potential sparse file
                                           * detected */


    num_blocks = nindir(fil_sys);
    data->total_blocks += (bsize(fil_sys) / fsize(fil_sys));

    /*
     * the triple-indirect block itself should be in sequence with the
     * data blocks
     */
    test_fragmentation (block, cur_pos, data);

    /*
     * get the t-i block
     */
    if (fseek (fsys, block * block_size, 0)){
        error (errno, "\nerror seeking triple indirect block %ld\n",
             block);
        /* NOTREACHED */
        }
    if (fread (tindirect_blk, sizeof (daddr_t), num_blocks, fsys) != num_blocks){
        error (errno, "\nerror reading triple indirect block %ld\n",
             block);
        /* NOTREACHED */
        }

    /*
     * scan through the t-i block
     */
    cur_pos = block;
    sparse = FALSE;
    for (i = 0; i < num_blocks; i++){
        if (tindirect_blk[i] == 0){
            sparse = TRUE;
            continue;
            }
        if (sparse){
            data->sparse++;
            }
        cur_pos = chk_double_indirects (tindirect_blk[i],
                        cur_pos,
                        data,
			inode,
                        lbn + (i * nindir (fil_sys) * nindir (fil_sys)));
        }
    return cur_pos;
    }

/*
 * chk_file : scans the data block numbers of an i-node, looking for
 * block numbers that are out of sequence, and would thus result in excess
 * track-to-track seeking.  This function also checks directory files for
 * indirection, and performs a simple consistency check on all file sizes.
 */
void chk_file (inode, inode_number, data)
struct dinode *inode;                  /* inode info structure */
int inode_number;                      /* inode number to be
                                        * checked */
struct file_data *data;                /* current file statistics */
{
    daddr_t pos;                       /* current block */
    daddr_t new_pos;                   /* next block in file */
    int i;                             /* loop counter */
    long file_size;                    /* file size computed by
                                        * actual byte count */
    boolean sparse;                    /* potential sparse file
                                        * detected */

    pos = blk_no (dta_blk (inode, 0)); /* first data block */

    /*
     * do some simple-minded statistics gathering
     */
    if (inode->di_nlink > 1){          /* multi-linked files */
        linked_files++;
        }

    if (FILE_TYPE (inode) == S_IFDIR){ /* got a directory */
        num_directories++;
        }

    /*
     * no fragmentation checks for special files or 0-length files
     */
    if (IS_SPECIAL (inode->di_mode)){
        num_specials++;
	if (debug)printf ("inode %d is special\n", inode_number);
        return;
        }

    if (inode->di_size == 0){
	if (debug)printf ("inode %d is empty\n", inode_number);
        return;                        /* ignore 0-size files */
        }

    data->data_blocks = data->total_blocks = (blk_size(fil_sys, inode, 1) / fsize (fil_sys));
    if (debug)printf ("blk_size returns %ld\n", blk_size (fil_sys, inode, 1));

    /*
     * scan the data blocks looking for numbers out of sequence
     */
    sparse = FALSE;
    for (i = 1; i < NDADDR; i++){
        new_pos = blk_no (dta_blk (inode, i));
        if (new_pos == 0){             /* end of file */
            sparse = TRUE;
            continue;
            }
        if (sparse){
            data->sparse++;
            }
        data->data_blocks += (blk_size (fil_sys, inode, i) / fsize (fil_sys));
        data->total_blocks += (blk_size (fil_sys, inode, i) / fsize (fil_sys));
	if (debug)printf ("  blk_size returns %ld\n", blk_size(fil_sys,inode,i));
        test_fragmentation (new_pos, pos, data);
        pos = new_pos;
        }

    /*
     * Indirect block 0, if non-zero, is the number of the block
     * which contains the next NINDIR data block numbers.  It should
     * also be in sequence with the data blocks.
     */
    if (blk_no (indir_blk (inode, 0))){
        indirects++;
        /*
         * if a directory contains indirection, it is too large for
         * efficient access.  Report it.
         */
        if (FILE_TYPE (inode) == S_IFDIR){
            printf ("inode %d is a large directory\n", inode_number);
            big_directories++;
            }
        pos = chk_indirects (blk_no (indir_blk (inode, 0)),
                     pos,
                     data,
		     inode,
                     NDADDR);
        }

    /*
     * Indirect block 1, if non-zero, is the number of the block which contains
     * the next NINDIR INDIRECT block numbers.  It should also be
     * in sequence with the data blocks.  This block is called a "double-
     * indirect" block.
     */
    if (blk_no (indir_blk (inode, 1))){
        double_indirects++;
        if (rpt_indirects){
            printf ("double indirection : %d\n", inode_number);
            }
        pos = chk_double_indirects (blk_no (indir_blk (inode, 1)),
                        pos,
                        data,
			inode,
                        NDADDR+nindir(fil_sys));
        }

    /*
     * Indirect block 2, if non-zero, is the number of the block which contains
     * the next NINDIR DOUBLE-INDIRECT block numbers.  It should
     * also be in sequence with the data blocks.  This block is called a
     * "triple-indirect" block.
     */
    if (blk_no (indir_blk (inode, 2))){
        triple_indirects++;
        if (rpt_indirects){
            printf ("triple indirection : %d\n", inode_number);
            }
        pos = chk_triple_indirects (blk_no (indir_blk (inode, 2)),
                        pos,
                        data,
			inode,
                        NDADDR + (nindir (fil_sys) * nindir (fil_sys)));
        }

    /*
     * detect and display sparse files
     */
    if (data->sparse){
        printf ("inode %d is sparse\n", inode_number);
        sparse_files++;
        }

    /*
     * do a simple check to detect possible file-size errors (a la
     * fsck phase 1)
     */
    file_size = (inode->di_size + (block_size - 1)) / block_size;
    data->wasted = (block_size - (inode->di_size % block_size));
    if (file_size != data->data_blocks){
        size_errors++;
        if (rpt_errors){
            printf ("inode %d, inconsistent file size : actual blocks = %ld, computed = %ld (%ld bytes)\n",
                inode_number, data->data_blocks, 
                file_size, inode->di_size);
            }
        }
    }

/*
 * scan : scan through each i-node of a file system, compiling statistics
 * regarding fragmentation and indirection.
 */
void scan (){
    int    i, j;                        /* loop counters */
    struct dinode i_node[MAXINOPB];     /* holds a block of inodes */
    struct file_data data;              /* per-inode statistics */

    for (i = first_inode; i < num_inodes(fil_sys); i+=inopb(fil_sys)){

        /*
         * for efficiency, read a block of i-nodes at a time
         */
        get_inodes (i, i_node, inopb(fil_sys));

        /*
         * scan through each i-node that was read in
         */
        for (j = 0; i+j <= num_inodes(fil_sys) && j < inopb(fil_sys); j++){
            if (debug){
                printf ("inode %d = mode %o\r", i+j, i_node[j].di_mode);
                }
            if (i+j <= 1)continue;                 /* don't scan i-node 1 */
            if (i_node[j].di_mode != 0){           /* unused i-node ? */
                init_stats (&data, i+j, i_node[j].di_size);
                chk_file (&i_node[j], i+j, &data); /* scan blocks in file */
                log_stats (&data);
                }
            else {
                free_inodes++;
                }
            }
        }
    }

/*
 * anal_file : analyzes a single file for fragmentation.
 */
void anal_file (ino, fname)
int ino;                        /* i-node number */
char *fname;                    /* filename of this inode */
{
    struct file_data data;      /* current file statistics */

    struct dinode i_node;       /* current inode data */

    get_inodes (ino, &i_node, 1);
    init_stats(&data, ino, i_node.di_size);
    chk_file (&i_node, ino, &data);
    log_stats (&data);
    printf ("   %-14s\t%6d    %6ld   %6ld   %6.2f%%  %6.1f  %ld\n",
        fname, data.inode, data.seeks+1, 
        data.total_blocks, data.fragm*100,
        data.rel_cost, data.wasted);
    }
