/*
 * fino - find inodes owned by a specific user
 *
 * With no arguments, fino lists all inodes in a file system along with
 * various information about those inodes, including the path name.  By
 * using the flags, it is possible to restrict the listing to those inodes
 * owned by a specific users and/or group, "open" inodes (refcount > 0),
 * and inodes with zero links.
 *
 * David A. Curry
 * Purdue University Engineering Computer Network
 * April, 1986
 */
#include <sys/param.h>
#include <sys/inode.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <sys/fs.h>
#include <stdio.h>
#include <fstab.h>
#include <nlist.h>

#define ISIZ		(MAXBSIZE / sizeof(struct dinode))
#define sblock		sb_un.u_sblock

/*
 * We're going to read in the inode table and the size of the
 * inode table out of memory.
 */
struct nlist nl[] = {
#define X_INODE		0
	{ "_inode", 0, 0, 0 },
#define X_NINODE	1
	{ "_ninode", 0, 0, 0 },
#define X_END		2
	{ 0, 0, 0, 0 }
};

/*
 * The superblock structure.
 */
union superblock {
	struct	fs u_sblock;
	char 	dummy[SBSIZE];
} sb_un;

/*
 * We do pathnames by makin an array of these, one for each inode.  The
 * structure for a given inode holds the "local" filename and a pointer
 * to the parent inode (directory).  Follow the list until you hit the
 * ROOTINO, and you have a pathname constructed in reverse.
 */
struct pathcomp {
	char *pc_name;
	struct pathcomp *pc_parent;
};

int ninode;			/* number of inodes in inode table	*/
int nfiles;			/* number of files (dinode entries)	*/
struct inode *inode;		/* the in-core inode table		*/
struct dinode *dinode;		/* the on-disk inode table		*/

int uid;			/* user id to restrict printing to	*/
int gid;			/* group id to restrict printing to	*/
int lines = 0;			/* number of lines (for header printing)*/

dev_t fs_dev;			/* device number of file system		*/
char *fs_cdev;			/* character device of file system	*/
char *fs_bdev;			/* block device of file system		*/
char *filsys = NULL;		/* mount point of file system		*/

int gflag = 0;			/* print only inodes owned by gid	*/
int oflag = 0;			/* print only "open" inodes		*/
int pflag = 1;			/* print pathnames of inodes		*/
int uflag = 0;			/* print only inodes owned by uid	*/
int zflag = 0;			/* print only inodes with zero links	*/

char *dirbuf;			/* block for reading directory blocks	*/
char *pathptr = NULL;		/* for constructing pathnames		*/
char pathbuf[MAXPATHLEN];
struct pathcomp *pathlist;	/* for building list of pathnames	*/

main(argc, argv)
int argc;
char **argv;
{
	register ino_t ino;
	struct inode *iget();
	register struct inode *ip;
	register struct dinode *dp;

	/*
	 * Process arguments.
	 */
	while (--argc) {
		if (**++argv != '-') {
			filsys = *argv;
			continue;
		}

		switch (*++*argv) {
		case 'g':		/* print inodes owned by gid */
			if (--argc <= 0)
				usage();

			gid = atoi(*++argv);
			gflag = 1;
			break;
		case 'o':		/* print "open" inodes only */
			oflag = 1;
			break;
		case 'p':		/* don't print paths	    */
			pflag = 0;
			break;
		case 'u':		/* print inodes owned by uid */
			if (--argc <= 0)
				usage();

			uid = atoi(*++argv);
			uflag = 1;
			break;
		case 'z':		/* print only inodes with 0 links */
			zflag = 1;
			break;
		default:
			usage();
			break;
		}
	}

	if (filsys == NULL)
		usage();

	/*
	 * Get the informatio about the file system.
	 */
	getfsinfo(filsys);

	/*
	 * Get the info we need from the kernel and disk.
	 */
	getkinfo();
	getdinfo();

	/*
	 * Get the pathname info.
	 */
	if (pflag)
		getpinfo();

	/*
	 * For each inode....
	 */
	for (ino = ROOTINO; ino < nfiles; ino++) {
		/*
		 * Find the disk and in-core copies.
		 */
		dp = &dinode[ino];
		ip = iget(ino);

		/*
		 * Skip unallocated inodes.
		 */
		if ((dp->di_mode & IFMT) == 0)
			continue;

		/*
		 * Check group.
		 */
		if (gflag) {
			if (dp->di_gid != gid)
				continue;
		}

		/*
		 * Check "open".  If not in-core,
		 * it can't be.
		 */
		if (oflag) {
			if (ip == NULL)
				continue;
		}

		/*
		 * Check user.
		 */
		if (uflag) {
			if (dp->di_uid != uid)
				continue;
		}

		/*
		 * Check link count.
		 */
		if (zflag) {
			if (dp->di_nlink != 0)
				continue;
		}

		/*
		 * Print it out.
		 */
		printout(ino, dp, ip);
	}

	exit(0);
}

/*
 * getfsinfo - find information about the file system.  We accept any
 *	       name: the block device, the raw device, or the mount point.
 */
getfsinfo(fsys)
char *fsys;
{
	struct stat sb;
	struct fstab *fs;
	char *rawdev(), *cookdev();

#ifdef DEBUG
	printf("getfsinfo()\n");
#endif

	if (stat(fsys, &sb) < 0) {
		perror(fsys);
		exit(1);
	}

	/*
	 * Figure out which name we got, and
	 * get the names we need.
	 */
	switch (sb.st_mode & S_IFMT) {
	case S_IFCHR:
		fs_cdev = fsys;
		fs_bdev = cookdev(fsys);
		break;
	case S_IFBLK:
		fs_bdev = fsys;
		fs_cdev = rawdev(fsys);
		break;
	case S_IFDIR:
	case S_IFREG:
		if ((fs = getfsfile(fsys)) == NULL) {
			fprintf(stderr, "fino: %s: not a mounted filesystem.\n", fsys);
			exit(1);
		}

		fs_bdev = fs->fs_spec;
		fs_cdev = rawdev(fs->fs_spec);
		break;
	default:
		fprintf(stderr, "fino: %s: not a mounted filesystem.\n", fsys);
		exit(1);
		break;
	}

#ifdef DEBUG
	printf("fs_cdev = %s\n", fs_cdev);
	printf("fs_bdev = %s\n", fs_bdev);
#endif

	/*
	 * Get the device number.
	 */
	if (stat(fs_cdev, &sb) < 0) {
		perror(fs_cdev);
		exit(1);
	}

	fs_dev = sb.st_dev;

#ifdef DEBUG
	printf("fs_dev = 0x%x\n", fs_dev);
#endif
}

/*
 * getkinfo - read information from the kernel.
 */
getkinfo()
{
	int inocmp();
	register int fd;
	struct inode *inode_addr;

#ifdef DEBUG
	printf("getkinfo()\n");
#endif

	/*
	 * Find things.
	 */
	nlist("/vmunix", nl);

	if (nl[0].n_value == 0) {
		fprintf(stderr, "fino: /vmunix: no name list.\n");
		exit(1);
	}

	if ((fd = open("/dev/kmem", O_RDONLY)) < 0) {
		perror("/dev/kmem");
		exit(1);
	}

	/*
	 * Get the size of the inode table.
	 */
	lseek(fd, nl[X_NINODE].n_value, L_SET);
	read(fd, &ninode, sizeof(int));

#ifdef DEBUG
	printf("ninode = %d\n", ninode);
#endif

	/*
	 * Allocate the inode table.
	 */
	if ((inode = (struct inode *) malloc(ninode * sizeof(struct inode))) == NULL) {
		fprintf(stderr, "fino: not enough memory for inode table.\n");
		exit(1);
	}

	/*
	 * Get the address of the inode table.
	 */
	lseek(fd, nl[X_INODE].n_value, L_SET);
	read(fd, &inode_addr, sizeof(struct inode *));

	/*
	 * Get the inode table itself.
	 */
	lseek(fd, (long) inode_addr, L_SET);
	read(fd, inode, ninode * sizeof(struct inode));

	/*
	 * Sort the inode table by inode number.
	 */
	qsort(inode, ninode, sizeof(struct inode), inocmp);

	close(fd);
}

/*
 * getdinfo - get the information we need from the disk.
 */
getdinfo()
{
	register int fd, ino, iblk;

#ifdef DEBUG
	printf("getdinfo()\n");
#endif

	/*
	 * Open the raw disk.
	 */
	if ((fd = open(fs_cdev, O_RDONLY)) < 0) {
		perror(fs_cdev);
		exit(1);
	}

	/*
	 * Update the disk.
	 */
	sync();

	/*
	 * Get the superblock.
	 */
	bread(fd, SBLOCK, &sblock, SBSIZE);
	 
	/*
	 * Number of files (maximum) is number of inodes
	 * per cylinder group times number of cylinder
	 * groups.
	 */
	filsys = sblock.fs_fsmnt;
	nfiles = sblock.fs_ipg * sblock.fs_ncg;

#ifdef DEBUG
	printf("filsys = %s\n", filsys);
	printf("nfiles = %d\n", nfiles);
#endif

	/*
	 * Allocate the disk inode table.
	 */
	if ((dinode = (struct dinode *) malloc(nfiles * sizeof(struct dinode))) == NULL) {
		fprintf(stderr, "fino: not enough memory for icommon blocks.\n");
		exit(1);
	}

	/*
	 * Read the inode tables out of each cylinder group.
	 */
	for (ino=0; ino < nfiles; ino += INOPB(&sblock)) {
		iblk = fsbtodb(&sblock, itod(&sblock, ino));
		bread(fd, iblk, &dinode[ino], sblock.fs_bsize);
	}

	close(fd);
}

/*
 * getpinfo - get pathname information from disk.  This is done by
 * 	      reading the raw disk blocks for each directory in the
 * 	      filesystem.
 */
getpinfo()
{
	register int fd, ino;

#ifdef DEBUG
	printf("getpinfo()\n");
#endif

	/*
	 * Allocate one path component structure for each inode.
	 */
	if ((pathlist = (struct pathcomp *) malloc(nfiles * sizeof(struct pathcomp))) == NULL) {
		fprintf(stderr, "fino: not enough memory to construct path list.\n");
		exit(1);
	}

	/*
	 * Get a disk block chunk.
	 */
	if ((dirbuf = (char *) malloc(sblock.fs_bsize)) == NULL) {
		fprintf(stderr, "fino: not enough memory to construct path list.\n");
		exit(1);
	}

	if ((fd = open(fs_cdev, O_RDONLY)) < 0) {
		perror(fs_cdev);
		exit(1);
	}

	/*
	 * We know the name of the root inode.
	 */
	pathlist[ROOTINO].pc_name = filsys;
	pathlist[ROOTINO].pc_parent = &pathlist[ROOTINO];

	/*
	 * For each inode, if it is a directory, read its disk
	 * blocks.
	 */
	for (ino = ROOTINO; (ino < nfiles) && (pflag != 0); ino++) {
		if ((dinode[ino].di_mode & IFMT) != IFDIR)
			continue;

		readdirblks(fd, ino, &dinode[ino]);
	}

	close(fd);
}

/*
 * readdirblks - reads raw disk blocks belonging to the directory with
 *		 inode number ino.
 */
readdirblks(fd, ino, dp)
int fd, ino;
struct icommon *dp;
{
	int n, count;
	char *savestr();
	register int i, len;
	register struct direct *dirp;

#ifdef DEBUG
	printf("readdirblks(ino = %u)\n", ino);
#endif

	count = dp->ic_size.val[0];

	/*
	 * We only handle direct blocks.  Hopefully there
	 * are no directories larger than 48K (for a 4K
	 * filesystem).
	 */
	for (i=0; (i < NDADDR) && (count > 0); i++) {
		lseek(fd, dtob(fsbtodb(&sblock, dp->ic_db[i])), 0);
		read(fd, dirbuf, n = MIN(count,sblock.fs_bsize));
		dirp = (struct direct *) dirbuf;
		count -= n;

		/*
		 * Treat the block as a directory and save each name
		 * at its inode, and set the parent to this inode.
		 */
		while (n > 0) {
			/*
			 * All done with this block.
			 */
			if (dirp->d_reclen <= 0)
				break;

			/*
			 * Don't save "." and "..", we want the real
			 * names.
			 */
			if (dirp->d_ino >= ROOTINO) {
				if ((strncmp(dirp->d_name, ".", dirp->d_namlen) != 0) && (strncmp(dirp->d_name, "..", dirp->d_namlen) != 0)) {
					pathlist[dirp->d_ino].pc_name = savestr(dirp->d_name, dirp->d_namlen);
					pathlist[dirp->d_ino].pc_parent = &pathlist[ino];
				}
			}

			/*
			 * Go to the next entry.
			 */
			dirp = (struct direct *) ((int) dirp + dirp->d_reclen);
			n -= dirp->d_reclen;
		}
	}

	if (count) {
		fprintf(stderr, "fino: directory (inode %u) too big, can't do path list.\n", ino);
		pflag = 0;
	}
}

/*
 * iget - look up ino in the in-core inode table.  Simple binary search.
 */
struct inode *iget(ino)
ino_t ino;
{
	struct inode *ip;
	register int l, m, u;

	l = 0;
	u = ninode - 1;

	while (l <= u) {
		m = l + (u - l) / 2;

#ifdef XDEBUG
		printf("l = %d, m = %d, u = %d\n", l, m, u);
		printf("ino = %u, inode[m].i_number = %u\n", ino, inode[m].i_number);
#endif
		if (ino < inode[m].i_number) {
			u = m - 1;
		}
		else if (ino > inode[m].i_number) {
			l = m + 1;
		}
		else {
			/*
			 * Back up to first inode with this number (since
			 * inodes are not unique across devices).
			 */
			while ((m > 0) && (inode[m-1].i_number == inode[m].i_number))
				m--;

			ip = &inode[m];

			/*
			 * See if we can find this inode.
			 */
			while (ip->i_number == ino) {
				if ((ip->i_dev == fs_dev) && (ip->i_count != 0))
					return(ip);
				ip++;
			}

			return(NULL);
		}
	}

	return(NULL);
}

/*
 * printout - print out the inode.
 */
printout(ino, dp, ip)
ino_t ino;
register struct dinode *dp;
register struct inode *ip;
{
	char *ctime();
	char *types = "cdb-ls";

	if ((lines % 22) == 0)
		printf("Inode   Uid   Gid   Mode  Links  Size/Dev      Modified    RefCnt Flags\n");

	printf("%6u %5d %5d ", ino, dp->di_uid, dp->di_gid);
	printf("%c ", types[((dp->di_mode & IFMT) >> 12) / 2 - 1]);
	printf("%04o  %3d  ", (dp->di_mode & 07777), dp->di_nlink);

	switch (dp->di_mode & IFMT) {
	case IFBLK:
	case IFCHR:
		printf("%4d, %4d ", major(dp->di_rdev), minor(dp->di_rdev));
		break;
	case IFREG:
	case IFLNK:
	case IFDIR:
		printf("%10u ", dp->di_size);
		break;
	case IFSOCK:
	default:
		printf("%10s ", "--");
		break;
	}

	printf("%.15s ", ctime(&dp->di_mtime) + 4);

	/*
	 * Print flags.
	 */
	if (ip) {
		printf("  %2d   ", ip->i_count);

		if (ip->i_flag & ILOCKED)
			printf("LOCKED ");
		if (ip->i_flag & IUPD)
			printf("UPD ");
		if (ip->i_flag & IACC)
			printf("ACC ");
		if (ip->i_flag & IMOUNT)
			printf("MOUNT ");
		if (ip->i_flag & IWANT)
			printf("WANT ");
		if (ip->i_flag & ITEXT)
			printf("TEXT ");
		if (ip->i_flag & ICHG)
			printf("CHG ");
		if (ip->i_flag & ISHLOCK)
			printf("SHLOCK ");
		if (ip->i_flag & IEXLOCK)
			printf("EXLOCK ");
		if (ip->i_flag & ILWAIT)
			printf("LWAIT ");
		if (ip->i_flag & IMOD)
			printf("MOD ");
		if (ip->i_flag & IRENAME)
			printf("RENAME ");
		if (ip->i_flag & IXMOD)
			printf("XMOD ");
	}

	printf("\n");

	/*
	 * Print the name of the inode.
	 */
	if (pflag) {
		printname(ino, dp);
		lines += 2;
	}
	else {
		lines++;
	}
}

/*
 * printname - print the pathname that goes with this inode.
 */
printname(ino, dp)
ino_t ino;
struct dinode *dp;
{
	register char *s;
	register struct pathcomp *pc;

	pc = &pathlist[ino];
	pathbuf[MAXNAMLEN-1] = NULL;
	pathptr = &pathbuf[MAXNAMLEN-2];

	if ((dp->di_nlink == 0) || (pc->pc_name == NULL) || (pc->pc_parent == NULL)) {
		printf("%43s%s", "", "<unable to determine path>\n");
		return;
	}

	/*
	 * Build the pathname in reverse working towards the root.
	 */
	while (pc != pc->pc_parent) {
#ifdef DEBUG
		printf("Add component %s\n", pc->pc_name);
#endif
		s = &(pc->pc_name[strlen(pc->pc_name)-1]);

		while (s >= pc->pc_name)
			*pathptr-- = *s--;

		*pathptr-- = '/';

		if ((pc = pc->pc_parent) == NULL) {
			printf("%43s???/%s\n", "", ++pathptr);
			return;
		}
	}

	s = &(pc->pc_name[strlen(pc->pc_name)-1]);

	while (s > pc->pc_name)
		*pathptr-- = *s--;

	if (*++pathptr == '/')
		pathptr++;

	printf("%43s/%s\n", "", pathptr);
}

/*
 * savestr - save a string in allocated memory.
 */
char *savestr(s, n)
register char *s;
register int n;
{
	register char *t;

	if ((t = (char *) malloc(n + 1)) == NULL) {
		fprintf(stderr, "fino: out of memory.\n");
		exit(1);
	}

	strcpy(t, s);
	t[n] = NULL;
	return(t);
}

/*
 * inocmp - compare two inodes (for qsort).
 */
inocmp(a, b)
register struct inode *a, *b;
{
	if (a->i_count == 0) {
		if (b->i_count == 0)
			return(0);
		return(1);
	}

	if (b->i_count == 0)
		return(-1);

	return(((unsigned) a->i_number) - ((unsigned) b->i_number));
}

/*
 * rawdev - turn cooked device name into raw device name.
 */
char *rawdev(s)
register char *s;
{
	char *rindex();
	register char *p, *q;
	static char buf[MAXNAMLEN];

	q = buf;

	if ((p = rindex(s, '/')) != NULL) {
		while (s <= p)
			*q++ = *s++;
	}

	*q++ = 'r';

	while (*q++ = *s++)
		;

	*q = NULL;
	return(buf);
}

/*
 * cookdev - turn raw device name into cooked device name.
 */
char *cookdev(s)
register char *s;
{
	char *rindex();
	register char *p, *q;
	static char buf[MAXNAMLEN];

	q = buf;

	if ((p = rindex(s, '/')) != NULL) {
		while (s <= p)
			*q++ = *s++;
	}

	s++;

	while (*q++ = *s++)
		;

	*q = NULL;
	return(buf);
}

/*
 * bread - read a block from disk.
 */
bread(fd, bno, buf, cnt)
int fd, cnt;
unsigned bno;
char *buf;
{
	lseek(fd, (long) bno * DEV_BSIZE, L_SET);
	read(fd, buf, cnt);
}

/*
 * perror - print error messages.
 */
perror(s)
char *s;
{
	extern int errno;
	extern char *sys_errlist[];

	fprintf(stderr, "fino: %s: %s\n", s, errno ? sys_errlist[errno] : "Error 0");
}

usage()
{
	fprintf(stderr, "Usage: fino [-g gid] [-o] [-p] [-u uid] [-z] filesystem\n");
	exit(1);
}
