/*
 * dbgrep - disk block grep
 *
 * This program searches for a pattern much like "grep" (regular expressions
 * are not supported) by reading the raw file system disk blocks.  When it
 * finds the pattern it prints out a line similar to "ls" which describes
 * the file.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Netowrk
 * davy@intrepid.ecn.purdue.edu
 * May, 1988
 */
#if sun || GOULD_PN || GOULD_NP1
#define NFS	1
#endif

#include <sys/param.h>
#include <sys/time.h>
#include <sys/stat.h>

#ifdef NFS
#include <sys/vnode.h>
#include <ufs/inode.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <ufs/fs.h>
#else
#include <sys/inode.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/fs.h>
#endif

#include <fstab.h>
#include <ctype.h>
#include <stdio.h>
#include <pwd.h>
#include <grp.h>

#define sblock		sb_un.u_sblock

/*
 * The superblock structure.
 */
union superblock {
	struct	fs u_sblock;
	char	dummy[SBSIZE];
} sb_un;

/*
 * A pathname component (for -p option).
 */
struct pathcomp {
	char	*pc_name;
	struct	pathcomp *pc_parent;
};

int	nfiles;			/* number of files in the file systems	*/
int 	filedes;		/* file descriptor of the raw disk dev	*/
struct	dinode *dinode;		/* table of on-disk inode structures	*/

dev_t	fs_dev;			/* device number (major/minor) of disk	*/
char	*fs_cdev;		/* character device name of the disk	*/
char	*filsys = NULL;		/* name of the file system we're doing	*/

char	*dirbuf;		/* contents of directory (for -p option)*/
char	*pathptr = NULL;	/* pointer to front of pathname		*/
struct	pathcomp *pathlist;	/* list of pathname components		*/
char	pathbuf[MAXPATHLEN];	/* area for constructing pathnames	*/

int	iflag = 0;		/* -i - ignore case in pattern matching	*/
int	pflag = 1;		/* -p - print file pathnames on match	*/

int	*flink;			/* failure links for the pattern match	*/
int	patlen;			/* length of the pattern to be matched	*/
int	patpos;			/* position in pattern during match	*/
char	*pattern = NULL;	/* the pattern we are looking for	*/

char	*pname;			/* program name (argv[0])		*/

char	*malloc();

main(argc, argv)
int argc;
char **argv;
{
	register ino_t ino;
	register struct dinode *dp;

	pname = *argv;

	/*
	 * Process arguments.
	 */
	while (--argc) {
		/*
		 * If it's not a flag, then it's either the pattern (first)
		 * or the file system name.
		 */
		if (**++argv != '-') {
			if (pattern)
				filsys = *argv;
			else
				pattern = *argv;

			continue;
		}

		/*
		 * Option.
		 */
		switch (*++*argv) {
		case 'e':		/* expression follows		*/
			if (--argc <= 0)
				usage();

			pattern = *++argv;
			break;
		case 'i':		/* ignore case in matches	*/
			iflag = 1;
			break;
		case 'p':		/* print pathnames of files	*/
			pflag = 0;
			break;
		default:
			usage();
		}
	}

	/*
	 * Gotta have both pattern and file system.
	 */
	if (!pattern || !filsys)
		usage();

	/*
	 * Compile the failure links for the pattern match.
	 */
	compile(pattern);

	/*
	 * Get the file system information and the information we
	 * need off disk.
	 */
	getfsinfo(filsys);
	getdinfo();

	/*
	 * If we're doing pathnames, get the pathname information.
	 */
	if (pflag)
		getpinfo();

	/*
	 * For each inode on the file system...
	 */
	for (ino = ROOTINO; ino < nfiles; ino++) {
		dp = &dinode[ino];

		/*
		 * We're not interested in anything but
		 * regular files.
		 */
		if ((dp->di_mode & IFMT) != IFREG)
			continue;

		/*
		 * Non-existent file.
		 */
		if (dp->di_nlink == 0)
			continue;

		/*
		 * Reset the pattern matcher.
		 */
		resetpatpos();

		/*
		 * Start by searching the direct blocks.
		 */
		if (search_direct(dp, ino))
			printout(dp, ino);
	}

	exit(0);
}

/*
 * getfsinfo - get information about the file system by figuring out the
 *	       name of the raw device whether they gave us that, the
 *	       block device, or the mount point.
 */
getfsinfo(fsys)
char *fsys;
{
	char *rawdev();
	struct stat sb;
	struct fstab *fs;

	if (stat(fsys, &sb) < 0) {
		fprintf(stderr, "%s: ", pname);
		perror(fsys);
		exit(1);
	}

	/*
	 * See what they gave us.
	 */
	switch (sb.st_mode & S_IFMT) {
	case S_IFCHR:			/* character device		*/
		fs_cdev = fsys;
		break;
	case S_IFBLK:			/* block device			*/
		fs_cdev = rawdev(fsys);
		break;
	case S_IFDIR:			/* mount point			*/
	case S_IFREG:
		if ((fs = getfsfile(fsys)) == NULL) {
			fprintf(stderr, "%s: %s: not a mounted filesystem.\n", pname, fsys);
			exit(1);
		}

		fs_cdev = rawdev(fs->fs_spec);
		break;
	default:
		fprintf(stderr, "%s: %s: not a mounted filesystem.\n", pname, fsys);
		exit(1);
	}

	if (stat(fs_cdev, &sb) < 0) {
		fprintf(stderr, "%s: ", pname);
		perror(fs_cdev);
		exit(1);
	}

	fs_dev = sb.st_dev;
}

/*
 * getdinfo - get the information about the file system off the disk.
 */
getdinfo()
{
	register ino_t ino;
	register daddr_t iblk;

	/*
	 * Open the character device.
	 */
	if ((filedes = open(fs_cdev, O_RDONLY)) < 0) {
		fprintf(stderr, "%s: ", pname);
		perror(fs_cdev);
		exit(1);
	}

	/*
	 * Update what's on the disk.
	 */
	sync();

	/*
	 * Read the super block.
	 */
	bread(SBLOCK, &sblock, SBSIZE);

	/*
	 * Copy the name of the last mount point, and
	 * get the number of files on the file system.
	 */
	filsys = sblock.fs_fsmnt;
	nfiles = sblock.fs_ipg * sblock.fs_ncg;

	/*
	 * Allocate space for the inodes.
	 */
	if ((dinode = (struct dinode *) malloc(nfiles * sizeof(struct dinode))) == NULL) {
		fprintf(stderr, "%s: out of memory.\n", pname);
		exit(1);
	}

	/*
	 * Read in each inode from disk.
	 */
	for (ino = 0; ino < nfiles; ino += INOPB(&sblock)) {
		iblk = fsbtodb(&sblock, itod(&sblock, ino));
		bread(iblk, &dinode[ino], sblock.fs_bsize);
	}
}

/*
 * getpinfo - get the path name information by reading the directory blocks
 *	      off the disk.
 */
getpinfo()
{
	register ino_t ino;

	/*
	 * We need one path component for each file.
	 */
	if ((pathlist = (struct pathcomp *) malloc(nfiles * sizeof(struct pathcomp))) == NULL) {
		fprintf(stderr, "%s: not enough memory to construct path list.\n", pname);
		pflag = 0;
		return;
	}

	/*
	 * We need room for a disk block worth of directory information.
	 */
	if ((dirbuf = (char *) malloc(sblock.fs_bsize)) == NULL) {
		fprintf(stderr, "%s: not enough memory to construct path list.\n", pname);
		pflag = 0;
		return;
	}

	/*
	 * Start the path.
	 */
	pathlist[ROOTINO].pc_name = filsys;
	pathlist[ROOTINO].pc_parent = &pathlist[ROOTINO];

	/*
	 * For each directory, read in the directory blocks.
	 */
	for (ino = ROOTINO; (ino < nfiles) && (pflag != 0); ino++) {
		if ((dinode[ino].di_mode & IFMT) != IFDIR)
			continue;

		readdirblks(ino, &dinode[ino]);
	}
}

/*
 * search_direct - search the direct blocks of the file for the pattern.
 */
search_direct(dp, ino)
struct dinode *dp;
ino_t ino;
{
	int count;
	char *pathname();
	register int i, n;
	char dblock[MAXBSIZE];

	count = dp->di_size;

	/*
	 * For each direct block in the file...
	 */
	for (i=0; (i < NDADDR) && (count > 0); i++) {
		/*
		 * Read in the block from disk.
		 */
		bread(fsbtodb(&sblock, dp->di_db[i]), dblock, n = min(count, sblock.fs_bsize));
		count -= n;

		if (search(pattern, dblock, n))
			return(1);
	}

	/*
	 * Not in the direct blocks; start searching the
	 * indirect blocks.
	 */
	for (i=0; (i < NIADDR) && (count > 0); i++) {
		if (search_indirect(dp->di_ib[i], i, &count))
			return(1);
	}

	/*
	 * Pattern not found.
	 */
	return(0);
}

/*
 * search_indirect - search the indirect blocks of the file for the pattern.
 *		     Level = 0 indicates singly indirect blocks, level = 1
 *		     indicates doubly indrect blocks, and so on.
 */
search_indirect(blkno, level, count)
ino_t blkno;
int *count;
int level;
{
	register int i, n;
	char dblock[MAXBSIZE];
	daddr_t idblk[MAXBSIZE / sizeof(daddr_t)];

	/*
	 * Read in the block from disk.
	 */
	if (blkno)
		bread(fsbtodb(&sblock, blkno), idblk, sblock.fs_bsize);
	else
		bzero(idblk, sblock.fs_bsize);

	/*
	 * If level is zero, then this block contains disk block
	 * addresses.
	 */
	if (level <= 0) {
		/*
		 * For each disk block...
		 */
		for (i=0; (i < NINDIR(&sblock)) && (*count > 0); i++) {
			/*
			 * Read in the block from disk.
			 */
			bread(fsbtodb(&sblock, idblk[i]), dblock, n = min(*count, sblock.fs_bsize));
			*count -= n;

			/*
			 * Search for the pattern.
			 */
			if (search(pattern, dblock, n))
				return(1);
		}

		/*
		 * Not found (but note this procedure is recursive,
		 * so this zero may not go all the way back).
		 */
		return(0);
	}

	level--;

	/*
	 * Handle the next level of indrection.
	 */
	for (i=0; i < NINDIR(&sblock); i++) {
		if (search_indirect(idblk[i], level, count))
			return(1);
	}

	/*
	 * Not found, and we've read the whole file.
	 */
	return(0);
}

/*
 * search - search for pat in buf.  Buf contains count bytes.  We use the
 *	    Knuth-Moyer-Pratt algorithm as described in "Computer Algorithms",
 *	    Sara Baase, pp. 175-183.
 */
search(pat, block, count)
char *pat, *block;
int count;
{
	register char c;
	register char *p, *endblock;

	p = block;
	endblock = &block[count];

	while (p < endblock) {
		while (patpos != -1) {
			if (iflag)
				c = isupper(*p) ? tolower(*p) : *p;
			else
				c = *p;

			if (c == pat[patpos])
				break;

			patpos = flink[patpos];
		}

		/*
		 * Found it.
		 */
		if (patpos == (patlen - 1))
			return(1);

		patpos++;
		p++;
	}

	return(0);
}
	
/*
 * pathname - print out the path name of the given inode.
 */
char *pathname(dp, ino)
struct dinode *dp;
ino_t ino;
{
	register char *s;
	register struct pathcomp *pc;
	char *err = "[unable to determine file name]";

	pc = &pathlist[ino];
	pathbuf[MAXNAMLEN-1] = NULL;
	pathptr = &pathbuf[MAXNAMLEN-1];

	/*
	 * File has been removed.
	 */
	if ((dp->di_nlink == 0) || (pc->pc_name == NULL) || (pc->pc_parent == NULL))
		return(err);

	/*
	 * We build the path name in reverse by following the
	 * parent links to the root.
	 */
	while (pc != pc->pc_parent) {
		if (pc->pc_name == NULL)
			return(err);

		s = &(pc->pc_name[strlen(pc->pc_name)-1]);

		while (s >= pc->pc_name)
			*pathptr-- = *s--;

		*pathptr-- = '/';

		if ((pc = pc->pc_parent) == NULL)
			return(pathptr);
	}

	if (pc->pc_name == NULL)
		return(err);

	s = &(pc->pc_name[strlen(pc->pc_name)-1]);

	while (s > pc->pc_name)
		*pathptr-- = *s--;

	if (*++pathptr != '/')
		*--pathptr = '/';

	return(pathptr);
}

/*
 * printout - print the information about the file in a manner similar to ls.
 */
printout(dp, ino)
struct dinode *dp;
ino_t ino;
{
	char *cp;
	struct group *gr;
	struct passwd *pw;
	char pbuf[16], gbuf[16];
	struct group *getgrgid();
	struct passwd *getpwuid();
	char *ctime(), *fmtmode(), *pathname();
	time_t now, sixmonthsago, onehourfromnow;

	/*
	 * Look up the owner of the file.
	 */
	if ((pw = getpwuid(dp->di_uid)) == NULL)
		sprintf(pbuf, "%d", dp->di_uid);
	else
		sprintf(pbuf, "%s", pw->pw_name);

	/*
	 * Look up the group of the file.
	 */
	if ((gr = getgrgid(dp->di_gid)) == NULL)
		sprintf(gbuf, "%d", dp->di_gid);
	else
		sprintf(gbuf, "%s", gr->gr_name);

	printf("%6d%5d ", ino, kbytes(dbtob(dp->di_blocks)));
	printf("-%s%3d", fmtmode(dp->di_mode), dp->di_nlink);
	printf("%9s/%-9s%8d", pbuf, gbuf, dp->di_size);

	time(&now);
	cp = ctime(&dp->di_mtime);
	onehourfromnow = now + 60L*60L;
	sixmonthsago = now - 6L*30L*24L*60L*60L;

	if ((dp->di_mtime < sixmonthsago) || (dp->di_mtime > onehourfromnow))
		printf(" %-7.7s %-4.4s", cp+4, cp+20);
	else
		printf(" %-12.12s", cp+4);

	if (pflag)
		printf(" %s", pathname(dp, ino));

	putchar('\n');
}

/*
 * compile - compile the failure links for the pattern.  We use the
 *	     Knuth-Moyer-Pratt algorithm as described in "Computer
 *	     Algorithms", Sara Baase, pp. 175-183.
 */
compile(pat)
char *pat;
{
	register int i, j;

	patlen = strlen(pat);

	/*
	 * If ignoring case, just convert to lower case.
	 */
	if (iflag) {
		for (i=0; i < patlen; i++)
			pat[i] = isupper(pat[i]) ? tolower(pat[i]) : pat[i];
	}

	if ((flink = (int *) malloc(patlen * sizeof(int))) == NULL) {
		fprintf(stderr, "%s: out of memory.\n", pname);
		exit(1);
	}

	flink[0] = -1;
	i = 1;

	while (i <= patlen) {
		j = flink[i-1];

		while ((j != -1) && (pat[j] != pat[i-1]))
			j = flink[j];

		flink[i++] = j + 1;
	}
}

/*
 * readdirblks - read in directory blocks for the given inode.
 */
readdirblks(ino, dp)
struct dinode *dp;
ino_t ino;
{
	int n, count;
	char *savestr();
	register int i, len;
	register struct direct *dirp;

	count = dp->di_size;

	/*
	 * We only handle direct blocks in directories; hopefully
	 * there's no directories larger than 96k (for a 8k
	 * file system).
	 */
	for (i=0; (i < NDADDR) && (count > 0); i++) {
		/*
		 * Read the block.
		 */
		bread(fsbtodb(&sblock, dp->di_db[i]), dirbuf, n = min(count, sblock.fs_bsize));
		dirp = (struct direct *) dirbuf;
		count -= n;

		/*
		 * Pull directory entries from the directory.
		 */
		while (n > 0) {
			if (dirp->d_reclen <= 0)
				break;

			/*
			 * Save the path name component.
			 */
			if (dirp->d_ino >= ROOTINO) {
				if ((strncmp(dirp->d_name, ".", dirp->d_namlen) != 0) && (strncmp(dirp->d_name, "..", dirp->d_namlen) != 0)) {
					pathlist[dirp->d_ino].pc_name = savestr(dirp->d_name, dirp->d_namlen);
					pathlist[dirp->d_ino].pc_parent = &pathlist[ino];
				}
			}

			dirp = (struct direct *) ((int) dirp + dirp->d_reclen);
			n -= dirp->d_reclen;
		}
	}

	/*
	 * This directory had indirect blocks.
	 */
	if (count) {
		fprintf(stderr, "%s: directory too big, can't do path list.\n", pname);
		pflag = 0;
	}
}

/*
 * bread - read cnt bytes of disk block bno into buf.
 */
bread(bno, buf, cnt)
daddr_t bno;
char *buf;
int cnt;
{
	int n;

	(void) lseek(filedes, (long) dtob(bno), L_SET);

#ifdef sun
	/*
	 * Sun has a botch in which a read can return MORE
	 * than cnt; it returns units of DEV_BSIZE.
	 */
	cnt = roundup(cnt, DEV_BSIZE);
#endif

	if ((n = read(filedes, buf, cnt)) != cnt) {
		fprintf(stderr, "%s: ", pname);

		if (n >= 0)
			fprintf(stderr, "read error on %s - asked for %d, got %d.\n", fs_cdev, cnt, n);
		else
			perror(fs_cdev);

		exit(1);
	}
}

int	m1[] = { 1, S_IREAD>>0, 'r', '-' };
int	m2[] = { 1, S_IWRITE>>0, 'w', '-' };
int	m3[] = { 3, S_ISUID|(S_IEXEC>>0), 's', S_IEXEC>>0, 'x', S_ISUID, 'S', '-' };
int	m4[] = { 1, S_IREAD>>3, 'r', '-' };
int	m5[] = { 1, S_IWRITE>>3, 'w', '-' };
int	m6[] = { 3, S_ISGID|(S_IEXEC>>3), 's', S_IEXEC>>3, 'x', S_ISGID, 'S', '-' };
int	m7[] = { 1, S_IREAD>>6, 'r', '-' };
int	m8[] = { 1, S_IWRITE>>6, 'w', '-' };
int	m9[] = { 2, S_ISVTX, 't', S_IEXEC>>6, 'x', '-' };

int	*m[] = { m1, m2, m3, m4, m5, m6, m7, m8, m9};

/*
 * fmtmode - format the mode.  Ripped off from ls.
 */
char *fmtmode(flags)
int flags;
{
	register int **mp;
	register char *lp;
	static char buf[16];

	lp = buf;

	for (mp = &m[0]; mp < &m[sizeof(m)/sizeof(m[0])]; ) {
		register int *pairp = *mp++;
		register int n = *pairp++;

		while (n-- > 0) {
			if ((flags & *pairp) == *pairp) {
				pairp++;
				break;
			}
			else {
				pairp += 2;
			}
		}			

		*lp++ = *pairp;
	}

	*lp = '\0';
	return(buf);
}

/*
 * rawdev - create a raw (character) device name from a cooked (block) name.
 */
char *rawdev(s)
register char *s;
{
	char *rindex();
	register char *p, *q;
	static char buf[MAXNAMLEN];

	q = buf;

	if ((p = rindex(s, '/')) != NULL) {
		while (s <= p)
			*q++ = *s++;
	}

	*q++ = 'r';

	while (*q++ = *s++)
		;

	*q = NULL;
	return(buf);
}

/*
 * savestr - save a string in allocated memory.
 */
char *savestr(s, n)
register char *s;
register int n;
{
	register char *t;

	if ((t = (char *) malloc(n + 1)) == NULL) {
		fprintf(stderr, "%s: out of memory.\n", pname);
		exit(1);
	}

	strcpy(t, s);
	t[n] = NULL;

	return(t);
}

/*
 * resetpatpos - reset pattern position to start of pattern.
 */
resetpatpos()
{
	patpos = 0;
}

/*
 * kbytes - return kilobytes from bytes.
 */
kbytes(n)
int n;
{
	return((n + 1023) / 1024);
}

/*
 * min - minimum function.
 */
min(x, y)
int x, y;
{
	return(x < y ? x : y);
}

usage()
{
	fprintf(stderr, "Usage: %s [-e] [-i] [-p] expr filesystem\n", pname);
	exit(1);
}
