#ifndef lint
static char *sccsid = "@(#)cobwebs.c	1.2 (Don Gworek) 9/21/85";
#endif

/*
 * cobwebs [-glq] [-d days] [-s size] [username ...]
 * 
 * -g general report     -- list all mailboxes, alphabetically
 * -l don't read LASTLOG -- program runs faster
 * -q quiet flag	 -- no top banner
 *
 * -d report mailboxes older than this age in days
 * -s report mailboxes larger than this size in bytes
 */

#include <sys/param.h>
#include <stdio.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include <lastlog.h>
#include <pwd.h>

#define MAILDIR "/usr/spool/mail"
#define LASTLOG "/usr/adm/lastlog"
#define DEF_SIZE 40000		/* default mbox size limit */
#define MAXUSERNAMES 400	/* max username args */
#define MAXPACKS 1000		/* max mailboxes to report */
#define NMAX 10			/* max username length */
#define HMAX 10			/* max hostname length */
#define TRUE 1
#define FALSE 0

/*
 * sortf values
 */
#define SIZE 0
#define ALPHA 1
#define DAYS 2

struct packet {
    char    name[NMAX];
    off_t   size;
    time_t  filetime;
};
struct packet  *packs[MAXPACKS];
struct packet **packp = packs;
struct packet  *calloc ();

char   *malloc (), *ctime(), *progname;
char    usernames[MAXUSERNAMES][NMAX];

int     now, scmp(), packcmp();
int     sizeval = DEF_SIZE;
int     daysval = 0;
long    dtimeval = 0;

int     mboxcount = 0;
int     namecount = 0;
int     packcount = 0;
long    sizeaccum = 0;
long    ageaccum = 0;

int     sortf = SIZE;
int     generalf = FALSE;
int     lastlogf = TRUE;
int	quietf = FALSE;

int     ARGC;			/* global argc and argv */
char  **ARGV;

main (argc, argv)
int     argc;
char   *argv[];
{
    ARGC = argc;
    ARGV = argv;
    (void) time (&now);
    progname = (progname = (char *) rindex (*ARGV, '/')) ? ++progname : *ARGV;
    while ((ARGC-- > 0) && (*(*++ARGV) == '-') && getoptions ());
    if (ARGC && !generalf) {
	while ((ARGC-- > 0) && (namecount < MAXUSERNAMES))
	    (void) strcpy (usernames[namecount++], *ARGV++);
	if (namecount >= MAXUSERNAMES) {
	    fprintf (stderr, "%s: MAXUSERNAMES %d reached\n",
		    progname, MAXUSERNAMES);
	    exit (namecount);
	}
    }
    if (namecount)
	qsort (usernames, namecount, sizeof usernames[0], scmp);
    check_mailboxes ();
    printout ();
}

getoptions () {
    while (*++(*ARGV))
	switch (**ARGV) {
	    case '\0': 
		ARGC--, ARGV++;
		return (FALSE);
	    case 's': 
		sortf = SIZE;
		if ((*++(*ARGV)) || (ARGC-- && *++ARGV && (**ARGV != '-')))
		    sizeval = atoi (*ARGV);
		else
		    usage();
		return (TRUE);
	    case 'd': 
		sortf = DAYS;
		if ((*++(*ARGV)) || (ARGC-- && *++ARGV && (**ARGV != '-')))
		    daysval = atoi (*ARGV);
		else
		    usage();
		dtimeval = now - 86400 * daysval;
		return (TRUE);
	    case 'g': 
		sortf = ALPHA;
		generalf = TRUE;
		break;
	    case 'l': 
		lastlogf = FALSE;
		break;
	    case 'q': 
		quietf = TRUE;
		break;
	    default: 
		usage ();
	}
    return (TRUE);
}

struct stat stbuf;

check_mailboxes () {
    DIR * etc;
    struct direct  *dp;
    if (chdir (MAILDIR) < 0) {
	perror (MAILDIR);
	exit (1);
    }
    if ((etc = opendir (".")) == NULL) {
	perror (MAILDIR);
	exit (1);
    }
    while (dp = readdir (etc)) {
	if (dp -> d_ino == 0)
	    continue;
	if (stat (dp -> d_name, &stbuf) < 0)
	    continue;
	if ((stbuf.st_mode & S_IFMT) == S_IFDIR)
	    continue;
	if (packcount >= MAXPACKS)
	    fprintf (stderr, "%s: MAXPACKS %d reached\n", progname, MAXPACKS);
	if (namecount) {
	    if (binsearch (dp -> d_name)) {
		take_note (dp -> d_name);
		if (packcount >= namecount)
		    return;
	    }
	    continue;
	}
	mboxcount++;
	ageaccum += (stbuf.st_ctime - now);
	sizeaccum += stbuf.st_size;
	if (generalf) {
	    take_note (dp -> d_name);
	    continue;
	}
	switch (sortf) {
	    case DAYS:
		if (stbuf.st_ctime < dtimeval)
		    take_note (dp -> d_name);
		break;
	    case SIZE:
		if (stbuf.st_size > sizeval)
		    take_note (dp -> d_name);
		break;
	}
    }
}

take_note (name)
char   *name;
{
    packcount++;
    *packp = calloc (1, sizeof (struct packet));
    (void) strcpy ((*packp) -> name, name);
    (*packp) -> size = stbuf.st_size;
    (*packp) -> filetime = stbuf.st_ctime;
    packp++;
}

char    agestring[50];

printout () {
    register struct packet **p;
    int     f;
    struct passwd  *pwd;
    struct lastlog  ll;
    int     i;
    char   *s, *nowyear;
    char host[HMAX];
    if (packcount == 0) {
	printf ("No Report.\n");
	return;
    }
    qsort (packs, packp - packs, sizeof packs[0], packcmp);
    s = ctime (&now);
    nowyear = s + 20;
    if (lastlogf && (f = open (LASTLOG, 0)) < 0) {
	perror (LASTLOG);
	lastlogf = FALSE;
    }
    if (!quietf) {
	for (i = 65; i-- > 0; putchar ('-'));
	putchar ('\n');
	(void) gethostname (host, 10);
	printf ("%10s     %.12s     %d Mailbox%s",
		host, (s + 4), packcount, (packcount==1) ? "  " : "es");
	if (!generalf && !namecount)
	    if (sortf == DAYS)
		printf (" Older Than %d Days", daysval);
	    else
		if (sortf == SIZE)
		    printf (" Larger Than %d", sizeval);
	putchar ('\n');
	if (!namecount) {
	    age_is ((ageaccum + now) / mboxcount);
	    printf ("\nTotal: %d   Average Size: %d   Average Age:%-16s\n",
		mboxcount, (sizeaccum / mboxcount), agestring);
	}
	printf ("\n  NAME    MAILBOX SIZE      MAILBOX AGE%s\n",
		(lastlogf) ? "           LAST LOGIN" : "");
	for (i = 65; i-- > 0; putchar ('-'));
	putchar ('\n');
    }
    for (p = packs; p < packp; p++) {
	printf ("%-8s", (*p) -> name);
	printf ("     %6d     ", (*p) -> size);
	age_is (now - (*p) -> filetime);
	printf ("%-16s", agestring);
	if (lastlogf) {
	    if (((pwd = getpwnam ((*p) -> name)) != NULL)
		    && (lseek (f, (long) pwd -> pw_uid *
			    sizeof (struct lastlog), 0) >= 0)
		    && (read (f, (char *) & ll, sizeof ll) == sizeof ll)
		    && (ll.ll_time <= 0))
		printf ("        no login record");
	    else {
		s = ctime (&ll.ll_time);
		printf ("        %.6s", (s + 4));
		if (strncmp (nowyear, (s + 20), 4))
		    printf (", %.4s", (s + 20));
		else
		    printf (" %.5s", (s + 11));
	    }
	}
	putchar ('\n');
    }
    (void) close (f);
}

age_is (period)
long    period;
{
    int     days, hours;
    char    temp1[20], temp2[20];
    days = period / 86400;
    hours = (period % 86400) / 3600;
    if (days)
	(void) sprintf (temp1, "%3d Day%s", days, (days == 1) ? " " : "s");
    else
	(void) sprintf (temp1, "        ");
    if (hours)
	(void) sprintf (temp2, "%2d Hour%s", hours, (hours == 1) ? " " : "s");
    else
	(void) sprintf (temp2, "%2d Mins ", (((period % 86400) % 3600) / 60));
    (void) sprintf (agestring, "%s %s", temp1, temp2);
}

binsearch (target)
char   *target;
{
    int     hi, lo, mid, val;
    lo = 0;
    hi = namecount - 1;
    while (TRUE)
	if (hi < lo)
	    return (FALSE);
	else
	    if ((val = strcmp (target, usernames[(mid = (lo + hi) / 2)])) == 0)
		return (TRUE);
	    else
		if (val > 0)
		    lo = mid + 1;
		else
		    hi = mid - 1;
}

scmp (p, q)
char   *p, *q;
{
    return (strcmp (p, q));
}

packcmp (p, q)
struct packet **p, **q;
{
    switch (sortf) {
	case ALPHA: 
	    return (strcmp ((*p) -> name, (*q) -> name));
	case DAYS: 
	    if ((*p) -> filetime == (*q) -> filetime)
		return (0);
	    if ((*p) -> filetime > (*q) -> filetime)
		return (1);
	    return (-1);
	case SIZE:
	    if ((*p) -> size == (*q) -> size)
		return (0);
	    if ((*p) -> size > (*q) -> size)
		return (-1);
	    return (1);
    }
    return (0);			/* keep lint quiet */
}

usage () {
    fprintf (stderr, "Usage: %s [-glq] [-d days]", progname);
    fprintf (stderr, " [-s size] [username ...]\n");
    exit (1);
}
