/*
 *	checkpasswd.c - Login password check routines.
 *
 *	Perform various sanity and security checks on a password candidate.
 *
 *	Written December, 1988
 *
 *	Copyright 1989 Clyde W. Hoover (Moose & Squirrel Software, NotInc.)
 *
 *		Clyde Hoover
 *		Computation Center
 *		The University of Texas at Austin
 *		Austin, Texas
 *		clyde@emx.utexas.edu
 */

#ifndef lint
static char sccsid[] = "%W% %G% (cc.utexas.edu) %P%";
#endif

#include <sys/types.h>
#include <strings.h>
#include <ctype.h>

#define	LOCAL	static

#include "checkpasswd.h"

/*
 *	Special string compare defines.
 */
#define	try(P,C,V)	{ if (cistrcmp(P,C) == 0) return(V); }
#define	mtry(P,C,V)	{ int i; if ((i = StrAllCmp(P,C,V)) != PWCK_OK) return(i); }

/*
 *	Table of operational parameter preferences.
 *	May be modified by the caller.
 */
EXPORT
struct pwck_preferences	pwck_preferences = {
	1,		/* single-case pwds ok */
	1,		/* control chars ok */
	3,		/* dup length = 3 */
	4,		/* minimum length */
	"PATH=/bin:/usr/bin:/usr/ucb; egrep"	/* How to find egrep */
};

/*
 *	Table of control chars best avoided -
 *	mostly commonly-used terminal special chars.
 *	May be modified by the caller.
 */
#define	ctrl(d)	('d' & 037)

EXPORT
char	pwck_illegalcc[128] = {
	ctrl(c), ctrl(d), ctrl(h), /* ctrl(i),*/  ctrl(j), ctrl(m),
	ctrl(o), ctrl(r), ctrl(s), ctrl(q), ctrl(z), ctrl(\\),
	ctrl([),	/* escape */
	'\0177',	/* rubout */
	0
};

#define	PWDICT		"/usr/dict/pwwords"
/*
 *	List of forbidden password dictionaries to look in.
 *	May be modified by the caller.
 */
static char	*pwck_dicitonaries[16] = {
	PWDICT,			/* illegal passwords list */
	0
};

/*
 *	The 'pwck_*' routines all use the PWCK_* return
 *	codes, which are then propigated up to the caller of checkpassword().
 *
 *	All pwck_* routines in the table below are called thusly:
 *		pwck_*(password, userid)
 *			password = plaintext password string to test.
 *			userid = the user id which wants to use <password>.
 *
 *	Table of check functions to be called by checkpassword()
 */
int	pwck_lexical(), pwck_local(), pwck_passwd(), pwck_dictionary();

typedef	int	(*function)();

LOCAL
function pwck_vector[] = {
	pwck_lexical,
	pwck_local,
	pwck_passwd,
	pwck_dictionary,
	0
};

/* ------------------------------------------------------------------- */

/*
 *	checkpassword - Password candidate sanity checker.
 *
 *	Arguments;
 *		password = plain text password string to check.
 *		userid = the uid whom the password is for, -1 to disable.
 *
 *	Returns:
 *		PWCK_OK if <password> is ok to use.
 *		PWCK_FAIL if something failed during the check process.
 *		PWCK_NULL if <password> is the null string
 *		PWCK_OBVIOUS if <password> is too "obvious".
 *			(equals login name, host name, 'zzzz' ).
 *		PWCK_FINGER if <password> is in the users' passwd/finger info.
 *		PWCK_INDICT if <password> is in the dictionaries checked.
 *		PWCK_ILLCHAR if <password> is lexically illegal.
 *		PWCK_SHORT if <password> is too short.
 *
 */
checkpassword(password, userid)
char	*password;		/* Plaintext of password to check */
int	userid;			/* The user this is for */
{
	int		rcode;		/* General purpose scratch */
	function	*checkfunc;	/* Check function pointer */

	if (password == 0 || *password == 0)
		return(PWCK_NULL);		/* Null password */

	for (checkfunc = pwck_vector; *checkfunc; checkfunc++) {
		if ((rcode = (*checkfunc)(password, userid)) != PWCK_OK)
			return(rcode);
	}
	return(PWCK_OK);
}


/* ------------------------------------------------------------------- */

#define	P_U	0x1 	/* Upper case in password */
#define	P_L	0x2 	/* Lower case in password */
#define	P_C	0x4 	/* Control chars in password */
#define	P_D	0x8 	/* Digits in password */
#define	P_P	0x10 	/* Punctutation chars in password */

#define	hasone(P)	(what |= (P))
#define	hasany(P)	((what & (P)) == (P))

#define	ccok	pwck_preferences.CtrlOk
#define	mcok	pwck_preferences.OneCaseOk
#define	runl	pwck_preferences.CharRunLen
#define	minl	pwck_preferences.MinPwLen

/*
 *	pwck_lexical - Perform lexical analysis of password candidate.
 *
 *	Things which are ok:
 *		Mixed case
 *		Digits
 *		Punctutation
 *		Control characters (except for those in the forbidden table)
 *			(controlled by the preferences)
 *
 *	Things which are NOT ok:
 *		Passwords less that 'minl' length
 *		Runs of more than <runl> of the same character (e.g. 'zzz')
 *		Single-case strings
 *			(controlled by the preferences)
 *
 *	Things not checked for:
 *		Cycles of character groups (e.g. 'aabbcc' or 'ababab')
 */
static int
pwck_lexical(password, userid)
char	*password;
int	userid;		/* NOTUSED */
{
	int	rc;		/* Duplicate character run count */
	char	*p = password;	/* Scratch */
	char	what = 0,	/* Lexical analysis result flags */
		last = 0;	/* Last character seen (for run checks) */

	if (minl && strlen(password) < minl)
		return(PWCK_SHORT);

	for (p = password; *p; p++) {
		if (*p != last) {
			last = *p;
			rc = 0;
		}
		else {		/* Run of same characters */
			if (runl && ++rc >= runl)
				return(PWCK_OBVIOUS);
		}
		if (*p < ' ') {		/* Control character */
			if (!ccok)
				return(PWCK_ILLCHAR);
			if (index (pwck_illegalcc, *p))
				return(PWCK_ILLCHAR);
			hasone(P_C);
		}
		else if (isupper(*p))	hasone(P_U);
		else if (islower(*p))	hasone(P_L);
		else if (ispunct(*p))	hasone(P_P);
		else if (isdigit(*p))	hasone(P_D);
	}
	if (hasany(P_U | P_L))	return(PWCK_OK);	/* UC+lc */
	if (hasany(P_D))	return(PWCK_OK);	/* Numbers */
	if (hasany(P_P))	return(PWCK_OK);	/* Punctutation chars */
	if (hasany(P_C))	return(PWCK_OK);	/* Control chars */
	/*
	 *	Check for mono-case passwords 
	 */
	if (!hasany(P_U) && mcok)	/* All lower case alpha */
		return(PWCK_OK);
	if (!hasany(P_L) && mcok)	/* All upper case alpha */
		return(PWCK_OK);

	return(PWCK_ILLCHAR);
}
#undef	hasone
#undef	hasany

/*
 *	pwck_local - Perform 'local' password checks.
 *
 *	Returns:
 *		PWCK_OBVIOUS if <password> == hostname
 *		PWCK_OK if otherwise
 */
LOCAL int
pwck_local(password, userid)
char	*password;
int	userid;		/* NOTUSED */
{
	char	myname[32];		/* Scratch */

	(void) gethostname(myname, sizeof(myname));
	try(password, myname, PWCK_OBVIOUS);
	/*
	 * Want to try full canoncalized hostname here in case gethostname
	 * didn't get that for us.
	 *
	 * Then look in users' .rhosts and try those strings (maybe)
	 */
	return(PWCK_OK);
}

/*
 *	pwck_dictionary - Look in the forbidden password dictionaries.
 *
 *	Returns:
 *		PWCK_INDICT if <password> was in any dictionary
 *		PWCK_OK if not
 */
LOCAL int
pwck_dictionary(password, userid)
char	*password;
int	userid;		/* NOTUSED */
{
	int	i,		/* Counter */
		rcode;		/* Return code temp */

	for (i = 0; pwck_dicitonaries[i]; i++) {
		if ((rcode =
		     IsInDictionary(pwck_dicitonaries[i], password)) != PWCK_OK)
			return(rcode);
	}
	return(PWCK_OK);
}

/*
 *	IsInDictionary - look for <password> in <dictionary>
 *
 *	Use a DBM version of the dictionary if present, 
 *	use egrep to search the flat file if not.
 *
 *	Returns:
 *		PWCK_INDICT if <password> was found in <dictionary>
 *		PWCK_OK if not
 */
#define	returnwith(code) { dbm_close(dbp); return(code); }
#define	EGREP	pwck_preferences.EgrepPath

#include <ndbm.h>

LOCAL int
IsInDictionary(dictionary, password)
char	*dictionary,		/* Pathname of dictionary */
	*password;		/* Plaintext of password */
{
	DBM	*dbp;		/* DBM database pointer */
	datum	k,		/* DBM lookup key */
		d;		/* DBM lookup datum */
	int	uc = isupper(password[0]);	/* Is first char UC? */
	char	pwtemp[128];	/* Scratch buffer */

	if ((dbp = dbm_open(dictionary, 0, 0)) == (DBM *)0) {
		char	command[128];	/* Command build buffer */
		int	rc;		/* System() return code */

		if (access(dictionary, 0) < 0)
			return(PWCK_OK);
		/*
		 * If the first letter is capitalized, look for
		 * "[wW]ord" else look for "word"
		 */
		if (uc) 
			(void) sprintf(command,
				"%s -s '^[%c%c]%s$' %s > /dev/null",
				EGREP, password[0], password[0] & ~040,
				&password[1], dictionary);
		else
			(void) sprintf(command, "%s -s '^%s$' %s > /dev/null",
				EGREP, password, dictionary);
		rc = system(command);
		if (rc == 0)
			return(PWCK_INDICT);
		else
			return(PWCK_OK);
	} 
	/*
	 * Look in the DBM version of the dictionary.
	 * Look for <password>, then if the first letter
	 * is capitalized, force to lower and look again.  I don't care
	 * if <password> is in the dictionary but has mixed case letters,
	 * but if the first letter has been capitalized, I care because
	 * that's not a sufficent permutation to be secure.
	 */
	(void) strcpy(pwtemp, password);
	k.dptr = pwtemp;
	k.dsize = strlen(pwtemp);
	d = dbm_fetch(dbp, k);
	if (d.dptr)
		returnwith(PWCK_INDICT);
	if (uc) {
		pwtemp[0] |= 040;
		d = dbm_fetch(dbp, k);
		if (d.dptr)
			returnwith(PWCK_INDICT);
	}
	returnwith(PWCK_OK);
}
#undef	returnwith


/*
 *	pwck_password - Check password candidate against the users' password
 *		file information, or any other information that is publicly
 *		available about this user that a bandit could use as
 *		password guesses.
 *
 *	Here is the place to search your 'finger' database.
 */
#include	<pwd.h>

static int
pwck_passwd(password, userid)
char	*password;
int	userid;
{
	char	temp[128];		/* Scratch */
	struct passwd	*pwp;		/* Pointer to user information */

	if (userid < 0)			/* Can't do user checks */
		return(PWCK_OK);

	pwp = getpwuid(userid);
	if (pwp == 0)
		return(PWCK_FAIL);

	try(password, pwp->pw_name, PWCK_OBVIOUS); /* Checks 'name' and 'Name' */

	(void) strcpy(temp, pwp->pw_name);
	(void) strcat(temp, pwp->pw_name);
	try(password, temp, PWCK_OBVIOUS);	/* Check 'namename' */

	(void) strcpy(temp, pwp->pw_name);
	MirrorString(temp);
	try(password, temp, PWCK_OBVIOUS);	/* 'eman' */

	/*
	 * Try every word in user's GECOS entry
	 */
	mtry(password, pwp->pw_gecos, PWCK_FINGER);
	return(PWCK_OK);
}
/* ------------------------------------------------------------------- */
/*
 *	StrAllCmp - Compare all sub-strings (delineated by white space)
 *
 *	Returns:
 *		PWCK_OK if no match found
 *		rc if match found
 */
LOCAL
StrAllCmp(s1, s2, rc)
char	*s1,		/* String to look for */
	*s2;		/* String to look for <s1> in */
int	rc;		/* What to return on match */
{
	int	l;		/* Temp */

	for (l = strlen(s1); *s2; s2++)
		if (cistrncmp(s1, s2, l) == 0)
			return (rc);
	return(PWCK_OK);
}

/*
 *	MirrorString - reverse a string in place
 */
LOCAL
MirrorString(s)
char	*s;		/* String to reverse */
{
	char	*p;	/* Scratch */
	char	t[128];	/* Scratch */
	
	(void) strcpy(t,s);
	p = t;
	while (*p) p++;		/* Find end of string */
	--p;
	for (; *s; )
		*s++ = *p--;
}

/*
 *	Case indepedant string comparasion routines swiped from
 *	the source to MIT Hesiod.
 */
/*
 * Copyright (c) 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */

/*
 * This array is designed for mapping upper and lower case letter
 * together for a case independent comparison.  The mappings are
 * based upon ascii character sequences.
 */

LOCAL char charmap[] = {
	'\000', '\001', '\002', '\003', '\004', '\005', '\006', '\007',
	'\010', '\011', '\012', '\013', '\014', '\015', '\016', '\017',
	'\020', '\021', '\022', '\023', '\024', '\025', '\026', '\027',
	'\030', '\031', '\032', '\033', '\034', '\035', '\036', '\037',
	'\040', '\041', '\042', '\043', '\044', '\045', '\046', '\047',
	'\050', '\051', '\052', '\053', '\054', '\055', '\056', '\057',
	'\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067',
	'\070', '\071', '\072', '\073', '\074', '\075', '\076', '\077',
	'\100', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
	'\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
	'\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
	'\170', '\171', '\172', '\133', '\134', '\135', '\136', '\137',
	'\140', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
	'\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
	'\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
	'\170', '\171', '\172', '\173', '\174', '\175', '\176', '\177',
	'\200', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
	'\210', '\211', '\212', '\213', '\214', '\215', '\216', '\217',
	'\220', '\221', '\222', '\223', '\224', '\225', '\226', '\227',
	'\230', '\231', '\232', '\233', '\234', '\235', '\236', '\237',
	'\240', '\241', '\242', '\243', '\244', '\245', '\246', '\247',
	'\250', '\251', '\252', '\253', '\254', '\255', '\256', '\257',
	'\260', '\261', '\262', '\263', '\264', '\265', '\266', '\267',
	'\270', '\271', '\272', '\273', '\274', '\275', '\276', '\277',
	'\300', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
	'\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
	'\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
	'\370', '\371', '\372', '\333', '\334', '\335', '\336', '\337',
	'\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
	'\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
	'\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
	'\370', '\371', '\372', '\373', '\374', '\375', '\376', '\377',
};

LOCAL
cistrcmp(s1, s2)
register char *s1, *s2;
{
	register char *cm = charmap;

	while (cm[*s1] == cm[*s2++])
		if (*s1++=='\0')
			return(0);
	return(cm[*s1] - cm[*--s2]);
}

LOCAL
cistrncmp(s1, s2, n)
register char *s1, *s2;
register n;
{
	register char *cm = charmap;

	while (--n >= 0 && cm[*s1] == cm[*s2++])
		if (*s1++ == '\0')
			return(0);
	return(n<0 ? 0 : cm[*s1] - cm[*--s2]);
}

/* END */
