/*
 * PRFORM - PRintout FORMatter by Richard Conn 
 *
 * PRFORM generates a formatted listing of a text file with headers at
 * the top of each page, wraparound if lines exceed the width of a
 * page, etc.  PRFORM sends its output to either standard output (the
 * default) or a text file. 
 *
 */

#include <stdio.h>
#include <time.h>

#define VERSION "PRFORM Version 1.7 (Beta Test)"

/* #define MSDOS /* Uncomment to compile under MSDOS & TURBO-C */

#define MAXLL 400
#define DEFSTART 1
#define DEFENDL 0
#define DEFNLINES 62
#define DEFLM 5
#define DEFLL 80
#define DEFTABS 8
#define DEFERASE 0
#define DEFHEAD "frp"
#define DEFEHEAD ""
#define DEFOHEAD ""
#define DEFHTEXT ""
#define DEFPNUM 1

#ifndef MSDOS
#define TEMPFILENAME "/tmp/prform"
#endif				/* MSDOS */

#define HEADLEN 400
#define FORMFEED '\014'

/* Options and their default values */
int             erase = DEFERASE;
int             fileout = 0;	/* 0 if standard out, 1 if file, 2 if
				 * none */
char            head[HEADLEN] = DEFHEAD;
char            ehead[HEADLEN] = DEFEHEAD;
char            ohead[HEADLEN] = DEFOHEAD;
char            htext[HEADLEN] = DEFHTEXT;
int             ll = DEFLL;
int             lm = DEFLM;
int             numberlines = 0;
int             nlines = DEFNLINES;
int             start = DEFSTART;
int             endl = DEFENDL;
int             tabs = DEFTABS;
int             onepass = 0;
int             fpnum = DEFPNUM;

int             totalpages = 0;
int             error = 0;
long            linenum;
int             lcount;
int             pagenum;
int             pnum;
int             prok;
char           *curdt;
FILE           *fpo;

main(argc, argv)
    int             argc;
    char           *argv[];
{
    int             i;
    long            seconds;
    char           *rover;
    char           *ctime();
    int             lastfile = 0;
    int             printdone = 0;
    int             origfileout;
    int             getpid();

#ifndef MSDOS
    char            tempfile[100];
    int             unlink();
    FILE           *fp, *fopen();
    char            inline[MAXLL];
#endif

    /* Get current time and set string */
    time(&seconds);
    curdt = ctime(&seconds);
    rover = curdt;
    while (*rover) {
	if (*rover == '\n')
	    *rover = '\0';
	else
	    rover++;
    }

#ifndef MSDOS
    /* Create temporary file name if needed */
    sprintf(tempfile, "%s%d", TEMPFILENAME, getpid());
#endif

    /* Process each argument as an option or file to print */
    for (i = 1; i < argc; i++) {
	if (*argv[i] == '-') {
	    if (lastfile) {
		lastfile = 0;
		resetopts();
	    }
	    option(&argv[i][1]);
	} else {
	    lastfile = 1;
	    if (!onepass) {
		origfileout = fileout;
		fileout = 2;
		print(argv[i], "");
		fileout = origfileout;
		totalpages = pnum - 1;
	    } else
		totalpages = 0;
	    print(argv[i], "");
	    printdone = 1;
	}
    }

    /* Print from standard input if no files specified */
    if (!printdone && !error) {

#ifndef MSDOS
	if ((fp = fopen(tempfile, "w")) == NULL) {
	    fprintf(stderr, " Cannot Create Temporary File %s\n", tempfile);
	    exit(1);
	}
	while (gets(inline, MAXLL))
	    fprintf(fp, "%s\n", inline);
	fclose(fp);
	totalpages = 0;
	if (!onepass) {
	    origfileout = fileout;
	    fileout = 2;
	    print("", tempfile);
	    fileout = origfileout;
	    totalpages = pnum - 1;
	} else
	    totalpages = 0;
	print("", tempfile);
	unlink(tempfile);
#else
	help();
#endif

    }
    /* Close output file if open */
    if (fileout == 1)
	fclose(fpo);

    /* Exit program */
    exit(0);
}

/* Reset options to default values */
resetopts()
{
    char           *rover1, *rover2;

    /* Reset -e default */
    endl = DEFENDL;

    /* Reset -E default */
    onepass = 0;

    /* Reset -h defaults */
    strcpy(head, DEFHEAD);
    strcpy(ehead, DEFEHEAD);
    strcpy(ohead, DEFOHEAD);

    /* Reset -l default */
    ll = DEFLL;

    /* Reset -m default */
    lm = DEFLM;

    /* Reset -M default */
    strcpy(htext, DEFHTEXT);

    /* Reset -n default */
    numberlines = 0;

    /* Do not reset -o */

    /* Reset -p default */
    nlines = DEFNLINES;

    /* Reset -s default */
    start = DEFSTART;

    /* Reset -S default */
    fpnum = DEFPNUM;

    /* Reset -t default */
    tabs = DEFTABS;

    /* Reset -T default */
    erase = 0;

}

/* Process options */
option(text)
    char           *text;
{
    char           *rover1, *rover2;
    FILE           *fopen();
    int             cont;

    cont = 1;
    while (cont && *text) {
	switch (*text) {
	case 'a':
	    erase = 0;
	    *head = '\0';
	    *ehead = '\0';
	    *ohead = '\0';
	    ll = MAXLL;
	    lm = 0;
	    numberlines = 0;
	    nlines = 0;
	    start = DEFSTART;
	    tabs = DEFTABS;
	    break;
	case 'e':
	    endl = atoi(++text);
	    cont = 0;
	    break;
	case 'E':
	    onepass = 1;
	    break;
	case 'h':
	    rover1 = ++text;
	    switch (*rover1) {
	    case 'e':
	    case 'E':
		rover1++;
		rover2 = ehead;
		break;
	    case 'o':
	    case 'O':
		rover1++;
		rover2 = ohead;
		break;
	    default:
		rover2 = head;
		break;
	    }
	    strcpy(rover2, rover1);
	    cont = 0;
	    break;
	case 'H':
	    help();
	    error = 1;
	    cont = 0;
	    break;
	case 'l':
	    ll = atoi(++text);
	    if (ll == 0)
		ll = DEFLL;
	    cont = 0;
	    break;
	case 'm':
	    lm = atoi(++text);
	    cont = 0;
	    break;
	case 'M':
	    strcpy(htext, ++text);
	    cont = 0;
	    break;
	case 'n':
	    numberlines = 1;
	    lm = 8;
	    break;
	case 'o':
	    if (fileout == 1)
		fclose(fpo);
	    if ((fpo = fopen(++text, "w")) == NULL) {
		fprintf(stderr, " Cannot Open Output File %s\n", text);
		exit(0);
	    }
	    fileout = 1;
	    cont = 0;
	    break;
	case 'p':
	    nlines = atoi(++text);
	    cont = 0;
	    break;
	case 's':
	    start = atoi(++text);
	    fpnum = start;
	    cont = 0;
	    break;
	case 'S':
	    fpnum = atoi(++text);
	    cont = 0;
	    break;
	case 't':
	    tabs = atoi(++text);
	    cont = 0;
	    if (tabs == 0)
		tabs = DEFTABS;
	    break;
	case 'T':
	    erase = 1;
	    break;
	default:
	    fprintf(stderr, " Invalid Option at %s\n", text);
	    help();
	    error = 1;
	    break;
	} if (cont)
	    text++;
    }
}

/* Print help message */
help()
{
    fprintf(stderr,
	    "%s\n", VERSION);
    fprintf(stderr,
	    " Parameters: [[-option ...] [filename ...]] ...\n");
    fprintf(stderr,
	    " Options:\n");
    fprintf(stderr,
    "  -a       as-is output (expand tabs and page breaks only)\n");
    fprintf(stderr,
	    "  -e#      end printing after the #th page\n");
    fprintf(stderr,
	    "  -E       no ending line number in page count\n");
    fprintf(stderr,
	    "  -hfMprt  sets the page heading\n");
    fprintf(stderr,
	    "     f   places the file name into the page heading\n");
    fprintf(stderr,
	    "     M   places the header message (defined by -M)\n");
    fprintf(stderr,
	    "          into the page heading\n");
    fprintf(stderr,
	 "     p   places the page number into the page heading\n");
    fprintf(stderr,
     "     r   places the rest of the heading on the right side\n");
    fprintf(stderr,
       "     t   places the date and time into the page heading\n");
    fprintf(stderr,
	    "         no option after h sets no page heading\n");
    fprintf(stderr,
	    "          (default option sequence is %s)\n", DEFHEAD);
    fprintf(stderr,
	    "  -hefMprt sets the page heading for even pages\n");
    fprintf(stderr,
	    "          (default option sequence is %s)\n", DEFEHEAD);
    fprintf(stderr,
	    "  -hofMprt sets the page heading for odd pages\n");
    fprintf(stderr,
	    "          (default option sequence is %s)\n", DEFOHEAD);
    fprintf(stderr,
	    "  -H       prints this help message on stderr and exits\n", DEFLL);
    fprintf(stderr,
	    "  -l#      is the length of a line (def %d)\n", DEFLL);
    fprintf(stderr,
	    "  -m#      is the length of the left margin (def %d)\n",
	    DEFLM);
    fprintf(stderr,
	    "  -M<text> sets the header message to <text>\n");
    fprintf(stderr,
      "  -n       causes each line to be numbered (-M set to 8)\n");
    fprintf(stderr,
	    "  -o<name> redirects the output listing to the indicated file\n");
    fprintf(stderr,
    "  -p#      is the number of lines/page (def %d)\n", DEFNLINES);
    fprintf(stderr,
	    "  -s#      start printing at the #th page (def %d)\n", DEFSTART);
    fprintf(stderr,
    "          (note: also does -S#, setting first page number)\n");
    fprintf(stderr,
	  "  -S#      sets the number of the first printed page\n");
    fprintf(stderr,
      "  -t#      is the number of columns per tab stop (def %d)\n",
	    DEFTABS);
    fprintf(stderr,
	    "  -T       truncate lines which are too long\n");

#ifdef MSDOS
    fprintf(stderr,
	 "Using PRFORM as a filter is not supported under MSDOS\n");
#endif

}

/* Main print function for a file */
print(filename, tempfile)
    char           *filename;
    char           *tempfile;
{
    FILE           *fp, *fopen();
    char            inline[MAXLL];

    if (error)
	return;
    pnum = fpnum;
    pagenum = 1;
    linenum = 1;
    if (start <= pagenum)
	prok = 1;
    else
	prok = 0;
    if (*filename) {
	if ((fp = fopen(filename, "r")) == NULL) {
	    fprintf(stderr, " Cannot Open %s\n", filename);
	    error = 1;
	    return;
	}
    } else {
	if ((fp = fopen(tempfile, "r")) == NULL) {
	    fprintf(stderr, " Cannot Open Temporary File %s\n", tempfile);
	    error = 1;
	    return;
	}
    }
    pagehead(filename);
    while (fgets(inline, MAXLL, fp) != NULL) {
	inline[strlen(inline) - 1] = '\0';
	prline(filename, inline, 0);
    }
    pageeject();
    fclose(fp);
}

/* Eject Page for next listing */
pageeject()
{
    prc(FORMFEED);
    pagenum++;
    if (start < pagenum)
	pnum++;
    if (start <= pagenum) {
	if (endl && (pagenum > endl))
	    prok = 0;
	else
	    prok = 1;
    } else
	prok = 0;
}

/* Print Page Heading */
pagehead(heading)
    char           *heading;
{
    char           *rover;
    char            rhead[HEADLEN];
    char            lhead[HEADLEN];
    char            outhead[HEADLEN];
    char           *output;
    int             olen;
    int             i;

    if (prok) {
	*rhead = '\0';
	*lhead = '\0';
	output = lhead;

	/* Output heading if any kind of heading has been defined */
	if (*head || *ehead || *ohead) {
	    prsp(0);
	    rover = head;
	    if ((pnum % 2) && *ohead)
		rover = ohead;
	    if ((pnum % 2 == 0) && *ehead)
		rover = ehead;
	    while (*rover) {
		switch (*rover) {
		case 'f':
		    sprintf(output, "%s    ", heading);
		    output = &output[strlen(output)];
		    break;
		case 'M':
		    sprintf(output, "%s", htext);
		    output = &output[strlen(output)];
		    break;
		case 'p':
		    if (totalpages)
			sprintf(output, "Page %d of %d    ", pnum,
				totalpages);
		    else
			sprintf(output, "Page %d    ", pnum);
		    output = &output[strlen(output)];
		    break;
		case 'r':
		    *output = '\0';
		    output = rhead;
		    break;
		case 't':
		    sprintf(output, "%s    ", curdt);
		    output = &output[strlen(output)];
		    break;
		default:
		    break;
		}
		rover++;
	    }
	    *output = '\0';

	    /* Strip trailing spaces off the right and left headings */
	    rover = rhead;
	    output = rover;
	    while (*rover) {
		if (*rover != ' ')
		    output = rover;
		rover++;
	    }
	    if (*output)
		*++output = '\0';
	    rover = lhead;
	    output = rover;
	    while (*rover) {
		if (*rover != ' ')
		    output = rover;
		rover++;
	    }
	    if (*output)
		*++output = '\0';

	    /*
	     * Build the composite heading from the left and right
	     * headings 
	     */
	    olen = (ll - lm) - (strlen(rhead) + strlen(lhead));
	    if (olen < 0)
		olen = 0;
	    output = outhead;
	    rover = lhead;
	    while (*rover)
		*output++ = *rover++;
	    for (i = 0; i < olen; i++)
		*output++ = ' ';
	    rover = rhead;
	    while (*rover)
		*output++ = *rover++;
	    *output = '\0';

	    /*
	     * Print the composite heading to a file or standard
	     * output 
	     */
	    if (fileout == 1)
		fprintf(fpo, "%s\n\n", outhead);
	    else {
		if (fileout == 0)
		    printf("%s\n\n", outhead);
	    }
	    /* Two lines have been output */
	    lcount = 2;
	} else
	    /* No lines have been output */
	    lcount = 0;
    } else {
	if (*head || *ehead || *ohead)
	    lcount = 2;
	else
	    lcount = 0;
    }
}

/* Print Line on page */
prline(heading, line, cont)
    char           *heading;
    char           *line;
    int             cont;
{
    char           *rover1, *rover2;
    int             len;
    int             cchar;
    int             looping;

    if (cont && !numberlines)
	len = ll - lm - 3;
    else
	len = ll - lm;
    if (nlines) {
	if (lcount >= nlines) {
	    pageeject();
	    pagehead(heading);
	}
    }
    if (cont && !numberlines)
	prs(">> ");
    if (cont)
	prsp(0);
    else
	prsp(1);
    rover1 = line;
    cchar = 0;
    looping = 1;
    while (looping) {
	switch (*rover1) {
	case '\t':
	    prs(" ");
	    cchar++;
	    while (cchar % tabs) {
		prs(" ");
		cchar++;
	    }
	    rover1++;
	    if (cchar >= len) {
		rover2 = rover1;
		while (*rover2 == ' ')
		    rover2++;
		if (*rover2) {
		    prs("\n");
		    lcount++;
		    if (!erase)
			prline(heading, rover1, 1);
		    looping = 0;
		}
	    }
	    break;
	case FORMFEED:
	    prs("\n");
	    pageeject();
	    pagehead(heading);
	    cchar = 0;
	    rover1++;
	    break;
	case '\0':
	    prs("\n");
	    lcount++;
	    looping = 0;
	    break;
	default:
	    prc(*rover1++);
	    cchar++;
	    if (cchar >= len) {
		rover2 = rover1;
		while (*rover2 == ' ')
		    rover2++;
		if (*rover2) {
		    prs("\n");
		    lcount++;
		    if (!erase)
			prline(heading, rover1, 1);
		    looping = 0;
		}
	    }
	    break;
	}
    }
}

/* Print leading spaces */
prsp(putnumber)
    int             putnumber;
{
    int             i;

    if (numberlines) {
	if (putnumber) {
	    if (fileout == 1)
		fprintf(fpo, "%7ld ", linenum++);
	    else {
		if (fileout == 0)
		    printf("%7ld ", linenum++);
	    }
	} else {
	    prs("        ");
	}
    } else {
	if (lm > 0)
	    for (i = 0; i < lm; i++)
		prs(" ");
    }
}

/* Print a character based on PROK flag */
prc(ch)
    char            ch;
{
    if (prok) {
	if (fileout == 1)
	    fprintf(fpo, "%c", ch);
	else {
	    if (fileout == 0)
		printf("%c", ch);
	}
    }
}

/* Print a string based on PROK flag */
prs(str)
    char           *str;
{
    if (prok) {
	if (fileout == 1)
	    fprintf(fpo, "%s", str);
	else {
	    if (fileout == 0)
		printf("%s", str);
	}
    }
}
