/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: utils.c
 * Utility routines used by several programs
 ***************************************************************************
 * Revision History: Created Sat Jan  9 15:02:12 CST 1988
 * $Log:	utils.c,v $
 * Revision 3.2  88/06/24  17:55:27  papowell
 * MODS for VAX 4.3BSD UNIX
 * 
 * Revision 3.1  88/06/18  09:35:55  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.4  88/05/19  10:34:26  papowell
 * Fixed open() calls to have a 0 parameter, ie: open(f, perms, 0), where needed
 * 
 * Revision 2.3  88/05/16  12:09:27  papowell
 * Spelling mistake in error messages
 * 
 * Revision 2.2  88/05/14  10:18:31  papowell
 * Use long format for job file names;
 * Added 'fd', no forward flag;
 * Control file has to have hostname and origination agree.
 * 
 * Revision 2.1  88/05/09  10:10:42  papowell
 * PLP: Released Version
 * 
 * Revision 1.7  88/04/27  20:27:34  papowell
 * Modified to remove unused variables
 * 
 * Revision 1.6  88/04/15  13:06:55  papowell
 * Std_environ() call added, to ensure that fd 0 (stdin), 1 (stdout), 2(stderr)
 * have valid file descriptors;  if not open, then /dev/null is used.
 * 
 * Revision 1.5  88/03/25  15:01:57  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.4  88/03/12  10:03:45  papowell
 * *** empty log message ***
 * 
 * Revision 1.3  88/03/11  19:29:25  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.2  88/03/05  15:01:28  papowell
 * Minor Corrections,  Lint Problems
 * 
 * Revision 1.1  88/03/01  11:09:27  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: utils.c,v 3.2 88/06/24 17:55:27 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"
/*
 * Time_str: return ctime() value, but without terminating '\n'
 */
char *
Time_str()
{
	time_t tvec;		/* time */
	char *ctime();
	static char s[40];

	(void)time(&tvec);
	(void)strcpy(s,ctime(&tvec));
	s[strlen(s)-1] = 0;
	return( s );
}

/***************************************************************************
 * Remove_job( FILE *fp; struct queue *q )
 * Remove a job and all associated files; if success == JSUCC, remove
 * U links as well
 * ACTIONS:
 * 1. Read the control file
 * 2. get data file names and unlink them if they match the job
 * 3. Unlink the control file
 ***************************************************************************/

Remove_job( fp, q )
	FILE *fp;
	struct queue *q;
{
	char parm[BUFSIZ];	/* line buffer */
	char *arg;			/* pointer to the argument */
	int c;				/* ACME Integers, Inc. */
	
	if(Debug>5)log(XLOG_DEBUG,"Remove_job: %s", q->q_name );
	clearerr( fp );
	if( fseek( fp, 0L, 0 ) < 0 ){
		logerr_die( XLOG_INFO,
			"Remove_job: fseek failed (%s)", q->q_name );
	} else {
		/*
		 * read through the file, looking for things to remove
		 */
		while(fgets( parm, sizeof(parm), fp )){
			if( (arg = index(parm, '\n')) == 0 ){
				log( XLOG_INFO, "Remove_job: file bad (%s), no endline",
					q->q_name);
				continue;
			}
			*arg = 0;
			arg = parm+1;
			c = parm[0];
			/*
			 * is it a format line?
			 */
			if( !isascii( c ) || !islower( c ) ){
				continue;
			}
			if( Job_match( q->q_name, arg ) == 0){
				if(Debug>4)log(XLOG_DEBUG,"Remove_job: bad file name '%s'",
					arg );
				continue;
			}
			(void)unlink_daemon( arg );
		}
	}
	if( unlink_daemon( q->q_name ) < 0 ){
		logerr( XLOG_INFO, "Remove_job: cannot unlink %s", q->q_name );
	}
}
/***************************************************************************
 * int Getq();
 * Scan the spool directory and make a list of server files sorted by
 * priority, creation time, etc.
 * Returns: the number of entries
 * Side Effects: sets struct queue *Queue to point to an array of pointers
 *   to queue entries
 ***************************************************************************/
static int arraysz;				/* number of entries in array */
static struct queue qb;			/* for zeroing purposes */

Getq()
{
	struct direct *d;		/* directory entry */
	int nitems;				/* ACME Integers, Inc. */
	struct stat stbuf;		/* for statting files */
	int compar();			/* for sorting files */
	DIR *dirp;				/* used by directory routines */
	char l[BUFSIZ];			/* for reading the control file */
	char fullname[BUFSIZ];	/* for the full pathname */
	char *filepart;			/* end of the pathname */
	FILE *cfp;				/* control file FP */
	struct queue *q;		/* pointer to queue entry */
	char *bp, *ep;			/* ACME Pointers, Inc. */

	if (SD == 0 || *SD == 0 ){
		fatal( XLOG_INFO, "no directory entry" );
	}
	if ((dirp = opendir(SD)) == NULL){
		logerr_die( XLOG_INFO, "cannot open spool directory %s", SD );
	}

	/*
	 * get the directory name and copy to buffer
	 */
	if( strlen(SD) + sizeof(q->q_name) + 2 > sizeof(fullname) ){
		logerr_die( XLOG_NOTICE,"INTERNAL: Getq, file name too big" );
	}
	(void)strcpy(fullname,SD);
	filepart = &fullname[strlen(fullname)];
	*filepart++ = '/';
	/*
	 * assume you have 100 entries to start with
	 */
	if( arraysz == 0 ){
		arraysz = 100;
		Queue = (struct queue *)malloc(
				(unsigned)(arraysz * sizeof(struct queue)));
		if (Queue == NULL){
			logerr_die( XLOG_INFO, "Getq: malloc failed" );
		}
	}
	/*
	 * set the item count to 0 and scan the directory
	 */
	nitems = 0;
	while ((d = readdir(dirp)) != NULL) {
		if(Debug>5)log(XLOG_DEBUG,"Getq: file %s", d->d_name );

		if (d->d_name[0] != 'c' || d->d_name[1] != 'f'){
			continue;	/* server control files only */
		}
		/*
		 * Check to make sure the array has space left and
		 * realloc the maximum size.
		 */
		if( nitems >= arraysz ){
			/* try to get another buffer */
			arraysz = arraysz + 20;
			Queue = (struct queue *)realloc((char *)Queue,
				(unsigned)((arraysz)*sizeof(struct queue)));
			if (Queue == NULL){
				logerr_die(XLOG_NOTICE,"malloc failed in Getq" );
			}
		}
		/*
		 * get next queue entry and clear it
		 */
		q = &Queue[nitems];
		*q = qb;	/* zero it out the easy way */

		if(Debug>5)log(XLOG_DEBUG,"Getq: control file %s", d->d_name );

		/* copy the name of the file */
		if( strlen( d->d_name ) >= CFNAMELEN){
			logerr_die(XLOG_NOTICE,"control file name %s too long", d->d_name );
		}
		(void)strcpy(q->q_name, d->d_name);
		(void)strcpy(filepart,d->d_name);
		if (stat(fullname, &stbuf) < 0){
			continue;	/* Doesn't exist */
		}
		/* get the modification time */
		q->q_time = stbuf.st_mtime;
		/* set a priority */
		q->q_priority = d->d_name[2];
		/* get the job number */
		q->q_num = atoi( &d->d_name[3] );
		
		/*
		 * now we open the control file and read the information
		 */
		if( (cfp = fopen_daemon( fullname, "r" )) == NULL ){
			/* it disappeared */
			continue;
		}
		if(Debug>5)log(XLOG_DEBUG,"Getq: control file %s readable", d->d_name );
		bp = q->q_data;
		ep = bp + sizeof(q->q_data);
		while( fgets( l, sizeof(l), cfp) ){
			/* clobber the \n */
			l[strlen(l)-1] = 0;
			if( islower( l[0] ) ){
				/* get size of file to be printed */
				(void)strcpy(filepart, &l[1] );
				if( stat( fullname,  &stbuf) < 0){
					(void)fclose(cfp);
					if(Debug>5)log(XLOG_DEBUG,
						"Getq: data file %s does not exist", l );
				} else {
					q->q_size += stbuf.st_size;
				}
			} else switch( l[0] ){
				case 'N':	/* name of file */
					bp = estrcp( bp, &l[1], ep );
					bp = estrcp( bp, " ", ep );
					break;
				case 'P':	/* name of user */
					(void)strncpy( q->q_user, &l[1], sizeof(q->q_user)-1);
					break;
				case 'D':	/* when spooled */
					q->q_sp = (time_t)atol( &l[1] );
					break;
				case 'T':	/* when to be unspooled */
					q->q_unsp = (time_t)atol( &l[1] );
					break;
				default: break;
			}
		}
		if(Debug>5)log(XLOG_DEBUG,"Getq: control file %s is ok", d->d_name );

		(void)fclose(cfp);
		++nitems;
	}
	closedir(dirp);

	/*
	 * sort the queue
	 */
	if (nitems){
		qsort((char *)Queue, nitems, sizeof(struct queue), compar);
	}
	return(nitems);
}


/*
 * Compare queue entries by
 * 1. Priority
 * 2. Time
 */
static
compar(c1, c2)
	char *c1, *c2;
{
	struct queue *p1, *p2;

	p1 = (struct queue *)c1;
	p2 = (struct queue *)c2;
	if ((p1)->q_priority < (p2)->q_priority)
		return(-1);
	if ((p1)->q_priority > (p2)->q_priority)
		return(1);
	if ((p1)->q_time < (p2)->q_time)
		return(-1);
	if ((p1)->q_time > (p2)->q_time)
		return(1);
	return(0);
}

/***************************************************************************
 * Match_entry( struct queue *q )
 * check the user name and job number against the parmlist starting at start;
 * Return: 1 if match found, 0 otherwise 
 ***************************************************************************/
int
Match_entry( q )
	struct queue *q;
{
	int i;		/* ACME Integer, Inc. */

	if(Debug>4)log(XLOG_DEBUG,"Match_entry on %s, %d", q->q_user, q->q_num );
	for( i = 0; i < Parmcount; ++i ){
		if( (Parms[i].num >= 0 && Parms[i].num == q->q_num)
			|| strcmp( Parms[i].str, q->q_user ) == 0 ){
			return(1);
		}
	}
	return(0);
}

/***************************************************************************
 * bpread( int fd ; char *buf; int size )
 * Almost bombproof way to read command line from socket
 * Returns: -1 if unable to read a line terminated with \n
 *     0 if 0 length line or first character is null (i.e.- no input)
 *     otherwise: length of line
 ***************************************************************************/
bpread( fd, buf, size )
	int fd;
	char *buf;
	int size;
{
	int n;			/* ACME Integer, Inc. */
	int i;

	/*
	 * Bombproof line reading
	 */
	for( i = 0; i < size; ++i ){
		n = read(fd, &buf[i], 1);
		if( n < 0 ){
			logerr(XLOG_INFO,"bpread: lost connection");
			return( -1 );
		} else if( n == 0 ){
			log(XLOG_INFO,"bpread: bad format %s", buf );
			return( 0 );
		}
		if( buf[i] == '\n' ){
			buf[i] = 0;
			if(Debug>3)log(XLOG_DEBUG,"bpread: '%d' %s", buf[0], &buf[1]);
			return( i );
		} else if( buf[i] == 0 && i == 0 ){
			if(Debug>3)log(XLOG_DEBUG,"bpread: zero byte first");
			return( i );
		}
	}
	log(XLOG_INFO,"bpread: bad line %s", buf );
	return( -1 );
}
/***************************************************************************
 * estrcp(char *s, *t, *e)
 * copies t to s, and returns the end position
 * Returns: end of string if the result is shorter than e, NULL otherwise
 * Note: this is a very useful routine.  Think about adding it to string
 * functions.
 ***************************************************************************/
char *
estrcp( s,t,e )
	char *s, *t, *e;
{
	if( s ){
		while( (*s++ = *t++) && (s < e) );
		if( s < e ){
			return( s-1 );
		} else {
			return( (char *)0 );
		}
	}
	return( s );
}
/***************************************************************************
 * splitline( char *line )
 *    splits a line into tokens, and places them in the Parms[] array.
 *    if firstarg is present then it is set to the first argument
 * Side Effects: modifies Parms, Parmcount
 ***************************************************************************/

splitline( bp )
	char *bp;
{
	/* find the start */
	while (*bp) {
		while( *bp && isspace(*bp))
			bp++;
		if( *bp ){
			if( Parmcount >= MAXPARMS ){
				fatal(XLOG_INFO,"splitline: too many requests");
			}
			Parms[Parmcount].str = bp;
			if (isdigit(*bp)) {
				Parms[Parmcount].num = atoi(bp);
			} else {
				Parms[Parmcount].num = -1;
			}
			Parmcount++;
		}
		while( *bp && !isspace(*bp))
			bp++;
		if( *bp )
			*bp++ = 0;
	}
}
/**********************************************************************
 * Shift_parms( int n )
 * left shift the entries in the Parms[] array by n
 * Side Effects: modifies Parms and Parmcount
 **********************************************************************/
Shift_parms( n )
	int n;
{
	int i;		/* ACME Chain and Integer, Inc. */
	while( Parmcount > 0 && n > 0 ){
		for( i = 1; i < Parmcount; ++i ){
			Parms[i-1] = Parms[i];
		}
		--Parmcount;
		--n;
	}
}

printstatus()
{
	if( ST && *ST ){
		if(Debug>4)log(XLOG_DEBUG,"printstatus: ST %s", ST );
		pr_stat_file( ST );
	}
	if( PS && *PS ){
		if(Debug>4)log(XLOG_DEBUG,"printstatus: PS %s", PS );
		pr_stat_file( PS );
	}
}

pr_stat_file( file )
	char *file;
{
	FILE *fp;
	char buf[BUFSIZ];

	if( (fp = fopen_daemon( file, "r" )) != NULL ){
		while( fgets( buf, sizeof(buf), fp) != NULL ){
			(void)fprintf(stdout, "  %s", buf );
		}
		(void)fflush(stdout);
		(void)fclose( fp );
	}
}
/***************************************************************************
 * char *Sigstr(n)
 * Return a printable form the the signal
 ***************************************************************************/

struct signame{
	char *str;
	int value;
} signals[] = {
{ "SIGHUP", SIGHUP }, { "SIGINT", SIGINT }, { "SIGQUIT", SIGQUIT },
{ "SIGILL", SIGILL }, { "SIGTRAP", SIGTRAP }, { "SIGIOT", SIGIOT },
{ "SIGEMT", SIGEMT }, { "SIGFPE", SIGFPE }, { "SIGKILL", SIGKILL },
{ "SIGBUS", SIGBUS }, { "SIGSEGV", SIGSEGV }, { "SIGSYS", SIGSYS },
{ "SIGPIPE", SIGPIPE }, { "SIGALRM", SIGALRM }, { "SIGTERM", SIGTERM },
{ "SIGURG", SIGURG }, { "SIGSTOP", SIGSTOP }, { "SIGTSTP", SIGTSTP },
{ "SIGCONT", SIGCONT }, { "SIGCHLD", SIGCHLD }, { "SIGTTIN", SIGTTIN },
{ "SIGTTOU", SIGTTOU }, { "SIGIO", SIGIO }, { "SIGXCPU", SIGXCPU },
{ "SIGXFSZ", SIGXFSZ }, { "SIGVTALRM", SIGVTALRM }, { "SIGPROF", SIGPROF } };
int nsignals = sizeof( signals )/ sizeof( struct signame );
char *
Sigstr( n )
	int n;
{
	int i;
	static char buf[40];

	for( i = 0; i < nsignals; ++i ){
		if( signals[i].value == n ){
			return( signals[i].str );
		}
	}
	(void)sprintf(buf,"unknown signal (%d)", n );
	return( buf );
}
/***************************************************************************
 * Decode_status( union wait *status )
 * returns a printable string encoding return status
 ***************************************************************************/

char *
Decode_status( status )
	union wait *status;
{
	static char msg[BUFSIZ];

	(void)sprintf( msg, "user: %d, system: %d%s",
		status->w_retcode, status->w_termsig,
		status->w_coredump ? ", core dump, " : ", " );
	if( status->w_termsig ){
		(void)sprintf(msg+strlen(msg),"error- %s",
			Sigstr((int)status->w_termsig));
	}
	return( msg );
}

/*
 * Checkactive()
 * find the currently active files in the spool queue.
 * 1.  check for the single server possibility first.
 * 2.  find the Names of the servers
 * 3.  find the active files for each of the servers
 */

int 
Checkactive()
{
	int i;				/* ACME Integers, Inc. */
	char buf[BUFSIZ];	/* Name of active file */
	static char server[BUFSIZ];	/* Name of server file */
	char *sp, *ep;			/* ACME Pointer */
	int pid;

	pid = 0;
	buf[0] = 0;
	if(Checklockfile(LO,&pid,buf,sizeof(buf),&LO_statb) && buf[0] ){
		for( i = 0; i < Jobcount; ++i ){
			if( strcmp( buf, Queue[i].q_name ) == 0 ){
				Queue[i].q_daemon = pid;
				Queue[i].q_server = Printer;
			} else {
				Queue[i].q_daemon = 0;
				Queue[i].q_server = 0;
			}
		}
	}
	/*
	 * check for each of the servers
	 */
	if( SV && *SV ){
		(void)strcpy( server, SV );
		for( sp = server; sp; sp = ep ){
			if( ep = index( sp, ',' ) ){
				*ep = 0;
				++ep;
			}
			/*
			 * get the lock file and the status from the server
			 */
			buf[0] = 0;
			if(Checklockfile(sp,&pid,buf,sizeof(buf),(struct stat *)0)
				&& buf[0] ){
				for( i = 0; i < Jobcount; ++i ){
					if( strcmp( buf, Queue[i].q_name ) == 0 ){
						Queue[i].q_daemon = pid;
						Queue[i].q_server = sp;
					} else {
						Queue[i].q_daemon = 0;
						Queue[i].q_server = 0;
					}
				}
			}
		}
	}
	return(pid);
}


/***************************************************************************
 * Get_Daemon()
 * Get the DAEMON uid and gid
 * Assume password entry has both values set for the user daemon
 ***************************************************************************/
Get_Daemon()
{
	struct passwd *passwd, *getpwnam();
	if( Daemon_uid == 0 ){
		if( (passwd = getpwnam(DAEMON)) == 0 ){
			logerr_die( XLOG_INFO,"Get_Daemon: getpwnam(%s) failed", DAEMON );
		}
		Daemon_uid = passwd->pw_uid;
		Daemon_gid = passwd->pw_gid;
		if(Debug>4)log(XLOG_INFO,"Get_Daemon: uid=%d, gid=%d",
			Daemon_uid, Daemon_gid);
	}
}

/***************************************************************************
 * Job_match( char *control, *data)
 * Check to see if the control and data file names are matching
 * 1. data file must start with "df"
 * 2. next must be letter
 * 3. next must be 3 digit sequence number
 * 4. control and data file sequence number must match
 * return 1 if OK, 0 if not
 ***************************************************************************/

Job_match( control, data )
	char *control;
	char *data;
{
	int c, i, j;	/* ACME Chain and Integers, Inc. */

	i = strlen( data );
	for( j = 0; j < i ; ++j ){
		c = data[j];
		if( !isascii( c ) || !isprint( c ) ){
			log(XLOG_INFO, "Job_match: bad char in '%s'", data );
			return( 0 );
		}
	}
	if( index( data, '/' ) ){
		/*
		 * tried to embed a / in the file name
		 */
		log(XLOG_INFO, "Job_match: / present '%s'",	data );
		return( 0 );
	}
	if( i > CFNAMELEN
		|| i <= STARTFR
		|| i != strlen( control )
		|| index( "cd", data[0] ) == 0
		|| (data[1] != 'f')
		|| !isalpha( data[STARTPR] )
		|| strcmp( control+STARTID, data+STARTID ) ){
		log(XLOG_INFO, "Job_match: bad match control '%s', data '%s'",	
			control, data );
		return( 0 );
	}
	for( j = 0; j < IDLEN; ++ j ){
		if( !isdigit( data[j+STARTID] ) ){
			log(XLOG_INFO, "Job_match: bad sequence control '%s', data '%s'",	
				control, data );
			return( 0 );
		}
	}
	if(Debug>5)log(XLOG_DEBUG,"Job_match: OK control '%s', data '%s'",	
		control, data );
	return( 1 );
}

/***************************************************************************
 * Std_environ()
 * Make sure that fd 0, 1, 2 exist by opening /dev/null for them if they
 * have not been provided.
 * Set up the Host information as well
 ***************************************************************************/

Std_environ()
{
	int fd;
	struct hostent *host_ent;	/* host entry from data base */

	/*
	 * close all file descriptors up to NOFILE
	 */
	for (fd = 3; fd < NOFILE; fd++){
		(void) close(fd);
	}
	while( (fd = open( "/dev/null", O_RDWR, 0 )) >= 0 && fd < 3 );
	if( fd < 0 ){
		logerr_die( XLOG_CRIT, "Std_environ: cannot open /dev/null" );
	} else {
		(void)close(fd);
	}
	/*
	 * get the Host computer Name
	 */
	if( gethostname(Host, sizeof(Host)) < 0 ){
		logerr_die( XLOG_INFO, "Std_environ: gethostname failed" );
	}
	host_ent = gethostbyname( Host );
	if( host_ent == 0 ){
		fatal( XLOG_INFO, "Std_environ: host entry for '%' not found", Host );
	}
	if( strlen( host_ent->h_name ) > sizeof( Host ) ){
		fatal( XLOG_INFO, "Std_enviorn: host name is too long: '%s'",
			host_ent->h_name);
	}
	(void)strcpy(Host, host_ent->h_name );
	/*
	 * Get Daemon UID
	 */
	Get_Daemon();
}

/***************************************************************************
 * int open_daemon()
 * FILE *fopen_daemon()
 * int unlink_daemon()
 * Do the above actions as DAEMON.
 * Note that this is very difficult to do,  and ensure that all the error
 * conditions are met.  What you want to do is perform the following as
 * an uninterrupted critical section.
 * 1. setreuid( root, daemon)
 * 2. perform the action
 * 3. setreuid( whatever, root )
 * This will only work if
 * 1. you are really root
 * 2. you do not ever get interrupts
 ***************************************************************************/
static	uid_t ruid, euid;	/* real and effective UIDs */
static	int uid_parm;		/* must be able to hold -1 */
static	int omask;			/* block these signals */


void
Set_uid( new_uid )
	int new_uid;
{
	/*
	 * get the DAEMON UID, GID
	 */
	omask = sigblock(sigmask(SIGCHLD)|sigmask(SIGHUP)
		|sigmask(SIGINT)|sigmask(SIGQUIT)|sigmask(SIGTERM));
	ruid = getuid();
	euid = geteuid();

#	ifdef SETREUID4BSD
	uid_parm = euid;
#	else
	uid_parm = -1;
#	endif SETREUID4BSD
	/*
	 * check to see if we are running without SUID
	 */
	if( euid && euid != getuid() && euid != new_uid ){
		log( XLOG_INFO, "Set_uid: euid %d, ruid %d, bad SUID?",
			euid, getuid() );
		/*
		 * NOTE: do not try and take normal exit!!!
		 */
		exit(1);
	} else if( euid == 0 && setreuid( uid_parm, new_uid ) < 0 ){
		logerr( XLOG_INFO, "Set_uid: setreuid( %d, %d) failed",
			uid_parm, new_uid );
		exit(1);
	}
	if(Debug>4)log(XLOG_DEBUG,"Set_uid: uid_parm %d, uid %d, euid %d",
		uid_parm, getuid(), geteuid());
}
/*
 * set the UID to DAEMON
 */
void
Set_daemon()
{
	/*
	 * get the DAEMON UID, GID
	 */
	Get_Daemon();
	Set_uid( Daemon_uid );
}
/*
 * set UID back to normal.  Note that there are two versions of setreuid
 * available.  The VAX 4.2/4.3 version allows you to swap RUID/EUID
 * and will allow you only to set your UID if you are EUID root.
 * The SUN 3.X and others will allow you ONE call to the SUID function,
 * and will allow you to set the UID back to the original.  The
 * uid_parm flag determines which strategy to use.
 * If uid_parm is -1,  then you have to use setreuid( -1, euid )
 * If it is not,  then we have to swap the EUID/UID, then set
 * the UID and EUID explicitly
 */
void
Clear_uid()
{
	/*
	 * handle the easy one first: set euid back to original
	 */
	if( uid_parm == -1 ){
		if( setreuid( -1, euid ) < 0 ){
			logerr( XLOG_INFO,
			"Clear_uid: setreuid( %d, %d) from %d, %d failed",
				-1, euid, getuid(), geteuid() );
			/*
			 * avoid problems, exit early
			 */
			exit(1);
		}
		(void)sigsetmask(omask);
		return;
	}
	/*
	 * otherwise we swap the UID and EUID, but first make sure that it is
	 * consistent;  uid should be the old (original) euid on entry to set_uid
	 */
	if( getuid() != euid ){
		logerr( XLOG_INFO,
			"Clear_uid: uid is %d, euid is %d, and uid should be %d",
				getuid(), geteuid(), euid );
		exit(1);
	}
	/*
	 * swap the RUID and EUID
	 */
	if( setreuid( geteuid(), getuid() ) < 0 ){
		logerr( XLOG_INFO,
			"Clear_uid: setreuid( %d, %d) from %d, %d failed",
				geteuid(), getuid(), getuid(), geteuid() );
		/*
		 * avoid problems, exit early
		 */
		exit(1);
	}
	/*
	 * Set the RUID and EUID explicitly
	 */
	if( setreuid( ruid, euid ) < 0 ){
		logerr( XLOG_INFO,
			"Clear_uid: setreuid( %d, %d) from %d, %d failed",
				ruid, euid, getuid(), geteuid() );
		/*
		 * avoid problems, exit early
		 */
		exit(1);
	}
	(void)sigsetmask(omask);
}

/*
 * Open a file as daemon.
 */
int
open_daemon( name, flag, perms )
	char *name;
	int flag;
	int perms;
{
	int fd, err;
	/*
	 * establish UID
	 */
	Set_daemon();
	/*
	 * now for the plunge into the unknown
	 */
	fd = open( name, flag, perms );
	err = errno;
	Clear_uid();
	errno = err;
	return( fd );
}

/*
 * fopen a file; the file has been created, and we want to do this as
 * daemon for permissions reasons.
 */
FILE *
fopen_daemon( name, how )
	char *name;
	char *how;
{
	int err;
	FILE *fp;
	/*
	 * establish UID
	 */
	Set_daemon();
	/*
	 * now for the plunge into the unknown
	 * NOTE: we are assuming that we only use this to READ a file already
	 * created.
	 */
	fp = fopen( name, how );
	err = errno;
	/*
	 * done with the daemon stuff
	 */
	Clear_uid();
	errno = err;
	return( fp );
}

/*
 * UNLINK a file as DAEMON
 */
unlink_daemon( name )
	char *name;
{
	int f, err;
	/*
	 * establish UID
	 */
	Set_daemon();
	/*
	 * now for the plunge into the unknown
	 */
	f = unlink( name );
	err = errno;
	/*
	 * done with the daemon stuff
	 */
	Clear_uid();
	errno = err;
	return( f );
}


/*
 * CHMOD a file as DAEMON
 */
chmod_daemon( name, perms )
	char *name;
	int perms;
{
	int f, err;
	/*
	 * establish UID
	 */
	Set_daemon();
	/*
	 * now for the plunge into the unknown
	 */
	f = chmod( name, perms );
	err = errno;
	/*
	 * done with the daemon stuff
	 */
	Clear_uid();
	errno = err;
	return( f );
}

/***************************************************************************
 * Rename_cf( char *tf )
 * rename the temporary file to a control file
 ***************************************************************************/

Rename_cf( tf )
	char *tf;
{
	char from[MAXPATHLEN+1];
	char to[MAXPATHLEN+1];
	int i, err;

	(void)sprintf( from, "%s/%s", SD, tf );
	(void)sprintf( to, "%s/c%s", SD, tf+1 );
	if(Debug>4)log(XLOG_DEBUG,"Rename_cf: from '%s' to '%s'",from, to);
	Set_daemon();
	i = rename( from, to );
	err = errno;
	Clear_uid();
	errno = err;
	if( i < 0 ){
		logerr_die(XLOG_INFO,"Rename_cf: rename '%s' to '%s' failed",from, to);
	}
}

/***************************************************************************
 * Tailor_names()
 * Appends the Host information to several file names.
 ***************************************************************************/
static
append_name( s, t )
	char *s, *t;
{
	if( (strlen(s) + strlen(t) + 2) > MAXPATHLEN ){
		fatal( XLOG_INFO,"append_name: names too long 's' + 't'", s, t);
	}
	(void)strcat( s, t);
}
Tailor_names()
{
	append_name( Lpdlogf, Host );
	append_name( Masterlock, Host );
	append_name( Permfile, Host );
	append_name( Printcap, Host );
}
