/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: Startprinter.c
 * Printer queue job handler
 * 
 * StartPrinter is responsible for handling the Printer queue.
 * 1.  Get the printcap entry.
 * 2.  Check the spool directory to see if printing is enabled
 * 3.  Check to see if there are any jobs
 * 4.  If a remote Printer, call remotePrinter()
 *     If a local Printer, call localPrinter()
 ***************************************************************************
 * Revision History: Created Sat Feb 13 09:11:38 CST 1988
 * $Log:	startprinter.c,v $
 * Revision 3.1  88/06/18  09:35:49  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.4  88/05/21  10:28:30  papowell
 * Minor editing
 * 
 * Revision 2.3  88/05/14  10:18:27  papowell
 * Use long format for job file names;
 * Added 'fd', no forward flag;
 * Control file has to have hostname and origination agree.
 * 
 * Revision 2.2  88/05/11  09:50:52  papowell
 * Remote printer file transfer error fixed.
 * 
 * Revision 2.1  88/05/09  10:10:30  papowell
 * PLP: Released Version
 * 
 * Revision 1.6  88/04/27  18:02:36  papowell
 * The SIGCHLD signal has an odd behaviour on some systems.  Modified so that
 * it will not get set UNLESS processes are started;  also,  it is reset
 * to SIG_DFL, not SIG_IGN.
 * 
 * Revision 1.5  88/04/07  21:24:27  papowell
 * changed calls of init to (*init)().  Sigh.
 * 
 * Revision 1.4  88/04/06  12:13:05  papowell
 * Minor updates, changes in error message formats.
 * Elimination of the AF_UNIX connections, use AF_INET only.
 * Better error messages.
 * 
 * Revision 1.3  88/03/25  15:01:47  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.2  88/03/11  19:29:20  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.1  88/03/01  11:09:24  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: startprinter.c,v 3.1 88/06/18 09:35:49 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"
extern int Printjob(), Sendjob();
extern int Printinit(), Setup_xfer();
extern void Printfinal(), Printerror(), Allsent(), Send_error();

Startprinter()
{
	int pid;				/* process id */
	char *ps, *st;			/* ACME Pointers */

	/*
	 * ignore SIGCHILD,  explicitly wait for them
	 */
	(void)signal(SIGCHLD, SIG_DFL);
	/*
	 * get the printcap entry
	 */
	if( Get_pc_entry(Printer, All_pc_vars, All_pc_len ) == 0){
		log( XLOG_INFO, "trying to start non-existent Printer" );
		exit(0);
	}
	if( SS && *SS ){
		/*
		 * we have a spool server that is to be started
		 */
		ps = PS;
		st = ST;
		if( Set_pc_entry(SS, All_pc_vars, All_pc_len ) == 0){
			log( XLOG_INFO, "queue %s does not entry", SS );
			exit(0);
		}
		PS = ps;
		ST = st;
		LO = Printer;
		SV = 0;
	}
	if( SD == 0 || *SD == 0 ){
		/*
		 * no spooling directory, not a Printer
		 */
		if(Debug>0)log(XLOG_DEBUG, "Startprinter: not a Printer");
		exit(0);
	}
	if( RM && NW ){
		(void)Remote_start();
		exit(0);
	}
	/* chdir to spool directory */
	if (chdir(SD) < 0) {
		logerr_die( XLOG_NOTICE,"Startprinter: cannot chdir to %s", SD);
	}
	/*
	 * check to see if there is a deamon running
	 */
	Lfd = Getlockfile( LO, &pid, (char *)0, 0, &LO_statb );
	if( Lfd == NULL ){
		if(Debug>2)log(XLOG_DEBUG, "Startprinter: server present, pid %u",pid);
		exit(0);
	}
	if( fseek( Lfd, 0L, 0 ) < 0 ){
		logerr_die( XLOG_INFO, "Startprinter: T1 fseek failed");
	}
	/*
	 * check for unspooling enabled
	 */
	if( LO_statb.st_mode & DISABLE_PRINT) {
		if(Debug>0)log( XLOG_DEBUG,"Startprinter: unspooling disabled");
		exit(0);
	}
	/*
	 * handle multiple servers
	 */
	if( SV && *SV ){
		Closelockfile(LO, Lfd );
		Lfd = 0;
		Start_server();
		/*
		 * Check on the status of the queue lock file
		 */
		Lfd = Getlockfile( LO, &pid, (char *)0, 0, &LO_statb );
		if( Lfd == NULL ){
			if(Debug>2)log(XLOG_DEBUG, "Startprinter: server present, pid %u",pid);
			exit(0);
		}
	}
	/*
	 * set up log file
	 */
	if( fseek( Lfd, 0L, 0 ) < 0 ){
		logerr_die( XLOG_INFO, "Startprinter: T2 fseek failed");
	}
	if(Debug>0)log(XLOG_DEBUG, "Startprinter: should log into %s",
		LF?LF:"/dev/null");
	Setuplog( LF, 0 );
	if( fseek( Lfd, 0L, 0 ) < 0 ){
		logerr_die( XLOG_INFO, "Startprinter: T3 fseek failed");
	}
	/*
	 * OK, you are the server, go to it.
	 * set process group to the queue handler process
	 */
	(void) setpgrp(0, getpid());

	Setlockfile( LO, Lfd, getpid(), "(checking queue)");
	setstatus( "checking queue at %s", Time_str());

	/*
	 * search the spool directory for work and sort by queue order.
	 */
	if(Debug>0)log(XLOG_DEBUG, "Startprinter: unqueueing jobs");
	if( RM ){
		unspool(Setup_xfer, Sendjob, Allsent, Send_error);
	} else {
		unspool(Printinit, Printjob, Printfinal,Printerror);
	}
	if(Debug>2)log( XLOG_DEBUG, "Startprinter: work done" );
	setstatus( "work done at %s", Time_str() );
	Closelockfile(LO,Lfd);
	exit( 0 );
}


/***********************************************************
 * unspool( init, jobhandler, closer)
 * 
 * do{
 *   check queue for new jobs
 *   jobdone = 0
 *   for(n = 0;n < Jobcount; ++n)
 *      if queue needs to be resorted then
 *         set jobdone and break from "for" loop;
 *      while not successful and retry count is less than a limit do
 *         init();
 *         success = jobhandler(job n);
 *     -- note: job can be busy (no action), done successfully,
 *              done unsuccessfully and should be abandoned,
 *              or done unsuccessfully and should be repeated
 *      if action was done then set jobdone and remove job
 *          else call closer();
 * }while( jobdone )
 * call closer()
 ***********************************************************/

unspool( init, dojob, closer, err )
	int (*init)();
	int (*dojob)();
	void (*closer)();
	void (*err)();
{
	int n;				/* job number */
	int retries;		/* number of retries */
	int success;		/* job success status */
	struct queue *q;	/* job entry  */
	int jobdone;		/* job done in this pass */
	FILE *cfp;			/* control file */
	int init_done;	/* initialization done */

	/*
	 * the big loop
	 */
	init_done = 0;
	do{
		/*
		 * search the spool directory for more work.
		 */
		Jobcount = Getq();
		/*
		 * reset "update queue order" flag
		 */
		if(fstat(fileno(Lfd), &LO_statb ) < 0 ){
			logerr_die( XLOG_NOTICE,"unspool: cannot stat %s", LO);
		}
		if(Debug>5)log(XLOG_DEBUG,"unspool: '%s' perms 0%o",LO,
			LO_statb.st_mode);
		if(fchmod(fileno(Lfd), (int)(LO_statb.st_mode & CLEAR_REQUE)) < 0){
			logerr_die( XLOG_NOTICE,"unspool: cannot chmod '%s'", LO);
		}
		jobdone = 0;
		/*
		 * scan the queue
		 */
		for(n=0; n < Jobcount; ++n ){
			q = &Queue[n];
			/*
			 * lock the control file
			 */
			if( (cfp = Lockcf(q->q_name) )!= NULL ){
				Setlockfile( LO, Lfd, getpid(), q->q_name );
				/*
				 * try printing this file until successful,
				 *  number of attempts exceeds limit, or abort
				 * try sending indefinately until successful
				 */
				success = JFAIL;
				for( retries = 0;
					success == JFAIL && retries <= RT;
					++retries ){
					/*
					 * STDIO file error reset, needed on some implementations
					 */
					clearerr( cfp );
					/*
					 * seek to the start of the file
					 */
					if( fseek( cfp, 0L, 0 ) < 0 ){
						logerr_die(XLOG_INFO,"unspool: fseek failed (%s)",
							q->q_name );
					}
					/*
					 * update status information
					 */
					if(Debug)log(XLOG_DEBUG,"unspool: doing %s, %d retry",
						q->q_name,retries);
					/*
					 * initialize
					 */
					if( init_done == 0 ){
						init_done = (*init)();
						if( init_done != JSUCC ){
							setstatus( "initialization failure, %s",Last_errormsg);
							return;
						}
					}
					setstatus( "processing, active job started %s, attempt %d",
						Time_str(), retries+1);
					/*
					 * try to print the job
					 */
					success = (*dojob)(cfp, q);
				}
				if(Debug>4)log(XLOG_DEBUG,"unspool: %s status %d", q->q_name,
					success );
				if( success == JFAIL ){
					(*err)();
					init_done = 0;
				}
				jobdone = 1;
				Remove_job( cfp, q );
				(void)fclose(cfp);
			}
			/*
			 * collect any garbage
			 */
			Reapchild();
		}
		/*
		 * collect any garbage
		 */
		Reapchild();
	} while( jobdone );
	/*
	 * collect any garbage
	 */
	Reapchild();
	/*
	 * queue finished
	 */
	if(Debug>0)log(XLOG_DEBUG, "unspool: no more work to do" );
	(*closer)();
}

/***************************************************************************
 * Start_server()
 * start multiple servers for a single queue.  The SV printcap parameter
 * has a list of the servers.  These servers use the spool queues
 * directory,  and have a lock file in the server directory.
 * This routine will extract the names of each server,  fork a process
 * for it,  and then will read the printcap information for the server
 * from the queue.
 * The printer names are separated by commas.
 ***************************************************************************/

Start_server()
{
	char *ps;
	char *last;
	int pid;
	
	last = 0;
	ps = SV;
	while( ps ){
		Printer = ps;
		if( last = index( ps, ',' ) ){
			*last = 0;
			ps = last+1;
		} else {
			ps = 0;
		}
		if( ps ){
			if( (pid = fork()) < 0 ){
				logerr_die( XLOG_INFO, "Start_server: fork failed" );
			} else if( pid == 0 ){
				ps = 0;
			}
		}
	}
	if( Set_pc_entry(Printer, All_pc_vars, All_pc_len ) == 0){
		fatal( XLOG_INFO, "trying to start non-existent server" );
	}
	LO = Printer;
	if(Debug>3)log(XLOG_DEBUG,"Start_server: started" );
}
