/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: servicereq.c
 * service requests to lpd
 ***************************************************************************
 * Revision History: Created Sat Jan  2 08:54:52 CST 1988
 * $Log:	servicereq.c,v $
 * Revision 3.1  88/06/18  09:35:39  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.1  88/05/09  10:10:17  papowell
 * PLP: Released Version
 * 
 * Revision 1.3  88/04/06  12:12:48  papowell
 * Minor updates, changes in error message formats.
 * Elimination of the AF_UNIX connections, use AF_INET only.
 * Better error messages.
 * 
 * Revision 1.2  88/03/25  15:01:39  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.1  88/03/01  11:09:16  papowell
 * Initial revision
 * 
 ***************************************************************************
 * Description:
 * The LPD daemon waits for a request from an associated process using a socket.
 * The main lpd daemon forks a server process,  which connects to the
 * requesting process.  The two processes communicate over the socket. 
 * A message packet containing a request is the first information transferred.
 * Some requests will need to have error messages and diagnostics generated
 * by LPD sent to the user;  the "Echo_on_stdout" flag causes errors to be
 * echoed there as well.
 * The following operations are supported:
 *	    \1Printer\n
 * 	    	check the queue for jobs and print any found.
 *          -- acknowlege request and close socket
 *          -- start up the Printer
 *	    \2Printer\n
 *	    	receive a job from another machine and queue it.
 *          -- a complex file transfer protocol is used to transfer files.
 *	    \3Printer [users ...] [jobs ...]\n
 *	    	return the current state of the queue (short form).
 *          -- errors and diagnostics returned as well
 *	    \4Printer [users ...] [jobs ...]\n
 *	    	return the current state of the queue (long form).
 *          -- errors and diagnostics returned as well
 *	    \5Printer Person [users ...] [jobs ...]\n
 *	    	remove jobs from the queue.
 *          -- errors and diagnostics returned as well
 *	    \6Printer Person operation
 *	    	enable/disable queueing, etc.
 *          -- errors and diagnostics returned as well
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: servicereq.c,v 3.1 88/06/18 09:35:39 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"

static char	command[BUFSIZ];	/* command line buffer */
static char	*cmd_names[] = {
	"ERROR",			/* 0 */
	"Startprinter",		/* 1 */
	"receive files",	/* 2 */
	"display [short]",	/* 3 */
	"display [long]",	/* 4 */
	"remove job",		/* 5 */
	"control operation"			/* 6 */
};

/****************************************************************
 * servicereq()
 *   1. Reads the first line from the socket
 *   2. Determine service request
 *   3. Call suitable dispatch function
 ****************************************************************/
servicereq()
{
	int n;			/* ACME Integer, Inc. */

	/*
	 * use almost bombproof way to read command line from socket
	 */
	n = bpread( 1, command, sizeof(command));
	if(n < 2 ){
		fatal(XLOG_INFO,"servicereq: bad command line");
	}
	Request = command[0];
	if (Request < 1 || Request > 6 ){
		fatal(XLOG_INFO,"servicereq: bad request (%d) %s",Request, &command[1]);
	}
	if (Debug>1)log( XLOG_DEBUG,"%s requests %d (%s) %s",From,Request,
		cmd_names[Request],&command[1]);
	splitline( &command[1] );
	if( Request != REQ_CONTROL ){
		/*
		 * extract Printer Name from command
		 */
		Printer = Parms[0].str;
		/*
		 * check to see if you can do the job
		 */
		if(chkhost() == 0){
			fatal(XLOG_INFO,"Host %s cannot access %s", From, Printer);
		}
	}
	switch (Request) {
	/*
	 * start Printer
	 */
	case REQ_START:
		/* signal the other end that you are doing the request */
		putchar( 0 );		/* fd 1 is the socket */
		(void)fflush(stdout);	/* send it */
		(void)close(1);			/* disconnect */
		if( dup2(0,1) < 0 ){
			logerr_die(XLOG_INFO,"dup2 failed");
		}
		Startprinter();		/* start Printer */
		break;
	/*
	 * get remote jobs
	 */
	case REQ_RECV:	/* get files from remote site */
		recvfiles();
		Startprinter();		/* start Printer */
		break;
	/*
	 * display current queue
	 */
	case REQ_DSHORT:	/* display the queue (short form) */
	case REQ_DLONG:		/* display the queue (long form) */
		/*	echo errors on stdout as well as stderr */
		Echo_on_stdout = 1;
		Short_format = (REQ_DSHORT==Request);
		Shift_parms(1);
		(void)Displayq();
		break;
	/*
	 * remove job from queue
	 */
	case REQ_REMOVE:	/* remove a job from the queue */
		/*	echo errors on stdout as well as stderr */
		Echo_on_stdout = 1;
		Person = Parms[1].str;
		Shift_parms(2);
		rmjob();
		break;
	/*
	 * perform control function
	 */
	case REQ_CONTROL:	/* remove a job from the queue */
		Echo_on_stdout = 1;
		Person = Parms[0].str;
		Shift_parms(1);
		(void)Control_ops();
		break;
	}
}

/**********************************************************************
 * chkhost()
 * check permissions to see if Host has access to the Printer
 * Return: 1 if permissions to use Printer, 0 otherwise
 **********************************************************************/
static int
chkhost()
{
	if( strcmp(From, Host) == 0	/* this Host */
		||( Permfile && *Permfile
			&& Checkperm(Permfile,From,(char*)0,Printer,(int*)0,(int*)0,0)) ){
		return(1);
	}
	return(0);
}
