/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: recvfiles.c
 * Receive files from a remote site
 ***************************************************************************
 * Revision History: Created Sat Jan 16 07:10:12 CST 1988
 * $Log:	recvfiles.c,v $
 * Revision 3.1  88/06/18  09:35:25  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.4  88/05/21  10:28:14  papowell
 * Minor editing
 * 
 * Revision 2.3  88/05/14  10:18:01  papowell
 * Use long format for job file names;
 * Added 'fd', no forward flag;
 * Control file has to have hostname and origination agree.
 * 
 * Revision 2.2  88/05/11  09:52:55  papowell
 * Remote printer file transfer error fixed.
 * 
 * Revision 2.1  88/05/09  10:09:55  papowell
 * PLP: Released Version
 * 
 * Revision 1.5  88/04/28  09:52:40  papowell
 * added casts to shut up lint
 * 
 * Revision 1.4  88/04/26  15:53:39  papowell
 * Fixed up a horribly silly bug in the add_files and File_name
 * routines;  sigh.  Would you believe an L (l) and a one (1) got mixed
 * up?
 * 
 * Revision 1.3  88/03/25  15:01:17  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.2  88/03/05  15:01:07  papowell
 * Minor Corrections,  Lint Problems
 * 
 * Revision 1.1  88/03/01  11:09:06  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: recvfiles.c,v 3.1 88/06/18 09:35:25 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"

/***************************************************************************
 * This module implements the file transfer protocol at the receiving
 * site. For each job:
 * 1.  The data files are transferred.
 * 2.  The control files are transferred.
 *     If the transfer is unsuccessful,  all job files are removed.
 * 
 * Individual files are transferred using the following protocol.
 * 
 * 1.  The sending site sends a line with the size and name of the file,
 * 	and a flag indicating if it is a data or control file.
 * 2.  The receiving site will acknowledge reception.
 * 3.  The remote site will send the file.
 * 4.  The receiving site will acknowledge reception.
 * 
 * The transfer protocol is implemented by a simple FSM:
 * INIT:  no files transferred
 * DATA:  data file transferred
 * CONTROL: control file transferred
 * 
 * A list of all files associated with the job is maintained,
 * and if any errors result,  the list of files will be deleted.
 * 
 ***************************************************************************/

/***************************************************************************
 * recvfiles()
 *
 * Gets a job from the remote end.
 * 
 * The old Berkeley protocol was to send data files and then the control
 * file.  The PLP protocol will send the control file first, then the data
 * files, followed by the name of the control file.
 * 
 * If there is an error during transfer, all the jobs associated with the
 * file are removed.
 * 
 * In order to be compatible with the Berkeley LPD, old Berkeley format is
 * also supported.
 ***************************************************************************/
recvfiles()
{
	int i;					/* ACME Integers */
	int succ;				/* success flag */
	int flag;				/* file kind */
	long size;				/* file size */
	char fname[CFNAMELEN+1];	/* file fname */
	char cfname[CFNAMELEN+1];	/* file fname */
	char tfname[CFNAMELEN+1];	/* file fname */
	int fd;					/* file descriptor */
	FILE *cfp;				/* FILE descriptor */
	int state;				/* state: IDLE, BDATA, BCNTRL, PCNTRL, PDATA */
	long jobsize;			/* job size */
#define IDLE	0	/* nothing doing */
#define BDATA	1	/* transferring Berkeley data file */
#define BCNTRL	2	/* transferring Berkeley control file */
#define PCNTRL	3	/* transferring PLP control file */
#define PDATA	4	/* transferring PLP data file */
#define PLAST	5	/* last file has been transferred */

	/*
	 * get the printcap entry
	 */
	if( Get_pc_entry(Printer, Status_pc_vars, Status_pc_len ) == 0){
		log( XLOG_INFO, "revcfiles: trying to start non-existent Printer" );
	}
	if( SD == 0 || *SD == 0 ){
		/*
		 * no spooling directory, not a Printer
		 */
		if(Debug>0)log(XLOG_DEBUG, "revcfiles: not a Printer");
		return;
	}
	/* chdir to spool directory */
	if (chdir(SD) < 0) {
		logerr_die( XLOG_NOTICE,"revcfiles: cannot chdir to %s", SD);
	}
	if(Debug>3)log(XLOG_DEBUG,"recvfiles: setting up log");
	Setuplog( LF, 1 );
	if(Debug>3)log(XLOG_DEBUG,"recvfiles: sending confirm");
	if( send_ack(0) != JSUCC ){
		if(Debug>3)log(XLOG_DEBUG,"recvfiles: confirm failed");
		goto error;
	}

	/*
	 * set up the file transfers
	 */
	state = IDLE;
	while( (succ = get_file_xfer(&flag, &size, fname )) == JSUCC && flag ){
		if(Debug>3)log(XLOG_DEBUG,"recvfiles: state %d, flag %d, file %s",
			state,flag,fname);
		switch( state ){
		case IDLE:
			cfp = NULL;
			Rec_cnt = 0;
			(void)strcpy( cfname, fname );
			jobsize = 0;
			for( i = 0; i < 26; ++i ){
				CFparm[i][0] = 0;
			}
			switch( flag ){
			case CNAME:
				state = PCNTRL; break;
			case DFILE:
				state = BDATA;	break;
			default:
				goto protocol;
			}
			break;
		case BDATA:
			switch( flag ){
			case DFILE:
				break;
			case CFILE:
				state = BCNTRL;
				break;
			default:
				goto protocol;
			}
			break;
		case PCNTRL:
			switch( flag ){
			case DFILE:
				state = PDATA;	break;
			default:
				goto protocol;
			}
			break;
		case PDATA:
			switch( flag ){
			case DFILE:
				break;
			case CEND:
				state = PLAST;
				break;
			default:
				goto protocol;
			}
			break;
		default:
			goto protocol;
		}
		/*
		 * trying to send a file with a different sequence number?
		 */
		if( Job_match( cfname, fname ) == 0 ){
			log( XLOG_INFO, "recvfiles: file with bad format %s", fname );
			succ = JFAIL;
			goto error;
		}
		switch( state ){
		case PDATA:
			/*
			 * add names to list
			 */
			if( Find_name( fname ) < 0 ){
				log( XLOG_INFO,
					"recvfiles: data file not in job '%s'", fname );
				succ = JFAIL;
				goto error;
			}
			break;
		case BCNTRL:
		case PCNTRL:
			(void)strcpy(tfname, fname );
			fname[0] = 't';
			break;
		}
		/*
		 * add the file name to the temporary list
		 */
		if( (i = add_recfiles( fname )) < 0 ){
			log( XLOG_INFO, "recvfiles: too many files (%s)", fname );
			succ = JFAIL;
			goto error;
		}
		/*
		 * check to see that file size does not exceed the total
		 */
		Parms[i].num = 1;
		Parms[i].size = ((size+1023)/1024);
		jobsize = jobsize + Parms[i].size;
		if( MX && jobsize > MX ){
			log( XLOG_INFO, "recvfiles: file (%s) exceeds MX",fname, MX);
			succ = JFAIL;
			goto error;
		}
		/*
		 * check to see if we have opened the file already
		 */
		if( state != PLAST ){
			/*
			 * we lock the file
			 */
			fd = Exlockcf( fname );
			if( fd < 0 ){
				/*
				 * The file is locked.  This can only happen if some other
				 * process  got hold of the file.  We either have several
				 * processes transferring files or horrible collision course.
				 * DONT remove already transferred files.
				 */
				logerr(XLOG_NOTICE,"recvfiles: %s IMPOSSIBLE collision %s",
					fname,From);
				exit(1);
			}
			succ = get_file( size, fd, fname );
			if( succ != JSUCC ){
				goto error;
			}
		}
		/*
		 * open a FILE for the control file, and check the entries in it.
		 * If we have just got the file in PLP, just scan for entries.
		 * Otherwise check to see that all are present.
		 */
		switch( state ){
		case BCNTRL:
		case PCNTRL:
			if( (cfp = fdopen( fd, "r" ) ) < 0 ){
				logerr_die( XLOG_INFO,"recvfiles: fdopen failed" );
			}
			if( Validate_cf( state == PCNTRL, cfp, fname ) == 0 ){
				goto error;
			}
			break;
		case PLAST:
			if( Validate_cf( 0, cfp, fname ) == 0 ){
				goto error;
			}
			break;
		}
		switch( state ){
		case PLAST:
		case BCNTRL:
			Rename_cf( fname );
			Rec_cnt = 0;
			(void)fclose(cfp);
			state = IDLE;
			break;
		case PCNTRL:
			break;
		default:
			(void)close( fd );
			break;
		}
		/*
		 * We can confirm transfer to the other end
		 */
		if( send_ack( 0 ) != JSUCC ){
			goto error;
		}
	}
	if( succ == JSUCC && flag == 0 && state == IDLE ){
		/*
		 * finished! start up Printer
		 */
		if(Debug>3)log(XLOG_DEBUG,"recvfiles: done, starting Printer");
		Link_close();
		Startprinter();
		return;
	}
	/*
	 * Error conditions
	 * 1. remove files
	 * 2. reply with error status to remote end
	 */
protocol:
	log(XLOG_INFO,
	"recvfiles: protocol violation, state %d, flag %d, file %s",
		state, flag, fname );
error:
	rm_recfiles();
	(void)send_ack(1);
	Startprinter();
	return;
}

/***************************************************************************
 * add_recfiles(char * Name)
 * add a file Name to the received files
 ***************************************************************************/

add_recfiles( filename )
	char *filename;
{
	int i;

	if(Debug>4)log(XLOG_DEBUG,"add_recfiles: %s", filename );
	/*
	 * add a control file name to the parameter list entries
	 * if there is nothing in the list, clobber the entry.
	 */
	if( Rec_cnt ==  0 ){
		Parmcount = 0;
	}
	if( (i = Add_name( filename )) < 0 ){
		return( -1 );
	}
	Rec_cnt = Parmcount;
	return( i );
}

/***************************************************************************
 * rm_recfiles()
 * remove the files that are in the receive list;
 ***************************************************************************/

rm_recfiles()
{
	int i;		/* ACME Integer, Inc. */

	if(Debug>4)log(XLOG_DEBUG,"rm_recfiles: removing files" );
	for( i = 0; i < Rec_cnt; ++i ){
		if(Debug>4)log(XLOG_DEBUG,"rm_recfiles: %s", Parms[i].filename );
		(void)unlink_daemon( Parms[i].filename );
	}
	Rec_cnt = 0;
}

/***************************************************************************
 * get_file_xfer(int *flag; long *size; char *name )
 * 1. read a line from the far end
 * 2. the line has the format <flag><size> <name>\n
 *      "%c%d %s\n" format
 * 3. extract the information from the line
 * 4. acknowledge the information
 ***************************************************************************/
get_file_xfer( flag, size, name )
	int *flag;
	long *size;
	char *name;
{
	char buf[BUFSIZ];		/* buffer for reading */
	int i;					/* ACME Integers, Inc. */
	char *cp;				/* ACME Pointers, Inc. */

	/*
	 * read a line from the remote end
	 * use Bomb-proof Read (bpread), which reads up to the first \n
	 */
	if(Debug>3)log(XLOG_DEBUG,"get_file_xfer: starting", buf );
	if( (i = bpread(1, buf, sizeof(buf)) ) < 0 ){
		logerr(XLOG_INFO, "get_file_xfer: read error from remote");
		*flag = 0;
		goto error;
	} else if( i == 0 ){
		if(Debug>3)log(XLOG_DEBUG,"get_file_xfer: end of input");
		*flag = 0;
		return( JSUCC );
	}
	/*
	 * UGLY UGLY UGLY:
	 * what I wanted to do is:
	 *    if( sscanf( buf, "%c%d %s", flag, size, name ) != 3 )...
	 * but the guilty-of-incestuous-rape sscanf is not portable across
	 * several implementations.
	 */
	if(Debug>3)log(XLOG_DEBUG,"get_file_xfer: %d'%s'", buf[0], &buf[1]);
	/*
	 * pull off the flag information
	 */
	i = buf[0];
	if( i != DFILE && i != CFILE && i != CNAME && i != CEND){
		log(XLOG_INFO, "get_file_xfer: bad first char (%d)", i);
		goto error;
	}
	*flag = i;
	/*
	 * now pull off the size information
	 */
	*size = 0;
	for( cp = &buf[1]; (i = *cp) && isdigit( i ); ++cp ){
		*size = 10*(*size) + (i - '0');
	}
	if( *cp != ' '){
		log(XLOG_INFO, "get_file_xfer: no space separator (%d)", *cp);
		goto error;
	}
	while( *cp == ' ' ) ++cp;
	if( Job_match( cp, cp ) == 0 ){
		log(XLOG_INFO, "get_file_xfer: bad job name '%s'", cp);
		goto error;
	}
	/*
	 * must be a data or control file only
	 */
	(void)strcpy( name, cp );
	/*
	 * send a confirmation
	 */
	if( send_ack( 0 ) != JSUCC ){
		goto error;
	}
	if(Debug>3)log(XLOG_DEBUG,"get_file_xfer: flag %d, size %d, name '%s'",
		*flag, *size, name );
	return( JSUCC );
	/*
	 * error, give up
	 */
error:
	Link_close();
	return( JFAIL );
}

/***************************************************************************
 * get_file(long size; int fd; char *name)
 * 1. copy size bytes from fd 1 to fd
 * 2. the next byte read should be 0; if not, then you have error
 ***************************************************************************/
get_file( size, fd, name )
	long size;
	int fd;
	char *name;
{
	char buf[BUFSIZ];		/* buffer for reading */
	long cnt;				/* number of bytes */
	int i;					/* ACME Integers, Inc. */

	/*
	 * we simply copy the file from the remote end
	 */
	if(Debug>3)log(XLOG_DEBUG,"get_file: get %d (%s) from %s",size,name,From);
	cnt = size;
	while( cnt > 0 ){
		if( cnt > sizeof(buf) ){
			i = sizeof(buf);
		} else {
			i = cnt;
		}
		i = read(1, buf, i);
		if( i < 0 ){
			logerr(XLOG_INFO, "get_file: read from %s failed", From );
			goto error;
		}
		if( i == 0 ){
			log(XLOG_INFO,"get_file: read 0 bytes from %s, assume dead",From);
			goto error;
		}
		if( write( fd, buf, i ) != i ){
			logerr(XLOG_INFO, "write to %s failed", name );
			goto error;
		}
		cnt = cnt - i;
	}
	if(Debug>3)log(XLOG_DEBUG,"get_file: success %d (%s) from %s",
		size,name,From);
	i = read(1, buf, 1);
	if( i < 0 ){
		logerr(XLOG_INFO, "get_file: end from %s failed", From );
		goto error;
	}
	if( i == 0 ){
		log(XLOG_INFO, "get_file: end, 0 bytes from %s, assume dead", From );
		goto error;
	}
	if( buf[0] != 0 ){
		log(XLOG_INFO, "get_file: bad end confirm, %d from %s",buf[0],From);
		goto error;
	}
	if(Debug>3)log(XLOG_DEBUG,"get_file: read %s from %s", name, From );
	return( JSUCC );
error:
	Link_close();
	return( JFAIL );
}

/***************************************************************************
 * send_ack( int c )
 * acknowledge by sending back single char c
 ***************************************************************************/
send_ack( c )
	int c;
{
	char buf[1];

	buf[0] = c;
	if( write( 1, buf, 1 ) != 1 ){
		logerr(XLOG_INFO,"send_ack: %d to %s failed", c, From );
		return( JFAIL );
	}
	if(Debug>3)log(XLOG_DEBUG,"send_ack: %d to %s succeeded", c, From );
	return( JSUCC );
}

/***************************************************************************
 * Validate_cf( scan, fp, fname )
 * Check to ensure that the files contained in this control file
 * were actually sent as part of the control file.
 * 1. fseek to the start of the file
 * 2. read the file, looking data file entries
 * 3. Check that the names of the data files are consistent with the
 *    name of the control file
 * 4. Check that the file was sent
 ***************************************************************************/
Validate_cf( scan, fp, fname )
	int scan;
	FILE *fp;
	char *fname;
{
	char buf[BUFSIZ];	/* Buffer, Inc. */
	int c, l;			/* AMCE Aluminum Siding and Integers, Inc. */
	char *bp;			/* Buffer */
	long size;			/* total job size */
	int perms;			/* permissions */
	int name_len;		/* length of hostname to check */

	if(Debug>6){
		(void)fprintf( stderr,"Validate_cf: files %d", Parmcount );
		for( c = 0; c < Parmcount; ++c ){
			(void)fprintf(stderr,", %d '%s'(%d)",c,
				Parms[c].filename,Parms[c].num);
		}
		(void)fprintf( stderr, "\n" );
	}
	size = 0;
	if( fseek( fp, 0L, 0 ) < 0 ){
		logerr_die( XLOG_INFO, "Validate_cf: fseek failed '%s'", fname );
	}
	while( fgets( buf, sizeof(buf), fp ) ){
		l = strlen( buf );
		if( l > MAXPARMLEN ){
			log( XLOG_INFO, "Validate_cf: '%s' line too long '%s'",fname,buf);
			return( 0 );
		}
		if( buf[l-1] != '\n' ){
			log( XLOG_INFO, "Validate_cf: '%s', missing \\n", fname );
			return( 0 );
		}
		buf[l-1] = 0;
		c = *buf;
		bp = buf+1;
		if( !isascii(c) || !isalnum(c)){
			log( XLOG_INFO, "Validate_cf: file %s, bad line'%s'",fname,bp);
			return( 0 );
		}
		/*
		 * Check to see that data file information is OK
		 */
		if( isupper(c) && c != 'L' && c != 'U' ){
			(void)strcpy( CFparm[c-'A'], bp );
		}
		if( islower(c) || c == 'U' ){
			/*
			 * check to see that the file is in the list
			 */
			if( Job_match( fname, bp ) == 0 ){
				log( XLOG_INFO,"Validate_cf: file %s, bad data file '%s'",
					fname,bp);
				return( 0 );
			}
			/*
			 * if we are scanning, just enter name in list, otherwise
			 * check to see that file is in the Parms[] list and that the
			 * total job size is within limits
			 */
			if( scan ){
				if( Add_name( bp ) < 0 ){
					log( XLOG_INFO,"Validate_cf: too many files in job %s (%s)",
						fname,bp);
					return( 0 );
				}
			} else if( (l = Find_name( bp )) < 0 ){
				log(XLOG_INFO,
					"Validate_cf: file %s, data file '%s' not in list",
					fname,bp);
				return( 0 );
			} else if( Parms[l].num == 0 ){
				log( XLOG_INFO,
					"Validate_cf: file %s, data file '%s' not transferred",
					fname,bp);
				return( 0 );
			} else if( islower( c ) ){
				size = size + Parms[l].size;
				if( MX && size > MX ){
					log( XLOG_INFO,"Validate_cf: job %s too large", fname);
					return( 0 );
				}
			}
		}
	}
	/*
	 * check to see if the remote submitter has valid authorizations
	 */
	if( LOGNAME[0] == 0 ){
		log( XLOG_INFO,"Validate_cf: job %s missing username", fname);
		return( 0 );
	}
	/*
	 * check for long or short name
	 */
	if( LH ){
		name_len = strlen( From );
	} else {
		name_len = strlen( &fname[STARTFR] );
	}
	if( strncmp( FROMHOST, &fname[STARTFR], name_len ) ){
		log( XLOG_INFO,"Validate_cf: bad filename '%s' FROMHOST '%s'",
			fname, FROMHOST);
		return( 0 );
	}
	/*
	 * check to see if you will accept forwarded files
	 * if FD is set, then no forwarded files allowed
	 */
	if( FD && strcmp( FROMHOST, From ) ){
		log( XLOG_INFO,"Validate_cf: forwarded job '%s' FROMHOST '%s'",
			fname, FROMHOST);
		return( 0 );
	}
	perms = 'R';
	if((Permfile && *Permfile &&
			!Checkperm(Permfile,FROMHOST,LOGNAME,First_name,&perms,(int *)0,0))
       ||(XU && *XU &&
			!Checkperm(XU,FROMHOST,LOGNAME,First_name,&perms,(int *)0,0 ) )){
		log(XLOG_INFO, "Validate_cf: %s@%s cannot use '%s'",
			LOGNAME, FROMHOST, First_name );
			return( 0 );
	}
	return( 1 );
}
