/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: printcap.c
 * Super slick printcap information extraction routines.
 * Note: these routines were designed after studying the CURSES routines,
 * which are HORRIBLE...  These may not be any better, but at least
 * they are a lot faster.
 * The design is based on the assumption that we will know all the printcap
 * entries that we want to know ahead of time,  and get them "en mass".
 * Further more,  these can be placed in sorted order.
 * Thus,  given a set of these,  we scan the printcap entry, extract the
 * field name,  determine if in the list,  and then update the information
 * only if in list.  Result:
 * 1.  1 scan of printcap file per lookup of printer name, finds K entries
 * 2.  Log N order search of list,  for a K log N search of printcap
 ***************************************************************************
 * Revision History: Created Sun Jan  3 19:12:33 CST 1988
 * $Log:	printcap.c,v $
 * Revision 3.1  88/06/18  09:35:22  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.1  88/05/09  10:09:50  papowell
 * PLP: Released Version
 * 
 * Revision 1.6  88/04/06  12:13:22  papowell
 * Minor updates, changes in error message formats.
 * Elimination of the AF_UNIX connections, use AF_INET only.
 * Better error messages.
 * 
 * Revision 1.5  88/03/25  15:01:10  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.4  88/03/12  10:04:29  papowell
 * Minor Bug Fixes,  Better Error Message Logging
 * 
 * Revision 1.3  88/03/11  19:29:48  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.2  88/03/05  15:01:37  papowell
 * Minor Corrections,  Lint Problems
 * 
 * Revision 1.1  88/03/01  11:09:04  papowell
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: printcap.c,v 3.1 88/06/18 09:35:22 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"
/*
 * char **All_printers()
 *  returns a pointer to an array of strings which contains the
 *  names of the printers in the printcap file.  The list is terminated
 *  with a 0 (NIL) entry.
 *
 * char *Get_first_printer()
 *  returns a pointer to the first printer in the printcap file.
 *
 * init_pc_entry( PC_ENTRY pc[]; int len)
 *     set the default values for variables listed in pc[]
 *
 * The following routines return 1 if printcap entry found, 0 otherwise.
 *     of the printcap entry.
 *
 * int Set_pc_entry(char *printer; PC_ENTRY pc[]; int len)
 *     set the values of the variables listed in pc from the printcap
 *  1. scan the printcap file and extract the printcap entry for
 *     printer.  Make a copy of the entry.
 *  2. find the name of the printer (first entry) and set the First_name
 *     variable to point to it.
 *  3. Initialize the printcap variables according to the defaults in
 *     the "pc" array.
 *  4. Scan the printcap entry, and for each tagged field in the entry:
 *     a. search the "pc" array for the tag using a binary search.
 *     b. if the entry is found, determine the type;
 *     c. if an integer, convert from string to integer and update variable.
 *     d. if a flag, set or clear it
 *     e. if a string, copy into a buffer and set the string variable to point
 *        to it.
 * int Get_pc_entry(char *printer; PC_ENTRY pc[]; int len)
 *     call init_pc_entry() and then Set_pc_entry.
 */

static	FILE *pfp = NULL;	/* printcap data base file pointer */
/*
 * The capbuf buffer holds printcap, the strbuf holds strings
 */
static	char	capbuf[BUFSIZ]; /* buffer to hold printcap */
static	char	strbuf[BUFSIZ]; /* buffer to hold strings */
static  char   *strloc;		/* next string location */

static  char *pr_names;		/* buffer to hold them */
static  int max_names;		/* max number of names in printcap */
static  char **pr_list;		/* printer names in printcap */
static  int max_list;		/* size of list */
/*
 * All_printers: reads the /etc/printcap file, and forms
 * a list of all the printer names in the file.
 */

char **
All_printers()
{
	int i;		/* ACME Integer, Inc. */
	char *bp, *cp, *ep;	/* ACME Pointer, Inc. */

loop:
	if(Debug>5)log(XLOG_DEBUG,"All_printers: max_names %d, max_list %d",
		max_names, max_list);
	if( max_names == 0 ){
		max_list = BUFSIZ;
		max_names = MAXPCNAMES;
		pr_names = (char *)malloc((unsigned)(max_list));
		if( pr_names == NULL ){
			logerr_die( XLOG_INFO, "All_printers: malloc failed" );
		}
		pr_list = (char **)malloc((unsigned)(max_names*sizeof(char *)));
		if( pr_list == NULL ){
			logerr_die( XLOG_INFO, "All_printers: malloc failed" );
		}
	}
	i = 0;
	bp = pr_names;
	ep = pr_names + max_list;
	closeprent();
	while( pc_entry( capbuf, sizeof(capbuf)) ){
		init_pc_entry( Status_pc_vars, Status_pc_len);
		getpcvars(Status_pc_vars, Status_pc_len);
			/* get the printcap variable values */
		if(Debug>6)show_pc( Status_pc_vars, Status_pc_len);
		if( SD != 0 && *SD != 0 ){
			pr_list[i] = bp;
			if( (cp = index( capbuf, '|')) == 0 ){
				cp = index( capbuf, ':');
			}
			*cp = 0;
			bp = estrcp( bp, capbuf, ep );
			if( bp ){
				++bp;
			}
			++i;
			if( i >= max_names || bp == 0 || bp >= ep){
				if(Debug>1)log(XLOG_DEBUG,"All_printers: need more space" );
				max_list += BUFSIZ;
				max_names += MAXPCNAMES;
				pr_names = (char *)realloc(pr_names,(unsigned)(max_list));
				if( pr_names == NULL ){
					logerr_die( XLOG_INFO, "All_printers: malloc failed" );
				}
				pr_list = (char **)
					realloc((char *)pr_list,(unsigned)max_names*sizeof(char *));
				if( pr_list == 0 ){
					logerr_die( XLOG_INFO, "All_printers: malloc failed" );
				}
				goto loop;
			}
			pr_list[i] = 0;
		}
	}
	closeprent();
	if(Debug>5){
		int j;
		(void)fprintf(stdout,"All_printers: %d,", i);
		for( j = 0; j < i; ++j ){
			(void)fprintf(stdout," %s", pr_list[j] );
		}
		(void)fprintf(stdout,"\n"); 
	}
	return( pr_list );
}

/*
 * First_printer: reads the /etc/printcap file
 * and finds the first printer name
 */
char *
First_printer()
{
	char *bp;
	static char first[PRNAMELEN+1];

	closeprent();
	while( pc_entry( capbuf, sizeof(capbuf)) ){
		init_pc_entry( Status_pc_vars, Status_pc_len);
		getpcvars(Status_pc_vars, Status_pc_len);
			/* get the printcap variable values */
		if(Debug>6)show_pc( Status_pc_vars, Status_pc_len);
		if( SD != 0 && *SD != 0 ){
			if( (bp = index( capbuf, '|')) == 0 ){
				bp = index( capbuf, ':');
			}
			*bp = 0;
			(void)strcpy( first, capbuf );
			closeprent();
			if(Debug>4)log(XLOG_DEBUG,"First_printer: %s", first );
			return( first );
		}
	}
	fatal( XLOG_INFO, "First_printer: bad printcap %s", Printcap );
	return( (char *)0 );
}

/*
 * pc_entry(char *tb; int size)
 * Used to scan the printcap file.
 * finds the next entry in the printcap file;
 * it places it in the tb buffer, and returns
 * 1 if entry found, 0 otherwise
 */
static int
pc_entry(tb, size)
	char *tb;
	int size;
{
	char buf[BUFSIZ];	/* read buffer */
	char *bp, *cp, *ep;	/* next read position */
	int l;				/* line length */

	/*
	 * get the printcap file
	 */
	if (pfp == NULL ){
		if( (pfp = fopen_daemon(Printcap, "r")) == NULL){
			logerr_die( XLOG_CRIT, "open %s failed", Printcap );
		}
	}
	/*
	 * We read a single printcap entry into the bp buffer.
	 * We scan lines until we hit one that does not start with a
	 * # or is a blank line.
	 */
	while( bp = fgets( buf, sizeof(buf), pfp) ){
		if( buf[0] != '#' && buf[0] != '\n' ){
			break;
		}
	}
	if( bp == 0 ){
		return(0);
	}
	/*
	 * read the file in, stripping off the \<newlines>
	 */
	ep = tb + size;
	bp = tb;
	do{
		cp = index( buf, '\n' );
		if( cp == 0 ){
			fatal( XLOG_INFO, "bad line in printcap file '%s'", buf );
		}
		*cp = 0;
		if( cp != buf ){
			l = (cp[-1] == '\\');
			if( l ){
				cp[-1] = 0;
			}
		} else {
			l = 0;
		}
		bp = estrcp( bp, buf, ep );
		if( bp == 0 ){
			fatal( XLOG_INFO, "printcap entry too long %s", tb );
		}
		if( l == 0 ){
			if(Debug>8)log(XLOG_DEBUG,"pc_entry: %s", tb );
			return( 1 );
		}
	} while( fgets( buf, sizeof(buf), pfp) );
	log( XLOG_INFO,"bad termcap entry %s", tb);
	return( 0 );
}

/*
 * init_pc_entry(  PC_ENTRY pc_vars[]; int pc_len)
 *   initialize the variables in pc_vars with their default values.
 *   sets the strloc to start of string buffer.
 */
init_pc_entry( pc_vars, pc_len)
	PC_ENTRY pc_vars[];	/* the printcap vars used for the printer */
	int pc_len;	/* the number of printcap vars */
{
	int i;		/* ACME Integer, Inc. */
	/*
	 * intialize with the defaults
	 */
	First_name = 0;
	for( i = 0; i < pc_len; ++ i ){
		switch( pc_vars[i].kind ){
		case PC_NUM:
			*(int *)pc_vars[i].var = pc_vars[i].idefault;
			break;
		case PC_FLAG:
			*(int *)pc_vars[i].var = pc_vars[i].idefault;
			break;
		case PC_STRING:
			*(char **)pc_vars[i].var = pc_vars[i].sdefault;
			break;
		default:
			fatal( XLOG_INFO,"getpcvars; bad kind %d in table entry %d",
				pc_vars[i].kind,i);
		}
	}
	strloc = strbuf;
	if(Debug>7)log(XLOG_DEBUG,"init_pc_entry: done");
}

/*
 * char *fix_str( char *str )
 *   makes a copy of str in strbuf and returns the start
 * Side Effect: modifies strloc
 */
static char *
fix_str(str)
	char *str;
{
	char *cp;	/* ACME Pointers, Inc. */
	int l;

	l = strlen(str);
	cp = strloc;
	if( ! ((strloc+l) < (strbuf+sizeof(strbuf))) ){
		fatal( XLOG_INFO,"fix_str: string table overflow" );
	}
	(void)strcpy(strloc,str);
	strloc += (l+1);
	return(cp);
}

/*
 * Set_pc_entry( char *name; PC_ENTRY pc_vars[]; int pc_len)
 *  1. gets the "name" printcap entry; capbuf[] holds it;
 *     strbuf is used to hold string constants.
 *  2. checks to see if the entry has a continuation (tc= field),
 *     and gets the continuations
 *  3. calls getpcvar routine to find fields in the printcap entry and set 
 *     variables.
 * Returns: 1 if successful, and printcap entry found; 0 otherwise.
 * Side Effect: sets the First_name to the first name in the printcap entry.
 */
Set_pc_entry(name, pc_vars, pc_len)
	char *name;		/* the printer name */
	PC_ENTRY pc_vars[];	/* the printcap vars used for the printer */
	int pc_len;	/* the number of printcap vars */
{
	int found;		/* ACME Integers and Pointers */
	char *cp;

	if(Debug>7)log(XLOG_DEBUG,"Set_pc_entry: %s", name);
	if( name == 0 || *name == 0 ){
		return( 0 );
	}
	found = 0;
	while( found == 0 && pc_entry(capbuf, sizeof(capbuf)) > 0 ) {
		if(Debug>7)log(XLOG_DEBUG,"Set_pc_entry: pc_entry %s", capbuf);
		if (Chk_name(capbuf, name)) {
			found = 1;
		}
	}
	closeprent();
	if( found == 0 ){
		return(0);
	}
	if( First_name == 0 ){
		if( (cp = index( capbuf, '|')) == 0 ){
			cp = index( capbuf, ':');
		}
		found = *cp;
		*cp = 0;
		First_name = fix_str( capbuf );
		*cp = found;
	}
	if(Debug>8){
		(void)fprintf(stderr,"printer %s, %s, len %d printcap ",
			name,First_name,pc_len);
		(void)fprintf(stderr,capbuf);
		(void)fprintf(stderr,"\n");
	}
	/*
	 * get the printcap variable values
	 */
	getpcvars(pc_vars, pc_len);	/* get the printcap variable values */
	if(Debug>6)show_pc( pc_vars, pc_len);
	return(1);
}

/*
 * Get_pc_entry( char *name; PC_ENTRY pc_vars[]; int pc_len)
 *  1. calls init_pc_entry() to initialize variables
 *  2. calls Set_pc_entry() to set them
 */
Get_pc_entry(name, pc_vars, pc_len)
	char *name;		/* the printer name */
	PC_ENTRY pc_vars[];	/* the printcap vars used for the printer */
{
	if(Debug>7)log(XLOG_DEBUG,"Get_pc_entry: %s", name);
	init_pc_entry( pc_vars, pc_len);
	return( Set_pc_entry(name, pc_vars, pc_len) );
}


/*
 * Chk_name deals with name matching.  The first line of the printcap
 * entry is a sequence of names separated by |'s, so we compare
 * against each such name.  The normal : terminator after the last
 * name (before the first field) stops us.
 */
static int
Chk_name(buf, name)
	char *buf, *name;
{
	int l;

	l = strlen(name);

	while( buf && *buf ){
		if( strncmp( name, buf, l ) == 0 && (buf[l] == ':' || buf[l] == '|')) {
			/* match */
			return(1);
		}
		if( buf = index(buf, '|' ) ){
			buf = buf + 1;
		}
	}
	/* no match */
	return(0);
}

/*
 * Return the (numeric) option id.
 * Numeric options look like
 *	li#80
 * i.e. the option string is separated from the numeric value by
 * a # character.  If the option is not found we return -1.
 * Note that we handle octal numbers beginning with 0.
 */
static int
getnum(bp)
	char *bp;
{
	int i, base;

	if (*bp != '#'){
		return(-1);
	}
	bp = bp + 1;
	i = 0;
	base = 10;
	if (*bp == '0'){
		base = 8;
	} 
	while(isdigit(*bp)){
		i = i*base + (*bp - '0');
		++bp;
	}
	return (i);
}

/*
 * Handle a flag option.
 * Flag options are given "naked", i.e. followed by a :, @ or end
 * of the buffer.  Return 1 if we find the option, or 0 if it is
 * not given.
 */
static int
getflag(bp)
	char *bp;
{
	if (*bp == 0 || *bp == ':'){
		return (1);
	}
	return(0);
}

/*
 * Get a string valued option.
 * These are given as xx=string
 * There is a limited amount of decoding done.
 * \n -> '\n', \nnn -> '\nnn'
 * ^<CHAR> -> CTRL-CHAR
 */
static char *
getstr(str)
	char *str;	/* points to string entry */
{
	char buf[BUFSIZ];
	char *bp, *cp, *op;		/* ACME Chain and Pointers, Inc */
	int c, i;			/* ACME Integers, Inc */
	static char norm[] = "E^\\:nrtbf";
	static char esc[] = "\033^\\:\n\r\t\b\f";

	bp = str;
	op = buf;	/* output area */
	if (*bp != '=')
		return( (char *)0 );
	bp++;

	while ((c = *bp++) && c != ':') {
		switch (c) {
		case '^':
			c = *bp++;
			if( c == 0 ){
				fatal(XLOG_INFO,"getstr: bad escape string in printcap" );
			}
			c = c & 037;
			break;
		case '\\':
			c = *bp++;
			if( c == 0 ){
				fatal(XLOG_INFO,"getstr: bad escape string in printcap" );
			}
			cp = index( norm, c );
			if( cp ){
				c = esc[cp - norm];
			} else if (isdigit(c)) {
				c = c - '0';
				for( i = 0; i < 3 && isdigit(*bp); ++i ){
					c = 8*c + *bp - '0';
					++bp;
				}
			}
			break;
		}
		*op++ = c;
	}
	*op++ = 0;
	if( strlen( buf ) > 0 ){
		return( fix_str(buf) );
	} else {
		return( (char *)0 );
	}
}

/*
 * getpcvars- passed a table of entries, tries to find them all
 */
getpcvars(pc_vars, pc_len)
	PC_ENTRY *pc_vars;
	int pc_len;
{
	int h, l, m;	/* high, low, middle */
	char *cp;	/* string value */
	int dir;	/* direction */
	
	/*
	 * now scan the printcap entry, getting the variables
	 */
	cp = capbuf;
	while( cp && (cp = index(cp, ':' ))){
		/*
		 * we are now positioned at the ":"
		 */
		cp = cp+1;
		if( cp[-2] == '\\' ){
			continue;
		}
		/* note: islower is a macro! do not combine the above statement! */
		if( !islower( cp[0] ) || !islower( cp[1] )){
			continue;
		}
		if(Debug>7)log(XLOG_DEBUG,"get_pc_vars: entry %c%c", cp[0], cp[1]);
		/*
		 * binary search the table
		 */
		l = 0; h = pc_len-1;
		dir = 1;
		while( dir && l <= h ){
			m = (l+h)/2;
			if(Debug>8)log( XLOG_DEBUG, "get_pc_vars: l %d, m %d, h %d",l,m,h);
			dir = strncmp( cp, pc_vars[m].pc_name, 2 );
			if( dir < 0 ){
				h = m-1; /* too high */
			} else {
				l = m + 1;	/* too low */
			}
		}
		if( dir == 0 ){
			/* bingo! we found it */
			if(Debug>7)log(XLOG_DEBUG,"get_pc_vars: found %c%c%c%c%c",
				cp[0], cp[1], cp[2], cp[3], cp[4] );
			fixentry( &pc_vars[m], cp+2 );
		}
		/*
		 * we finished, on to next
		 */
	}
}

/*
 * fixentry: fix up the value for the entry in the printcap
 */

fixentry( pc_vars, cp )
	PC_ENTRY *pc_vars;
	char *cp;
{
	if(Debug>6)log(XLOG_DEBUG,"found %c%c", cp[-2], cp[-1] );
	switch( pc_vars->kind ){
	case PC_NUM:
		*(int *)pc_vars->var = getnum(cp);
		break;
	case PC_FLAG:
		*(int *)pc_vars->var = getflag(cp);
		break;
	case PC_STRING:
		*(char **)pc_vars->var = getstr(cp);
		break;
	default:
		fatal( XLOG_INFO, "fixentry: bad kind in the pc_vars" );
		break;
	}
}

/*
 * Debugging and display information
 */

show_pc( pc_vars, pc_len )
	PC_ENTRY *pc_vars;
	int pc_len;
{
	int i;

	(void)fprintf(stderr, "printcap entry %s: \n", First_name );
	for( i = 0; i < pc_len; ++i ){
		(void)fprintf(stderr, "%s: ", pc_vars[i].pc_name );
		switch( pc_vars[i].kind ){
		case PC_NUM:
			(void)fprintf(stderr, "PC_NUM %d\n", *(int *)pc_vars[i].var );
			break;
		case PC_FLAG:
			(void)fprintf(stderr, "PC_FLAG %d\n", *(int *)pc_vars[i].var );
			break;
		case PC_STRING:
			(void)fprintf(stderr, "PC_STRING %s\n", *(char **)pc_vars[i].var );
			break;
		}
	}
	(void)fflush(stderr);
}
/***************************************************************************
 * closeprent
 * closes the file that the printcap was read from
 ***************************************************************************/
closeprent()
{
	if (pfp != NULL)
		(void) fclose(pfp);
	pfp = NULL;
}
