/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: Print_support.c
 * handle the actual output to the Printer
 ***************************************************************************
 * Revision History: Created Wed Jan 13 16:21:39 CST 1988
 * $Log:	print_support.c,v $
 * Revision 3.1  88/06/18  09:35:18  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.4  88/05/27  08:28:01  papowell
 * Added a SIGCONT to start up the output filter.
 * 
 * Revision 2.3  88/05/19  10:34:19  papowell
 * Fixed open() calls to have a 0 parameter, ie: open(f, perms, 0), where needed
 * 
 * Revision 2.2  88/05/14  10:21:16  papowell
 * Modified -X flag handling
 * 
 * Revision 2.1  88/05/09  10:09:44  papowell
 * PLP: Released Version
 * 
 * Revision 1.4  88/03/25  15:01:04  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.3  88/03/11  19:29:04  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.2  88/03/05  15:01:11  papowell
 * Minor Corrections,  Lint Problems
 * 
 * Revision 1.1  88/03/01  11:09:01  papowell
 * Initial revision
 * 
 ***************************************************************************
 * void Print_open(): opens the Printer
 * static int Print_of_fd(): makes an 'of' filter if needed
 * void Print_close(): closes the Printer
 * int Print_ready(): combines Print_open() and Print_of_fd()
 * static int of_stop(): stops the 'of' filter
 * int Print_string( str ): prints a string through 'of' or to Printer
 * int Print_copy( fd ): copies a file through 'of' or to Printer
 * int Print_filter( file, cmd ): makes a filter and copies file
 * int Print_banner(): prints a banner through 'of' or to Printer
 * Note: the above procedures which return values return JSUCC on success,
 *   and JFAIL or JABORT on failure
 ***************************************************************************/

#ifndef lint
static char id_str1[] =
	"$Header: print_support.c,v 3.1 88/06/18 09:35:18 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"

/***************************************************************************
 * Print_open()
 * 	Open the Printer, and set the Print_fd variables
 * 	If the RW printcap flag is set, output is opened RW, otherwise
 * 	opened writeonly in append mode.
 * 
 * 	If the Printer is a tty (i.e.- isatty returns non-zero),
 * 	then the baud rate is set.  The baud rate table is stolen from
 * 	the original LPD code, which stole it from STTY, etc.
 * Side Effect:
 * 	sets the Print_fd variable to a non-zero value
 *  terminates if Printer is unable to be opened
 ****************************************************************************/
struct bauds {
	char *string;
	int	baud;
	int	speed;
} bauds[] = {
	"110", 110,	B110,
	"134", 134,	B134,
	"150", 150,	B150,
	"300", 300,	B300,
	"600", 600,	B600,
	"1200", 1200,	B1200,
	"1800", 1800,	B1800,
	"2400", 2400,	B2400,
	"4800", 4800,	B4800,
	"9600", 9600,	B9600,
	"19200", 19200,	B19200,
	"38400", 38400,	B38400,
	(char *)0, 0,	0
};

static int of_pid;		/* OF process pid */
static int of_fd;		/* pipe to OF process */
static int of_running;	/* of filter running */
static char filter_stop[] = "\031\001";	/* what to send filter */

Print_open()
{
	int err;

	/*
	 * check to see if it is open
	 */
	if( Print_fd ){
		return;
	}
	if( LP == 0 || *LP == 0 ){
		fatal( XLOG_INFO, "Missing LP value for local Printer");
	}
	setstatus("waiting for %s to become ready since %s (offline?)",
		Printer, Time_str());
	Print_fd = open(LP, RW ? ( O_RDWR|O_APPEND ) : ( O_WRONLY|O_APPEND ), 0);
	if( Print_fd < 0 ){
		err = errno;
		setstatus("Print_open: '%s' open(%s) failed, %s",Printer,LP,
			Errormsg(err));
		errno = err;
		logerr_die( XLOG_INFO, "Print_open: cannot open %s", LP);
	} else if( Print_fd == 0 ){
		fatal( XLOG_INFO, "Print_open: cannot happen- Print_fd == 0");
	}
	/*
	 * if it is a tty, set the baud rates and control information
	 */
	if (isatty(Print_fd)){
		Do_stty(Print_fd);
	}
	if(Debug>3)log(XLOG_DEBUG,"Print_open: %s is fd %d", LP, Print_fd );
}

/*
 * Start up an output filter, if needed.
 */
Print_of_fd()
{
	int p[2];
	char *cmd;

	if( OF == 0 || *OF == 0 || of_pid ){
		return;
	}
	/*
	 * set OF command line
	 */
	cmd = Setup_filter('o',OF);
	if( cmd == 0 || *cmd == 0 ){
		fatal(XLOG_INFO, "bad OF entry");
	}
	if(Debug>2)log( XLOG_DEBUG, "starting OF, '%s'", cmd);
	/*
	 * create pipe and fork
	 */
	if( pipe(p) < 0 ){
		logerr_die( XLOG_NOTICE, "Print_of_of: pipe failed" );
	}
	if( p[0] < 3 || p[1] < 3 ){
		fatal( XLOG_INFO, "Print_of_fd: IMPOSSIBLE- pipe fd %d %d", p[0], p[1]);
	}
	if ((of_pid = fork()) == 0) {	/* child */
		if( dup2(p[0], 0) < 0		/* pipe is std in */
		 || dup2(Print_fd, 1) < 0 ){		/* Printer is std out */
			logerr_die( XLOG_NOTICE, "Print_of_fd: dup2 failed" );
		}
		if(Debug>3)log( XLOG_DEBUG, "Print_of_fd: forked %d OF: %s",
			getpid(), cmd );
		if( geteuid() == 0 && setreuid( Daemon_uid, Daemon_uid ) < 0 ){
			logerr_die( XLOG_NOTICE, "Print_of_fd: setreuid failed" );
		}
		mexecv( cmd );
		logerr_die( XLOG_NOTICE,"Print_of_fd: execv failed %s", cmd);
	} else if( of_pid < 0 ){
		logerr_die( XLOG_NOTICE, "Print_of_fd: fork failed" );
	}
	(void)close(p[0]);		/* close input side */
	of_fd = p[1];
	of_running = of_pid;
	if(Debug>3)log(XLOG_DEBUG,"started OF process %d, fd %d", of_pid, of_fd);
}

/***************************************************************************
 * Print_close()
 * 1. Close the pipes and outputs.
 * 2. Signal the of process that it is to do something.
 *	Note that there is a Reapchild() call that will collect the
 *	children;  this will make sure that there is a minimal number of
 *	outstanding processes active.
 ***************************************************************************/

void
Print_close()
{
	if( of_fd ){
		(void)close(of_fd);
		of_fd = 0;
	}
	if( Print_fd ){
		(void)close(Print_fd);	/* close Printer */
		Print_fd = 0;
	}
	/*
	 * when you kill the printer, make sure that you do it neatly
	 */
	if( of_pid ){
		(void)kill(of_pid, SIGCONT);
	}
	of_pid = 0;
}

/***************************************************************************
 * Print_ready()
 * 1. open the Printer if neccessary
 * 2. start up the output filter
 * 3. if the filter has been stopped, start it
 * Return: JSUCC if started, JFAIL if not
 ***************************************************************************/
Print_ready()
{
	/*
	 * open the Printer
	 */
	Print_open();
	Print_of_fd();
	if( of_pid && of_running == 0 ){
		if (kill(of_pid, SIGCONT) < 0) {
			logerr( XLOG_INFO,"cannot restart output filter");
			Print_close();
			return( JFAIL );
		}
		of_running = of_pid;
	}
	return( JSUCC );
}

/***************************************************************************
 * of_stop()
 * stop the output filter if neccessary
 * 1. open and create the filter
 * 2. flush output
 * 3. kill(SIGSTOP) it
 * Return: JSUCC if stopped, JFAIL if not
 ***************************************************************************/

int
of_stop()
{
	union wait statb;
	int pid;

	Print_open();
	Print_of_fd();
	/*
	 * stop the OF filter
	 */
	if( of_pid && of_running ){
		if(Debug>3)log( XLOG_DEBUG, "stopping output filter" );
		if( write( of_fd, filter_stop, strlen(filter_stop))
			!= strlen(filter_stop) ){
			logerr( XLOG_NOTICE,"of_stop: cannot write to OF");
			Print_close();
			return( JFAIL );
		}
		/*
		 * wait until OF blocks
		 */
		if(Debug>3)log( XLOG_DEBUG, "of_stop: waiting for OF %d", of_pid );
		while ((pid = wait3(&statb,WUNTRACED,(struct rusage *)0)) != -1
			&& pid != of_pid) ;
		if( pid < 0 || statb.w_stopval != WSTOPPED ){
			logerr( XLOG_INFO, "of_stop: OF %d died (%s)", of_pid,
				Decode_status( &statb ) );
			Print_close();
			return( JFAIL );
		}
		if(Debug>3)log( XLOG_DEBUG, "of_stop: output filter stopped" );
		of_running = 0;
	}
	return( JSUCC );
}

/***************************************************************************
 * Print_string( char *str )
 * print a string through a filter 
 * 1. Enable the line Printer
 * 2. get the filter or device fd;
 * 3. put out the string
 * 4. if unsuccessful, close the Printer
 * Return: JSUCC if successful, JFAIL otherwise
 ***************************************************************************/
int
Print_string( str )
	char *str;
{
	int f;
	int l;

	if( Print_ready() != JSUCC ){
		return( JFAIL );
	}
	if( of_fd ){
		f = of_fd;
	} else {
		f = Print_fd;
	}
	l = strlen( str );
	if( write( f, str, l ) != l ){
		logerr( XLOG_INFO, "Print_string: write error");
		Print_close();
		return( JFAIL );
	}
	return( JSUCC );
}


/***************************************************************************
 * Print_copy( int fd )
 * copy a file through a filter
 * 1. ready the Printer
 * 2. copy the file to the appropriate output device
 * 3. if an error, close the Printer
 ***************************************************************************/
int
Print_copy( fd )
	int fd;
{
	int f;
	long cnt;
	char buf[BUFSIZ];
	int in;			/* bytes read */
	int out;		/* bytes written out */

	cnt = 0;
	if( Print_ready() != JSUCC ){
		return( JFAIL );
	}
	if( of_fd ){
		f = of_fd;
	} else {
		f = Print_fd;
	}
	while( (in = read( fd, buf, sizeof(buf) )) > 0 ){
		out = write( f, buf, in );
		if( in != out ){
			logerr( XLOG_INFO, "Print_copy: write error");
			Print_close();
			return( JFAIL );
		}
		cnt = cnt + out;
	}
	/*
	 * completed the reading
	 */
	if( in < 0 ){
		logerr( XLOG_INFO, "Print_copy: read error");
		Print_close();
		return( JFAIL );
	}
	if(Debug>3)log(XLOG_DEBUG,"Print_copy: printed %d bytes", cnt);
	return( JSUCC );
}

/***************************************************************************
 *Print_filter( int fd, char *cmd )
 *  spawn a subprocess to do the printing
 * 1. stop the Printer
 * 2. fork a process
 * 3. wait for process to complete
 * 4. restart the process
 * Return: JSUCC if successful, JFAIL otherwise
 ***************************************************************************/
int
Print_filter( file, cmd )
	int file;
	char *cmd;
{
	int succ;			/* success code */
	int filter_pid;		/* filter process */
	int pid;			/* daughter pid */
	union wait status;	/* daughter status */
	int err;			/* saved value of errno */
	
	/*
	 * stop the Printer
	 */
	succ = of_stop();
	if( succ != JSUCC ){
		return( succ );
	}
	/*
	 * fork a process,  and connect file to fd 0, Printer to fd 1.
	 */
	if ((filter_pid = fork()) == 0) {	/* daughter */
		/*
		 * dup input file to standard in and Printer to stdout
		 */
		if( dup2(file, 0)<0 || dup2(Print_fd, 1)<0 ){
			logerr_die(XLOG_NOTICE,"Print_filter: dup2 failed filter %s",cmd);
		}
		mexecv(cmd);
		logerr_die( XLOG_NOTICE,"Print_filter: cannot execv %s", cmd);
	} else if( filter_pid < 0 ){
		logerr( XLOG_NOTICE, "Print_filter: fork failed" );
		Print_close();
		return( JFAIL );
	}
	/*
	 *	Wait for filter to complete 
	 */
	if(Debug>2)log(XLOG_DEBUG,"Print_filter: waiting for pid %d",filter_pid);
	while ((pid = wait(&status)) > 0 && pid != filter_pid){
		if(Debug>3)log( XLOG_DEBUG, "Print_filter:caught %d (%s)",
			pid, Decode_status(&status) );
	}
	err = errno;
	if(Debug>2)log( XLOG_DEBUG, "Print_filter: filter %d finished (%s)",
		pid,Decode_status(&status));
	errno = err;
	/*
	 * Check to see how filter terminated
	 */
	if( pid < 0 || !WIFEXITED(status)
		|| (unsigned)status.w_retcode > 1 ){
		/*
		 * died for bad reasons, don't run this again
		 */
		log(XLOG_INFO,"Print_filter:Filter '%s' Malfunction (%s)",
			cmd, Decode_status(&status));
		Print_close();
		return( JABORT );
	} else if (status.w_retcode != 0){
		/*
		 * try again
		 */
		log( XLOG_INFO, "Print_filter:Filter '%s' Retry wanted", cmd );
		Print_close();
		return(JFAIL);
	}
	return(JSUCC);
}

/*
 * Print_banner()
 * 1. get the Printer ready
 * 2. call the banner() routine with the correct parameter
 */
Print_banner()
{
	int f;

	if(Debug>3)log(XLOG_DEBUG,"Print_banner: printing banner");
	if( Print_ready() != JSUCC ){
		return( JFAIL );
	}
	if( of_fd ){
		f = of_fd;
	} else {
		f = Print_fd;
	}
	if( banner(f) != JSUCC ){
		if(Debug>3)log(XLOG_DEBUG,"Print_banner: banner failed");
		Print_close();
		return( JFAIL );
	}
	return( JSUCC );
}

/*
 * Fri Feb 26 08:44:53 CST 1988 Patrick Powell
 * set terminal modes
 * This code was based on a public domain version of public domain version
 * of stty. I suppose that I could have created if from scratermctrlh,
 * but I have seen the same table appearing
 * in many "public domain" display terminal modes programs.
 */

struct tchars termctrl;
struct ltchars linectrl;
struct sgttyb mode;
struct
{
	char	*string;
	int	set;
	int	reset;
	int	lset;
	int	lreset;
} modes[] = {
	"bs0",		BS0, BS1, 0, 0,
	"bs1",		BS1, BS1, 0, 0,
	"cbreak",	CBREAK, 0, 0, 0,
	"-cbreak",	0, CBREAK, 0, 0,
	"cooked",	0, RAW, 0, 0,
	"cr0",		CR0, CR3, 0, 0,
	"cr1",		CR1, CR3, 0, 0,
	"cr2",		CR2, CR3, 0, 0,
	"cr3",		CR3, CR3, 0, 0,
	"decctlq",	0, 0, LDECCTQ, 0,
	"-decctlq",	0, 0, 0, LDECCTQ,
	"echo",		ECHO, 0, 0, 0,
	"-echo",	0, ECHO, 0, 0,
	"even",		EVENP, 0, 0, 0,
	"-even",	0, EVENP, 0, 0,
	"ff0",		FF0, FF1, 0, 0,
	"ff1",		FF1, FF1, 0, 0,
	"lcase",	LCASE, 0, 0, 0,
	"-lcase",	0, LCASE, 0, 0,
	"litout",	0, 0, LLITOUT, 0,
	"-litout",	0, 0, 0, LLITOUT,
	"nl",		0, CRMOD, 0, 0,
	"-nl",		CRMOD, 0, 0, 0,
	"nl0",		NL0, NL3, 0, 0,
	"nl1",		NL1, NL3, 0, 0,
	"nl2",		NL2, NL3, 0, 0,
	"nl3",		NL3, NL3, 0, 0,
	"noflsh",	0, 0, LNOFLSH, 0,
	"-noflsh",	0, 0, 0, LNOFLSH,
	"nohang",	0, 0, LNOHANG, 0,
	"-nohang",	0, 0, 0, LNOHANG,
	"odd",		ODDP, 0, 0, 0,
	"-odd",		0, ODDP, 0, 0,
	"raw",		RAW, 0, 0, 0,
	"-raw",		0, RAW, 0, 0,
	"tab0",		TAB0, XTABS, 0, 0,
	"tab1",		TAB1, XTABS, 0, 0,
	"tab2",		TAB2, XTABS, 0, 0,
	"tabs",		0, XTABS, 0, 0,
	"-tabs",	XTABS, 0, 0, 0,
	"tandem",	TANDEM, 0, 0, 0,
	"-tandem",	0, TANDEM, 0, 0,
	"tilde",	0, 0, LTILDE, 0,
	"-tilde",	0, 0, 0, LTILDE,
	"tn300",	CR1, ALLDELAY, 0, 0,
	"tty33",	CR1, ALLDELAY, 0, 0,
	"tty37",	FF1+CR2+TAB1+NL1, ALLDELAY, 0, 0,
	"vt05",		NL2, ALLDELAY, 0, 0,
	0,
};


struct	special {
	char	*name;
	char	*cp;
	char	def;
} special[] = {
	"stop",		&termctrl.t_stopc,		CSTOP,
	"start",	&termctrl.t_startc,		CSTART,
	0
};

Do_stty( fd )
	int fd;
{
	int i;
	int	localmode;
	int	linedisc;
	char buf[BUFSIZ], *bp, *ep, *arg;

	if( ioctl(fd, TIOCGETP, &mode) < 0
	 || ioctl(fd, TIOCGETC, &termctrl) < 0
	 || ioctl(fd, TIOCLGET, &localmode) < 0
	 || ioctl(fd, TIOCGLTC, &linectrl) < 0 ){
		logerr_die( XLOG_INFO,"cannot get tty parameters");
	}
	if(Debug>3)log(XLOG_DEBUG,"stty: before mode 0x%x, lmode 0x%x, speed 0x%x",
			mode.sg_flags, localmode, mode.sg_ispeed );
	if( BR ){
	  for(i=0; bauds[i].baud && BR != bauds[i].baud; i++);
	  if( i == 0){
		  fatal(XLOG_INFO,"illegal baud rate %d", BR);
	  }
	  mode.sg_ispeed = mode.sg_ospeed = bauds[i].speed;
	}
	mode.sg_flags &= ~FC;
	mode.sg_flags |= FS;
	localmode &= ~XC;
	localmode |= XS;
 

	if( TY && *TY ){
		(void)strcpy(buf, TY);
		ep = buf;
	} else {
		ep = 0;
	}
	while( ep && *ep ){
		for( ; *ep && isspace(*ep) ; ++ ep );
		for( arg = ep; *ep && !isspace(*ep) ; ++ ep );
		if( *ep ){
			*ep = 0;
			++ep;
		}
		for(i=0; modes[i].string && strcmp(modes[i].string,arg); i++);
		if(modes[i].string) {
			if(Debug>4)log(XLOG_DEBUG,
				"stty: modes %s, mc 0x%x ms 0x%x lc 0x%x ls 0x%x",
				modes[i].string, modes[i].reset, modes[i].set,
				modes[i].lreset, modes[i].lset );
			mode.sg_flags &= ~modes[i].reset;
			mode.sg_flags |= modes[i].set;
			localmode &= ~modes[i].lreset;
			localmode |= modes[i].lset;
			continue;
		}
		for (i = 0; special[i].name && strcmp(special[i].name,arg); i++);
		if( special[i].name ){
			for( ; *ep && isspace(*ep) ; ++ ep );
			for( bp = ep; *ep && !isspace(*ep) ; ++ ep );
			if( *ep ){
				*ep = 0;
				++ep;
			}
			if( *bp == 0 ){
				fatal( XLOG_INFO, "stty: missing parameter for %s", arg );
			}
			if (bp[0] == '^'){
				if( bp[1] == '?' ){
					*special[i].cp = 0177;
				} else {
					*special[i].cp = 037 & bp[1];
				}
			} else {
				*special[i].cp = bp[0];
			}
			if(Debug>4)log(XLOG_DEBUG,"stty: special %s %s", arg, bp );
			continue;
		}
		for(i=0; bauds[i].string && strcmp(bauds[i].string,arg); i++);
		if(bauds[i].string) {
			if(Debug>4)log(XLOG_DEBUG,"stty: speed %s", arg );
			mode.sg_ispeed = mode.sg_ospeed = bauds[i].speed;
			continue;
		}
		if (!strcmp("new", arg)){
			if(Debug>4)log(XLOG_DEBUG,"stty: ldisc %s", arg );
			linedisc = NTTYDISC;
			if (ioctl(fd, TIOCSETD, &linedisc)<0)
				logerr_die(XLOG_INFO,"stty: TIOCSETD ioctl failed");
			continue;
		}
		if (!strcmp("old",arg)){
			if(Debug>4)log(XLOG_DEBUG,"stty: ldisc %s", arg );
			linedisc = 0;
			if (ioctl(fd, TIOCSETD, &linedisc)<0)
				logerr_die(XLOG_INFO,"stty: TIOCSETD ioctl failed");
			continue;
		}
		fatal(XLOG_INFO,"unknown mode: %s\n", arg);
	}
	if(Debug>3)log(XLOG_DEBUG,"stty: after mode 0x%x, lmode 0x%x, speed 0x%x",
			mode.sg_flags, localmode, mode.sg_ispeed );
	if( ioctl(fd, TIOCSETN, &mode) < 0
		|| ioctl(fd, TIOCSETC, &termctrl) < 0
		|| ioctl(fd, TIOCSLTC, &linectrl) < 0
		|| ioctl(fd, TIOCLSET, &localmode) < 0 ){
		logerr_die( XLOG_NOTICE,"cannot set tty parameters");
	}
}
