/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: pac.c
 * Accounting Summary
 ***************************************************************************
 * Revision History: Created Thu Feb  4 15:39:38 CST 1988
 * $Log:	pac.c,v $
 * Revision 3.1  88/06/18  09:35:14  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.2  88/05/14  10:20:47  papowell
 * Modified -X flag handling
 * 
 * Revision 2.1  88/05/09  10:09:39  papowell
 * PLP: Released Version
 * 
 * Revision 1.6  88/05/05  20:07:32  papowell
 * Minor changes to shut up lint.
 * 
 * Revision 1.5  88/04/15  13:06:00  papowell
 * Std_environ() call added, to ensure that fd 0 (stdin), 1 (stdout), 2(stderr)
 * have valid file descriptors;  if not open, then /dev/null is used.
 * 
 * Revision 1.4  88/04/07  09:11:50  papowell
 * Removed redundant break;  strangely, lint did not pick this up
 * 
 * Revision 1.3  88/03/25  15:00:57  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.2  88/03/11  19:28:52  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.1  88/03/01  11:08:58  papowell
 * Initial revision
 * 
 ***************************************************************************/

#ifndef lint
static char id_str1[] =
	"$Header: pac.c,v 3.1 88/06/18 09:35:14 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint
#include "lp.h"
/***************************************************************************
 *      pac - printer/ploter accounting information
 * SYNOPSIS
 *      pac [ -Pprinter ] [ -pprice ] [ -a ] [ -e ] [ -s ]
 * DESCRIPTION
 *      Pac reads the printer/plotter accounting files, accumulating
 *      the number of pages (the usual case) or feet (for raster
 *      devices) of paper consumed by each user, and printing out
 *      how much each user consumed in pages or feet and dollars.
 *      -Pprinter
 *           Selects the printer.  If not specified, the value of
 *           the environment variable PRINTER is used, and if it is
 *           not defined, the default printer for the site.
 *      -pprice
 *           The price per page to be used, instead of the printcap
 *           co (cost) entry.  If none is specified, the default
 *           price of $0.02 per page is used.
 *      -a   Accounting is done for all printers.  A cumulative sum-
 *           mary of usage of all the entries will be printed.
 *      -e   Accounting is done for each printer entry in turn.
 *      -s   Summarize the accounting information in the summary
 *           accounting file.  Used with the -a or -e flag all the
 *           accounting information for each printer can be done at
 *           once.  This option has the side effect of truncating
 *           the original file.  It copies the original file to a
 *           backup before doing any action.
 *      The accounting file is specified in the printcap af entry.
 *      The summary accounting file has the .sum suffix added.
 * FILES
 *      printcap af entry
 * NOTES
 *      The accounting information is generated by the various
 *      filters used by the LPD daemon.  For each user file they
 *      should write a line to the accounting file in the following
 *      format.
 *           user host printer pages format date
 *           papowell umn-cs 10 C April 13 1988
 *      The summary file will produce entries of the form:
 *           user host printer pages format runs cost
 *           papowell umn-cs 1320 17 32.00
 ***************************************************************************/
int	All;			/* do for all printers */
int Each;			/* do for each printer */
int Summary;		/* do a summary for each printer */
int Xpert;
int Page_price;			/* cost per page to be used */
float	pagecost = 0.0;			/* cost per page (or what ever) */

/*
 * Hash Table
 * Each user has his own entry
 */

#define	HASHSIZE	97			/* Number of hash buckets */

struct entry {
	struct entry *next;		/* Forward hash link */
	char	user[32];		/* user name */
	char	host[32];		/* host name */
	char	printer[32];	/* host name */
	long	pages;			/* Feet or pages of paper */
	char	format[32];		/* format */
	long	runs;			/* Number of runs */
	float	price;			/* cost for this */
	struct entry *chain;	/* list link */
};

struct	entry	*Hash_table[HASHSIZE];	/* Hash table proper */
struct entry *All_user, *Sum_user, *Raw_user;


main( argc, argv )
	int argc;
	char **argv;
{
	char **list;

	/*
	 * set umask to avoid problems with user umask
	 */
	(void)umask(0);
	/*
	 * Set fd 0, 1, 2 to /dev/null if not open
	 */
	Std_environ();
#	ifdef XPERIMENT
		Setup_test();
#	endif XPERIMENT
	/*
     * set up the pathnames for information files
	 */
	Tailor_names();

	/*
	 * process the parameters
	 */
	Get_parms(argc, argv);

	if( All | Each ){
		for( list = All_printers(); *list; ++list ){
			Printer = *list;
			Process();
		}
	} else {
		Process();
	}
	if( All ){
		Sort_file( &All_user );
		Dump_info(All_user, "All");
	}
}

/***************************************************************************
 * Get_parms()
 * get the parameters
 ***************************************************************************/
static char *optstr = "D:P:Xaep:s";
Get_parms( argc, argv )
	int argc;
	char **argv;
{
	int option;

	while( (option = Getopt(argc,argv,optstr)) != EOF ){
		switch( option ){
		case 'p':
			Check_int_dup( option, &Page_price, Optarg );
			break;
		case 'D':
			Check_int_dup( option, &Debug, Optarg );
			break;
		case 'P':
			if( Printer ){
				Check_str_dup( option, Printer, Optarg );
			}
			Printer = Optarg;
			break;
		case 'X':
			Check_dup( option, &Xpert );
#			ifdef DEBUG
				Setup_test();
				Tailor_names();
#			else
				Diemsg( "-X not allowed" );
#			endif DEBUG
			break;
		case 'e':
			Check_dup( option, &Each );
			break;
		case 'a':
			Check_dup( option, &All );
			break;
		case 's':
			Check_dup( option, &Summary );
			break;
		case '?':
			break;
		default:
			fatal(XLOG_INFO, "Get_parms: badparm %c", option );
		}
	}

	/*
	 * set up the Parms[] array
	 */
	for( ; Optind < argc; ++Optind ){
		if( Parmcount < MAXPARMS ){
			Parms[Parmcount].str = argv[Optind];
			++Parmcount;
		} else {
			Diemsg( "too many files to print; break job up" );
		}
	}
	Get_Printer(0);
	/*
	 * set default format
	 */
	if( FROMHOST[0] == 0 ){
		(void)strcpy(FROMHOST, Host);
	}
	if(Debug>4){
		(void)fprintf( stdout, "Printer %s, All %s, Each %s, Summary %s",
			Printer, All, Each, Summary );
	}
}

/***************************************************************************
 * Check_int_dup( int option, int *value, char *arg )
 * 1.  check to see if value has been set
 * 2.  if not, then get integer value from arg
 ***************************************************************************/
 
Check_int_dup( option, value, arg )
	int option;
	int *value;
	char *arg;
{
	if( *value ){
		Diemsg( "duplicate option %c", option );
	}
	if( sscanf( arg, "%d", value ) != 1 || *value <= 0){
		Diemsg( "option %c parameter (%s) is not positive integer",
			option, arg);
	}
}
/***************************************************************************
 * Check_str_dup( int option, char *value, char *arg )
 * 1.  check to see if value has been set
 * 2.  if not, then set it
 ***************************************************************************/
Check_str_dup( option, value, arg )
	int option;
	char *value;
	char *arg;
{
	if( *value ){
		Diemsg( "duplicate option %c", option );
	}
	if( strlen( arg ) > MAXPARMLEN ){
		Diemsg( "option %c arguement too long (%s)", option, arg );
	}
	(void)strcpy(value, arg );
}
/***************************************************************************
 * Check_dup( int option, int *value )
 * 1.  check to see if value has been set
 * 2.  if not, then set it
 ***************************************************************************/
Check_dup( option, value )
	int option;
	int *value;
{
	if( *value ){
		Diemsg( "duplicate option %c", option );
	}
	*value = 1;
}


/***************************************************************************
 * Process
 * Get the accounting file.
 * 1. Read printcap entry.
 * 2. Get the accounting file information
 * 3. If we are to summarize,  read the summyary
 * 4. Combine the summary and other file information.
 * 5. if we are to summarize, copy the information to the output summary file
 *    save the original file, and truncate it.
 ***************************************************************************/
Process()
{
	char buf[BUFSIZ];
	FILE *fp, *bp, *sp;		/* Accounting and summary file */

	fp = bp = sp = 0;
    if(Get_pc_entry(Printer, Status_pc_vars, Status_pc_len) == 0){
        Warnmsg("Printer %s does not exist\n", Printer );
        return;
    }
    if( SD == 0 || *SD == 0 ){
        if(Debug>2)log(XLOG_DEBUG,"not a Printer");
        return;
    }
    /* chdir to spool directory */
    if (chdir(SD) < 0) {
        logerr_die( XLOG_NOTICE,"cannot chdir to %s", SD);
    }
	if( AF == 0 || *AF == 0 ){
		Warnmsg("Printer %s does not have accounting file", Printer );
		return;
	}
	if( (fp = fopen( AF, "r+" )) == 0 ){
		Warnmsg("Printer %s does not have accounting file", Printer );
		return;
	}
	(void)strcpy(buf, AF );
	(void)strcat(buf, ".bac" );
	if( (bp = fopen( buf, "w" )) == NULL ){
		logerr_die( XLOG_NOTICE,"cannot open backup file %s", buf);
	}
	if( Summary ){
		(void)strcpy(buf, AF );
		(void)strcat(buf, ".sum" );
		if( (sp = fopen( buf, "r+" )) == NULL ){
			logerr_die( XLOG_NOTICE,"cannot open summary file %s", buf);
		}
	}
	Get_raw_file(fp,bp);
	if( Summary ){
		Get_summary(sp);
		Merge_raw_sum();
		Sort_file( &Sum_user );
		Update_sum(sp);
		if( Each ){
			Dump_info( Sum_user, Printer );
		}
	} else if( !All ){
		Dump_info( Raw_user, Printer );
	}
	if( All ){
		Merge_raw_all();
	}
	if( Sum_user ){
		Free_entry( Sum_user );
		Sum_user = 0;
	}
	Free_entry( Raw_user );
	Raw_user = 0;
	if( sp ){
		(void)fclose(sp);
	}
	if( bp ){
		(void)fclose(bp);
	}
	if( fp ){
		(void)fclose(fp);
	}
}

cleanup()
{
	exit(1);
}
/*
 * Get_raw_file()
 * read and copy the raw accounting file into the backup,
 * set up the Raw information
 */

Get_raw_file(f, b)
	FILE *f, *b;
{
	char buf[BUFSIZ];
	char user[BUFSIZ];
	char host[BUFSIZ];
	char printer[BUFSIZ];
	char format[BUFSIZ];
	int  pages;
	int  i;
	struct entry *p, *Add_entry();
	Raw_user = 0;
	
	Clean_hash();

	while( fgets(buf, sizeof(buf), f ) ){
		(void)fputs(buf, b);
		/* break out the fields */
		i = sscanf(buf,"%s %s %s %d %s\n",user,host,printer,&pages,format);
		if( i < 4 || strlen(user) > 32 || strlen(host)>32
			|| pages < 0 || strlen(format)>32){
			Warnmsg( "bad format of accounting information: %s\n", buf );
		}
		p = Add_entry( &Raw_user, user, host, printer, format );
		p->pages += pages;
		p->runs += 1;
	}
}
/*
 * Get_summary()
 * read and copy the raw accounting file into the backup,
 * set up the Raw information
 */

Get_summary(f)
	FILE *f;
{
	char buf[BUFSIZ];
	char user[BUFSIZ];
	char host[BUFSIZ];
	char printer[BUFSIZ];
	char format[BUFSIZ];
	int  pages;
	int  jobs;
	int  i;
	struct entry *p, *Add_entry();

	Sum_user = 0;
	
	Clean_hash();

	while( fgets(buf, sizeof(buf), f ) ){
		/* break out the fields */
		i = sscanf(buf,"%s %s %s %d %s %d\n",
			user,host,printer,&pages,format,&jobs);
		if( i < 4 || strlen(user) > 32 || strlen(host)>32
			|| pages < 0 || strlen(format)>32){
			Warnmsg( "bad format of accounting information: %s\n", buf );
		}
		p = Add_entry( &Sum_user, user, host, printer, format );
		p->pages += pages;
		p->runs += jobs;
	}
}

Clean_hash()
{
	int i;
	for(i = 0; i < HASHSIZE; ++i ){
		Hash_table[i] = 0;
	}
}

int
Hash( s )
	char *s;
{
	int i;
	
	i = s[1]*3 + s[0] + s[strlen(s) -1 ]*7;
	return( i % HASHSIZE );
}

struct entry *
Find_entry( user, host, printer, format )
	char *user, *host, *printer, *format;
{
	struct entry *p;
	int i;

	i = Hash( user );

	p = Hash_table[i];

	while( p ){
		if( strcmp(user, p->user ) == 0
			&& strcmp(host, p->host ) == 0
			&& strcmp(printer, p->printer ) == 0
			&& strcmp(format, p->format ) == 0 ){
			return(p);
		}
		p = p->next;
	}
	return( p );
}
	
static struct entry *entry_list;
#define NENTRY 25

struct entry *
Make_entry( list, user, host, printer, format )
	struct entry **list;
	char *user, *host, *printer, *format;
{
	struct entry *p;
	static struct entry blank;	/* all zeros */
	int i;

	if( entry_list == 0 ){
		p = entry_list =
			(struct entry *)malloc( sizeof(struct entry) * NENTRY );
		for( i = 0; i < (NENTRY - 1); ++i ){
			p[i].next = &p[i+1];
		}
		p[i].next = 0;
	}
	p = entry_list;
	entry_list = p->next;

	*p = blank;
	p->next = 0;
	p->chain = 0;
		
	i = Hash( user );
	p->next = Hash_table[i];
	Hash_table[i] = p;
	(void)strcpy( p->user, user );
	(void)strcpy( p->host, host );
	(void)strcpy( p->printer, printer );
	(void)strcpy( p->format, format );
	p->chain = *list;
	*list = p;
	return( p );
}

Free_entry( p )
	struct entry *p;
{
	struct entry *q;
	while( p ){
		q = p->chain;
		p->next = entry_list;
		entry_list = p;
		p = q;
	}
}

struct entry *
Add_entry( list, user, host, printer, format )
	struct entry **list;
	char *user, *host, *printer;
	char *format;
{
	struct entry *p;
	p = Find_entry( user, host, printer, format );
	if( p == 0 ){
		p = Make_entry( list, user, host, printer, format );
	}
	return( p );
}

Merge_raw_sum()
{
	Merge_raw(&Sum_user);
}
Merge_raw_all()
{
	Merge_raw(&All_user);
}
Merge_raw(s)
	struct entry **s;
{
	struct entry *p, *q;

	/*
	 * assume that you have just read in the summary file
	 */
	p = *s;
	while( p ){
		q = Make_entry( s, p->user, p->host, p->printer, p->format);
		q->runs += p->runs;
		q->pages += p->pages;
		p = p->chain;
	}
}

Update_sum( fp )
	FILE *fp;
{
	struct entry *p;

	/*
	 * next, truncate the file
	 */
	if( ftruncate( fileno(fp), 0 ) < 0 ){
		logerr_die( XLOG_INFO, "cannot truncate summary file" );
	}
	for( p = Sum_user; p; p = p->chain ){
		(void)fprintf( fp, "%s %s %s %d %s %d\n", p->user, p->host, p->printer,
			p->pages, p->format, p->runs );
	}
}

cmpx( l, r )
	struct entry **l, **r;
{
	int c;
	struct entry *sl, *sr;
	sl = *l;
	sr = *r;
	if( (c = strcmp( sl->user, sr->user )) == 0
	 && (c = strcmp( sl->host, sr->host )) == 0 ){
		c = strcmp( sl->printer, sr->printer );
	}
	return( c );
}
Sort_file(st)
	struct entry **st;
{
	int i, n;
	struct entry *p, **q;
	
	/* count the number of entries */
	
	for( i = 0, p = *st; p; p = p->chain, ++i );
	n = i;
	if( n > 0 ){
		q = (struct entry **)malloc( (unsigned)n*sizeof(struct entry *));
		if( q == 0 ){
			logerr_die( XLOG_INFO, "malloc failed" );
		}
		for( i = 0, p = *st; i < n; p = p->chain, ++i ){
			q[i] = p;
		}
		qsort( (char *)q, i, sizeof(struct entry *), cmpx );
		for( i = 0; i < n-1;  ++i ){
			q[i]->chain = q[i+1];
		}
		q[i]->chain = 0;
	}
	
}

Dump_info( p, header )
	struct entry *p;
	char *header;
{
	(void)fprintf( stdout, "%s\n", header );
	for( ; p; p = p->chain ){
		(void)fprintf( stdout, "%s %s %s %d %s %d\n", p->user, p->host, p->printer,
			p->pages, p->format, p->runs );
	}
}
