/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: lpr_job.c
 * Submit job in the spool directory
 ***************************************************************************
 * Revision History: Created Sat Jan 30 18:11:41 CST 1988
 * $Log:	lpr_job.c,v $
 * Revision 3.2  88/06/24  17:55:45  papowell
 * MODS for VAX 4.3BSD UNIX
 * 
 * Revision 3.1  88/06/18  09:34:57  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.1  88/05/09  10:09:19  papowell
 * PLP: Released Version
 * 
 * Revision 1.6  88/05/06  07:08:11  papowell
 * Modified getwd() call so that a setreuid() is done first.  This way there
 * are no problems with root perms on different systems.
 * 
 * Revision 1.5  88/05/05  20:07:24  papowell
 * Minor changes to shut up lint.
 * 
 * Revision 1.4  88/04/06  12:08:06  papowell
 * Minor change, the 'A'printer control line not put in control file.
 * This allows the existing version to send control files to older versions.
 * 
 * Revision 1.3  88/03/25  15:00:35  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.2  88/03/11  19:28:08  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.1  88/03/01  11:08:49  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: lpr_job.c,v 3.2 88/06/24 17:55:45 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lpr.h"

char *Make_link();
char *Full_path();
extern char *getwd();


/***************************************************************************
 * Make_job()
 * 1. Get a control file (actually a temp file)
 * 2. Write the header information into the file
 * 3. If you have a filter output, queue that
 * 4. otherwise if you have a simple file, queue that
 * 5. otherwise copy all of the files 
 ***************************************************************************/

Make_job()
{
	char *cf;			/* control file name */
	FILE *cfp;			/* control file fp */
	int fd;				/* data file fd */
	int i, c;			/* ACME Integers, Inc. */
	long j, jobsize;	/* jobsize is a long */
	char *s;			/* ACME Pointers, Inc. */
	struct stat f_stat, c_stat;	/* source and copy files status */

	/*
	 * get the control file
	 */
	jobsize = 0;
	cf = Get_cf();
	i = Open_SD( cf );
	if( (cfp = fdopen( i, "w")) == NULL ){
		logerr_die( XLOG_INFO, "Make_job: IMPOSSIBLE- fdopen failed %s",cf);
	}
	/*
	 * place in the time
	 */
	(void)sprintf( WHENSP, "%lu", (unsigned long)time( (time_t *)0 ));
	/*
	 * Put entries in for each character in the Parms list
	 */
	for( i = 'A'; i <= 'Z'; ++i ){
		s = CFparm[i-'A'];
		if( s[0] ){
			Entry( i, s, cfp );
		}
	}
	/* set the job name to the first file if not specified */
	if( Parmcount == 0 ){
		Entry( 'N', "(stdin)", cfp );
		if( JOBNAME[0] == 0 ){
			Entry( 'J', "(stdin)", cfp );
		}
	} else {
		for( i = 0; i < Parmcount; ++i ){
			if( Parms[i].str ){
				Entry( 'N', Parms[i].str, cfp );
				if( JOBNAME[0] == 0 ){
					Entry( 'J', Parms[i].str, cfp );
					JOBNAME[0] = 'X';
				}
			}
		}
	}
	/* set the class name to the priority if not specified */
	if( CLASSNAME[0] == 0 ){
		CLASSNAME[0] = Priority;
		CLASSNAME[1] = 0;
		Entry( 'C', CLASSNAME, cfp );
	}
	/*
	 * put out the created files or copy the ones passed
	 * First find if there are any.
	 */
	s = Filter_out? Filter_out : Read_stdin;
	if( s ){
		Get_stat( SD, s, &f_stat );
		j = (f_stat.st_size + 1023) / 1024;
		for( c = 0; c == 0 || c < Copies; ++c ){
			Entry( Format, s, cfp );
			jobsize += j;
			if( MX && jobsize > MX ){
				Diemsg( "job too large. split up" );
			}
			Entry( 'U', s, cfp );
		}
	} else {
		/*
		 * we create an entry for each file in the list
		 */
		for( i = 0; i < Parmcount; ++i ){
			if( Parms[i].str ){
				/*
				 * repeat the check for printability
				 */
				if( (fd = Is_printable( Parms[i].str, &f_stat )) < 0 ){
					fatal(XLOG_CRIT, "Make_job: file %s become bad, %s@%s",
						Parms[i].str, Person, Host );
				}
				if( Use_links ){
					/*
				 	 * try to make a symbolic link to the file
					 */
					s = Make_link(Parms[i].str, &c_stat );
				} else {
					/*
				 	 * make a copy in the spool directory
					 */
					s = Copy_file( fd, Parms[i].str, &c_stat);
				}
				(void)strcpy(Parms[i].filename, s);
				(void)close(fd);
				if( f_stat.st_size != c_stat.st_size ){
					fatal(XLOG_CRIT,"Make_job: file %s changed, %s@%s",
						Parms[i].str, Person, Host );
				}
				Parms[i].size = (f_stat.st_size + 1023)/1024;
				if(Debug>4)log(XLOG_DEBUG,"Make_job: %s -> %s, %d",
					Parms[i].str, Parms[i].filename, Parms[i].size );
			}
		}
		for( c = 0; c == 0 || c < Copies; ++c ){
			for( i = 0; i < Parmcount; ++i ){
				if( Parms[i].str ){
					Entry( Format, Parms[i].filename, cfp );
					jobsize += Parms[i].size;
					if( MX && jobsize > MX ){
						Diemsg( "job too large (over %ldK). split up", jobsize);
					}
				}
			}
		}
		/*
		 * Remove the file after we have used it
		 */
		for( i = 0; i < Parmcount; ++i ){
			if( Parms[i].str ){
				Entry( 'U', Parms[i].filename, cfp );
				if( Remove && Use_links == 0 ){
					/* set the EUID to user */
					Set_uid( getuid() );
					if( unlink( Parms[i].str ) < 0 ){
						Warnmsg( "could not remove %s- %s",Parms[i].str,
							Errormsg(errno) );
					}
					Clear_uid();
				}
			}
		}
	}
	/*
	 * flush the control file and rename it
	 */
	if( fflush( cfp ) == EOF ){
		logerr_die(XLOG_INFO, "cannot flush %s", cf );
	}
	if( fclose( cfp ) == EOF ){
		logerr_die(XLOG_INFO, "cannot close %s", cf );
	}
	Rename_cf(cf);
}

/***************************************************************************
 * Entry( char c; char *cp; FILE *fp )
 * Prints a line out in the control file
 ***************************************************************************/
Entry( c, cp, fp )
	int c;
	char *cp;
	FILE *fp;
{
	int d;

	d = 0;
	if(Debug>4)log(XLOG_DEBUG,"Entry: %c '%s'", c, cp );
	if( strlen( cp ) > MAXPARMLEN ){
		d = cp[MAXPARMLEN];
		cp[MAXPARMLEN] = 0;
	}
	if( fprintf( fp, "%c%s\n", c, cp ) == EOF ){
		logerr_die( XLOG_INFO, "error while writing control file" );
	}
	if( d ){
		cp[MAXPARMLEN] = d;
	}
}


/***************************************************************************
 * Make_link( char *target, struct stat *statb )
 * makes a link to the target, and puts out entry
 ***************************************************************************/
static char *
Make_link( target, statb )
	char *target;
	struct stat *statb;
{
#ifdef NOSYMLINK
	fatal( XLOG_INFO, "no symbolic links allowed" );
#else  NOSYMLINK
	char *lf;	/* link file */
	char buf[MAXPATHLEN];
	uid_t ruid, euid;

	euid = geteuid();
	/*
	 * get temp file name
	 */
	lf = Get_tmp_data();
	(void)sprintf(buf,"%s/%s", SD, lf );
	/*
	 * generate absolute path name
	 */
	target = Full_path( target );
	if(Debug>3)log(XLOG_DEBUG,"Make_link: from %s to %s", buf, target );
	Set_daemon();
	if( symlink( target, buf ) < 0 ){
		logerr_die( XLOG_INFO, "Make_link: symbolic link from %s to %s failed",
			buf, target );
	}
	if( stat( target, statb ) < 0 ){
		logerr_die( XLOG_INFO, "Make_link: stat failed '%s'", target );
	}
	Clear_uid();
	return( lf );
#endif NOSYMLINK
}


/***************************************************************************
 * Get_stat( char * dir, char *file , struct stat *statb)
 * stat the file in the specified directory
 ***************************************************************************/

Get_stat(dir, file, statb )
	char *dir;
	char *file;
	struct stat *statb;		/* status buffer */
{
	static char buf[BUFSIZ];	/* for holding the pathname */

	if( dir && *dir ){
		(void)sprintf( buf, "%s/%s", dir, file );
	} else {
		(void)strcpy( buf, file );
	}
	if( stat( buf, statb ) < 0 ){
		logerr_die( XLOG_INFO, "Get_stat: cannot stat %s", buf );
	}
	if(Debug>5)log(XLOG_DEBUG,"Get_stat: %s is %ld", buf, statb->st_size );
}

/***************************************************************************
 * Get_home()
 * get the current directory
 ***************************************************************************/
static char dir[MAXPATHLEN];

char *
Get_home()
{
	/*
	 * get current working directory
	 */
	if( dir[0] == 0 ){
		Set_uid( getuid() );
		if( getwd(dir) == 0 ){
			logerr_die(XLOG_INFO, "Get_home: getwd() failed" );
		}
		Clear_uid();
	}
	return( dir );
}
/***************************************************************************
 * Full_path( char *file )
 * return the full path name
 ***************************************************************************/

static char full[MAXPATHLEN];
char *
Full_path( file )
	char *file;
{
	if( file[0] == '/' ){
		(void)strcpy( full, file );
	} else {
		(void)sprintf( full, "%s/%s", Get_home(), file );
	}
	return( full );
}
