/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: Link_support.c
 ***************************************************************************
 * Revision History:  Created Fri Jan 15 20:13:48 CST 1988
 * $Log:	link_support.c,v $
 * Revision 3.1  88/06/18  09:34:21  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.1  88/05/09  10:08:20  papowell
 * PLP: Released Version
 * 
 * Revision 1.5  88/04/06  12:12:12  papowell
 * Minor updates, changes in error message formats.
 * Elimination of the AF_UNIX connections, use AF_INET only.
 * Better error messages.
 * 
 * Revision 1.4  88/03/25  14:59:39  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.3  88/03/12  10:03:50  papowell
 * *** empty log message ***
 * 
 * Revision 1.2  88/03/11  19:28:26  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.1  88/03/01  11:08:29  papowell
 * Initial revision
 * 
 * Support for the inter-machine communications
 * Link_open(int retry)
 *  opens a link to the remote host
 * Link_close()
 *  closes the link to the remote Printer
 * Link_send( char *l)
 *  sends a line to the remote host
 * Link_line( int retry, char *l)
 *  opens the link (with or without retry)
 *  sends a line to the remote host
 * Link_confirm()
 *  gets a single character back, confirming the action
 * Link_ack( c )
 *  sends a single character back, confirming the action
 * Link_copy( FILE *fp, count)
 *  copies a file to the remote host;
 *  if count bytes not transfered, an error is generated
 * Link_get()
 *  gets and prints all information  on stdout
 * Link_port_num()
 *	gets remote port number for connection
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: link_support.c,v 3.1 88/06/18 09:34:21 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"

static int Link_fd;			/* fd for the socket */

/***************************************************************************
 * Link_open(retry)
 * 1. Set up an inet socket;  a socket has a local host/local port and
 *    remote host/remote port address part.  The LPR software runs SUID
 *    root,  and will attempt to open a privileged port (number less than
 *    PRIV);  at the remote end this is checked to make sure that the
 *    remote machine is running SUID root.  Primitive,  but it is adequate
 *    in a trusting environment.
 *
 * (See Advanced 4.3 BSD IPC Tutorial for more information) The server on
 * the remote machine will be listening on the port given by the entry in
 * the /etc/service for the particular service desired.  If we are using
 * the "real" version of lpd this will be the "Printer" service, the test
 * version will use a "test" service.  The LPD daemon will open a socket
 * and bind to the appropriate "port" number.  The
 * getservbyname("printer","tcp"), is used to get the port information
 * (sp->s_port), which is used in a bind call.
 * 
 * When we want to communicate with the lpd daemon on the remote machine,
 * listening on that particular port, we call getseverbyname(...) to get
 * the port number and protocol.  The remote host expects the local port
 * to be in a range that is available only to a root UID process, i.e.-
 * less than IPPORT_RESERVED.  When we open the local port, we get a local
 * port in this range.  At the remote end, the port number is checked to
 * ensure that it is in a valid range.  Since the reserved ports can only
 * be accessed by UID 0 (root) processes, this would appear to prevent
 * ordinary users from directly contacting the remote daemon.
 *
 ***************************************************************************/
/*
 * getport()
 *   gets a port,  in the correct range,  to the remote machine
 *   This code comes from a description of the RLOGIN and LPD
 *   materials
 */
static int
getport()
{
	struct hostent *host;			/* host entry pointer */
	int port_num;					/* port number to connect to */
	struct sockaddr_in sin;			/* inet socket address */
	int sock;						/* socket */

	/*
	 * !!! Warning: zero out the sockaddr_in struct
	 */
	bzero( (char *)&sin, sizeof( sin ));
	/*
	 * Get the host address and port number to connect to.
	 */
	if(Debug>4)log( XLOG_DEBUG, "getport: host %s", RM );
	host = gethostbyname(RM);
	if (host == NULL){
		logerr_die( XLOG_INFO,"getport: unknown host %s", RM);
	}
	/*
	 * get the server name and protocol information from /etc/services
	 */
	port_num = Link_port_num();
	/*
	 * set up the address information
	 */
	bzero((char *)&sin, sizeof(sin));
	bcopy(host->h_addr, (caddr_t)&sin.sin_addr, host->h_length);
	sin.sin_family = host->h_addrtype;
	sin.sin_port = port_num;

	/*
	 * Try connecting to the server.
	 */
	if(Debug>3)log( XLOG_DEBUG, "trying connection to %s", RM );
	sock = reserveport(Maxportno,Minportno);
	if(sock < 0){
		return(-1);
	}
	if(connect(sock, (struct sockaddr *)&sin, sizeof(sin)) < 0){
		if(Debug>2)logerr( XLOG_DEBUG, "connect failed to %s", RM );
		(void) close(sock);
		return(-1);
	}
	return(sock);
}

/*
 * reserveport(int port_no, min_port_no)
 * Reserve a port, starting at port_no, down to min_port_no.
 * while port_no > min_port_no
 *   try to grab the port;
 *   if you can, then return socket
 *   else return -1
 * Returns: socket if successful, -1 if not
 */
static int
reserveport(port_no, min_port_no)
	int port_no, min_port_no;
{
	struct sockaddr_in sin;
	int sock;

	sock = socket(AF_INET, SOCK_STREAM, 0);
	if(sock < 0){
		logerr_die( XLOG_INFO, "reserveport socket call failed" );
	} else if( sock == 0 ){
		logerr_die( XLOG_INFO, "reserveport: socket returns 0" );
	}
	while( port_no >= min_port_no ){
		bzero( (char *)&sin, sizeof( sin ));
		sin.sin_family = AF_INET;
		sin.sin_addr.s_addr = 0;
		sin.sin_port = htons((u_short) port_no);
		if (bind(sock, (struct sockaddr *)&sin, sizeof(sin)) >= 0){
			if(Debug>4)log( XLOG_DEBUG, "reserveport got socket %d", port_no );
			return(sock);
		}
		if(Debug>4)logerr( XLOG_DEBUG,"reserveport bind failed on %d",port_no );
		if (errno != EADDRINUSE && errno != EADDRNOTAVAIL){
			logerr_die( XLOG_INFO, "reserveport: bind failed" );
		}
		--port_no;
	}
	(void)close(sock);
	return(-1);
}

Link_open(retry)
	int retry;
{
	unsigned int i = 1;
	int l;

	if( Link_fd ){
		return( JSUCC );
	}
	while( Link_fd == 0 ){
		if(Debug>3)log( XLOG_DEBUG,"Link_open: retry %d, attempt %d",retry,i );
		l = getport();
		if( l < 0 ){
			if( retry == 0 ){
				if(Debug>3)log( XLOG_DEBUG,"Link_open: failed" );
				return( JFAIL );
			} else if( retry > 0 ){
				--retry;
			}
			sleep((unsigned)i);
			if( i < 512 ){
				i = i << 1;
			}
		} else if( l == 0 ){
			fatal( XLOG_INFO, "Link_open: cannot happen- fd 0" );
		} else {
			Link_fd = l;
		}
	}
	if(Debug>3)log( XLOG_DEBUG, "made connection to %s", RM );
	return( JSUCC );
}

/***************************************************************************
 * Link_close()
 * 1. close the link
 ***************************************************************************/
void
Link_close()
{
	if(Debug>4)log(XLOG_DEBUG,"Link_close: closing link (%d)", Link_fd);
	if( Link_fd ){
		(void)close( Link_fd );
	}
	Link_fd = 0;
}


/***************************************************************************
 * Link_line( int retry; char *line )
 * send a line to the remote end
 * if retry != 0, blocks until connection made
 ***************************************************************************/

int
Link_line( retry, str )
	char *str;
{
	if( Link_open(retry) != JSUCC ){
		return( JFAIL );
	}
	return( Link_send( str ));
}

int
Link_send( str )
	char *str;
{
	int i, l;		/* ACME Integers, Inc. */

	l = strlen( str );
	if( Link_fd == 0 ){
		log(XLOG_INFO, "Link_send: link to %s not open", RM);
		return( JFAIL );
	}
	if( l != 0 && (i = write( Link_fd, str, l )) != l){
		if( i < 0 ){
			logerr( XLOG_INFO, "write error to remote site %s", RM);
		}
		Link_close();
		return( JFAIL );
	}
	if(Debug>4)log(XLOG_DEBUG,"Link_send: sent to %s- '%d'%s",RM,*str,str+1);
	return( JSUCC );
}

/***************************************************************************
 * Link_confirm()
 *  gets a single character back, confirming the action
 ***************************************************************************/
Link_confirm()
{
	char buf[1];	/* buffer */
	int n;

	if( Link_fd == 0 ){
		log( XLOG_INFO, "link to %s not open", RM);
		return( JFAIL );
	}
	if( (n = read( Link_fd, buf, 1 )) != 1 ){
		if( n == 0 ){
			log( XLOG_INFO, "Link_confirm: nothing from remote site %s", RM);
		} else {
			logerr( XLOG_INFO, "Link_confirm: error from remote site %s", RM);
		}
		Link_close();
		return( JFAIL );
	}
	if( buf[0] != 0 ){
		if(Debug>4)log(XLOG_DEBUG,"Link_confim: failed (%d) from %s",*buf,RM);
		Link_close();
		return( JFAIL );
	}
	if(Debug>4)log( XLOG_DEBUG, "successful confirm from %s", RM);
	return( JSUCC );
}

/***************************************************************************
 * Link_copy( FILE *fp; long count; char *name)
 *  copies a file to the remote nost;
 *  if count bytes not transfered, an error is generated
 ***************************************************************************/
Link_copy( fp, count, name )
	FILE *fp;
	long count;
	char *name;
{
	char buf[BUFSIZ];	/* buffer */
	int i, l;				/* ACME Integer, Inc. */
	long total;			/* total number of bytes */

	if(Debug>4)log( XLOG_DEBUG, "Link_copy: %s- %d bytes to %s",name,count,RM);
	if( Link_fd == 0 ){
		log( XLOG_INFO, "Link_copy: Link_fd is not open");
		goto error;
	}

	total = 0;
	while( (i = fread( buf, 1, sizeof(buf), fp )) > 0 ){
		total = total + i;
		if( total > count ){
			log(XLOG_DEBUG,
			"Link_copy: file '%s', length %d instead of %d bytes",
				name, total, count );
			goto error;
		}
		if( (l = write( Link_fd, buf, i )) != i ){
			if( l < 0 ){
			logerr(XLOG_INFO,"Link_copy: write error while sending '%s' to %s",
				name, RM);
			} else {
			logerr(XLOG_INFO,"Link_copy: partial write sending '%s' to %s",
				name, RM);
			}
			goto error;
		}
	}
	if( i < 0 ){
		logerr(XLOG_INFO,"Link_copy: file '%s' read error", name);
		goto error;
	} else if( total != count ){
		log(XLOG_DEBUG,
		"Link_copy: file '%s', copied %d instead of %d bytes to %s",
			name, total, count, RM);
		goto error;
	}
	if(Debug>4)log(XLOG_DEBUG,"Link_copy: file '%s' %d bytes to %s",
		name,count,RM);
	return( JSUCC );
error:
	Link_close();
	return( JFAIL );
}

Link_ack( c )
	int c;
{
	char buf[1];
	int succ;

	buf[0] = c;
	succ = JFAIL;

	if( write( Link_fd, buf, 1 ) != 1 ){
		if(Debug>4)logerr(XLOG_DEBUG,"ack '%d' write error to %s",c,RM);
	} else {
		succ = JSUCC;
		if(Debug>4)log(XLOG_DEBUG,"ack '%d' sent to site %s",c,RM);
	}
	if( succ != JSUCC ){
		Link_close();
	}
	return( succ );
}

/***************************************************************************
 * Link_get()
 * reads all information from the link, and prints on stdout
 ***************************************************************************/
Link_get()
{
	int i;				/* ACME Integers, Inc. */
	char buf[BUFSIZ];	/* buffer */

	if( Link_fd == 0 ){
		fatal( XLOG_INFO, "Link_copy: Link_fd is not open");
	}

	while( (i = read( Link_fd, buf, sizeof(buf)) ) > 0 ){
		(void)fwrite( buf, 1, i, stdout );
	}
}

/***************************************************************************
 * Link_port_num()
 * - look up the service in the service directory using getservent
 * - if the port number has been set, don't do it a second time.
 * - Note that Setup_test will set the port number if necessary.
 ***************************************************************************/

Link_port_num()
{
	static char *name = SERVERNAME;
	static char *prot = SERVERPROT;
	struct servent *sp;
	if( Lpr_port_num == 0 ){
		if( ( sp = getservbyname( name, prot )) == 0 ){
			logerr_die( XLOG_CRIT, "Get_port_nun: getservbyname(%s,%s) failed",
				name, prot );
		}
		Lpr_port_num = sp->s_port;
	}
	return( Lpr_port_num );
}
