/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: rmjob.c
 * Remove jobs from a printer queue
 ***************************************************************************
 * Revision History: Created Sat Jan  9 20:08:26 CST 1988
 * $Log:	rmjob.c,v $
 * Revision 3.1  88/06/18  09:35:34  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.2  88/05/14  10:21:11  papowell
 * Modified -X flag handling
 * 
 * Revision 2.1  88/05/09  10:10:09  papowell
 * PLP: Released Version
 * 
 * Revision 1.4  88/03/25  15:01:32  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.3  88/03/11  19:29:16  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.2  88/03/05  15:01:42  papowell
 * Minor Corrections,  Lint Problems
 * 
 * Revision 1.1  88/03/01  11:09:11  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: rmjob.c,v 3.1 88/06/18 09:35:34 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"

/***************************************************************************
 * rmjob()
 *  1. get the printcap entries
 *  2. get the queue entries
 *  3. get the active server files and see if they are in queue
 *  4. scan the queue, checking each job for removal
 *
 *  user   local  job   from  remove_all    perms inlist    remove?
 *  ===============================================================
 *  root   yes    *     *     yes           *     *         yes
 *  root   yes    *     *     no            *     yes       yes
 *  root   no     *     mach  yes           *     *         yes
 *  root   no     *     mach  no            *     yes       yes
 *  user   -      user  mach  -             R     yes       yes
 *  user   -      *     *     -             C     yes       yes
 *
 *  5. remove the job;  this may necessitate stopping or killing a deamon
 ***************************************************************************/
static int remove_all;	/* remove all jobs */
static int control_perms;	/* has C perms */

rmjob()
{
	int i;				/* ACME Integers, Inc. */
	struct queue *q;	/* job entry */
	int perms;			/* hold perms values */

	/*
	 * get the printcap entry
	 */
	if(Get_pc_entry(Printer, Status_pc_vars, Status_pc_len) == 0){
		(void)fprintf(stdout, "Printer %s does not exist\n", Printer );
		(void)fflush(stdout);
		return;
	}
	if( SD == 0 || *SD == 0 ){
		if(Debug>2)log(XLOG_DEBUG,"not a Printer");
		return;
	}
	/* chdir to spool directory */
	if (chdir(SD) < 0) {
		logerr_die( XLOG_NOTICE,"cannot chdir to %s", SD);
	}
	/*
	 * set the flags needed
	 */
	Is_local = strcmp( From, Host ) == 0;
	Is_root  = strcmp( Person, "root" );
	remove_all = (Parmcount > 0 && (strcmp( Parms[0].str, "-all" ) == 0));
	/*
	 * check to see that the user has RMJOB privs on this machine
	 */
	perms = 'R';	/* must be able to at least use the Printer */
	if( !Is_root && (
		(Permfile && *Permfile
			&& !Checkperm( Permfile,From,Person,First_name,&perms,(int *)0,0 ))
		|| (XU && *XU
			&& !Checkperm( XU,From,Person,First_name,&perms,(int *)0,0 )))) {
		(void)fprintf(stdout, "No remove permission on %s", First_name);
		return;
	}
	perms = 'C';	/* check for control perms */
	control_perms = 0;
	if( !Is_root && (
		(Permfile && *Permfile
			&& Checkperm( Permfile,From,Person,First_name,&perms,(int *)0,0 ))
		|| (XU && *XU
			&& Checkperm( XU,From,Person,First_name,&perms,(int *)0,0 )))) {
		control_perms = 1;
	}
	if(Debug>4)log(XLOG_DEBUG,
		"rmjob: Is_root %d, Is_local %d, remove_all %d, control_perms %d",
		Is_root, Is_local, remove_all, control_perms );
	/*
	 * check for remote machine and networked file system
	 */
	if( RM && NW ){
		Remote_remove();
		return;
	}
	/*
	 * get the job queue
	 */
	Jobcount = Getq();
	(void)Checkactive();
	/*
	 * run down list
	 */
	(void)fprintf(stdout,"Printer '%s' (%s):\n", Printer, Host );
	(void)fflush(stdout);
	for( i = 0; i < Jobcount; ++i ){
		q = &Queue[i];
		if( shouldremove( q ) ){
			(void)fprintf(stdout,"removing %s, job %d owner %s\n",
				q->q_name, q->q_num, q->q_user);
			(void)fflush(stdout);
			doremove(q);
		}
	}
	/*
	 * check for remote machine
	 */
	if( RM ){
		Remote_remove();
	}
	/*
	 * give the server a kick
	 */
	(void)Startserver();
}

/***********************************************************************
 * shouldremove( q )
 *  a simple application of the removal decision table
 *
 *  user   local  job   from  remove_all    perms inlist    remove?
 *  ===============================================================
 *  root   yes    *     *     yes           *     *         yes
 *  root   yes    *     *     no            *     yes       yes
 *  root   no     *     mach  yes           *     *         yes
 *  root   no     *     mach  no            *     yes       yes
 *  user   -      user  mach  -             R     yes       yes
 *  user   -      *     *     -             C     yes       yes
 *
 * Returns: 1 if removal is indicated, 0 otherwise
 ***********************************************************************/
int
shouldremove( q )
	struct queue *q;
{
	int i, same_host;

	same_host = (strcmp( From, &q->q_from ) == 0);
	i = Match_entry(q);
	if( Is_root && Is_local && remove_all ) return(1);
	if( Is_root && Is_local && !remove_all && i ) return(1);
	if( Is_root && !Is_local && same_host && remove_all ) return(1);
	if( Is_root && !Is_local && same_host && !remove_all && i ) return(1);
	if( strcmp( Person, q->q_user) == 0 && same_host && i ) return(1);
	if( i && control_perms ) return(1);
	return( 0 );
}

/***************************************************************************
 * doremove(struct queue *q)
 * remove the job
 * 1. Lock the control file.
 * 2. If unsuccessful, find the server PID and kill it off.
 * 3. Use brute force and remove the files.
 ***************************************************************************/

doremove( q )
	struct queue *q;
{
	FILE *cfp;

	if( (cfp = Lockcf( q->q_name )) == NULL ){
		/* hmmm... looks like an active server */
		if( (cfp = fopen_daemon( q->q_name, "r" )) == NULL ){
			/* nope, the file has really gone */
			logerr(XLOG_INFO,"control file %s not readable", q->q_name );
			return;
		}
		/* well, we will just have to kill of the server */
		if( q->q_daemon == 0){
			/*
			 * Hmmm... we have this fellow running the file, and it is
			 * locked.  That means that it just started running this
			 * guy. Better check again.
			 */
			(void)Checkactive();
		}
		if( q->q_daemon ){
			(void)fprintf( stdout, "killing off %s server %d",
				q->q_server,q->q_daemon );
			if( killpg( q->q_daemon, SIGINT) < 0 ){
				if(Debug>2)log(XLOG_DEBUG,
						"server %s (%d) was not alive",
							q->q_server,q->q_daemon );
			}
		}
	}
	/* use brute force;  we simply remove files */
	if(Debug>3)log(XLOG_DEBUG,"removing files for job %s",q->q_name);
	Remove_job( cfp, q );
	(void)fclose( cfp );
}
