/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: lpr_filters.c
 * filter programs for LPR
 * Supports the reading and enqueuing of input
 ***************************************************************************
 * Revision History: Created Sat Jan 30 08:06:50 CST 1988
 * $Log:	lpr_filters.c,v $
 * Revision 3.1  88/06/18  09:34:53  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.1  88/05/09  10:09:11  papowell
 * PLP: Released Version
 * 
 * Revision 1.3  88/03/25  15:00:27  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.2  88/03/11  19:28:36  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.1  88/03/01  11:08:45  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: lpr_filters.c,v 3.1 88/06/18 09:34:53 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint
#include "lpr.h"

/***************************************************************************
 * Run_pr()
 *   Copies input the PR program; temp file name left in Read_stdin
 *   Returns: 0 if nothing, 1 otherwise
 * Copy_stdin()
 *   Copies input to a temporary file; temp file name left in Read_stdin
 *   Returns: 0 if nothing, 1 otherwise
 * char *Copy_file( int infile, char *name ) 
 *    copies file to temp file, returns name of temp file
 * int Open_SD(char *s) Open the file in the spool directory
 ***************************************************************************/

/***************************************************************************
 * Run_pr()
 * 1. Gets the name of a temporary file for the output
 * 2. Sets up the PR command
 * 3. Forks a process
 * 4. Arranges the process output goes to the output file
 * 5. Waits for PR to complete
 ***************************************************************************/
char * estr3();		/* copies 3 strings to end */

Run_pr()
{
	int fd;				/* Data file */
	char cmdbuf[BUFSIZ];	/* build up the command here */
	char buf[BUFSIZ];		/* local buf up the command here */
	char *bp;				/*  bp points to cmdbuf */
	char *ep;				/*  ep is limit */
	int pid;				/* pid of pr process */
	union wait status;		/* the status.w_status is the integer value */
	int i;					/* ACME Integer, Inc. */
	struct stat statb;		/* for stating the output file */

	/*
	 * get name of the data file
	 */
	Read_stdin = Get_tmp_data();
	fd = Open_SD( Read_stdin );
	/*
	 * set up the printer name
	 */
	if( PR == 0 || *PR == 0 ){
		Diemsg( "there is no pr program specified for -p format");
	}
	/* set up the command */
	ep = cmdbuf + sizeof(cmdbuf);
	bp = estr3( cmdbuf, PR,(char *)0,(char *)0, ep );
	/*
	 * set the width, length, and title flags
	 */

	if( PWIDTH[0] ){
		bp = estr3( bp, " -w", PWIDTH, " ", ep );
	} else if( PW ){
		(void)sprintf( buf, " -w%d ", PW );
		bp = estrcp( bp, buf, ep );
	}
	if( PL ){
		(void)sprintf( buf, "-l%d ", PL );
		bp = estrcp( bp, buf, ep );
	}
	if( PRTITLE[0] ){
		bp = estr3( bp, "-h '", PRTITLE, "' ", ep );
	} else if( JOBNAME[0] ){
		bp = estr3( bp, "-h '", JOBNAME, "' ", ep );
	} else if( Parmcount == 0 ){
		bp = estr3( bp, "-h '", "(stdin)", "' ", ep );
	}
	/*
	 * Put the file names in the list
	 */
	for( i = 0; i < Parmcount; ++i ){
		bp = estr3( bp, Parms[i].str, " ", (char *)0, ep );
	}
	if( bp == 0 ){
		Diemsg( "pr command is too long: %s", cmdbuf);
	}
	if(Debug>2)log(XLOG_DEBUG, "pr command is '%s'", cmdbuf);
	/*
	 * start up the pr process
	 */
	if( (pid = fork()) < 0 ){
		logerr_die(XLOG_INFO,"Run_pr: fork failed");
	} else if( pid == 0 ){
		if( dup2(fd,1) < 0 ){
			logerr_die(XLOG_INFO,"Run_pr: dup2 failed" );
		}
		if( setreuid( getuid(), getuid() )< 0 ){
			logerr_die(XLOG_INFO,"setreuid failed" );
		}
		mexecv( cmdbuf );
		exit(1);
	}
	/* wait for printer */
	while ((i = wait(&status)) > 0 && i != pid);
	if( i < 0 || status.w_status ){
		logerr_die(XLOG_INFO,"pr failed (%s)", Decode_status(&status) );
	}
	if(Debug>3)log(XLOG_DEBUG,"pr done successfully");
	if( fstat( fd, &statb ) <0 ){
		logerr_die(XLOG_INFO, "Run_pr: cannot stat output file %s", Read_stdin);
	}
	if(Debug>3)log(XLOG_DEBUG,"Run_pr: %s is %d",Read_stdin,statb.st_size);
	(void)close(fd);
	return( statb.st_size != 0 );
}

static char *
estr3( s, s1, s2, s3, e )
	char *s, *s1, *s2, *s3, *e;
{
	if( s1 ) s = estrcp( s, s1, e );
	if( s2 ) s = estrcp( s, s2, e );
	if( s3 ) s = estrcp( s, s3, e );
	return( s );
}

/***************************************************************************
 * int Open_SD(char *s)
 * Open the name of the file in the spool directory
 * 1. Prepend the SD name
 * 2. Create the file with the desired perms
 * Returns: fd if successful; dies otherwise
 ***************************************************************************/

Open_SD( s )
	char *s;
{
	char buf[BUFSIZ];
	int fd;

	(void)sprintf(buf, "%s/%s", SD, s);
	fd = Exlockcf( buf );
	if( fd < 0 ){
		logerr_die( XLOG_INFO, "Open_SD: could not open and lock %s", buf );
	}
	if(Debug>3)log(XLOG_DEBUG,"Open_SD: file %s, fd %d", buf, fd );
	return( fd );
}

/***************************************************************************
 * Copy_stdin()
 * 1. Gets the name of a temporary file for the output
 * 2. Copies stdin to the file
 * 3. Returns 0 if empty file, 1 otherwise
 ***************************************************************************/

Copy_stdin()
{
	int fd;					/* Data file */
	char buf[BUFSIZ];		/* local buf up the command here */
	int n, i;				/* bytes read */
	int count;				/* number of bytes */
	long blocks;				/* number of blocks */

	/*
	 * get name of the data file
	 */
	Read_stdin = Get_tmp_data();
	fd = Open_SD( Read_stdin );

	count = 0;
	blocks = 0;
	while( (n = fread( buf, 1, sizeof(buf), stdin )) > 0 ){
		count += n;	/* update count */
		while( count >= 1024 ){
			count -= 1024;
			++blocks;
		}
		if( MX && blocks > MX ){
			Diemsg( "input from stdin exceeds maximum file size limits" );
		}
		i = write( fd, buf, n );
		if( i != n ){
			logerr_die(XLOG_INFO,"Copy_stdin: cannot write %s",Read_stdin);
		}
	}
	if( ferror( stdin ) ){
		Diemsg( "error reading from stdin" );
	}
	(void)close(fd);
	if( count ){
		++blocks;
	}
	if(Debug>3)log(XLOG_DEBUG,"Copy_stdin: %s is %d blocks", Read_stdin,blocks);
	return( blocks != 0 );
}
/***************************************************************************
 * char *Copy_file( int *in_fp, char *in_file )
 * 1. Gets the name of a temporary file for the output
 * 2. Copies file to the temporary file
 * Returns: temporary file name
 ***************************************************************************/

char *
Copy_file( in_fd, in_file, statb )
	int in_fd;
	char *in_file;
	struct stat *statb;
{
	int out_fd;				/* Data file */
	char buf[BUFSIZ];		/* local buf for IO */
	int n, i;				/* bytes read */
	int count;				/* number of bytes */
	long blocks;			/* number of blocks */
	char *fname;			/* file name */

	/*
	 * get name of the data file
	 */
	fname = Get_tmp_data();
	out_fd = Open_SD( fname );

	count = 0;
	blocks = 0;
	while( (n = read( in_fd, buf, sizeof(buf))) > 0 ){
		count += n;	/* update count */
		while( count >= 1024 ){
			count -= 1024;
			++blocks;
		}
		if( MX && blocks > MX ){
			Diemsg( "file %s too large",in_file);
		}
		i = write( out_fd, buf, n );
		if( i != n ){
			logerr_die(XLOG_INFO,"Copy_file: cannot write to file %s",in_file);
		}
	}
	if( n < 0 ){
		logerr_die( XLOG_INFO,"Copy_file: error reading from %s", in_file );
	}
	if( fstat( out_fd, statb ) < 0 ){
		logerr_die( XLOG_INFO,"Copy_file: cannot stat %s", fname);
	}
	(void)close( out_fd );
	return( fname );
}
/***************************************************************************
 * Do_prefilter( char *cmd, char *file )
 *
 *	Prefilter handler
 *  1. Sets up a prefilter command.
 *  filtername arguments \   <- from filtername
 *      -PPrinter -wwidth -llength -xwidth -ylength [-c] [-iindent] \
 *      [-Zoptions] [-Cclass] [-Jjob] [-Raccntname] -nlogin -hHost
 *      -Fformat [file]
 *     Note: the filter parameter is of the form "filter,X", where
 *     output format is specified.  If there is no specified format,
 *     'f' will be used.
 *  2. Gets a temporary file for output
 *  3. If file is explicitly named,  uses that for input instead of
 *     parameter files.
 *  4. Forks and runs the command.
 *  5. Filter must return 0 for successful status, otherwise we abort
 ***************************************************************************/

Do_prefilter( cmd, file )
	char *cmd;
	char *file;
{
	char cmdbuf[BUFSIZ];	/* build up the command here */
	union wait status;		/* the status.w_status is the integer value */
	char *cp, *ep;			/* ACME Pointers, Inc. */
	int i, l, pid;			/* ACME Integers, Inc. */
	int new_format = 'f';	/* new format */
	int fd;					/* the output file */
	struct stat statb;		/* for stating the output file */
	
	/*
	 * get the output format
	 */
	(void)strcpy(cmdbuf, cmd);
	l = strlen( cmdbuf );
	if( l > 2 ){
		if( cmdbuf[l-2] == ',' ){
			new_format = cmdbuf[l-1];
			cmdbuf[l-2] = 0;
		}
	}
	if( !isascii(new_format) || !islower(new_format)){
		fatal(XLOG_INFO, "bad prefilter new format, %s", cmd);
	}
	/*
	 * set up the basic filter command 
	 */
	AF = 0;
	cp = Setup_filter( Format, cmdbuf );
	Format = new_format;
	ep = cmdbuf+sizeof(cmdbuf);
	/*
	 * copy command to buffer
	 */
	cp = estrcp( cmdbuf, cp, ep );

	/*
	 * add the file names
	 */
	if( file ){
		/*
		 * we have an explicitly named file
		 */
		cp = estrcp( cp, " ", ep );
		cp = estrcp( cp, file, ep );
	} else {
		for( l = 0; l < Parmcount; ++l ){
			cp = estrcp( cp, " ", ep );
			cp = estrcp( cp, Parms[l].str, ep );
		}
	}
	if( cp == 0 ){
		fatal( XLOG_INFO, "Do_prefilter: command too long: %s", cmd);
	}
	if(Debug>3)log(XLOG_DEBUG, "Do_prefilter: command '%s'", cmdbuf);
	/*
	 * get the output file name
	 */
	Filter_out = Get_tmp_data();
	fd = Open_SD( Filter_out );

	/*
	 * start the prefilter up and wait for output
	 */
	if( (pid = fork()) < 0 ){
		logerr_die(XLOG_INFO,"Do_prefilter: fork failed");
	} else if( pid == 0 ){
		if( dup2(fd,1) < 0 ){
			logerr_die(XLOG_INFO,"Do_prefilter: dup2 failed" );
		}
		if( setreuid( getuid(), Daemon_uid ) < 0 ){
			logerr_die(XLOG_INFO,"Do_prefilter: setreuid failed" );
		}
		mexecv( cmdbuf );
		logerr_die(XLOG_INFO,"Do_prefilter: execv failed");
	}
	/*
	 * wait for prefilter
	 */
	while ((i = wait(&status)) > 0 && i != pid) ;
	if( i < 0 || status.w_status ){
		logerr_die(XLOG_INFO,"Do_prefilter: prefilter %d failed (%s)", i,
			Decode_status( &status ) );
	}
	if(Debug>3)log(XLOG_DEBUG,"Do_prefilter: prefilter successful");
	if( fstat( fd, &statb ) <0 ){
		logerr_die(XLOG_INFO, "Do_prefilter: cannot stat output file %s",
			Filter_out);
	}
	if(Debug>3)log(XLOG_DEBUG,"Do_prefilter: %s is %ld",
			Filter_out,statb.st_size);
	(void)close(fd);
	return( statb.st_size != 0 );
}
