/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: lpd.c
 * lpd main program
 ***************************************************************************
 * Revision History: Created Fri Jan  1 16:19:30 CST 1988
 * $Log:	lpd.c,v $
 * Revision 3.1  88/07/25  19:51:03  papowell
 * Distribution version
 * 
 * 
 * Revision 3.1  88/06/18  09:34:40  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.3  88/05/19  10:34:08  papowell
 * Fixed open() calls to have a 0 parameter, ie: open(f, perms, 0), where needed
 * 
 * Revision 2.2  88/05/14  10:18:06  papowell
 * Use long format for job file names;
 * Added 'fd', no forward flag;
 * Control file has to have hostname and origination agree.
 * 
 * Revision 2.1  88/05/09  10:08:46  papowell
 * PLP: Released Version
 * 
 * Revision 1.9  88/04/27  20:26:17  papowell
 * Modified to remove unused variables
 * 
 * Revision 1.8  88/04/26  15:51:31  papowell
 * Added a Reapchild() call in cleanup and each time a server is started
 * This should gather up the orphans that are left in some UNIX implementations
 * 
 * Revision 1.7  88/04/15  13:06:17  papowell
 * Std_environ() call added, to ensure that fd 0 (stdin), 1 (stdout), 2(stderr)
 * have valid file descriptors;  if not open, then /dev/null is used.
 * 
 * Revision 1.6  88/04/07  12:31:11  papowell
 * 
 * Revision 1.5  88/04/06  12:12:53  papowell
 * Minor updates, changes in error message formats.
 * Elimination of the AF_UNIX connections, use AF_INET only.
 * Better error messages.
 * 
 * Revision 1.4  88/03/25  15:00:06  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.3  88/03/11  19:28:58  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.2  88/03/05  15:01:20  papowell
 * Minor Corrections,  Lint Problems
 * 
 * Revision 1.1  88/03/01  11:08:37  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: lpd.c,v 3.1 88/07/25 19:51:03 papowell Locked $ PLP Copyright 1988 Patrick Powell";
#endif lint
/***************************************************************************
 * lpd -- line Printer daemon.
 * 1. Get options 
 * 2. Check for existence of another daemon and exit if it exists.
 *    Note: this is done by locking a file
 * 3. Initialize error logging
 * 4. Start up all the queue handlers
 * 5. Set up communication sockets
 * 6. Loop forever doing:
 *     Listen for a connection and perform the requested operation.
 *     Operations are:
 *	    \1Printer\n
 *	    	check the queue for jobs and print any found.
 *	    \2Printer\n
 *	    	receive a job from another machine and queue it.
 *	    \3Printer [users ...] [jobs ...]\n
 *	    	return the current state of the queue (short form).
 *	    \4Printer [users ...] [jobs ...]\n
 *	    	return the current state of the queue (long form).
 *	    \5Printer Person [users ...] [jobs ...]\n
 *	    	remove jobs from the queue.
 *	    \6Printer Person operation
 *	    	enable/disable queueing, etc.
 *    Note: a process is forked for each operation
 *
 * Strategy to maintain protected spooling area:
 *	1. Spooling area is writable only by daemon and daemon group
 *	2. lpr runs setuid ROOT; it uses the user UID
 *	   to access any file it wants (verifying things before
 *	   with access(2)) and sets ownership of files in the spooling
 *     directory to be owner DAEMON, group DAEMON.
 *	3. Critical files in spooling area are owned by DAEMON, group DAEMON
 *	   with mode 660.
 *	4. lpd, lpq and lprm run setuid ROOT, and change group to DAEMON
 *	   Users can't get to anything w/o help of lpq and lprm programs.
 *	5. LPD forks a server process to service each device
 *	   in the printcap entry.  These processes run setuid root
 *	   and setgrp to their PID.  They open neccessary files, and
 *	   fork "filter" processes to process the output.  Filter
 *	   processes run UID DAEMON.
 *	6. The server process group is set to the main server PID.  This is used
 *	   by the LPRM program, which signals them with killpg(2)
 *
 * Error reporting:
 *	lpd will open a standard logging file. If not already present,
 *  no messages will be logged.  Critial messages will also be logged
 *  using syslog(8).
 *	The individual daemon processes will open log files also.  If
 *	they cannot be opened, then no logging will be done for the process.
 ***************************************************************************/

#include "lp.h"

int	cleanup();		/* file system scullery maid, cleans up files */
static int lpdpid;			/* lpd process id */

main(argc, argv)
	int argc;
	char **argv;
{
	int f;						/* Acme Integer, Inc. strikes again */
	int finet;					/* Internet socket */
	int defreadfds, readfds;	/* select()ed connections */
	int	req;					/* request on this socket */
	int port_num;				/* used to get the inet server information */
	struct sockaddr_in sock_in, frominet; /* networking addresses */
	int fromlen;
	int nfds;					/* accept return value */

	/*
	 * explicitly set umask
	 */
	(void)umask(0);

	/*
	 * Set fd 0, 1, 2 to /dev/null if not open.  This protects against
	 * being started in strange environments.
	 */
	Std_environ();
	/*
	 * we set up the alternate version if XPERIMENT is defined
	 */
#	ifdef XPERIMENT
		Setup_test();
#	endif XPERIMENT

	/*
	 * process command line options
	 */
	Getoptions(argc, argv);

	/*
     * set up the pathnames for information files
	 */
	Tailor_names();
	/*
	 * check to see that the is not a spooler present.  The LO is used
	 * when we exit to clean out the lock file information.
	 * Note that we check first, as the initial process, and then we fork
	 * another process.  This allows the output message to be printed
	 * on the stdout device in a synchronous manner.
	 */
	LO = Masterlock;
	if( (Lfd = Getlockfile(LO, &f, (char *)0, 0, (struct stat *)0)) == NULL ){
		(void)fprintf( stdout, "active LPD %d\n", f );
		exit(0);
	}
	(void)fclose(Lfd);
	/* We can let parent die */
	if ((f = fork()) < 0 ){
		logerr_die( XLOG_INFO, "lpd: fork failed" );
	} else if( f ){
		exit(0);
	}
	/*
	 * We can now start logging
	 */
	Setuplog(Lpdlogf, 0 );
	/*
	 * Note that we do the locking at this point, as the lock may not
	 * continue across a fork.
	 */
	if((Lfd=Getlockfile(LO,&f,(char *)0, 0, (struct stat *)0)) == NULL){
		log(XLOG_INFO, "active LPD %d\n", f );
		exit(0);
	}
	/*
	 * save the PID of the LPD process;  it will be useful later
	 */
	lpdpid = getpid();
	/*
	 * drop the control tty.  This prevents the user which started
	 * LPD from inadvertenly generation a signal and clobbering LPD
	 */
	if( (f = open("/dev/tty", O_RDWR, 0) ) >= 0) {
		(void)ioctl(f, TIOCNOTTY, (struct sgttyb *)0);
		(void)close(f);
	}
	/* set pgrp to pid, so we can kill our kids later */
	(void)setpgrp(0, lpdpid);

	/* put PID and time started in the lockfile */
	Setlockfile(LO, Lfd, lpdpid,Time_str());

	/*
	 * set up signals; note that SIGPIPE may not be needed, but it
	 * never hurts to make sure.
	 */
	(void)signal(SIGCHLD, Reapchild);
	(void)signal(SIGPIPE, SIG_IGN);
	(void)signal(SIGHUP, cleanup);
	(void)signal(SIGINT, cleanup);
	(void)signal(SIGQUIT, cleanup);
	(void)signal(SIGTERM, cleanup);
	/*
	 * Restart all the Printers.
	 */
	Startup();

	/*
	 * Set up the IPC.  This is very ugly,  and is stolen directly from
	 * the original LPD code and the 4.3 BSD Interprocess Communication
	 * Tutorial, which seems to have borrowed it from RLOGIN, etc.
	 *
	 * Set up INET socket
	 */
	if( ( finet = socket(AF_INET, SOCK_STREAM, 0) ) < 0 ){
		logerr_die( XLOG_CRIT,"lpd: cannot create AF_INET socket");
	}
	port_num = Link_port_num();
	/*
	 * zero the sockaddr structure before using it
	 */
	bzero( (char *)&sock_in, sizeof(sock_in) );
	/*
	 * the INADDR_ANY will allow different nets to be used
	 */
	sock_in.sin_family = AF_INET;
	sock_in.sin_addr.s_addr = INADDR_ANY;
	sock_in.sin_port = port_num;
	if (bind(finet, (struct sockaddr *)&sock_in, sizeof(sock_in)) < 0) {
		logerr_die( XLOG_CRIT,"lpd: bind failed internet domain");
	} 
	if( listen(finet, 5) < 0 ){
		logerr_die( XLOG_CRIT, "lpd: listen failed internet domain" );
	}
	if(Debug>2)log( XLOG_DEBUG, "lpd: AF_INET socket %d", finet );

	/*
	 * Set up the sockets on which we will do an accept
	 */
	defreadfds = (1 << finet);

	/*
	 * Main loop: accept, do a request, continue.
	 */
	for (;;) {
		/*
		 * Defensive Coding a la Brain Damaged Unix Implementations
		 * A particular UNIX implementation will not cause SIGCHLD
		 * signals in a particularly reliable way.  Apparently if there
		 * is a SIGCHLD while it is masked off (error printing does this)
		 * IT IS IGNORED!!!  So we put a Reapchild() here;  at least
		 * we will gather them up each time we start a server
		 */
		Reapchild();
		if(Debug>2)log( XLOG_DEBUG,"lpd: starting select");

		readfds = defreadfds;
		nfds = select(20, (fd_set *)&readfds, (fd_set *)0, (fd_set *)0,
			(struct timeval *)0); /* wait patiently */
		if (nfds <= 0) {
			if (nfds < 0 && errno != EINTR) {
				logerr_die( XLOG_CRIT,"lpd: select error");
			}
			continue;
		}
		/*
		 * incredible, but true: could have NO requests
		 */
		if( readfds ){
			if(Debug>2)log( XLOG_DEBUG,"lpd: doing select 0x%x", readfds );
			/* get the socket causing the select */
			fromlen = sizeof(frominet);
			req = accept(finet, (struct sockaddr *)&frominet, &fromlen);
			if (req < 0) {
				logerr( XLOG_CRIT,"lpd: finet accept");
				continue;
			}
			if (!From_host(&frominet)){
				logerr(XLOG_INFO,"lpd: From_host failed");
				continue;
			}
			/*
			 * Fork process to handle activity requested
			 */
			(void) fflush(stdout);
			(void) fflush(stderr);
			if ((f = fork()) == 0) {	/* daughter */
				(void) fclose(Lfd);
				Lfd = NULL;
				if( dup2(req, 1) < 0 ){
					logerr_die( XLOG_CRIT, "lpd: dup2 for server failed" );
				}
				(void) close(req);
				(void) close(finet);
				servicereq();
				exit(0);
			} else if( f < 0 ){
				logerr( XLOG_CRIT, "lpd: fork failed" );
			}
			(void) close(req);
		}
	}
}

/***************************************************************************
 * Getoptions(argv, argc)
 *    char **argv; int argc;
 * Purpose:
 * extracts options and arguments from command line using Getopt(2)
 * Side Effects:
 *	-X : calls setup_test() to modify defaults 
 *  -D nn : sets Debug level
 *  -L file : sets log file
 ***************************************************************************/
Getoptions(argc, argv)
	int argc;
	char **argv;
{
	int c;
	while ((c = Getopt(argc, argv, "XD:L:")) != EOF){
		switch(c){
			case 'X':	/* test version */
#			ifdef DEBUG
				Setup_test();
				Tailor_names();
#			else
				(void)fprintf( stderr, "%s: -X not allowed\n", Name );
				exit(1);
#			endif DEBUG
				break;
			case 'D':		/* turn on Debugging */
				if( Optarg == NULL ){
					exit(1);
				}
				Debug= atoi( Optarg );
				break;
			case 'L':
				if( Optarg == NULL ){
					exit(1);
				}
				(void)strcpy(Lpdlogf, Optarg);
				break;
			default:
				exit(1);
		}
	}
	if( Optind < argc ){
		(void)fprintf( stderr, "%s: extra argument %s\n", Name, argv[Optind] );
		exit(1);
	}
}
/***************************************************************************
 * Setuplog( char *logfile, int saveme )
 * Purpose: to set up a standard error logging environment
 * saveme will prevent stdin from being clobbered
 *	 1.  open /dev/null on fd 0;
 *   2.  if saveme not 1, dup 0 to fd 1
 *   3.  If logfile is "-" or NULL, output file is alread opened
 *   4.  Open logfile; if unable to, then open /dev/null for output
 ***************************************************************************/
Setuplog( logfile, saveme )
	int saveme;
	char *logfile;
{
	int fd;

	/*
	 * we want 0 (stdin) to be /dev/null
	 */
	(void)fflush(stdout);
	(void)fflush(stderr);
	(void)close(0);
    if( (fd = open( "/dev/null", O_RDWR, 0 )) != 0 ){
		logerr_die( XLOG_CRIT, "Setuplog: /dev/null opened as %d", fd);
	}
	/*
	 * do the dup if necessary
	 */
	if( saveme != 1 ){
		(void)close(1);
		if( dup2(0,1) < 0){
			logerr_die( XLOG_CRIT, "Setuplog: dup2 failed" );
		}
	}
	if(Debug>4)log(XLOG_DEBUG,"Setuplog: opening log file %s", logfile );
	/*
	 * open logfile;  if it is "-", use stderr
	 */
	if( logfile && *logfile && strcmp(logfile, "-") ){
		if( (fd = open_daemon(logfile, O_WRONLY|O_APPEND, 0)) < 0 ){
			if(Debug>0)logerr(XLOG_DEBUG,"cannot open logfile %s",logfile);
			/* write to stderr if debugging, /dev/null otherwise */
			if(Debug>0){
				fd = 2;
			} else {
				fd = 0;
			}
		}
		if( fd != 2 && dup2(fd,2) < 0){
			logerr_die( XLOG_CRIT, "Setuplog: dup2 failed" );
		}
		if( fd > 2 ){
			(void)close(fd);
		}
	}
}

/*
 *	When a child dies, it will generate a SIGCHLD,
 *	and Reapchild() will lay the body to rest with a wait3().
 */
Reapchild()
{
	int pid;
	union wait status;

	while ((pid =wait3(&status, WNOHANG, (struct rusage *)0)) > 0){
		if(Debug>3)log(XLOG_DEBUG,"process %d, status %s", pid,
			Decode_status( &status ) );
	}
}

/*
 *	Clean up; kill off children and remove sockets and files
 */
cleanup()
{
	int pid;

	(void)sigblock(sigmask(SIGCHLD)|sigmask(SIGHUP)
		|sigmask(SIGINT)|sigmask(SIGQUIT)|sigmask(SIGTERM));
	if( Lfd ){
		Closelockfile(LO, Lfd);
		Lfd = 0;
	}
	pid = getpid();
	if(Debug>0)log( XLOG_DEBUG, "cleanup" );
	/*
	 * if the recfiles server, clean up files
	 */
	rm_recfiles();
	if( pid == getpgrp(0) ){
		/* Kill children too */
		(void)kill(0, SIGINT);
		(void)kill(0, SIGCONT);
		if( pid == lpdpid){
			log(XLOG_CRIT, "cleanup: lpd process %d terminating", pid);
		}
	}

	/*
	 * Wait for children until the bodies rot
	 */
	Reapchild();
	exit(Errorcode);
}

/***********************************************************************
 * From_host( sockaddr_in *f )
 *   This routine will extract the remote Host Name and remote Host
 *   port number information.  The remote process must connect on
 *   a port in a "reserved" range;  these are reserved for use
 *   by priviledged processes (setuid/uid root).
 * Side Effects: sets the "from" string to the Name of the remote Host.
 * Returns: 1 if privileged and successful, 0 if not.
 * NOTE: this code was derived from the orignal 4.2BSD LPR source,
 *   the 4.3BSD Interprocess Communications Tutorials, and other places.
 ***********************************************************************/

From_host(f)
	struct sockaddr_in *f;
{
	struct hostent *hp;			/* Host entry */
	extern char *inet_ntoa();	/* inet address translation routine */
	int port;					/* from port number */

	port = ntohs(f->sin_port);
	if (f->sin_family != AF_INET ){
		log(XLOG_CRIT,
		"From_host: malformed from address, family %d, not AF_INET %d",
		f->sin_family, AF_INET);
		return(0);
	}
	if ( port > Maxportno){
		log(XLOG_CRIT,"From_host: from port %d, max is %d",port, Maxportno);
		return(0);
	}
	hp = gethostbyaddr((char *)(&f->sin_addr), sizeof(struct in_addr),
				f->sin_family);
	if (hp == 0){
		logerr(XLOG_INFO,"From_host: Host Name for address '%s' unknown",
			inet_ntoa(f->sin_addr));
		return(0);
	}
	if( (From = malloc( (unsigned)strlen(hp->h_name)+1 )) == 0 ){
		logerr_die(XLOG_CRIT,"malloc failed in From_host" );
	}
	(void)strcpy(From, hp->h_name);
	return(1);
}

/************************************************************************
 * Startup()
 * start queue servers
 ************************************************************************/
static
Startup()
{
	int pid;
	char **pr;
	union wait status;

	/*
	 * Restart the server.
	 * Start scanning the printcap for entries
	 */
	if ((pid = fork()) < 0) {
		logerr( XLOG_CRIT,"server for %s: cannot fork", Printer);
	} else if(pid == 0 ) {
		if(Debug>4)log( XLOG_DEBUG, "starting servers" );
		for( pr = All_printers(); *pr; ++pr ){
			Printer = *pr;
			if ((pid = fork()) < 0) {
				logerr( XLOG_CRIT,"server for %s: cannot fork", Printer);
				continue;
			} else if(pid == 0) {
				if(Debug>2)log( XLOG_DEBUG, "started %s server, pid %d",
					Printer,getpid());
				(void)fclose(Lfd);
				Lfd = 0;
				Startprinter();
				exit(0);
			}
			sleep( (unsigned)1 );
		}
		while ((pid=wait(&status)) > 0){
			if(Debug>3)log(XLOG_DEBUG,"process %d, status %s", pid,
				Decode_status( &status ) );
		}
		if(Debug>4)log( XLOG_DEBUG, "all servers done" );
		exit(0);
	}
}
