/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: lockfile.c
 * lock file manipulation procedures.
 ***************************************************************************
 * Revision History: Created Sat Jan  2 06:59:04 CST 1988
 * $Log:	lockfile.c,v $
 * Revision 3.1  88/06/18  09:34:27  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.2  88/05/19  10:34:01  papowell
 * Fixed open() calls to have a 0 parameter, ie: open(f, perms, 0), where needed
 * 
 * Revision 2.1  88/05/09  10:08:33  papowell
 * PLP: Released Version
 * 
 * Revision 1.6  88/05/03  11:45:45  papowell
 * Added LOCKF and FLOCK support
 * 
 * Revision 1.5  88/04/28  10:04:42  papowell
 * minor definitions to keep lint happy
 * 
 * Revision 1.4  88/04/27  20:23:42  papowell
 * Added flock() and lockf() functionality
 * 
 * Revision 1.3  88/04/06  12:13:32  papowell
 * Minor updates, changes in error message formats.
 * Elimination of the AF_UNIX connections, use AF_INET only.
 * Better error messages.
 * 
 * Revision 1.2  88/03/25  14:59:49  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.1  88/03/01  11:08:33  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: lockfile.c,v 3.1 88/06/18 09:34:27 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint
/***************************************************************************
 * File Locking Routines:
 * FILE * Lockcf( char *filename )
 *     locks the indicated file if it exists
 *     Returns: a FILE * for RW if successful, NULL otherwise;
 * FILE * Readlockfile(char *filename,int *pid,char *str,int len,struct stat *s)
 *     reads the information in the lock file.
 * int Checklockfile(char *filename,int *pid,char *str,int len,struct stat *s)
 *     Checks for the presence of a server by using kill(0,pid).
 *     Returns 1 if server present, 0 if not.
 *     Side Effect: if the server is present, returns PID and string from file
 *       in pid and str, stats the file as well
 * FILE * Getlockfile(char *filename,int *pid,char *str,int len,struct stat *s)
 *     locks the indicated file;
 *     Returns a FILE * if successful, NULL otherwise;
 *     Side Effect: if the file is locked, returns PID and string from file
 *       in pid and str, stats the file as well
 * Setlockfile( FILE *fp; int pid; char *str);
 *     write PID and str into the file
 * Closelockfile( FILE *fp );
 *     truncates the lock file and closes it.
 * int Exlockcf( char *filename )
 *    creates and locks the file.
 *    Returns: FILE * for writing, NULL if file exists and is locked
 ***************************************************************************
 * Lock File Manipulation:
 * Each active server has a lock file, which it uses to record its
 * activity.  The lock file is created and then locked;
 * the deamon will place its PID and an activity in the lock file.
 * Programs wanting to know the server status will read the file.
 * Note:  only active server, not status programs, will lock the file.
 * This prevents a status program from locking out a server.
 * Note that the information in the status file may be stale,  as the
 * may update the file without the knowledge of the checker.  This
 * appears to be a small price to pay;  the only place where it may
 * have an effect is when a job is being removed.
 ***************************************************************************/
#include "lp.h"
extern long lseek();
static int do_lock();

FILE *
Lockcf( filename )
	char *filename;		/* Name of file */
{
	int fd;				/* fd for file descriptor */
	FILE *fp;			/* fp for FILE * */
	int lock;			/* lock status */

	/*
	 * Open the lock file for RW
	 */
	fp = NULL;
	if( (fd = open_daemon(filename, O_RDWR, 0)) < 0 ){
		logerr(XLOG_DEBUG,"Lockcf: cannot open file %s", filename);
	} else if( fd < 3 ){
		fatal( XLOG_CRIT, "Lockcf: open '%s' fd is %d", filename, fd);
	} else if( (lock = do_lock(fd, filename)) < 0) {
		logerr_die( XLOG_CRIT, "Lockcf: lock '%s' failed", filename);
	} else if( lock == 0 ){
		if(Debug>3)logerr( XLOG_DEBUG, "Lockcf: lock failed '%s' active",
			filename);
		(void)close( fd );
		fd = -1;
	} else if( (fp = fdopen( fd, "r+")) == NULL ){
		logerr_die( XLOG_CRIT, "Lockcf: fdopen '%s' failed", filename);
	}
	if(Debug>5)log(XLOG_DEBUG,"Lockcf: file '%s', fd %d", filename, fd );
	return( fp );
}
/***************************************************************************
 * File *Readlockfile( .... )
 * Utility function used by Getlockfile and Checklockfile
 * Implements the lock manipulation and reading of the lock file.
 ***************************************************************************/
FILE *
Readlockfile( filename, pid, str, len, sb )
	char *filename;		/* Name of file */
	int *pid;			/* pid */
	char *str;			/* string */
	int len;			/* max string length */
	struct stat *sb;	/* stat buffer */
{
	int fd;				/* file descriptor */
	FILE *fp = NULL;	/* stream */
	int p;				/* ACME Integer and Buggy Works, Inc. */
	char buffer[BUFSIZ]; /* holds the pid read from file */
	char *cp;			/* ACME Pointer, Inc. */
	struct stat s;		/* try stating first */

	/*
	 * Open the lock file, creating if necessary
	 */
	if(Debug>4)log(XLOG_DEBUG,"Readlockfile: lockfile '%s'", filename );
	if( (fd = open_daemon(filename, O_RDWR|O_CREAT, 0644) ) < 0 ){
		logerr_die(XLOG_DEBUG,
			"Readlockfile: cannot create lock file '%s'", filename);
	}
	/*
	 * check to see that you are not clobbering stdin, stdout, stderr
	 */
	if( fd < 3 ){
		fatal( XLOG_CRIT, "Readlockfile: open '%s' fd is %d", filename, fd);
	}
	/*
	 * need a FILE * for stdio
	 */
	if( (fp = fdopen( fd, "r+" )) == NULL ){
		logerr_die(XLOG_CRIT,"Readlockfile: fdopen '%s' failed",filename);
	}
	/*
	 * stat the file for use by others
	 */
	if( fstat( fd, &s ) < 0 ){
		logerr_die( XLOG_INFO,"Readlockfile: fstat '%s' failed", filename);
	}
	if( sb ){
		*sb = s;
	}
	if(Debug>4)log(XLOG_DEBUG,"Readlockfile: %s, perms 0%o",
		filename,s.st_mode);
	/* read the process number */
	if( fseek( fp, 0L, 0 ) < 0 ){
		logerr_die( XLOG_INFO,"Readlockfile: fseek '%s' failed", filename);
	}
	p = 0;
	if( fgets( buffer, sizeof(buffer), fp ) ){
		p = atoi( buffer );
	}
	if(Debug>2)log(XLOG_DEBUG, "Readlockfile: '%s' pid %d len %d",
		filename,p, s.st_size);
	if( fgets( buffer, sizeof(buffer), fp ) ){
		if( cp = index(buffer, '\n') ){
			*cp = 0;
		}
		if(Debug>2)log(XLOG_DEBUG, "Readlockfile: info '%s'", buffer );
		if( str ){
			(void)strncpy(str,buffer,len);
		}
	}
	if( pid ){
		*pid = p;
	}
	if( fseek( fp, 0L, 0 ) < 0 ){
		logerr_die( XLOG_INFO,"Readlockfile: fseek '%s' failed", filename);
	}
	return( fp );
}


/***************************************************************************
 * int Checklockfile( ... )
 * Calls Readlockfile() to read the interesting information, passes
 * things onwards.
 ***************************************************************************/

Checklockfile( filename, pid, str, len, sb )
	char *filename;		/* Name of file */
	int *pid;			/* pid */
	char *str;			/* string */
	int len;			/* max string length */
	struct stat *sb;	/* stat buffer */
{
	FILE *fp;			/* stream */
	int p;				/* ACME Integer and Buggy Works, Inc. */
	/*
	 * Read the information
	 */
	fp = Readlockfile( filename, &p, str, len, sb );
	/* check to see if deamon present by using KILL */
	if( p != 0 && kill( p, 0 ) < 0 ){
		if(Debug>4)log(XLOG_DEBUG, "Checklockfile: server %d not present", p );
		/* sigh... not present */
		p = 0;
	}
	if( pid ){
		*pid = p;
	}
	if( fp ){
		(void)fclose( fp );
	}
	if(Debug>2)log(XLOG_DEBUG, "Checklockfile: %s server %d", filename,p );
	return( p );
}
 

/***************************************************************************
 * FILE * Getlockfile( ... )
 * Calls Readlockfile() to read the interesting information, passes
 * things onwards.  Will try to lock the file; if it fails, assume status
 * correct.  Slight race condition here.
 ***************************************************************************/

FILE *
Getlockfile( filename, pid, str, len, statb )
	char *filename;		/* Name of file */
	int *pid;			/* pid */
	char *str;			/* string */
	int len;			/* max string length */
	struct stat *statb;	/* status buffer */
{
	FILE *fp;		/* lockfile */
	int p;
	int lock;

	fp = Readlockfile( filename, &p, str, len, statb );
	if( pid ){
		*pid = p;
	}
	if( fp != NULL ){
		if( (lock = do_lock(fileno(fp), filename) ) < 0) {
			logerr_die( XLOG_CRIT, "Getlockfile: lock '%s' failed", filename);
		} else if( lock == 0 ){
			if(Debug>3)log(XLOG_DEBUG,"Getlockfile: locked by %d",p);
			(void)fclose( fp );
			fp = NULL;
		}
	}
	if(Debug>2)log(XLOG_DEBUG,"Getlockfile: %s %s",
		filename,fp!=NULL?"success":"fail");
	return( fp );
}

/***************************************************************************
 * Setlockfile( FILE *lockfile; int pid; char *str )
 * will write things into the lock file
 * This has the format:
 * line 1: pid\n
 * line 2: string\n
 ***************************************************************************/

Setlockfile(name, fp, pid, str)
	char *name;
	FILE *fp;
	int pid;
	char *str;
{
	/* clear the lock file */
	if( fseek(fp, 0L, 0) < 0 ){
		logerr_die( XLOG_CRIT, "Setlockfile: fseek failed %s",name);
	}
	if( ftruncate(fileno(fp), (off_t)0) < 0 ){
		logerr_die( XLOG_CRIT, "Setlockfile: ftruncate failed %s", name);
	}
	if( fprintf( fp, "%d\n%s\n", pid, str?str:"" ) == EOF ){
		logerr_die( XLOG_CRIT, "Setlockfile: fprintf failed %s", name);
	}
	if( fflush( fp ) == EOF ){
		logerr_die( XLOG_CRIT, "Setlockfile: fflush failed %s", name);
	}
	if(Debug>4)log(XLOG_DEBUG,"Setlockfile: %s (%d), %s",name,pid,
		str?str:"(nil)");
}

/***************************************************************************
 * Closelockfile( FILE *lockfile)
 * Truncate the file.
 ***************************************************************************/
Closelockfile(name, fp )
	char *name;
	FILE *fp;
{
	if(Debug>4)log(XLOG_DEBUG,"Closelockfile: closing %s", name );
	if( fp == NULL ){
		return;
	}
	/* clear the lock file */
	(void)fflush(fp);
	if( ftruncate(fileno(fp), (off_t)0) < 0 ){
		logerr( XLOG_CRIT, "Closelockfile: cannot truncate %s", name);
		return;
	}
	if( fclose( fp ) == EOF ){
		logerr( XLOG_CRIT, "Closelockfile: fclose failed %s", name);
		return;
	}
}

/***************************************************************************
 * int Exlockcf( char *userfile )
 * Create a user file in the directory and lock it. Prevents things
 * from getting confused.
 ***************************************************************************/

Exlockcf( filename )
	char *filename;		/* Name of file */
{
	int fd;				/* fd for file descriptor */
	int lock;

	/*
	 * Open the lock file, creating if necessary
	 */
	fd = open_daemon(filename, O_RDWR|O_CREAT, FILMOD);
	if (fd < 0) {
		logerr_die(XLOG_DEBUG,"EXlockcf: cannot create file %s", filename);
	}
	if( (lock = do_lock(fd, filename)) < 0) {
		logerr_die( XLOG_CRIT, "Exlockcf: lock '%s' failed", filename);
	} else if( lock == 0 ){
		if(Debug>4)log( XLOG_DEBUG, "Exlockcf: '%s' locked", filename);
		(void)close( fd );
		fd = -1;
	} else if( ftruncate( fd, (off_t)0) < 0 ){
		logerr_die( XLOG_CRIT, "Exlockcf: cannot truncate %s", filename);
	}
	return( fd );
}

/***************************************************************************
 * do_lock( fd , char *filename)
 * does a lock on a file;
 * Returns: 1 if successful; 0 if locked; -1 otherwise
 ***************************************************************************/
static int
do_lock( fd, filename )
	int fd;
	char *filename;
{
	int code;

	code = -1;
	if(Debug>5)log(XLOG_DEBUG,"do_lock: file %s, fd %d", filename, fd );
#ifndef NO_LOCKF
	/*
	 * want to try F_TLOCK
	 */
	if( lockf(fd, F_TLOCK, 0L) < 0) {
		code = 0;
	} else {
		code = 1;
	}
#else
	if( flock(fd, LOCK_EX|LOCK_NB) < 0) {
		if (errno == EWOULDBLOCK){
			code = 0;
		}
	} else {
		code = 1;
	}
#endif NO_LOCKF
	if(Debug>5)log(XLOG_DEBUG,"do_lock: status %d", code );
	return( code );
}
