/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: localprinter.c
 * local Printer queue job handler
 ***************************************************************************
 * Revision History: Created Wed Jan 13 15:34:42 CST 1988
 * $Log:	localprinter.c,v $
 * Revision 3.1  88/06/18  09:34:24  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.4  88/05/21  10:27:48  papowell
 * Minor editing
 * 
 * Revision 2.3  88/05/19  10:33:53  papowell
 * Fixed open() calls to have a 0 parameter, ie: open(f, perms, 0), where needed
 * 
 * Revision 2.2  88/05/14  10:17:51  papowell
 * Use long format for job file names;
 * Added 'fd', no forward flag;
 * Control file has to have hostname and origination agree.
 * 
 * Revision 2.1  88/05/09  10:08:26  papowell
 * PLP: Released Version
 * 
 * Revision 1.5  88/05/09  10:03:18  papowell
 * Revised effects of -h option
 * 
 * Revision 1.4  88/05/05  20:08:11  papowell
 * Added a NOHEADER option that allows user to suppress banner
 * 
 * Revision 1.3  88/03/25  14:59:44  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.2  88/03/05  15:01:02  papowell
 * Minor Corrections,  Lint Problems
 * 
 * Revision 1.1  88/03/01  11:08:31  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: localprinter.c,v 3.1 88/06/18 09:34:24 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"

char *Setup_filter();			/* setup filter */
static int prjobs;				/* numbers of jobs done */

/**********************************************************************
 * Printinit()
 * 1. set up the state to indicated 0 completed jobs
 * 2. get the printer ready
 **********************************************************************/
Printinit()
{
	int n;

	if(Debug>3)log(XLOG_DEBUG,"Printinit: called with prjobs %d", prjobs );
	n = Print_ready();
	return( n );
}

/**********************************************************************
 * Printerror()
 * called when there is an error in the job handling
 * 1. set up the state to indicated 0 completed jobs
 * 2. close the printer
 **********************************************************************/
void
Printerror()
{
	if(Debug>3)log(XLOG_DEBUG,"Printerror: called with prjobs %d", prjobs );

	prjobs = 0;
	Print_close();
}
/**********************************************************************
 * Printfinal()
 * called when there is an error in the job handling
 * 1. if there are completed jobs, print the trailer
 * 2. close the printer
 **********************************************************************/
void
Printfinal()
{
	if(Debug>3)log(XLOG_DEBUG,"Printfinal: called with prjobs %d", prjobs );
	/*
	 * print out trailer on close with a job out
	 */
	if( prjobs ){
		/*
		 * FQ set means FF needed on open
		 */
		if( FQ && FF && *FF ){
			if(Debug>3)log( XLOG_DEBUG, "Printfinal: FF on final");
			(void)Print_string( FF );
		}
		if( TR && *TR ){
			/*
			 * TR is sent out
			 */
			if(Debug>2)log( XLOG_DEBUG, "Printfinal: TR '%s'", TR);
			(void)Print_string( TR );
		}
	}
	prjobs = 0;
	Print_close();
}

/**********************************************************************
 * Printjob
 *   1.  First scan extracts information which controls printing
 *   2.  Set any default values not passed
 *   3.  Second scan does the printing.
 * Returns:  JBUSY, JFAIL, JSUCC, JABORT
 * Side effects:  sets information vector  CFparm[].
 **********************************************************************/
Printjob(cfp, q)
	FILE *cfp;			/* control file */
	struct queue *q;	/* job entry */
{
	int i;				/* ACME Integer, Inc. */
	char parm[BUFSIZ];	/* holds a line read in */
	char *arg;			/* control line argument */
	char opt;			/* control line option */
	int  jstatus;	    /* job status */
	int  perms = 'R';	/* file perms */
	long jobsize;		/* job size */

	if( fseek( cfp, 0L, 0 ) < 0 ){
		logerr_die(XLOG_INFO, "Printjob: start- fseek failed" );
	}
	/*
	 * set job status
	 */
	jstatus = JABORT;	/* default */
	/*
	 * initialize the CFparm array, which holds value read from file
	 */
	for( i = 0; i < 26; ++i ){
		CFparm[i][0] = 0;
	} 

	/*
	 * read the control file and extract user information
	 */
	jobsize = 0;
	while(fgets( parm, sizeof(parm), cfp )){
		if( (arg = index(parm, '\n')) == 0 ){
			log(XLOG_INFO,"Printjob: bad control file (%s), no endline",
				q->q_name);
			goto error;
		}
		*arg = 0;
		opt = parm[0];
		arg = parm+1;
		if( !isascii(opt) || ! isalnum( opt )){
			log( XLOG_INFO, "Printjob: bad control file (%s), line('%d'%s)",
				q->q_name,opt, arg);
			goto error;
		}
		/*
		 * copy it into the appropriate place if needed
		 */
		if( isupper( opt ) ){
			switch( opt ){
			case 'U':
			case 'N': break;
			default:
				if( CFparm[opt - 'A'][0] ){
					log(XLOG_INFO,
						"Printjob: duplicate %c parm '%s', previous:'%s'",
						opt, arg, CFparm[opt-'A']);
					goto error;
				}
				if( strlen( arg ) >= MAXPARMLEN ){
					log(XLOG_INFO,
						"Printjob: control file %s line too long:'%s'",
						q->q_name,arg);
					goto error;
				}
				(void)strcpy( CFparm[opt - 'A'], arg );
				break;
			}
		} else if( islower( opt )){
			if( Job_match( q->q_name, arg) == 0 ){
				logerr(XLOG_INFO,"Printjob: bad control file '%s' entry '%s'",
					q->q_name,parm);
				goto error;
			}
			if( stat( arg, &LO_statb ) < 0 ){
				logerr( XLOG_INFO,"Printjob: cannot stat file %s",arg);
				goto error;
			}
			if( (i = Add_name( arg )) < 0 ){
				logerr( XLOG_INFO,"Printjob: too many files %s",q->q_name);
				goto error;
			}
			jobsize = jobsize + (LO_statb.st_size + 1023)/1024;
			if( MX && jobsize > MX ){
				logerr( XLOG_INFO,"Printjob: job too big %s",q->q_name);
				goto error;
			}
			switch( opt ){
				case 'f':	/* uses the IF filter */
				case 'l':
					break;
				default:
					if( Filter_name[opt-'a'] == 0){
						log( XLOG_INFO,"Printjob: no %c filter",opt);
						goto error;
					}
					break;
			}
		}
	}
	/*
	 * Set up any parameters that were not provided to default values
	 */
	if( PWIDTH[0] = 0){
		(void)sprintf( PWIDTH, "%d", PW );
	}
	/*
	 * check permissions
	 */
	if( strcmp( &q->q_from, FROMHOST ) ){
		log(XLOG_INFO,
			"Printjob: control file origin '%s' and H entry '%s' do not match",
			&q->q_from, FROMHOST );
			goto error;
	}
	if((Permfile && *Permfile &&
			!Checkperm(Permfile,FROMHOST,LOGNAME,First_name,&perms,(int *)0,0))
       ||(XU && *XU &&
			!Checkperm(XU,FROMHOST,LOGNAME,First_name,&perms,(int *)0,0 ) )){
		log(XLOG_INFO,
			"Sorry %s@%s, you don't have permission to use '%s'",
			LOGNAME, FROMHOST, First_name );
			goto error;
	} else if((Permfile && *Permfile &&
			!Checkperm(Permfile,FROMHOST,LOGNAME,First_name,&perms,(int *)0,1))
       ||(XU && *XU &&
			!Checkperm(XU,FROMHOST,LOGNAME,First_name,&perms,(int *)0,1 ) )){
		log( XLOG_INFO, "Sorry %s@%s, no more pages allowed on '%s'",
			LOGNAME, FROMHOST, First_name );
			goto error;
	}
	/*
	 * See if there are any files to print
	 */
	if( Parmcount == 0 ){
		/* no files */
		if(Debug>4)log(XLOG_DEBUG,"Printjob: no files");
		jstatus = JSUCC;
		goto error;
	}
	/*
	 * Pass 2: Do the printing now
	 */
	if( fseek(cfp, 0L, 0) < 0 ){
		/* this is impossible */
		logerr( XLOG_NOTICE, "Printjob: fseek %s failed",q->q_name);
		goto error;
	}
	if( QH ){
		/*
		 * queuejob returns one of the Job Status settings
		 */
		jstatus = queuejob(cfp, q);
		goto error;
	}
	/*
	 * Put out FF on open and between jobs
	 */
	if( prjobs == 0 ){
		/*
		 * FO set means FF needed on open
		 */
		if( LD && *LD ){
			if(Debug>3)log( XLOG_DEBUG, "Printjob: putting out LD on open");
			jstatus = Print_string( LD );
			if( jstatus != JSUCC ){
				log( XLOG_INFO, "Printjob: LD on open printing failed" );
				goto error;
			}
		}
		if( FO && FF && *FF ){
			if(Debug>3)log( XLOG_DEBUG, "Printjob: putting out FF on open");
			jstatus = Print_string( FF );
			if( jstatus != JSUCC ){
				log( XLOG_INFO, "Printjob: FF on open printing failed" );
				goto error;
			}
		}
		prjobs = 1;
	} else if( SF == 0 && FF && *FF ){
		if(Debug>3)log(XLOG_DEBUG,"Printjob: putting out FF between jobs");
		jstatus = Print_string( FF );
		if( jstatus != JSUCC ){
			log( XLOG_INFO, "Printjob: FF printing failed" );
			goto error;
		}
	}
	/*
	 * print banner
	 * if it has not been disabled by the SH flag
	 * or the lpr -h flag was used and the AB (always banner) is clear
	 */
	if( !(SH || (NOHEADER[0] && !AB)) ){
		jstatus = Print_banner();
		if( jstatus != JSUCC ){
			log( XLOG_INFO, "Printjob: banner printing failed" );
			goto error;
		}
		prjobs = 1;
	}
	/*
	 * Banner Printing Program
	 */
	if( BP && *BP ){
		jstatus = Banner_print(BP);
		if( jstatus != JSUCC ){
			log( XLOG_INFO, "Printjob: BP program %s failed", BP);
			goto error;
		}
		prjobs = 1;
	}
	/*
	 * print individual jobs
	 */
	while(fgets( parm, sizeof(parm), cfp )){
		if( (arg = index(parm, '\n')) == 0 ){
			log( XLOG_INFO,"Printjob: bad control file format (%s), no endline",
				q->q_name);
			goto error;
		}
		*arg = 0;
		opt = parm[0];
		arg = parm+1;
		if( opt == 0 || ! isalnum( opt )){
			log( XLOG_INFO, "bad control file (%s), line(%s)",
				q->q_name,parm);
			goto error;
		}
		if( islower( opt ) ){
			/*
			 * print the file
			 */
			prjobs = 1;
			jstatus = printfile( opt, arg );
			if( jstatus != JSUCC ){
				goto error;
			}
		}
	}
	/*
	 * End Printing Program
	 */
	if( EP && *EP ){
		jstatus = Banner_print(EP);
		if( jstatus != JSUCC ){
			log( XLOG_INFO, "Printjob: EP program %s failed", EP);
			goto error;
		}
		prjobs = 1;
	}

	/*
	 * error and status reporting
	 */
error:
	if( MAILNAME[0] ){
		sendmail(q, jstatus);
	}
	if(Debug>3)log(XLOG_DEBUG,"Printjob: '%s' status %d",q->q_name,jstatus);
	return (jstatus);
}

/***********************************************************************
 * printfile( int format; char *datafile )
 *   1. determine the format and corresponding filter command
 *   2. if 'p' format, set up pr process to format output
 *   3. output to Printer
 * Returns: JFAIL if retry, JSUCC if success, JABORT if ugly
 ***********************************************************************/
static int
printfile(format, file)
	int format;
	char *file;
{
	int dfd;			/* data file fd */
	int status;			/* return status */
	char *filter;		/* program Name */
	int nofilter = 0;	/* no filter, just copy */
	char buf[BUFSIZ];	/* hold the Printer command */
	int p[2];			/* pipe file descriptors */
	int prchild;		/* Printer process */

	if(Debug>3)log( XLOG_DEBUG, "printfile: format %c, file %s", format, file);
	/*
	 * Open input file
	 */
	if ((dfd = open_daemon(file, O_RDONLY, 0)) < 0) {
		logerr( XLOG_NOTICE,"printfile: open failed '%s'", file);
		return(JABORT);
	}
	/*
	 * find filter chain to be generated
	 */
	switch (format) {
	/*
	 * use the filter Name tagged by 'Xf', where X is format
	 */
	default:
		filter = Setup_filter( format, Filter_name[format-'a']);
		break;
	/*
	 * 'f' and 'l' formats use the IF filter, and OF if not present
	 */
	case 'f':	/* default */
	case 'l':	/* ignore control */
	case 'p':	/* print file using 'pr' */
		if( IF == 0 || *IF == 0){
			/*
			 * you have to use the OF filter; we don't set up a filter
			 */
			nofilter = 1;
		} else {
			filter = Setup_filter( 'f', IF);
		}
		if( format != 'p' ){
			break;
		}
		/*
		 * create a process to invoke 'pr'
		 * first, set up the pr command
		 */
		(void)sprintf(buf, "%s -w%d -l%d %s %s",
			PR, PWIDTH, PL, PRTITLE[0]?"-h":"", PRTITLE);
		if(Debug>3)log( XLOG_DEBUG, "printfile: pr command '%s'", buf);
		/*
		 * create PR process
		 */
		if( pipe(p) < 0 ){
			logerr_die( XLOG_NOTICE, "printfile: cannot make pipe" );
		}
		/*
		 * make a pipe, fork PR process
		 * Run this process as DAEMON
		 */
		if ((prchild = fork()) == 0) {	/* child */
			if( dup2(dfd, 0) < 0		/* file is stdin */
			 || dup2(p[1], 1) < 0 ){	/* pipe stdout */
				logerr_die( XLOG_NOTICE, "printfile: dup2 failed, PR process" );
			}
			/* set the uid to be safe */
			if( geteuid() == 0 && setreuid( Daemon_uid, Daemon_uid ) < 0 ){
				logerr_die( XLOG_INFO, "printfile: setreuid failed" );
			}
			mexecv(buf);
			logerr_die( XLOG_NOTICE,"printfile: cannot execv %s", buf);
		} else if (prchild < 0) {
			logerr_die( XLOG_NOTICE,"printfile: fork for pr failed");
		}
		(void) close(p[1]);		/* close output side */
		(void) close(dfd);
		dfd = p[0];	/* use pipe for input */
		break;
	}
	/*
	 * start filter
	 */
	if( nofilter ){
		status = Print_copy( dfd );
	} else if (filter == 0 || *filter==0) {
		log( XLOG_INFO,"printfile: no %c filter, file %s", format,file);
		status = JABORT;
	} else {
		/*
		 * start filter up
		 */
		if(Debug>0)log( XLOG_DEBUG,
			"printfile: format %c, file %s, filter %s", format,file,filter);
		status = Print_filter( dfd, filter );
	}
	(void)close( dfd );
	return( status );
}

/********************************************************************
 * Banner_print( char *prog)
 * Print the banner using the user specified program
 * 1. setup the filter
 * 2. start program
 * 3. wait for completion.
 ********************************************************************/
Banner_print( prog )
	char *prog;
{
	char *filter;
	int status;

	if(Debug>3)log( XLOG_DEBUG, "Banner_print %s", prog);
	filter = Setup_filter( 'f', prog);
	if (filter == 0 || *filter==0) {
		log( XLOG_INFO,"Banner_printer: bad program %s", prog );
		status = JABORT;
	} else {
		/*
		 * start filter up
		 */
		if(Debug>3)log( XLOG_DEBUG, "Banner_print: filter %s", filter);
		status = Print_filter( 0, filter );
	}
	return( status );
}
/********************************************************************
 * queuejob()
 * Use the user specified Queue Handler.
 * exec'd as `qh -PPrinter -B[nsl] control_file'.
 *  exit codes are used to indicate the different success:
 *   0 - successful; 1 retry; anything else- abandon
 * Returns: JSUCC, etc.
 * NOTE: the queue handler is invoked with the control file as
 *  stdin and the Printer as stdout.
 ********************************************************************/

queuejob(cfp, q)
	FILE *cfp;				/* control file */
	struct queue *q;		/* queue entry */
{
	union wait status;
	int id, pid, ret;
	char buf[BUFSIZ];

	/*
	 * set up arguments
	 */
	(void)sprintf(buf, "%s %s", QH, Printer, q->q_name);
	if(Debug>0)log( XLOG_DEBUG,"queuejob: %s", buf);
	/*
	 * Open the Printer
	 */
	Print_open();
	if( (pid = fork()) == 0 ){	/* child */
		if( dup2(fileno(cfp), 0) < 0 ){	/* stdin is cf */
			logerr_die( XLOG_NOTICE, "queuejob: dup2 failed" );
		}
		if( dup2(Print_fd, 1) < 0 ){	/* stdin is cf */
			logerr_die( XLOG_NOTICE, "queuejob: dup2 failed" );
		}
		if( geteuid() == 0 && setreuid( Daemon_uid, Daemon_uid ) < 0 ){
			logerr_die( XLOG_INFO, "queuejob: setreuid failed" );
		}
		mexecv(buf);
		logerr_die( XLOG_INFO, "queuejob: exec failed" );
	} else if( pid < 0 ){
		logerr_die( XLOG_INFO, "queuejob: fork failed" );
	}
	/*
	 * wait for process
	 */
	if(Debug>0)log( XLOG_DEBUG, "waiting for queuejob %d", pid );
	while ((id = wait(&status) ) > 0 && pid!=id){
		if(Debug>3)log(XLOG_DEBUG,"queuejob: caught %d, %s", id,
			Decode_status(&status) );
	}
	if(Debug>0)log( XLOG_DEBUG, "queuejob: pid %d status %s", id,
		Decode_status(&status ) );
	if( id < 0 || status.w_stopval != 0 || (unsigned)status.w_retcode > 1 ){
		logerr( XLOG_INFO, "queuejob process %d failed, status (%s)",
			id, Decode_status(&status));
		ret = JABORT;
	} else if( status.w_retcode == 1 ){
		ret = JFAIL;
	} else {
		ret = JSUCC;
	}
	return (ret);
}
