/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: Errormsg.c
 * error messages and logging
 * log:
 *  --message on stderr and on stdout if "Echo_on_stdout" is set
 *  --if the severity is high enough, will also log using syslog().
 *  --saves the last "non-Debug" log message
 * logerr:
 *  --formats the error message corresponding to "errno" and calls log();
 * fatal:
 *  -- log() and exit() with Errorcode
 * logerr_die:
 *  -- logerr() and exit() with Errorcode
 ***************************************************************************
 * Revision History: Created Sun Jan  3 18:41:01 CST 1988
 * $Log:	errormsg.c,v $
 * Revision 3.1  88/06/18  09:34:09  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.1  88/05/09  10:08:01  papowell
 * PLP: Released Version
 * 
 * Revision 1.4  88/04/26  15:46:11  papowell
 * Added a datestamp to all messages
 * 
 * Revision 1.3  88/04/06  12:12:43  papowell
 * Minor updates, changes in error message formats.
 * Elimination of the AF_UNIX connections, use AF_INET only.
 * Better error messages.
 * 
 * Revision 1.2  88/03/25  14:59:24  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.1  88/03/01  11:08:23  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: errormsg.c,v 3.1 88/06/18 09:34:09 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"

/*
 * Errormsg( err ) returns a printable form of the errormessage
 * corresponding to errno.
 */

char *
Errormsg( err )
	int err;
{
	char *cp;
	static char msgbuf[30];

	if( err >= 0 && err <= sys_nerr ){
		cp = sys_errlist[err];
	} else {
		(void)sprintf(msgbuf,"errno=%d",err);
		cp = msgbuf;
	}
	return(cp);
}

struct msgkind {
	int *var;
	char *str;
};
static struct msgkind msg_name[] = {
	{ &XLOG_ERR, "LOG_ERR" },
	{ &XLOG_CRIT, "LOG_CRIT" },
	{ &XLOG_WARNING, "LOG_WARNING" },
	{ &XLOG_NOTICE, "LOG_NOTICE" },
	{ &XLOG_INFO, "LOG_INFO" },
	{ &XLOG_DEBUG, "LOG_DEBUG" },
	{ 0 }
};

static
char *
logmsg( kind )
	int kind;
{
	int i;
	static char b[35];

	for( i = 0; msg_name[i].var; ++i ){
		if( *msg_name[i].var == kind ){
			return( msg_name[i].str );
		}
	}
	(void)sprintf(b, "bad message type %d",  kind );
	return(b);
}

/*
 * log( int severity; char *msg; args )
 *  	1. format the message; msg and args are similar to printf
 *		2. output on stderr
 *		3. if "Echo_on_stdout" output on stdout
 *		4. if severity high enough,  log using syslog()
 *		5. if not a Debug message, save in Last_errormsg[]
 */
/*VARARGS2*/
log(kind, msg, a1, a2, a3, a4, a5, a6, a7, a8, a9)
	char *msg, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8, *a9;
{
	char buf[BUFSIZ];
	int omask;

	omask = sigblock(sigmask(SIGCHLD)|sigmask(SIGHUP)
		|sigmask(SIGINT)|sigmask(SIGQUIT)|sigmask(SIGTERM));
	buf[0] = 0;
	if(Name && *Name){
		(void)sprintf(buf+strlen(buf),"%s: ", Name);
	}
	if(Printer && *Printer){
		(void)sprintf(buf+strlen(buf),"%s- ", Printer);
	}
	if(Debug){
		(void)sprintf(buf+strlen(buf),"pid=%d, %s, ",getpid(),logmsg(kind));
	}
	(void)sprintf(buf+strlen(buf), msg, a1, a2, a3, a4, a5, a6, a7, a8, a9);
	(void)sprintf(buf+strlen(buf)," at %s",Time_str() );

	if(Echo_on_stdout){
		(void)fprintf(stdout,"%s\n", buf);
		(void)fflush(stdout);
	}
	(void)fprintf(stderr,"%s\n", buf);
	(void)fflush(stderr);
	if( kind <= XLOG_INFO ){
		use_syslog( kind, buf);
	}
	if( kind != XLOG_DEBUG ){
		(void)strcpy( Last_errormsg, buf );
	}
	(void)sigsetmask(omask);
}

/*VARARGS2*/
fatal(kind, msg, a1, a2, a3, a4, a5, a6, a7, a8, a9)
	char *msg, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8, *a9;
{
	log( kind, msg, a1, a2, a3, a4, a5, a6, a7, a8, a9);
	cleanup();
	exit(Errorcode);
}

/*VARARGS2*/
logerr( kind,msg, a1, a2, a3, a4, a5, a6, a7, a8, a9)
	char *msg, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8, *a9;
{
	int err = errno;
	char buf1[BUFSIZ];

	(void)sprintf(buf1, msg, a1, a2, a3, a4, a5, a6, a7, a8, a9);
	log( kind,"%s - %s", buf1, Errormsg(err) );
	errno = err;
}

/*VARARGS2*/
logerr_die( kind,msg, a1, a2, a3, a4, a5, a6, a7, a8, a9)
	char *msg, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8, *a9;
{
	logerr( kind,msg, a1, a2, a3, a4, a5, a6, a7, a8, a9);
	cleanup();
	exit(Errorcode);
}

/*
 * use the syslog(8) deamon to do our logging
 */
#ifdef NOSYSLOG
static FILE *f;
#else NOSYSLOG
#include <syslog.h>
#endif NOSYSLOG
static int init;

use_syslog( kind, msg )
	int kind;
	char * msg;
{
	if(Debug>7){
		(void)fprintf(stderr, "syslog: %s\n", msg );
		(void)fflush(stderr);
	}
#ifdef NOSYSLOG
	if( f != NULL || (f = fopen( "/dev/console", "w" ) ) != NULL ){
		(void)fprintf( f, "syslog: LPD %s\n", msg );
		(void)fflush(f);
	}
#else NOSYSLOG
	if(init == 0 ){
		init = 1;
	}
#	ifdef IS_SUN
		(void)syslog(kind, "%s", msg);
#	endif
#	ifdef IS_VAX
		(void)syslog(LOG_LPR|kind, msg);
#	endif
#endif NOSYSLOG
}
