/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: Checkperm.c
 * printer permission checking
 ***************************************************************************
 * Revision History: Created Sun Jan  3 19:08:26 CST 1988
 * $Log:	checkperm.c,v $
 * Revision 3.1  88/06/18  09:33:59  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.4  88/05/27  08:27:23  papowell
 * Fixed the '~' permissions line
 * 
 * Revision 2.3  88/05/21  10:26:22  papowell
 * Added check for '~' or always
 * 
 * Revision 2.2  88/05/16  09:44:32  papowell
 * Modified match() to handle null strings better
 * 
 * Revision 2.1  88/05/09  10:07:46  papowell
 * PLP: Released Version
 * 
 * Revision 1.3  88/04/07  09:11:14  papowell
 * Modified 'while' loop; removed break
 * 
 * Revision 1.2  88/03/25  14:59:09  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.1  88/03/01  11:08:18  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: checkperm.c,v 3.1 88/06/18 09:33:59 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint
/******************************************************************************
 * Checkperm - determine if a user has permissions to act on a printer queue
 *
 * Input:
 *  permfile - name of the permissions file,
 *    each line of which has the form:
 *		hostname userid printer_name op pr max current
 * 		hostname - the host name of the invoker
 * 		userid - the user name of the invoker
 * 		printer_name - the printer queue to be checked
 *		op -the maximum operation level, A is 'Highest', Z is 'lowest'
 *			Note: R is for LPR, C is for LPC, * allows all
 *		pr - the maximum priority level, A is 'Highest', Z is 'lowest'
 *		max - max number of pages allowed
 *		current - current number of pages
 *
 *	# -- comment
 *	#hostname username printername op pr maxpages currentpages
 *	attila    papowell imagen      C  A  1000     100
 *  !*        cs9*     lpr         *  *
 *	*         jsmith   imagen      *  *  0        0
 *	*         root     *           *  *
 *	
 *	Note: * is the wildcard indicator and matches all characters
 *	Note: if maxpages == 0, then page count is unlimited.
 *  Note: if a ! is in the first column  and there is a match
 *    (i.e.- we should grant permission), then permission is refused.
 *    In the example above, this prevents users whose names start with
 *    cs9 will be forbidden access to the lpr queue.
 *
 * Output:
 * 0 - if userid does not have permissions to act on printer queue
 * nonzero - permission granted
 *
 * If any of arguments is not specified (i.e.- 0 or NULL value),
 * then checking is not done for the field.
 *****************************************************************************/

#include "lp.h"

int
Checkperm(permfile, host, user, printerq, op, pr_level, pages)
	char	*permfile;	/* file to read for permissions */
	char	*host;		/* hostname to check for */
	char	*user;		/* username to check for */
	char	*printerq;	/* printername to check for */
	int		*op;		/* operation level to check for */
	int		*pr_level;	/* priority level to check for */
	int		pages;		/* check page limits flag */
{
	FILE	*fp;
	int		i;
	char	buf[BUFSIZ];
	char	username[MAXPARMLEN+1];
	char	hostname[MAXPARMLEN+1];
	char	printerqname[MAXPARMLEN+1];
	char	priority[MAXPARMLEN+1];
	char	operation[MAXPARMLEN+1];
	char	*str;
	int		max, current;
	int		permission = 0;
	int		invert, must, found;

	must = 0;
	invert = 0;
	if( permfile == 0 ){
		logerr( XLOG_INFO, "Checkperm: no perm file" );
		return(permission);
	}
	if(Debug>5)log(XLOG_DEBUG,
	"Checkperm file %s,host=%s,user=%s,printerq=%s,op=%c,pr_level=%c,pages=%d",
		permfile, host?host:"NONE", user?user:"NONE", printerq?printerq:"NONE",
		 op?*op:'?', pr_level?*pr_level:'?', pages);
	/*
	 * open permfile for reading
	 */
	if((fp = fopen_daemon(permfile,"r")) == NULL){
		logerr( XLOG_CRIT, "Checkperm: cannot open perms file %s",permfile);
		return(permission);
	}

	/*
	 * scan through the file looking for a match
	 */
	found = 0;
	while( found == 0 && fgets( buf, sizeof(buf), fp ) != NULL ){
		/* read in the host name */
		/* sscanf returns number of fields converted */
		if( buf[0] == '\n' || buf[0] == '#' ){
			continue;
		}
		must = 0;
		invert = 0;
		max = current = 0;
		priority[0] = 0;
		operation[0] = 0;
		str = buf;
		switch( buf[0] ){
			case '!': invert = 1; ++str; break;
			case '~': must = 1; ++str; break;
		}
		i = sscanf(str, "%s%s%s%s%s%d%d", hostname, username,
				printerqname,operation,priority,&max,&current);
		if( i == 0 ){
			continue;
		} else if( i < 4 ){
			log(XLOG_CRIT,
			"Error in %s, '%s'\n  Please inform system guru.", permfile, buf);
			(void)fclose(fp);
			return(permission);
		}
		if(Debug>6)log(XLOG_DEBUG,
			"host=%s,user=%s,printer=%s,oper=%s,priority=%s,max%d,current%d",
			hostname, username, printerqname,operation,priority,max,current );

		/*
		 * if the hostname field is '*', this means any host can do it
		 * otherwise we should check to see if a valid host name
		 */
		if( !match(host,hostname) ){
			if( must ){
				found = 1;
			}
			continue;
		}
		if(Debug>6)log(XLOG_DEBUG, "Checkperm: host ok, %s, %s",
			host?host:"<NONE>",hostname);

		/* compare the user names */
		if( !match(user,username) ){
			if( must ){
				found = 1;
			}
			continue;
		}
		if(Debug>6)log(XLOG_DEBUG, "Checkperm: user ok, %s, %s",
			user?user:"<NONE>",username);

		/* compare the printer queue names */
		if( !match(printerq,printerqname) ){
			if( must ){
				found = 1;
			}
			continue;
		}
		if(Debug>6)log(XLOG_DEBUG, "Checkperm: printerq ok, %s, %s",
			printerq?printerq:"<NONE>",printerqname);

		/* compare the operations */
		if( op && operation[0] != '*' && *op < operation[0]){
			if( must ){
				found = 1;
			}
			continue;
		}
		if(Debug>6)log(XLOG_DEBUG,
			"Checkperm: OK, must %d, wanted %c(%d), have %c(%d), invert %d",
			must, op?*op:'?', op?*op:'?', operation[0], operation[0], invert);
		/*
		 * Well, you had to have a match, and you did.  So this means that
		 * you passed, but you do not use this entry.
		 */
		if( must ){
			continue;
		}
		if(Debug>6)log(XLOG_DEBUG,
			"Checkperm: pages- %d, max %d, current %d", pages,max,current);
		found = 1;
		permission = 1;
		if( pages && max && current > max ){
			/* page limit exceeded in first matching entry: bad news */
			permission = 0;
		}
		if( pr_level && priority[0] && priority[0] != '*'){
			/* we want to check priory and there are limits */
			if( priority[0] > *pr_level ){
				/* sorry, this is too high */
				*pr_level = priority[0];
			}
		}
	}
	if( permission && invert ){
		permission = 0;
	}
	if(Debug>4)log(XLOG_DEBUG,"Checkperm %d in %s, printerq %s for %s@%s",
		permission, permfile,
		printerq?printerq:"NONE",user?user:"NONE",host?host:"NONE");
	(void)fclose(fp);
	return(permission);
}

/*
 * match( str, pattern )
 *  -- str is null (match returns 1 )
 *	-- pattern is <fixed>*, str is <fixed><anything> (match returns 1)
 *  -- pattern is !<fixed>*, str is <fixed><anything> (match returns 0)
 *	-- if not above, match returns 0
 */

static int
match( str, pattern)
	char *str, *pattern;
{
	int c;

	if(Debug>8)log(XLOG_DEBUG,"match str '%s', pattern '%s'\n", str, pattern );
	if( str == 0 || pattern == 0) return(1);
	while( c = *pattern ){
		if(Debug>8)log(XLOG_DEBUG,"match partial str '%s', pattern '%s'\n",
			str, pattern );
		switch( c ){
			/*
			 * match 0 or more characters in the string
			 */
			case '*':
				if( match( str, pattern+1 ) ){
					return( 1 );
				}
				if( *str && match( str+1, pattern ) ){
					return( 1 );
				}
				return( 0 );
			case '?':
				if( *str == 0 ){
					return( 0 );
				}
				break;
			default:
				if( c != *str ){
					return( 0 );
				}
				break;
		}
		++pattern; ++str;
	}
	return( *pattern == *str );
}
