/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: remoteprinter.c
 * send a job to a remote Printer
 ***************************************************************************
 * Revision History: Creation Fri Jan 15 19:28:22 CST 1988
 * $Log:	remoteprinter.c,v $
 * Revision 3.1  88/06/18  09:35:31  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.3  88/05/14  10:18:22  papowell
 * Use long format for job file names;
 * Added 'fd', no forward flag;
 * Control file has to have hostname and origination agree.
 * 
 * Revision 2.2  88/05/11  09:52:49  papowell
 * Remote printer file transfer error fixed.
 * 
 * Revision 2.1  88/05/09  10:10:04  papowell
 * PLP: Released Version
 * 
 * Revision 1.4  88/04/06  12:12:35  papowell
 * Minor updates, changes in error message formats.
 * Elimination of the AF_UNIX connections, use AF_INET only.
 * Better error messages.
 * 
 * Revision 1.3  88/03/25  15:01:27  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.2  88/03/12  10:03:32  papowell
 * *** empty log message ***
 * 
 * Revision 1.1  88/03/01  11:09:10  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: remoteprinter.c,v 3.1 88/06/18 09:35:31 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"

/***************************************************************************
 * int Setup_xfer()
 * sends the "\02Printer" message to the remote Host
 * Returns: successful JSUCC, otherwise JFAIL
 ***************************************************************************/
int
Setup_xfer()
{
	char buf[BUFSIZ];

	if( RP == 0 || *RP == 0 ){
		fatal(XLOG_INFO, "no RP specified for RM (%s)", RM );
	}
	(void)sprintf( buf, "%c%s\n",  REQ_RECV, RP );
	if(Debug>3)log(XLOG_DEBUG,
		"Setup_xfer: request %d, (REQ_RECV) Host '%s' pr '%s'",
		REQ_RECV,RM,RP);
	if( JSUCC == Link_line( -1, buf )){
		if(Debug>3)log(XLOG_DEBUG,"Setup_xfer: request sent");
		if( JSUCC == Link_confirm()){
			if(Debug>3)log(XLOG_DEBUG,"Setup_xfer: request to %s confirmed",RM);
			return(JSUCC);
		}
		if(Debug>3)log(XLOG_DEBUG,"Setup_xfer: request to %s not confirmed",RM);
	} else {
		if(Debug>3)log(XLOG_DEBUG,"Setup_xfer: request not sent");
	}
	setstatus("request for file transfer to %s failed %s", RM, Time_str() );
	fatal( XLOG_INFO, "Setup_xfer: cannot start transfer to %s", RM );
	/*NOTREACHED*/
}

/***************************************************************************
 * Sendjob( FILE *cfp; struct queue *q)
 *   1.  First scan extracts information which controls printing
 *			and finds the names of the files to send.
 *	 2.	 The data files are then sent.
 *	 3.	 Control file is then sent.
 *   4.  Job files are removed
 *   5.  Control file is removed
 * Returns:  JBUSY, JFAIL, JSUCC, JABORT
 * NOTE: this code was based on a description of the LPD file transfer
 *    protocol.
 *
 * File transfer protocol:
 * Sender: sends line of form 'flag' length filename
 * Receiver: acknowledges with 0 (single byte)
 * Sender: sends file (length bytes of information), then a teminating 0
 * Receiver: acknowledges with 0 (single byte)
 *
 * Tne end is signalled by closing the stream, which results in a read error.
 * 
 *************************************************************************/
Sendjob(cfp, q)
	FILE *cfp;			/* control file */
	struct queue *q;	/* job entry */
{
	FILE *fp;					/* File to send */
	char parm[MAXPATHLEN];			/* holds a line read in */
	char *arg;					/* control line argument */
	char opt;					/* control line option */
	int  jstatus;			    /* job status */
	int  i;						/* ACME Integers, Inc. */

	/*
	 * set job status
	 */
	jstatus = JFAIL;	/* default */

	/*
	 * read the control file and extract user information
	 */
	Parmcount = 0;
	while(fgets( parm, sizeof(parm), cfp )){
		if( (arg = index(parm, '\n')) == 0 ){
			log( XLOG_INFO, "Sendjob: bad control file format (%s), no endline",
				q->q_name);
			jstatus = JABORT;
			goto error;
		}
		*arg = 0;
		opt = parm[0];
		arg = parm+1;
		if( !isascii(opt) || !isalnum( opt ) || strlen( arg ) > MAXPARMLEN ){
			log( XLOG_INFO, "Sendjob: bad control file (%s), line(%s)",
				q->q_name,parm);
			jstatus = JABORT;
			goto error;
		}
		if( !islower( opt ) && opt != 'U' ){
			continue;
		}
		if( Job_match( q->q_name, arg ) == 0 ){
			log( XLOG_INFO, "Sendjob: bad file name format (%s)", arg );
			jstatus = JABORT;
			goto error;
		}
		if( islower( opt )){
			if( Add_name(arg) < 0 ){
				log( XLOG_INFO, "Sendjob: too many files (%s)", arg );
				jstatus = JABORT;
				goto error;
			}
		} else {
			if( (i = Find_name(arg) ) < 0 ){
				log(XLOG_INFO,"Sendjob: job(%s)U specified for '%s' not in job",
					q->q_name,parm);
				jstatus = JABORT;
				goto error;
			}
		}
	}
	/*
	 * We send the control file first if we are using the PLP protocol
	 */
	if( FJ ){
		jstatus = sendfile( cfp, q->q_name, CNAME);
		if( jstatus != JSUCC ){
			setstatus("transfer %s to %s failed %s",q->q_name,RM,Time_str());
			log(XLOG_INFO,"Sendjob: sendname %s to %s failed",q->q_name,RM);
			goto error;
		}
	}
	/*
	 * send the jobs in the file
	 */
	for( i = 0; i < Parmcount; ++i ){
		arg = Parms[i].filename;
		if( (fp = fopen_daemon( arg, "r" )) == NULL ){
			jstatus = JABORT;
			logerr( XLOG_INFO,"Sendjob: cannot open data file %s",arg);
			goto error;
		}
		/*
		 * send the file
		 */
		if(Debug>3)log(XLOG_DEBUG,"Sendjob: sending file %s", arg );
		jstatus = sendfile( fp, arg, DFILE );
		(void)fclose(fp);
		if( jstatus != JSUCC ){
			setstatus("transfer %s to %s failed %s",arg,RM,Time_str());
			log(XLOG_INFO,"Sendjob: transfer %s to host %s failed",arg,RM);
			goto error;
		}
	}
	/*
	 * send the control file
	 */
	if(Debug>3)log(XLOG_DEBUG,"sending file %s", q->q_name );
	if( FJ ){
		jstatus = sendname( cfp, q->q_name, CEND );
	} else {
		jstatus = sendfile( cfp, q->q_name, CFILE );
	}
	if( jstatus != JSUCC ){
		setstatus("transfer %s to %s failed %s",q->q_name,RM,Time_str());
		log(XLOG_INFO,"Sendjob: transfer %s to host %s failed",q->q_name,RM);
		goto error;
	}
	/*
	 * finished!
	 */
	jstatus = JSUCC;
error:
	if( jstatus != JSUCC ){
		Link_close();
	}
	return (jstatus);
}

/***************************************************************************
 * sendfile( FILE* fp; char *name; int flag )
 *   sendfile protocol:
 * 1. line containing the flag, number of bytes, and file name is sent.
 *     This has the format:
 *        "%c%d %s", flag, filesize, name
 *    DFILE (data file) flag is 03,  a CFILE (control file) flag is 02
 * 2. the remote end responds by sending back a single 0 character;
 *    anything else is an error condition and terminates transfer
 * 3. the file is then copied to the remote end and terminated with 0
 * 4. if an error is detected, we send a non-zero character
 *    and close the link.
 ***************************************************************************/

sendfile( fp, name, flag )
	FILE *fp;
	char *name;
	int flag;
{
	int succ;			/* ACME Integer, Inc. */
	struct stat statb;	/* status buffer */

	/*
	 * stat the file
	 */
	if( fstat( fileno(fp), &statb) < 0 ){
		logerr( XLOG_INFO, "sendname: could not stat %s", name );
		return( JFAIL );
	}
	/*
	 * rewind the file
	 */
	if( fseek(fp, 0L, 0) < 0 ){
		logerr_die( XLOG_INFO, "sendfile: fseek failed '%s'", name );
	}
	succ = sendname( fp, name, flag );
	if( succ == JSUCC ){
		succ = Link_copy( fp, (long)statb.st_size, name );
	}
	if( succ == JSUCC ){
		succ = Link_ack( 0 );
	}
	if( succ == JSUCC ){
		succ = Link_confirm();
	}
	if( succ != JSUCC ){
		succ = JFAIL;
		Link_close();
	}
	return( succ );
}

/***************************************************************************
 * Send_error()
 * Called when you have retried a couple of times; bad job, needs attention
 ***************************************************************************/
void
Send_error()
{
	log(XLOG_CRIT, "job cannot be sent from %s to %s", Host, RM );
	exit( 0 );
}

/***************************************************************************
 * sendname( FILE *fp; char *name; int flag )
 *   sendname protocol:
 * 1. line containing the flag and file name is sent
 *     This has the format:
 *        "%c%d %s", flag, filesize, name
 *    DFILE (data file) flag is 03,  a CFILE (control file) flag is 02
 * 2. the remote end responds by sending back a single 0 character;
 *    anything else is an error condition and terminates transfer
 ***************************************************************************/
sendname( fp, name, flag )
	FILE *fp;
	char *name;
	int flag;
{
	struct stat statb;	/* status buffer */
	int succ;			/* ACME Integer, Inc. */
	char buf[BUFSIZ];

	/*
	 * stat the file
	 */
	if( fstat( fileno(fp), &statb) < 0 ){
		logerr( XLOG_INFO, "sendfile: could not stat %s", name );
		return( JFAIL );
	}
	(void)sprintf( buf, "%c%d %s\n", flag, statb.st_size, name);
	succ = Link_send( buf );
	if( succ == JSUCC ){
		succ = Link_confirm();
	}
	return( succ );
}

/***************************************************************************
 * Allsent( )
 *    all files have been sent; send a closing 0 and shut down link
 ***************************************************************************/
void
Allsent()
{
	(void)Link_ack( 0 );
	Link_close();
}
