/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: lpr_parms.c
 * get the parameters for lpr program
 ***************************************************************************
 * Revision History: Created Mon Jan 25 17:29:45 CST 1988
 * $Log:	lpr_parms.c,v $
 * Revision 3.1  88/06/18  09:35:00  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.4  88/05/25  15:42:37  papowell
 * No header flag modification
 * 
 * Revision 2.3  88/05/19  09:00:18  papowell
 * Fixed lint unused variables
 * 
 * Revision 2.2  88/05/14  10:18:40  papowell
 * Use long format for job file names;
 * Added 'fd', no forward flag;
 * Control file has to have hostname and origination agree.
 * 
 * Revision 2.1  88/05/09  10:09:23  papowell
 * PLP: Released Version
 * 
 * Revision 1.8  88/05/09  10:03:47  papowell
 * Revised effects of -h option
 * 
 * Revision 1.7  88/05/05  20:08:45  papowell
 * Added a NOHEADER option that allows user to suppress banner
 * 
 * Revision 1.6  88/04/07  12:31:53  papowell
 * Modified to use Getopt
 * 
 * Revision 1.5  88/04/06  12:13:46  papowell
 * Minor updates, changes in error message formats.
 * Elimination of the AF_UNIX connections, use AF_INET only.
 * Better error messages.
 * 
 * Revision 1.4  88/03/25  15:00:39  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.3  88/03/11  19:28:41  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.2  88/03/05  15:01:51  papowell
 * Minor Corrections,  Lint Problems
 * 
 * Revision 1.1  88/03/01  11:08:50  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: lpr_parms.c,v 3.1 88/06/18 09:35:00 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lpr.h"

/***************************************************************************
 * Setup_parms( int argc; char **argv )
 * 1. get the parameters;
 * 2. set up the get the printcap entry
 * 3. check to see if any parameters violate the printcap restrictions
 * 4. get the sequence number
 ***************************************************************************/

Setup_parms( argc, argv )
	int argc;
	char **argv;
{
	/*
	 * get the parameters
	 */
	Get_parms( argc, argv );
	/*
	 * print the parameters passed
	 */
	if(Debug>4)Show_parms();
	/*
	 * get the printcap entry
	 */
	Get_Printer(1);
	/*
	 * check authorizations
	 */
	Can_use();
	/*
	 * check consistency of parameters
	 */
	Check_parms();
	/*
	 * get the sequence number
	 */
	Get_sequence();
	if(Debug>4)Show_parms();
}
 
/***************************************************************************
 * Get_parms( int argc; char **argv )
 * 1. Scan the argument list and get the flags
 ***************************************************************************/

static char *optstr = "#:C:D:F:J:P:R:T:U:XZ:bcdfghi:lm?nprstvw:";
Get_parms( argc, argv )
	int argc;
	char **argv;
{
	int option;
	int i;

	while( (option = Getopt(argc,argv,optstr)) != EOF ){
		switch( option ){
		case '#':
			Check_int_dup( option, &Copies, Optarg );
			break;
		case 'C':
			Check_str_dup( option, CLASSNAME, Optarg );
			break;
		case 'D':
			Check_int_dup( option, &Debug, Optarg );
			break;
		case 'F':
			if( strlen( Optarg ) != 1 ){
				Diemsg( "bad -F format string '%s'\n",Optarg);
			}
			if( Format ){
				Diemsg( "duplicate format specification -F%s\n", Optarg);
			} else {
				Format = *Optarg;
			}
			break;
		case 'J':
			Check_str_dup( option, JOBNAME, Optarg );
			break;
		case 'P':
			if( Printer ){
				Check_str_dup( option, Printer, Optarg );
			}
			if( *Optarg == 0 ){
				Diemsg( "missing printer name in -P option\n" );
			}
			Printer = Optarg;
			break;
		case 'R':
			Check_str_dup( option, ACCNTNAME, Optarg );
			break;
		case 'T':
			Check_str_dup( option, PRTITLE, Optarg );
			break;
		case 'U':
			Root_only( option );
			Check_str_dup( option, BNRNAME, Optarg );
			(void)strcpy( LOGNAME, Optarg );
			break;
		case 'X':
			Check_dup( option, &Exper );
#			ifdef DEBUG
				Setup_test();
				Tailor_names();
#			else
				Diemsg( "-X not allowed" );
#			endif DEBUG
			break;
		case 'Z':
			Check_str_dup( option, CLASSNAME, Optarg );
			break;
		case 'b':
			Check_dup( option, &Binary );
			break;
		case 'h':
			Check_dup( option, &Noheader );
			/*
			 * set the NOHEADER flag for lpd
			 */
			NOHEADER[0] = 'X';
			break;
		case 'i':
			Check_str_dup( option, INDENT, Optarg );
			break;
		case 'm':
			/*
			 * -m[Mailname]
			 */
			if( Optarg == 0 ){
				Check_str_dup( option, MAILNAME, Person );
				(void)sprintf(MAILNAME,"%s@%s",Person,Host);
			} else {
				Check_str_dup( option, MAILNAME, Optarg );
			}
			break;
		case 'r':
			Check_dup( option, &Remove );
			break;
		case 's':
			Check_dup( option, &Use_links );
			break;
		case 'c':
		case 'd':
		case 'f':
		case 'g':
		case 'l':
		case 'n':
		case 'p':
		case 't':
		case 'v':
			if( Format ){
				Diemsg( "duplicate format specification -%c\n", option);
				exit( 1 );
			} else {
				Format = option;
			}
			break;
		case '?':
			break;
		case 'w':
			Check_str_dup( option, PWIDTH, Optarg );
			i = atoi( PWIDTH );
			if( i <= 0 ){
				Diemsg( "-w <width>, width value '%s' bad", PWIDTH );
				exit( 1 );
			}
			(void)sprintf( PWIDTH, "%d", i );
			break;
		default:
			fatal(XLOG_INFO, "Get_parms: badparm %c", option );
		}
	}

	/*
	 * set up the Parms[] array
	 */
	for( ; Optind < argc; ++Optind ){
		if( Parmcount < MAXPARMS ){
			Parms[Parmcount].str = argv[Optind];
			++Parmcount;
		} else {
			Diemsg( "too many files to print; break job up" );
		}
	}
	/*
	 * set default format
	 */
	if( Format == 0 ){
		Format = 'f';
	}
	if( FROMHOST[0] == 0 ){
		(void)strcpy(FROMHOST, Host);
	}
}

/***************************************************************************
 * Check_int_dup( int option, int *value, char *arg )
 * 1.  check to see if value has been set
 * 2.  if not, then get integer value from arg
 ***************************************************************************/
 
Check_int_dup( option, value, arg )
	int option;
	int *value;
	char *arg;
{
	if( *value ){
		Diemsg( "duplicate option %c", option );
	}
	if( sscanf( arg, "%d", value ) != 1 || *value <= 0){
		Diemsg( "option %c parameter (%s) is not positive integer",
			option, arg);
	}
}
/***************************************************************************
 * Check_str_dup( int option, char *value, char *arg )
 * 1.  check to see if value has been set
 * 2.  if not, then set it
 ***************************************************************************/
Check_str_dup( option, value, arg )
	int option;
	char *value;
	char *arg;
{
	if( *value ){
		Diemsg( "duplicate option %c", option );
	}
	if( strlen( arg ) > MAXPARMLEN ){
		Diemsg( "option %c arguement too long (%s)", option, arg );
	}
	(void)strcpy(value, arg );
}
/***************************************************************************
 * Check_dup( int option, int *value )
 * 1.  check to see if value has been set
 * 2.  if not, then set it
 ***************************************************************************/
Check_dup( option, value )
	int option;
	int *value;
{
	if( *value ){
		Diemsg( "duplicate option %c", option );
	}
	*value = 1;
}
/***************************************************************************
 * Root_only( int option );
 * 1.  check to see if root
 ***************************************************************************/
Root_only( option )
	int option;
{
	if( Is_root == 0 ){
		Diemsg( "option %c can be used only by root", option );
	}
}
/***************************************************************************
 * Can_use()
 * Check for permissions and priority
 * 1. RG - restrict access by group: must be in group
 * 2. printcap: check for restrictions
 * 3. XU: check for restrictions
 ***************************************************************************/
Can_use()
{
	int prior;		/* find the max priority */
	int op;	/* 'R' for lpr */
	char buf[BUFSIZ];
	char *pf;		/* XU perm file */

	/*
	 * get the full path name
	 */
	pf = XU;
	if( pf && *pf && pf[0] != '/' ){
		(void)sprintf( buf, "%s/%s", SD, pf );
		pf = buf;
	}
	/*
	 * check for priority
	 */
	Priority = CLASSNAME[0];
	if( Priority == 0 ){
		Priority = 'Z';
	}
	if( !isascii(Priority) || !isupper(Priority) ){
		Priority = 'Z';
		Warnmsg( "Priority set to %c", Priority );
	}
	/*
	 * check to see if we have access restricted by group perms
	 */
	if( !Is_root && RG && *RG && !Checkgrp( Person,RG ) ){
		Diemsg( "Sorry %s@%s, you don't have permission to use the %s",
			Person, Host, Printer );
	}
	/*
	 * check to see if we have restricted access
	 */
	prior = Priority;
	op = 'R';	/* able to use lpr */
	if((Permfile && *Permfile
		&& !Checkperm(Permfile,Host,Person,First_name,&op,(int *)0,0))
		||(pf && *pf
			&& !Checkperm(pf,Host,Person,First_name,&op,(int *)0,0 ) )){
		Diemsg( "Sorry %s@%s, you don't have permission to use '%s'",
			Person, Host, Printer );
	}
	if((Permfile && *Permfile
			&& !Checkperm(Permfile,Host,Person,First_name,&op,&prior,1 ))
		||(pf && *pf
			&& !Checkperm(pf,Host,Person,First_name,&op,&prior,1 ) )){
		Diemsg( "Sorry %s@%s, no more pages allowed on '%s'",
			Person, Host, Printer );
	}
	/*
	 * now check to see if you have not exceeded your page limits
	 */
	if( prior > Priority ){
		Warnmsg( "maximum Priority allowed is %c", prior );
		Priority = prior;
	}
	if(Debug>2)log(XLOG_DEBUG,"Can_use: %s can use %s, priority %c",
		Person, Printer, Priority );
}

/***************************************************************************
 * Checkgrp( name, list )
 * -- check to see if person is a member of one of the groups
 * Returns: 1 if  person is a member, 0 if not
 ***************************************************************************/

Checkgrp( name, list )
	char *name, *list;
{
	char buf[BUFSIZ];
	char *cp, *bp;
	struct group *gr;
	char **grplist;

	if(Debug>4)log(XLOG_DEBUG,"Checkgrp: checking for %s in %s", name, list );
	cp = list;
	while( cp && *cp ){
		for( bp = buf; *cp && (*bp = *cp) && (*bp != ','); ++bp, ++cp );
		*bp = 0;
		if( *cp ){
			++cp;
		}
		if(Debug>5)log(XLOG_DEBUG,"Checkgrp: checking group %s", buf );
		if( gr = getgrnam(buf) ){
			for( grplist = gr->gr_mem; *grplist; ++grplist ){
				if( strcmp( name, *grplist ) == 0 ){
					if(Debug>4)log(XLOG_DEBUG,"Checkgrp: %s in %s",
						name, gr->gr_name );
					return(1);
				}
			}
		}
	}
	if(Debug>4)log(XLOG_DEBUG,"Checkgrp: %s not in any group" );
	return( 0 );
}
/***************************************************************************
 * Check_parms()
 *	Check for parameter consistency
 *  1. Check to see if the format is allowed
 *  2. Check on the multiple copies
 ***************************************************************************/
Check_parms()
{
	char *msg;
	/* check format and options */
	if( FX && *FX && index( FX, Format ) == 0 ){
		msg=NULL;
		switch(Format){
		case 'f': msg = "Normal or plain files";	break;
		case 't': msg = "Troff files, use -Fn for Ditroff";	break;
		case 'n': msg = "Ditroff files, use -Ft for (old) troff files";break;
		case 'v': msg = "Varian raster images";		break;
		case 'd': msg = "TeX intermediate files (DVI)";	break;
		case 'g': msg = "Plot intermediate files";	break;
		case 'c': msg = "Cifplot intermediate files";	break;
		default:  msg = "format 'X' files"; msg[8]=Format;
		}
		Diemsg("Printer %s does not know how to interpret %s\n", Printer, msg);
	}
	if( SC && Copies > 1){
		Warnmsg("multiple copies are not allowed");
		Copies = 0;
	}
	if(MC > 0 && Copies > MC){
		Warnmsg("only %d copies are allowed", MC);
		Copies = MC;
	}
	if( Remove && !Is_root
		&& (LN == 0 || *LN == 0 || !Checkgrp( Person, LN ))){
		Warnmsg( "-r (remove) not allowed" );
		Remove = 0;
	}
#ifdef NOSYMLINK
	if( Use_links ){
		Warnmsg("-s (symbolic links) disabled");
		Use_links = 0;
	}
#else
	if( Use_links ){
		/*
		 * check to see if we are a member of the right group
		 */
		if( !Is_root && (LN == 0 || *LN == 0 || !Checkgrp( Person, LN ) )){
			Warnmsg("-s (symbolic links) not allowed by user %s", Person);
			Use_links = 0;
		} else if( RM && NW ){
			Warnmsg("-s (symbolic links) not available for this queue" );
			Use_links = 0;
		}
	}
#endif NOSYMLINK
	if( Remove && !Use_links ){
		Warnmsg( "-r (remove) not allowed without -s" );
		Remove = 0;
	}
}

/***************************************************************************
 * Get_sequence()
 * Get the job sequence number
 * 1. Check to see if queuing is enabled
 * 2. Get the job number from the .job file
 ***************************************************************************/

Get_sequence()
{
	char buf[MAXPATHLEN];		/* holds the pathname */
	FILE *fp;				/* for sequence number */
	int i;					/* waiting time */

	/*
	 * check to see if printing is enabled
	 */
	if( LO == 0 || *LO == 0 || SD == 0 || *SD == 0 ){
		fatal( XLOG_CRIT, "Get_sequence: bad printcap entry" );
	}
	if( LO[0] == '/' ){
		(void)strcpy( buf, LO );
	} else {
		(void)sprintf( buf, "%s/%s", SD, LO );
	}
	/*
	 * get the sequence file name
	 */
	(void)sprintf( buf, "%s/.seq.%s", SD, Host );
	if(Debug>3)log(XLOG_DEBUG,"sequence file name '%s'", buf );
	/*
	 * lock the sequence file and get a new number
	 */
	for( i = 0;
		(fp = Getlockfile( buf, &Job_number,(char *)0,0,&LO_statb )) == NULL
			&& i < 3;
		++i ){
		sleep( (unsigned)(i+ (getpid()&1) ) );
	}
	if( fp == NULL ){
		Diemsg("cannot lock sequence file %s, try later (%s)", buf,
			Errormsg( errno ) );
	}
	if( !Is_root && (LO_statb.st_mode & DISABLE_QUEUE) ){
		Diemsg( "printer %s- queueing disabled", Printer );
	}
	/*
	 * set the sequence number to the new value mod 1000;
	 */
	Job_number = (Job_number+1) % 1000;
	Setlockfile( buf, fp, Job_number, Time_str() );
	(void)fclose(fp);
	if(Debug>4)log(XLOG_DEBUG, "Get_sequence: number %d", Job_number);
}

/***************************************************************************
 * Show_parms()
 * Display the values of the parameters that were read.
 * 1. Print the CFparm array
 * 2. Print the other parameters
 ***************************************************************************/

Show_parms()
{
	int i;
	char *s;

	/*
	 * CFparm first:
	 */
	(void)fprintf(stderr,"CFparm:\n");
	for( i = 'A'; i <= 'Z'; ++i ){
		s = CFparm[i-'A'];
		if( s[0] ){
			(void)fprintf(stderr," %c  '%s'\n", i, s );
		}
	}
	(void)fprintf(stderr, " Format: %c\n",  Format );
	(void)fprintf(stderr, " Copies: %d\n",  Copies );
	(void)fprintf(stderr, " Binary: %d\n",  Binary );
	(void)fprintf(stderr, " Use_links: %d\n",  Use_links );
	(void)fprintf(stderr, " Exper: %d\n",  Exper );
	(void)fprintf(stderr, " Priority: %d\n",  Priority );
	(void)fprintf(stderr, " Job_number: %d\n",  Job_number );
	if( Read_stdin ){
		(void)fprintf(stderr, " Read_stdin: %s\n",  Read_stdin );
	}
	if( Filter_out ){
		(void)fprintf(stderr, " Filter_out: %s\n",  Filter_out );
	}
	(void)fprintf(stderr, " Temp_count: %d\n",  Temp_count );
	(void)fprintf(stderr, " Temp_max: %d\n",  Temp_max );
	for( i = 0; i < Temp_count; ++i ){
		s  = Temp_file[i];
		(void)fprintf(stderr," %d  '%s'\n", i, s );
	}
}
