/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: Control_ops.c
 ***************************************************************************
 * Revision History: Created Mon Jan 18 07:48:12 CST 1988
 * $Log:	control_ops.c,v $
 * Revision 3.2  88/06/24  17:55:14  papowell
 * MODS for VAX 4.3BSD UNIX
 * 
 * Revision 3.1  88/06/18  09:34:02  papowell
 * Version 3.0- Distributed Sat Jun 18 1988
 * 
 * Revision 2.2  88/05/14  10:21:23  papowell
 * Modified -X flag handling
 * 
 * Revision 2.1  88/05/09  10:07:50  papowell
 * PLP: Released Version
 * 
 * Revision 1.9  88/04/28  11:02:11  papowell
 * removed unused variables,  shuts up lint
 * 
 * Revision 1.8  88/04/27  17:59:14  papowell
 * Added debugging statements to C_lpq, C_lprm
 * 
 * Revision 1.7  88/04/07  12:31:32  papowell
 * 
 * Revision 1.6  88/04/06  12:12:24  papowell
 * Minor updates, changes in error message formats.
 * Elimination of the AF_UNIX connections, use AF_INET only.
 * Better error messages.
 * 
 * Revision 1.5  88/03/25  14:59:13  papowell
 * Debugged Version:
 * 1. Added the PLP control file first transfer
 * 2. Checks for MX during file transfers
 * 3. Found and fixed a mysterious bug involving the SYSLOG facilities;
 * 	apparently they open files and then assume that they will stay
 * 	open.
 * 4. Made sure that stdin, stdout, stderr was available at all times.
 * 
 * Revision 1.4  88/03/12  10:03:38  papowell
 * *** empty log message ***
 * 
 * Revision 1.3  88/03/11  19:29:57  papowell
 * Minor Changes, Updates
 * 
 * Revision 1.2  88/03/05  15:01:56  papowell
 * Minor Corrections,  Lint Problems
 * 
 * Revision 1.1  88/03/01  11:08:19  papowell
 * Initial revision
 * 
 ***************************************************************************/
#ifndef lint
static char id_str1[] =
	"$Header: control_ops.c,v 3.2 88/06/24 17:55:14 papowell Exp $ PLP Copyright 1988 Patrick Powell";
#endif lint

#include "lp.h"

/***************************************************************************
 * Control_ops()
 *  1. get the printcap entries
 *  2. check for permissions
 *  3. determine the function to be carried out
 *  4. carry out the function
 ***************************************************************************/
static int op_init;		/* flag for operation initialization */


/*
 * dispatch is a data structure that contains the names and flagss
 * of control commands.
 */
struct dispatch{
	char *name;
	int distinct;
	int (*routine)();
	int flags;
	char *summary;
};

int C_bad(), C_help(), C_abort(), C_clean(), C_disable(),
	C_enable(), C_exit(), C_kill(), C_restart(), C_lpq(), C_lprm(),
	C_start(), C_status(), C_stop(), C_topq(), C_remote(), C_lpd();

#define NEED_OPT		1	/* we check for other options */
#define ALL_ALLOWED		2	/* check for all */
#define NEED_PRIV		4	/* privileged operation */
#define ALL_DEF			8	/* if no parameter, all is default */
#define NO_PR_LIST		0x10	/* printer list */
#define REMOTE_OP		0x20	/* remote allowed */
#define SERVER_OP		0x40	/* server allowed */
#define ON_REMOTE			0x80	/* not if you have RM and NW */
#define IS_OPT(x) ((x)&NEED_OPT)
#define IS_ALL(x) ((x)&ALL_ALLOWED)
#define IS_PRIV(x) ((x)&NEED_PRIV)
#define IS_ALL_DEF(x) ((x)&ALL_DEF)
#define IS_NO_PR_LIST(x) ((x)&NO_PR_LIST)
#define IS_REMOTE(x) ((x)&REMOTE_OP)
#define IS_SERVER(x) ((x)&SERVER_OP)
#define IS_ON_REMOTE(x) ((x)&ON_REMOTE)

static struct dispatch
dispatch[] = {
{ "", 0, 0, 0, "not a command" },
{ "?", 1, C_help, 0, "? is same as 'help'" },
{ "abort", 1, C_abort,
	NEED_OPT | NEED_PRIV |ALL_ALLOWED |REMOTE_OP |SERVER_OP |ON_REMOTE,
	"abort ( all | Printer ): kill off server and disable printing" },
{ "clean", 1, C_clean, NEED_OPT | NEED_PRIV |ALL_ALLOWED |REMOTE_OP |ON_REMOTE,
	"clean ( all | Printer ): remove all crud from queue (dangerous!)"},
{ "disable", 1, C_disable, NEED_OPT | NEED_PRIV |ALL_ALLOWED |REMOTE_OP |ON_REMOTE,
	"disable ( all | Printer ): disable queueing"},
{ "enable", 2, C_enable, NEED_OPT | NEED_PRIV |ALL_ALLOWED |REMOTE_OP |ON_REMOTE,
	"enable ( all | Printer ): enable queuing"},
{ "exit", 2, C_exit, 0, "exit: terminate" },
{ "help", 1, C_help, 0 ,"help [all] [command]: print command summary"},
{ "kill", 1, C_kill,
	NEED_OPT | NEED_PRIV |ALL_ALLOWED |REMOTE_OP |SERVER_OP |ON_REMOTE,
	"kill ( all | Printer ): kill off server and then restart printing" },
{ "lpd", 3, C_lpd, 0,
	"lpd : check out lpd process"},
{ "lpq", 3, C_lpq, 0,
	"lpq ( parms ) : call lpq"},
{ "lprm", 3, C_lprm, 0,
	"lprm ( parms ) : call lpq"},
{ "quit", 1, C_exit, 0, "quit (same as exit): terminate" },
{ "remote", 3, C_remote, 0,
	"remote command: do the action on remote printer"},
{ "restart", 3, C_restart, NEED_OPT | ALL_ALLOWED |REMOTE_OP |SERVER_OP,
	"restart ( all | Printer ): start a server"},
{ "start", 4, C_start,
	NEED_OPT | NEED_PRIV |ALL_ALLOWED |REMOTE_OP |SERVER_OP |ON_REMOTE,
	"start ( all | Printer ): enable printing and start server"},
{ "status", 4, C_status, ALL_DEF |ALL_ALLOWED | NEED_OPT,
	"status [all] [Printer]: print status"},
{ "stop", 3, C_stop,
	NEED_OPT | NEED_PRIV |ALL_ALLOWED |REMOTE_OP |SERVER_OP |ON_REMOTE,
	"stop ( all | Printer ): disable further printing"},
{ "topq", 1, C_topq, NO_PR_LIST | NEED_OPT |REMOTE_OP | ON_REMOTE| NEED_PRIV,
	"topq Printer (Host | jobnumber)*: move job to top of queue"},
};
int ncmds = sizeof(dispatch)/sizeof(struct dispatch);

/***************************************************************************
 * Control_ops()
 * This routine is called with the parameters in the Parms[] array.
 * It will check to see if the command is valid,  and if so, will
 * do any other actions.
 ***************************************************************************/
int
Control_ops()
{
	int command;		/* command we are going to do */
	char **list;		/* list of Printers */
	int i, all;			/* doing for all Printers */

	/*
	 * set the flagss needed
	 */
	Is_local = ( strcmp( From, Host ) == 0 );
	Is_root  = ( strcmp( Person, "root" ) == 0);
	op_init = 0;
	command = 0;
	all = 0;
	/*
	 * check to see that we have a valid command
	 */
	if(Debug>4)Show_ops();
	if( Parmcount < 1 ){
		(void)fprintf( stdout, "no command\n" );
		return( 0 );
	}
	command = find_command( Parms[0].str );
	if( command == 0 ){
		(void)fprintf( stdout, "unknown command %s\n", Parms[0].str );
		return( 0 );
	}
	if(Debug>3)log(XLOG_DEBUG,"Control_ops: command %s, flags 0x%x, %s",
	dispatch[command].name,dispatch[command].flags,dispatch[command].summary);
	if( !Is_local && !IS_REMOTE(dispatch[command].flags) ){
		log(XLOG_INFO,"command %s cannot be done remotely", Parms[0].str);
		return( 0 );
	}

	/*
	 * call the appropriate routine
	 */
	Shift_parms(1);
	/*
	 * If no option processing just dispatch
	 */
	if( !IS_OPT(dispatch[command].flags) ){
		return( (*dispatch[command].routine)(&dispatch[command]));
	}
	/*
	 * if no options and IS_ALL is the default, set up the all flag
	 */
	if( Parmcount == 0 ){
		if( IS_ALL_DEF(dispatch[command].flags) ){
			all = 1;
		} else {
			(void)fprintf( stdout, "no parameters, use: %s\n",
				dispatch[command].summary);
			return( 0 );
		}
	} else if( strcmp( Parms[0].str, "all" ) == 0 ){
		if( IS_ALL(dispatch[command].flags) ){
			all = 1;
		} else {
			(void)fprintf( stdout, "'all' not allowed, use: %s\n",
				dispatch[command].summary);
			return( 0 );
		}
	}
	/*
	 * set up Printer from parameters
	 */
	if( all ){
		if(Debug>2)log(XLOG_DEBUG,"'all' parameter" );
		for( list = All_printers(); *list; ++list ){
			Printer = *list;
			if( service( &dispatch[command] ) == 0 ){
				return( 0 );
			}
		}
	} else {
		for( i = 0; i < Parmcount; ++i ){
			Printer = Parms[0].str;
			Shift_parms(1);
			if( service( &dispatch[command] ) == 0 ){
				return( 0 );
			}
			if( IS_NO_PR_LIST(dispatch[command].flags) ){
				break;
			}
		}
	}
	return( 1 );
}

/***************************************************************************
 * service( struct dispatch *cmd );
 * 1. check on the printcap entry
 * 2. chdir to the spool directory
 * 3. dispatch the particular routine
 ***************************************************************************/

service( cmd )
	struct dispatch *cmd;
{
	char *st, *ps;
	int perms = 'C';	/* Permission Checking */

	if(Debug>4)log(XLOG_DEBUG,"service: printer '%s', cmd '%s'",
		Printer, cmd->name );
	if(Get_pc_entry(Printer, Status_pc_vars, Status_pc_len) == 0){
		log(XLOG_INFO,"service: Printer %s does not have printcap entry", Printer);
		return( 0 );
	}
	if( RM && NW && IS_ON_REMOTE(cmd->flags) ){
		(void)fprintf( stdout,
	"remote printer and network file system, use \'remote %s %s\'\n",
		cmd->name, Printer );
		return( 0 );
	}
	/*
	 * we may have a server specified
	 */
	if( SS && *SS ){
		ps = PS;
		st = ST;
		if( Set_pc_entry(SS, Status_pc_vars, Status_pc_len) == 0){
			log(XLOG_INFO,"service: Server %s queue %s does not have printcap entry",
			Printer,SS);
			return( 0 );
		}
		PS = ps;
		ST = st;
		LO = Printer;
		SV = 0;
	}
	if( SD == 0 || *SD == 0 ){
		log( XLOG_INFO,"service: Printer %s does not have spool directory", Printer);
		return( 0 );
	}
	/* chdir to spool directory */
	if (chdir(SD) < 0) {
		logerr( XLOG_INFO,"service: cannot chdir to %s", SD);
		return( 0 );
	}
	/*
	 * check on the privileges needed
	 * You have to be local and root OR have C privs
	 */
	if( IS_PRIV(cmd->flags) && !( Is_local && Is_root )
			&& !( Permfile && *Permfile
			  && Checkperm(Permfile,From,Person,First_name,&perms,(int *)0,0))){
		(void)fprintf(stdout,
			"You do not have Printer control perms on '%s'\n", Printer);
		return( 0 );
	}
	return( (*cmd->routine)(cmd));
}

/***************************************************************************
 * int find_command( char *str )
 * look in the command table for a match.  The "distinct" entry is used
 * to determine the numbers of characters for a match.
 ***************************************************************************/

int
find_command( str )
	char *str;
{
	int i;

	for( i = 1; i < ncmds; ++ i ){
		if( strncmp( str, dispatch[i].name, dispatch[i].distinct ) == 0
			&& strncmp( str, dispatch[i].name, strlen(str)) == 0 ){
			return( i );
		}
	}
	return( 0 );
}

/***************************************************************************
 * C_bad()
 * Cannot decide what to do with the command
 ***************************************************************************/
int
C_bad()
{
	(void)fprintf(stdout, "bad command %s\n", Parms[0].str );
	return( 1 );
}
/***************************************************************************
 * C_help()
 * Print a help message for each command listed,
 * or a simple list of commands
 ***************************************************************************/
int
C_help()
{
	int i, cmd;

	if( Parmcount < 2 || strcmp( Parms[1].str, "all" ) == 0 ){
		for( i = 1; i < ncmds; ++i ){
			Print_sum( i );
		}
	} else {
		for( i = 1; i < Parmcount; ++i ){
			cmd = find_command( Parms[i].str );
			if( cmd > 0 ){
				Print_sum( cmd );
			} else {
				(void)fprintf(stdout, "not a command: %s", Parms[i].str );
			}
		}
	}
}

/***************************************************************************
 * Print_sum( cmd )
 * prints the command summary line
 ***************************************************************************/
Print_sum( cmd )
{
	(void)fprintf(stdout, "%s\n", dispatch[cmd].summary );
}


/***************************************************************************
 * C_exit()
 * terminate gracefully
 ***************************************************************************/
C_exit()
{
	exit( 0 );
}
/***************************************************************************
 * C_stop()
 * Sets the DISABLE_PRINT perm bit in the lockfile perms
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
C_stop()
{
	int s;

	if(Debug>4)log(XLOG_DEBUG,"C_stop: printer %s, lock '%s'", Printer, LO );
	(void)Checklockfile( LO, (int *)0,(char *)0,0,&LO_statb );
	if((s=chmod_daemon(LO,(int)(LO_statb.st_mode&0777) |DISABLE_PRINT)) < 0){
		logerr(XLOG_INFO,"cannot chmod lockfile %s", LO);
	} else {
		(void)fprintf(stdout,"%s: printing disabled\n", Printer);
	}
	return( s >= 0 );
}
/***************************************************************************
 * C_start()
 * 1. Clears the DISABLE_PRINT perm bit in the lockfile perms
 * 2. Starts the Printer
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
C_start()
{
	int s;

	if(Debug>4)log(XLOG_DEBUG,"C_start: printer %s, lock '%s'", Printer, LO );
	(void)Checklockfile( LO, (int *)0,(char *)0,0,&LO_statb );
	if((s=chmod_daemon(LO, (int)(LO_statb.st_mode & ENABLE_PRINT))) < 0){
		logerr(XLOG_INFO,"cannot chmod lockfile %s", LO);
	} else {
		(void)fprintf(stdout,"%s: printing enabled\n", Printer);
	}
	/*
	 * start the server
	 */
	if( s >= 0 ){
		s = C_restart();
	}
	return( s >= 0 );
}
/***************************************************************************
 * C_disable()
 * Sets the DISABLE_QUEUE perm bit in the lockfile perms
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
C_disable()
{
	int s;

	if(Debug>4)log(XLOG_DEBUG,"C_disable: printer %s, lock '%s'", Printer, LO );
	(void)Checklockfile( LO, (int *)0,(char *)0,0,&LO_statb );
	if((s=chmod_daemon(LO,(int)(LO_statb.st_mode&0777) | DISABLE_QUEUE)) < 0){
		logerr(XLOG_INFO,"cannot chmod lockfile %s", LO);
	} else {
		(void)fprintf(stdout,"%s: queueing disabled\n", Printer);
	}
	return( s >= 0 );
}
/***************************************************************************
 * C_enable()
 * 1. Clears the DISABLE_QUEUE perm bit in the lockfile perms
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
C_enable()
{
	int s;

	if(Debug>4)log(XLOG_DEBUG,"C_enable: printer %s, lock '%s'", Printer, LO );
	(void)Checklockfile( LO, (int *)0,(char *)0,0,&LO_statb );
	if((s=chmod_daemon(LO, (int)(LO_statb.st_mode & ENABLE_QUEUE))) < 0){
		logerr(XLOG_INFO,"cannot chmod lockfile %s", LO);
	} else {
		(void)fprintf(stdout,"%s: queueing enabled\n", Printer);
	}
	return( s >= 0 );
}
/***************************************************************************
 * C_restart()
 * 1. Attempts to fire up the server
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
C_restart()
{
	/*
	 * start the server
	 */

	if(Debug>4)log(XLOG_DEBUG,"C_restart: printer %s, lock '%s'", Printer, LO );
	(void)Startserver();
	return( 1 );
}

/*
 * killserver(): kills the current server to stop printing
 */
static int
killserver()
{
	int s;		/* Success of operation */
	int pid;	/* server PID */
	/*
	 * Kill the current server to stop printing now.
	 */
	s = 1;
	if( Checklockfile( LO, &pid, (char *)0, 0,&LO_statb )){
		if( kill(pid, SIGINT) < 0 ){
			(void)logerr(XLOG_INFO,"server (pid %d) not killed", pid);
			s = 0;
		} else {
			(void)fprintf(stdout,"%s: server (pid %d) killed\n",
				Printer, pid);
		}
	} else {
		(void)fprintf(stdout,"%s: no server present\n", Printer);
	}
	return(s);
}
/***************************************************************************
 * C_abort()
 * 1. Does C_stop()
 * 2. kills off server if there is one
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
C_abort()
{
	int s;		/* Success of operation */

	if(Debug>4)log(XLOG_DEBUG,"C_abort: printer %s, lock '%s'", Printer, LO );
	s = C_stop();
	if( s ){
		s = killserver();
	}
	return( s );
}
/***************************************************************************
 * C_kill()
 * 1. Does C_abort()
 * 2. Does C_start()
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
C_kill()
{
	int s;		/* Success of operation */

	if(Debug>4)log(XLOG_DEBUG,"C_kill: printer %s, lock '%s'", Printer, LO );
	s = C_abort();
	if( s ){
		s = C_start();
	}
	return( s );
}

/***************************************************************************
 * C_clean()
 * 1. Removes all entries in the specified spool directory.
 * Returns: 1 if successful, 0 if not;
 ***************************************************************************/
C_clean()
{
	int c;				/* ACME Integers, Inc. */
	DIR *dirp;
	struct direct *d;

	if(Debug>4)log(XLOG_DEBUG,"C_clean: printer %s, lock '%s'", Printer, LO );
	if ((dirp = opendir(SD)) == NULL) {
		logerr(XLOG_INFO,"cannot examine spool directory %s");
		return( 0 );
	}
	while ((d = readdir(dirp)) != NULL) {
		c = d->d_name[0];
		if ((c == 'c' || c == 't' || c == 'd') && d->d_name[1]=='f') {
			if (unlink_daemon(d->d_name) < 0){
				(void)fprintf(stdout,"cannot remove %s\n", d->d_name);
			} else {
				(void)fprintf(stdout,"removed %s\n", d->d_name);
			}
		}
	}
	closedir(dirp);
	return( 1 );
}
/*
 * Print the status of each queue listed or all the queues.
 */
static char *hdr_format =  "%-12.12s %-4s  %-10s %s\n";
static char *data_format = "%-12.12s %4d  %-10s %s%s%s\n";
C_status()
{
	int active, pid;			/* active server and its pid */
	char buf[BUFSIZ];			/* buffer */
	char *prstat, *actstat, *qstat, *rqstat;	/* Printer and queue status */
	char *st, *ps;
	char sbuf[BUFSIZ];			/* status buffer */
	char servers[BUFSIZ];
	char *sp, *ep, *sr;			/* ACME Pointers, Inc. */

	if(Debug>3)log(XLOG_INFO,"C_status: printer %s", Printer);
	if(Get_pc_entry(Printer, Status_pc_vars, Status_pc_len) == 0){
		log(XLOG_INFO,"Printer %s does not have printcap entry", Printer);
		return( 0 );
	}
	/* we have a server here */
	if( SS && *SS ){
		ps = PS;
		st = ST;
		if( Set_pc_entry(SS, Status_pc_vars, Status_pc_len) == 0){
			log(XLOG_INFO,"Server %s queue %s does not have printcap entry",
				Printer,SS);
			return( 0 );
		}
		PS = ps;
		ST = st;
		LO = Printer;
		SV = 0;
	}
	if( SD == 0 || *SD == 0 ){
		log( XLOG_INFO,"Printer %s does not have spool directory", Printer);
		return( 0 );
	}
	/* chdir to spool directory */
	if (chdir(SD) < 0) {
		logerr( XLOG_INFO,"cannot chdir to %s", SD);
		return( 0 );
	}
	/*
	 * start by getting active server information
	 */
	buf[0] = 0;
	active = Checklockfile( LO, &pid, buf, sizeof( buf),&LO_statb);

	/*
	 * get numbers of jobs in queue
	 */

	Jobcount = Getq();

	/*
	 * now format the info appropriately
	 */
	qstat = (LO_statb.st_mode & DISABLE_QUEUE) ? "disabled " : "enabled ";
	prstat = ( LO_statb.st_mode & DISABLE_PRINT) ? "disabled " : "enabled ";
	/*
	 * get active server
	 */
	if(Debug>4)log(XLOG_DEBUG,"C_status: active '%d', Jobcount '%d', '%s'",
		active, Jobcount, buf );
	if( SV == 0 || *SV == 0 ){
		if( Jobcount == 0 && active == 0 ){
			(void)sprintf( sbuf,"" );
		} else if( Jobcount == 0 && active ){
			(void)sprintf( sbuf,"(server %d)", pid, buf );
		} else if( Jobcount && active == 0 ){
			(void)sprintf( sbuf,"(no server)" );
		} else {
			(void)sprintf( sbuf,"(server %d, job %s)", pid, buf );
		}
		actstat = sbuf;
	} else {
		(void)strcpy( servers, SV );
		(void)sprintf( sbuf, hdr_format, "", "", "X", "" );
		if( (sp = index( sbuf, 'X' ) ) == 0 ){
			fatal(XLOG_INFO,"C_status: header format bad" );
		}
		*sp = 0;
		for( sp = servers; sp; sp = ep ){
			ep = index( sp, ',');
			if( ep ){
				*ep = 0;
				ep = ep + 1;
			}
			active = Checklockfile( sp, &pid,buf,sizeof(buf),&LO_statb );
			if( LO_statb.st_mode & DISABLE_PRINT){
				sr = "dis";
			} else {
				sr = "enb";
			}
			if( active == 0 ){
				(void)sprintf( sbuf+strlen(sbuf),"(%s: %s, no server)",sp, sr);
			} else {
			(void)sprintf( sbuf+strlen(sbuf),"(%s: %s, %d, job %s)",
				sp, sr, pid, buf );
			}
		}
		actstat = sbuf;
	}
	if( LO_statb.st_mode & FORCE_REQUE ){
			rqstat = ", reorder requested";
	} else {
			rqstat = "";
	}
	/* displays heading if not already displayed */
	if (!op_init) {
		(void)fprintf(stdout,hdr_format,
			SS ? "Server" : "Queue", "Jobs", "Queueing", "Printing" );
		op_init = 1;
	}
	/* prints the queue status */
	(void)fprintf(stdout,data_format, Printer, Jobcount, qstat,
			prstat,rqstat,actstat);
	/* prints the Printer status */
	printstatus();
	(void)fflush(stdout);
	return(1);
}

/*
 * Put the specified jobs at the top of Printer queue.
 */
C_topq()
{
	int i;					/* ACME Integer, Inc. */
	int changed;			/* We changed the queue */
	char *cfname;			/* control file name */

	(void)fprintf(stdout,"%s:\n", Printer);

	/*
	 * put the parameters in the list in the correct position
	 */
	if( Parmcount == 0 ){
		if(Debug>4)log(XLOG_DEBUG,"C_topq: no parameters" );
		return( 0 );
	}
	if(Debug>4)log(XLOG_DEBUG,"C_topq: '%s'(%d)", Parms[0].str, Parms[0].num );
	/* get number of jobs in the queue */
	Jobcount = Getq();
	/*
	 * go through the queue and find the jobs to be promoted
	 */
	changed = 0;
	for( i = 0; i < Jobcount; ++i ){
		if(Match_entry( &Queue[i] )){
			/*
			 * Reposition the job by setting its priority to high level
			 * and then fix the control file
			 */
			if(promote(Queue[i].q_name)) {
				changed++;
				Queue[i].q_name[0] = 0;
			}
		}
	}
	/*
	 * Put the other high priority jobs lower
	 */
	if( !changed ){
		return( 1 );
	}
	for( i = 0; i < Jobcount; i++) {
		cfname = Queue[i].q_name;
		if(*cfname && cfname[2] == 'A'){
			touch(cfname);
		}
	}
	(void)fprintf(stdout, "queue order changed\n");
	/*
	 * Turn on the public execute bit of the lock file to
	 * get lpd to rebuild the queue after the current job.
	 */
	(void)Checklockfile( LO, (int *)0,(char *)0,0,&LO_statb );
	if( chmod_daemon(LO, (int)(LO_statb.st_mode & 0777) | FORCE_REQUE) < 0 ){
		logerr( XLOG_INFO, "cannot force requeue on lockfile %s", LO );
		return( 0 );
	}
	return( 1 );
} 

/***************************************************************************
 * promote( char *cfname )
 * promote a file to the A priority, and head of the list.
 ***************************************************************************/
promote(cfname)
	char *cfname;
{
	char buf[BUFSIZ];

	(void)strcpy( buf, cfname);
	buf[STARTPR] = 'A';
	if(Debug>0)log( XLOG_DEBUG, "renaming %s to %s", cfname, buf );
	if( strcmp( buf, cfname) && rename( cfname, buf ) < 0 ){
		logerr( XLOG_INFO,"cannot rename %s to %s", cfname, buf );
		return(0);
	}
	(void)fprintf( stdout, "promoting %s to %s\n", cfname, buf );
	return( 1 );
}


/* 
 * Change the modification time of the file.
 *	Returns boolean if successful.  
 */
touch(cfname)
	char *cfname;
{
	FILE *fp;
	int i;

	if( (fp = Lockcf(cfname)) == NULL ){
		logerr(XLOG_INFO,"cannot open %s\n", cfname);
	} else if( (i = getc(fp)) == EOF ){
		logerr(XLOG_INFO,"cannot read %s\n", cfname);
	} else if(fseek(fp, 0L, 0) < 0 ){;
		/* set pointer back to top of file */
		logerr(XLOG_INFO,"cannot seek %s\n", cfname);
	} else if( putc( i, fp) == EOF ){
		logerr(XLOG_INFO,"cannot write %s\n", cfname);
	} else if( fflush( fp ) == EOF ){
		logerr(XLOG_INFO,"cannot flush %s\n", cfname);
	}
	if( fp != NULL ){
		(void)fclose(fp);
	}
}

/*
 * Show_ops()
 * show the values of the local options
 */

Show_ops()
{
	int i;
	(void)fprintf(stdout,"From: %s, Host %s, Person %s\n",From,Host,Person);
	(void)fprintf(stdout,"Is_local %d, Is_root %d\n", Is_local, Is_root );
	(void)fprintf(stdout,"Parmcount %d ", Parmcount );
	for( i = 0; i < Parmcount; ++i ){
		(void)fprintf(stdout, " '%s'(%d)", Parms[i].str, Parms[i].num );
	}
	(void)fprintf(stdout,"\n");
	(void)fflush(stdout);
}

/***************************************************************************
 * C_lpq()
 * invoke Lpq with parameters.
 * use the "system()" facility to do this.
 ***************************************************************************/
C_lpq()
{
	char buf[BUFSIZ];
	char *bp, *ep;			/* ACME Pointers, Inc. */
	int i;

	ep = buf+sizeof(buf);
	bp = estrcp( buf, "lpq", ep );
	for( i = 0; i < Parmcount; ++i ){
		bp = estrcp( bp, " ", ep );
		bp = estrcp( bp, Parms[i].str, ep );
	}
	if(Debug>4)log(XLOG_DEBUG,"C_lpq: '%s'", buf );
	i = system( buf );
	if(Debug>4)log(XLOG_DEBUG,"C_lpq: status %d", i );
	return( i );
}

/***************************************************************************
 * C_lprm()
 * invoke Lprm with parameters.
 * use the "system()" facility to do this.
 ***************************************************************************/
C_lprm()
{
	char buf[BUFSIZ];
	char *bp, *ep;			/* ACME Pointers, Inc. */
	int i;

	ep = buf+sizeof(buf);
	bp = estrcp( buf, "lprm", ep );
	for( i = 0; i < Parmcount; ++i ){
		bp = estrcp( bp, " ", ep );
		bp = estrcp( bp, Parms[i].str, ep );
	}
	if(Debug>4)log(XLOG_DEBUG,"C_lprm: '%s'", buf );
	i =  system( buf );
	if(Debug>4)log(XLOG_DEBUG,"C_lprm: status %d", i );
	return( i );
}

/***************************************************************************
 * C_remote()
 * do the indicated command on the remote host
 * command has the form: "remote op printer"
 * 1. get the command and see if it can be done remotely
 *    Parms[0] will be op, Parms[1] will be printer
 * 2. check the printer and make sure that it is a remote printer
 * 3. send the remote command to the far end, and get the information
 *    sent back.
 ***************************************************************************/

C_remote(cmd)
	struct dispatch *cmd;
{
	char **list;		/* list of printers */
	int i;				/* ACME Integers and Bearings, Inc. */
	int command;		/* command we are going to do */

	if( Parmcount < 2 ){
		(void)fprintf( stdout, "no or missing command or parameters\n" );
		return( 0 );
	}
	command = find_command( Parms[0].str );
	if( command == 0 ){
		(void)fprintf( stdout, "unknown command %s\n", Parms[0].str );
		return( 0 );
	}
	if(Debug>3)log(XLOG_DEBUG,"C_remote: command %s, %d, %s",
	dispatch[command].name,dispatch[command].flags,dispatch[command].summary);

	if( !IS_REMOTE(dispatch[command].flags) ){
		(void)fprintf( stdout, "command %s cannot be done remotely\n",
			Parms[0].str );
	}

	/*
	 * check on the printer
	 */
	if( strcmp("all", Parms[0].str) == 0){
		if(Debug>2)log(XLOG_DEBUG,"'all' parameter" );
		for( list = All_printers(); *list; ++list ){
			Printer = *list;
			if( remote_cando(dispatch[command].flags) ){
				Remote_control(dispatch[command].name);
			}
		}
	} else {
		for( i = 1; i < Parmcount; ++i ){
			Printer = Parms[i].str;
			if( remote_cando(dispatch[command].flags) ){
				Remote_control(dispatch[command].name);
			}
		}
	}
	return( 1 );
}

/***************************************************************************
 * remote_cando()
 * returns 0 if not able to do to remote site, 1 otherwise
 *    
 ***************************************************************************/
remote_cando(flags)
{
	int perms = 'C';	/* Permission Checking */

	if(Get_pc_entry(Printer, Status_pc_vars, Status_pc_len) == 0){
		log(XLOG_INFO,"Printer %s does not have printcap entry", Printer);
		return( 0 );
	}
	if( RM == 0 || RP == 0 ){
		if(Debug>5)log(XLOG_DEBUG,"%s not remote", Printer );
		return( 0 );
	}
	/*
	 * check on the privileges needed
	 * You have to be local and root OR have C privs
	 */
	if( IS_PRIV(flags) && !( Is_local && Is_root )
		&& !( Permfile && *Permfile
		  && Checkperm(Permfile,From,Person,First_name,&perms,(int *)0,0))){
	(void)fprintf(stdout,
			"You do not have printer control perms on '%s'\n", Printer);
		return( 0 );
	}
	return( 1 );
}

/***************************************************************************
 * C_lpd()
 * 1. Check to see if there is an active lpd server by checking the
 *    lock file.
 * 2. Check to see if the /dev/printer is available as well.
 ***************************************************************************/

C_lpd()
{
	int f;

    (void)Checklockfile(Masterlock, &f, (char *)0, 0, &LO_statb );
    if(f){
        (void)fprintf(stdout, "active LPD %d\n", f );
    } else {
		(void)fprintf( stdout, "LPD is not active\n" );
	}
	(void)fflush(stdout);
}
