/*MANUAL.TH LABEL "1WI" "November 1985" "Wang Institute" "UNIX User's Manual"
*/

/*MANUAL.SH NAME
label \- print label sheet
*/

/*MANUAL.SH USAGE
.I label
[-cU] [-d\ delim] [-h\ height] [-i\ indent] [-l\ count] [-n\ repeat]
.ti +.5i
[-s\ skip] [-w\ width] [-] [files]
*/

#include <stdio.h>
#include <ctype.h>

#define	MAXFIELDS    30        /* max number of labels across */

typedef	int       Status;      /* return/exit status of functions */
#define	SUCCESS   ((Status) 0)
#define	FAILURE   ((Status) 1)
typedef	int       Boole;       /* no Boolean type in C */
#define	TRUE      ((Boole) 1)
#define	FALSE     ((Boole) 0)
#define	EOS       '\0'

char	*Argv0;            /* program name */
char	*File;             /* current file name */
int 	Labelcount;        /* current label number in File */
int 	Linecount;         /* current line number in File */

/*MANUAL.SH OPTIONS
.I label
options adjust to different label page formats by
controlling the number of labels across a page,
the label height and width, and so on..
.de OP\" option-letter [optional value]
.TP
.B -\\$1 \\$2
..
*/

Boole	Center = FALSE;    /*MANUAL.OP c
Center text on each line in the label.
This option resets the label indent to 0.
*/

char	Delim = ':';       /*MANUAL.OP d delim
The label line field delimiter (default: `:' the colon) tells
.I label
how to separate fields in a line.
This character should not appear in the fields you want to print.
Several field delimiters in a row (blank fields) are skipped.
If you really want a blank line inside a label, use a space.
*/

int 	Realheight;        /* Labheight - Labskip */
int 	Labheight = 6;     /*MANUAL.OP h height
The height of a label (default: 6 lines).
*/

int 	Labindent = 2;     /*MANUAL.OP i indent
The number of spaces to indent before each label (default: 2).
*/

int 	Nlabels = 3;       /*MANUAL.OP l labels
The number of labels to fit across a page (default: 3).
*/

int 	Repeat = 1;        /*MANUAL.OP n number
The number of times to repeat each label (default: 1).
*/

int 	Labskip = 1;       /*MANUAL.OP s skip
The number of lines to skip at the top of each label (default: 1).
*/

/*MANUAL.OP U
Print a usage summary and exit.
*/

int 	Realwidth;         /* usable width of label (Labwidth - Labindent) */
int 	Labwidth = 33;     /*MANUAL.OP w width
The label width in spaces (default 33).
*/

/*FUNCTION printusage: on-line help */
void
printusage ()
	{
	fprintf (stderr, "%s: format label sheet\n", Argv0);
	fprintf (stderr, "-c	center fields on each label line\n");
	fprintf (stderr, "-d C	field delimiter character (%c)\n", Delim);
	fprintf (stderr, "-h N	height in lines of labels (%d lines)\n", Labheight);
	fprintf (stderr, "-i N	indent before each label (%d spaces)\n", Labindent);
	fprintf (stderr, "-l N	number of label across page (%d)\n", Nlabels);
	fprintf (stderr, "-n N	repeat each label N times (%d times)\n", Repeat);
	fprintf (stderr, "-s N	skip lines at label top (%d lines)\n", Labskip);
	fprintf (stderr, "-U	print this usage message\n");
	fprintf (stderr, "-v	verbose information printed\n");
	fprintf (stderr, "-w N	label width (%d spaces)\n", Labwidth);
	}

/*FUNCTION getint: set integer, checking type and range */
Status
getint (flag, string, varptr, min, max)
int 	flag;
char	*string;
int 	*varptr;
int 	min;
int 	max;
	{
	int 	retval = FAILURE;
	if (number (string) == 1) /* an integer */
		{
		int 	value = atoi (string);
		if (value >= min && value <= max)
			{
			*varptr = value;
			retval = SUCCESS;
			}
		else
			{
			fprintf (stderr, "\007%s: -%c value (%s) must be between %d and %d\n",
				Argv0, flag, string, min, max);
			}
		}
	else
		fprintf (stderr, "\007%s: -%c value (%s) must be an integer\n",
			Argv0, flag, string);
	return (retval);
	}

void
nputchar (c, n)
	{
	while (n-- > 0)
		putchar (c);
	}

/*FUNCTION initial: returns local version of optind, index to first operand */
int
initial (argc, argv) char **argv;
	{
	extern	char *optarg;    /* option value accessed through this by getopt */
	extern	int  optind;     /* will be index to first operand */
	int 	opterr = 0;      /* count of number of errors */
	int 	flag;            /* option flag characters read in here */
	char	*optstring =     /* getopt string */
		"cd:h:i:l:n:s:Uw:";
	char	*usage =         /* variable part of usage summary */
		"[-cU] [-d delim] [-h height] [-i indent] [-l count] [-n repeat]\n\t[-s skip] [-w width] [-] [files]";

	Argv0 = argv[0];

	while ((flag = getopt (argc, argv, optstring)) != EOF)
		switch (flag) /* put option cases here */
			{
			default:
				opterr++;
				break;
			case 'U':
				printusage ();
				exit (0);
			case 'c':
				Center = TRUE;
				Labindent = 0;
				break;
			case 'd':
				Delim = *optarg;
				break;
			case 'h':
				if (getint (flag, optarg, &Labheight, 0, 100) == FAILURE)
					opterr++;
				break;
			case 'i':
				if (getint (flag, optarg, &Labindent, 0, 100) == FAILURE)
					opterr++;
				break;
			case 'l':
				if (getint (flag, optarg, &Nlabels, 1, MAXFIELDS) == FAILURE)
					opterr++;
				break;
			case 'n':
				if (getint (flag, optarg, &Repeat, 1, 100) == FAILURE)
					opterr++;
				break;
			case 's':
				if (getint (flag, optarg, &Labskip, 0, 100) == FAILURE)
					opterr++;
				break;
			case 'w':
				if (getint (flag, optarg, &Labwidth, 10, 150) == FAILURE)
					opterr++;
				break;
			}

	if (opterr)
		{
		fprintf (stderr, "\007Usage: %s %s\n", argv[0], usage);
		exit (1);
		}

	Realwidth = Labwidth - Labindent;
	Realheight = Labheight - Labskip;

	return (optind);
	}

/*FUNCTION main: loop through files in classic UNIX filter style */
void
main (argc, argv)
int 	argc;     /* argument count */
char	**argv;   /* argument vector */
	{
	Status 	label ();   /* label (file, ioptr) will filter files */
	Status	status;     /* return status of filter () */
	int 	firstfile;  /* first file name index returned by initial */
	firstfile = initial (argc, argv);
	status = filter (argc, argv, firstfile, label);
	exit (status);
	}

/*FUNCTION geline: get non-blank input line and duplicate as needed */
char *
getline (line, size, ioptr)
char	*line;
FILE	*ioptr;
	{
	static	char	buf[BUFSIZ];
	static	int 	nrepeat = 0;
	char	*ptr = buf;

	if (nrepeat == 0)
		{
		do
			{
			if (fgets (ptr = buf, size, ioptr) == NULL)
				return (NULL);
			Linecount++;
			while (isspace (*ptr))
				ptr++;
		} while (*ptr == '\0');
		Labelcount++;
		checklabel (buf);
		}
	strncpy (line, buf, size);
	if (++nrepeat == Repeat)
		nrepeat = 0;
	return (line);
	}

/*FUNCTION fieldprint: print a field in a label */
char *
fieldprint (lptr)
char	*lptr;
	{
	char	*eptr = lptr;
	int 	len;           /* field length */
	int 	pad;           /* pad needed for field */

	while (*eptr && *eptr != Delim && *eptr != '\n') eptr++;
	while (*eptr == Delim)
		*eptr++ = EOS;
	if (*eptr == '\n')
		*eptr = EOS;

	nputchar (' ', Labindent);
	len = strlen (lptr);
	if (len > Realwidth)
		{
		lptr[Realwidth] = '\0';
		pad = 0;
		}
	else
		pad = Realwidth - len;

	if (Center)
		{
		nputchar (' ', pad / 2);
		pad -= pad / 2;
		}
	printf (lptr);
	nputchar (' ', pad);

	return (eptr);
	}

/*MANUAL.SH DESCRIPTION
.PP
.I label
prints a series of labels in a matrix on a page
that is then suitable to use as a copy master for a page of stick on labels.
If the labels are read in order: 1 2 3 4 5 6,
and printed three across a page,
then the output is arranged like:
.nf
	1	2	3
	4	5	6
.fi
.SS Input
.I label
reads non-blank lines that contain a series of delimited fields.
The delimiter is a single character that separates the fields.
Each field is used as one line in a label.
Example labels:
.ce 2
Gary Perlman:Wang Institute:Tyng Road:Tyngsboro, MA 01879
Copyright (c) 1985:Wang Institute:All Rights Reserved
.SS Output
.I label
reads label lines until it has enough to print a row of labels.
*/

/*MANUAL.SH EXAMPLE
.nf
Print six-high labels, four across, for 12 pitch print (100 wide):
	label  -h 6  -l 4  -w 25
.fi
*/

/*MANUAL.SH AUTHOR
Gary Perlman
*/

/*FUNCTION checklabel: print warnings about problems with label */
checklabel (line)
char	*line;
	{
	char	*ptr;
	int 	nfields = 0;        /* number of fields in label */
	Boole	truncate = FALSE;   /* are fields truncated? */

	for (;;)
		{
		for (ptr = line; *ptr && *ptr != '\n' && *ptr != Delim; ptr++);
		if (ptr - line > Realwidth)
			truncate = TRUE;
		while (*ptr == Delim)
			ptr++;
		nfields++;
		if (*ptr == EOS || *ptr == '\n')
			break;
		line = ptr;
		}

	if (nfields > Realheight)
		fprintf (stderr, "\007%s: Field ignored in label %d on line %d in %s\n",
			Argv0, Labelcount, Linecount, File);

	if (truncate == TRUE)
		fprintf (stderr, "\007%s: Field truncated in label %d on line %d in %s\n",
			Argv0, Labelcount, Linecount, File);
	}

/*FUNCTION label: main routine */
Status
label (file, ioptr)
char	*file;
FILE	*ioptr;
	{
	char	labbuf[MAXFIELDS][BUFSIZ], *lptr[MAXFIELDS];
	int 	line, col;

	File = file;
	Labelcount = 0;

	for (;;)
		{
		for (col = 0; col < Nlabels; col++)
			{
			if (getline (labbuf[col], BUFSIZ, ioptr) == NULL)
				if (col == 0) /* no labels read in, just quit */
					return (SUCCESS);
				else /* at least one label read in, use blanks to pad */
					labbuf[col][0] = EOS;
			lptr[col] = labbuf[col];
			}
		nputchar ('\n', Labskip);
		for (line = 0; line < Labheight-Labskip; line++)
			{
			for (col = 0; col < Nlabels; col++)
				lptr[col] = fieldprint (lptr[col]);
			putchar ('\n');
			}
		}
	}
