/*
 * imagen - print things on an Imagen laser printer.
 *
 * David A. Curry
 * Purdue University
 * Engineering Computer Network
 * davy@intrepid.ecn.purdue.edu
 */
#include <stdio.h>
#include <ctype.h>

/*
 * Booleans.
 */
#define TRUE		1
#define FALSE		2

/*
 * Argument types.
 */
#define TY_INT		1
#define TY_CHAR		2
#define TY_BOOL		3

/*
 * Emulations.
 */
#define EM_NONE		0x1
#define EM_EPSON	0x2
#define EM_DIABLO	0x4
#define EM_PRINTER	0x8
#define EM_TEKTRONIX	0x10

/*
 * Option structure.
 */
struct option {
	char	*o_name;		/* name as on command line	*/
	char	*o_text;		/* what to print to the laser	*/
	char	 o_type;		/* type of argument		*/
	char	 o_flag;		/* boolean value arg represents	*/
	char	 o_emul;		/* valid emulations		*/
	char	 o_given;		/* non-zero if this arg given	*/

#define o_ival	o_val.u_i
#define o_cval	o_val.u_c

	union {				/* value given with the arg	*/
		int	 u_i;
		char	*u_c;
	} o_val;
};

/*
 * All the options we know about.
 */
struct option options[] = {
	{ "-jr",	"jamresistance",	TY_BOOL,	TRUE,
		EM_NONE,	0 },
	{ "-pc",	"pagecollation",	TY_BOOL,	TRUE,
		EM_NONE,	0 },
	{ "-pr",	"pagereversal",		TY_BOOL,	TRUE,
		EM_NONE,	0 },
	{ "-paper",	"paper",		TY_CHAR,	0,
		EM_NONE,	0 },
	{ "-pm",	"papermargin",		TY_CHAR,	0,
		EM_NONE,	0 },
	{ "-pre",	"prerasterization",	TY_BOOL,	TRUE,
		EM_NONE,	0 },
	{ "-lf",	"lfiscrlf",		TY_BOOL,	TRUE,
		EM_EPSON | EM_DIABLO,	0 },
	{ "-lf",	"lfiscrlf",		TY_BOOL,	FALSE,
		EM_TEKTRONIX,	0 },
	{ "-cr",	"criscrlf",		TY_BOOL,	TRUE,
		EM_EPSON,	0 },
	{ "-cr",	"criscrlf",		TY_BOOL,	FALSE,
		EM_TEKTRONIX,	0 },
	{ "-auto",	"autocrlf",		TY_BOOL,	TRUE,
		EM_EPSON | EM_DIABLO,	0 },
	{ "-auto",	"autocrlf",		TY_BOOL,	FALSE,
		EM_TEKTRONIX,	0 },
	{ "-stroke",	"stroke",		TY_INT,		0,
		EM_TEKTRONIX,	0 },
	{ "-bm",	"bottommargin",		TY_INT,		0,
		EM_DIABLO,	0 },
	{ "-ecs",	"ecs",			TY_BOOL,	TRUE,
		EM_DIABLO,	0 },
	{ "-seven",	"eightbit",		TY_BOOL,	FALSE,
		EM_DIABLO,	0 },
	{ "-tab",	"htabs",		TY_BOOL,	TRUE,
		EM_DIABLO,	0 },
	{ "-lm",	"leftmargin",		TY_INT,		0,
		EM_DIABLO | EM_PRINTER,	0 },
	{ "-pw",	"printwheel",		TY_CHAR,	0,
		EM_DIABLO,	0 },
	{ "-rm",	"rightmargin",		TY_INT,		0,
		EM_DIABLO,	0 },
	{ "-tm",	"topmargin",		TY_INT,		0,
		EM_DIABLO,	0 },
	{ "-vmi",	"vmi",			TY_INT,		0,
		EM_DIABLO,	0 },
	{ "-cs",	"characterset",		TY_CHAR,	0,
		EM_EPSON,	0 },
	{ "-lpi",	"linesperinch",		TY_INT,		0,
		EM_EPSON,	0 },
	{ "-st",	"SelecType",		TY_INT,		0,
		EM_EPSON,	0 },
	{ "-it",	"italics",		TY_BOOL,	TRUE,
		EM_EPSON,	0 },
	{ "-ul",	"underline",		TY_BOOL,	TRUE,
		EM_EPSON,	0 },
	{ "-ll",	"linelength",		TY_INT,		0,
		EM_EPSON,	0 },
	{ "-zs",	"zeroslash",		TY_BOOL,	TRUE,
		EM_EPSON,	0 },
	{ "-fl",	"formlength",		TY_INT,		0,
		EM_PRINTER,	0 },
	{ "-fw",	"formwidth",		TY_INT,		0,
		EM_PRINTER,	0 },
	{ "-fp",	"formsperpage",		TY_INT,		0,
		EM_PRINTER,	0 },
	{ "-lm",	"leftmargin",		TY_INT,		0,
		EM_PRINTER,	0 },
	{ "-o",		"outlines",		TY_BOOL,	TRUE,
		EM_PRINTER,	0 },
	{ "-r",		"rules",		TY_BOOL,	TRUE,
		EM_PRINTER,	0 },
	{ NULL,		NULL,			0,		0,
		0,		0 }
};

char	*pname;				/* program name			*/

int	copies;				/* number of copies we want	*/
int	emulation;			/* emulation we're using	*/
int	notdone = 1;			/* have we done options yet?	*/

main(argc, argv)
int argc;
char **argv;
{
	FILE *fp;
	register struct option *op;

	pname = *argv;

	/*
	 * Defaults: one copy, no emulations.
	 */
	copies = 1;
	emulation = EM_NONE;

	/*
	 * Process arguments.
	 */
top:	while (--argc) {
		argv++;

		/*
		 * -# where # is a number is the number of copies
		 * they want.
		 */
		if ((**argv == '-') && (isdigit((*argv)[1]))) {
			copies = -atoi(*argv);
			continue;
		}

		/*
		 * Check for emulations.
		 */
		if (!strcmp(*argv, "-printer")) {
			setemul(EM_PRINTER);
			continue;
		}
		else if (!strcmp(*argv, "-diablo")) {
			setemul(EM_DIABLO);
			continue;
		}
		else if (!strcmp(*argv, "-tek")) {
			setemul(EM_TEKTRONIX);
			continue;
		}
		else if (!strcmp(*argv, "-fx") || !strcmp(*argv, "-epson")) {
			setemul(EM_EPSON);
			continue;
		}

		/*
		 * Special arguments which set up environments for
		 * portrait and landscape printing.
		 */
		if (!strcmp(*argv, "-port") || !strcmp(*argv, "-port2") ||
		    !strcmp(*argv, "-land") || !strcmp(*argv, "-land2")) {
			setemul(EM_PRINTER);

			setoption2("-fw", (*argv)[1] == 'p' ? 80 : 132);
			setoption2("-fp", (*argv)[5] == '2' ? 2 : 1);
			continue;
		}

		/*
		 * Check for normal options.
		 */
		for (op = options; op->o_name != NULL; op++) {
			/*
			 * Only need to check things valid in this
			 * emulation.
			 */
			if ((op->o_emul != EM_NONE) && ((op->o_emul & emulation) == 0))
				continue;

			/*
			 * Match?
			 */
			if (strcmp(op->o_name, *argv) != 0)
				continue;

			/*
			 * Set it.
			 */
			setoption(op, &argc, &argv);
			goto top;
		}

		/*
		 * If we got through the loop above, we found either
		 * an option we don't recognize, or a file name.
		 */
		break;
	}

	/*
	 * No more options, print the options and read standard input.
	 */
	if (argc == 0) {
		dooptions();
		readfile(stdin);
	}
	else {
		/*
		 * Handle the rest of the arguments.
		 */
		while (argc--) {
			/*
			 * Minus means standard input, otherwise
			 * things starting with - are unrecognized
			 * options.
			 */
			if (!strcmp(*argv, "-")) {
				/*
				 * If needed, print the options.
				 */
				if (notdone)
					dooptions();

				/*
				 * Print the file.
				 */
				readfile(stdin);
			}
			else if (**argv == '-') {
				usage(*argv);
			}
			else {
				/*
				 * Open the file.
				 */
				if ((fp = fopen(*argv, "r")) == NULL) {
					fprintf(stderr, "%s: cannot open \"%s\".\n", pname, *argv);
					exit(1);
				}

				/*
				 * If needed, print the options.
				 */
				if (notdone)
					dooptions();

				/*
				 * Print it.
				 */
				readfile(fp);
				fclose(fp);
			}

			argv++;
		}
	}

	exit(0);
}

/*
 * setemul - set the emulation.
 */
setemul(em)
int em;
{
	/*
	 * If we already have an emulation, they
	 * can't change emulations.
	 */
	if (emulation != EM_NONE) {
		fprintf(stderr, "%s: only one emulation at a time.\n", pname);
		exit(1);
	}

	emulation = em;
}

/*
 * setoption - set the option pointed to by op to the value in the
 *	       argument list.
 */
setoption(op, argc, argv)
register struct option *op;
char ***argv;
int *argc;
{
	register int ac;
	register char **av;

	/*
	 * Mark this argument as given on the command line.
	 */
	op->o_given = 1;

	ac = *argc;
	av = *argv;

	/*
	 * If it's an integer or string, we need the next argument.
	 */
	if ((op->o_type == TY_INT) || (op->o_type == TY_CHAR)) {
		if (--ac <= 0)
			usage(NULL);

		if (op->o_type == TY_INT)
			op->o_ival = atoi(*++av);
		else
			op->o_cval = *++av;
	}

	*argc = ac;
	*argv = av;
}

/*
 * setoption2 - set the option named by opt to the value val.
 */
setoption2(opt, val)
char *opt;
int val;
{
	register struct option *op;

	/*
	 * Find the option and set it.
	 */
	for (op = options; op->o_name != NULL; op++) {
		if (strcmp(op->o_name, opt) != 0)
			continue;

		op->o_given = 1;
		op->o_ival = val;
	}
}

/*
 * dooptions - print out the options they gave us.
 */
dooptions()
{
	register char *comma;
	register struct option *op;

	/*
	 * All commands start with this.
	 */
	printf("@document(");

	/*
	 * Print the emulator they wanted.
	 */
	switch (emulation) {
	case EM_NONE:
		comma = "";
 		break;
	case EM_EPSON:
		printf("language fx");
		comma = ", ";
		break;
	case EM_DIABLO:
		printf("language diablo");
		comma = ", ";
		break;
	case EM_PRINTER:
		printf("language printer");
		comma = ", ";
		break;
	case EM_TEKTRONIX:
		printf("language tektronix");
		comma = ", ";
		break;
	}

	/*
	 * Print the number of copies they want.
	 */
	if (copies > 1) {
		printf("%scopies %d", comma, copies);
		comma = ", ";
	}

	/*
	 * Print out all options they set.
	 */
	for (op = options; op->o_name != NULL; op++) {
		if (op->o_given == 0)
			continue;

		printf("%s%s", comma, op->o_text);
		comma = ", ";

		switch (op->o_type) {
		case TY_BOOL:
			printf(" %s", op->o_flag == TRUE ? "on" : "off");
			break;
		case TY_CHAR:
			printf(" %s", op->o_cval);
			break;
		case TY_INT:
			printf(" %d", op->o_ival);
			break;
		}
	}

	printf(")");

	notdone = 0;
}

/*
 * readfile - read from the file, write to standard output.
 */
readfile(fp)
FILE *fp;
{
	register int i;
	char buf[BUFSIZ];

	while ((i = fread(buf, 1, BUFSIZ, fp)) > 0)
		fwrite(buf, 1, i, stdout);
}

/*
 * usage - handle the usage message
 */
usage(opt)
char *opt;
{
	int didone;
	register int i;
	register struct option *op;

	/*
	 * If they sent us a string, we are looking for a specific
	 * flag which caused an error.
	 */
	if (opt != NULL) {
		/*
		 * See if we can find the flag.
		 */
		didone = 0;
		for (op = options; op->o_name != NULL; op++) {
			/*
			 * Yup... print out which emulators
			 * support this flag.
			 */
			if (!strcmp(op->o_name, opt)) {
				if (didone == 0) {
					fprintf(stderr, "%s: %s only valid under ", pname, opt);
					didone = 1;
				}
				
				if (op->o_emul & EM_EPSON)
					fprintf(stderr, "epson ");
				if (op->o_emul & EM_DIABLO)
					fprintf(stderr, "diablo ");
				if (op->o_emul & EM_PRINTER)
					fprintf(stderr, "printer ");
				if (op->o_emul & EM_TEKTRONIX)
					fprintf(stderr, "tek ");
			}
		}

		if (didone) {
			fprintf(stderr, "emulation(s).\n");
			exit(1);
		}
	}

	/*
	 * Print out what everything does.
	 */
	fprintf(stderr, "Usage: %s [emulation] [options] files ...\n", pname);
	fprintf(stderr, "  Emulations:  printer, diablo, epson, tektronix\n\n");
	fprintf(stderr, "    Option    Meaning             Emulators\n");
	fprintf(stderr, "    -#        # of copies         all\n");
	fprintf(stderr, "    -port     portrait            printer\n");
	fprintf(stderr, "    -port2    2 portrait          printer\n");
	fprintf(stderr, "    -land     landscape           printer\n");
	fprintf(stderr, "    -land2    2 landscape         printer\n");

	for (op = options; op->o_name != NULL; op++) {
		fprintf(stderr, "    %-10s%s", op->o_name, op->o_text);
		i = strlen(op->o_text);

		if (op->o_type == TY_BOOL) {
			fprintf(stderr, " %s", op->o_flag == TRUE ? "on" : "off");
			i += op->o_flag == TRUE ? 3 : 4;
		}

		while (i++ < 20)
			putc(' ', stderr);

		if (op->o_emul & EM_NONE)
			fprintf(stderr, "all ");
		if (op->o_emul & EM_EPSON)
			fprintf(stderr, "epson ");
		if (op->o_emul & EM_DIABLO)
			fprintf(stderr, "diablo ");
		if (op->o_emul & EM_PRINTER)
			fprintf(stderr, "printer ");
		if (op->o_emul & EM_TEKTRONIX)
			fprintf(stderr, "tek ");

		fprintf(stderr, "\n");
	}
}
