/*
 * dvipage: DVI Previewer Program for Suns
 *
 * Neil Hunt (hunt@spar.slb.com)
 *
 * This program is based, in part, upon the program dvisun,
 * distributed by the UnixTeX group, extensively modified by
 * Neil Hunt at the Schlumberger Palo Alto Research Laboratories
 * of Schlumberger Technologies, Inc.
 *
 * From the dvisun manual page entry:
 *	Mark Senn wrote the early versions of [dvisun] for the
 *	BBN BitGraph. Stephan Bechtolsheim, Bob Brown, Richard
 *	Furuta, James Schaad and Robert Wells improved it. Norm
 *	Hutchinson ported the program to the Sun. Further bug fixes
 *	by Rafael Bracho at Schlumberger.
 *
 * Copyright (c) 1988 Schlumberger Technologies, Inc 1988.
 * Anyone can use this software in any manner they choose,
 * including modification and redistribution, provided they make
 * no charge for it, and these conditions remain unchanged.
 *
 * This program is distributed as is, with all faults (if any), and
 * without any warranty. No author or distributor accepts responsibility
 * to anyone for the consequences of using it, or for whether it serves any
 * particular purpose at all, or any other reason.
 *
 * HISTORY
 *
 * 12 April 1988 - Neil Hunt
 *	Version 2.0 released for use.
 *
 * 11 April 1988 - Neil Hunt
 *	Applied fixes supplied by Rafael Bracho (Schlumberger Austin)
 *	for operation on Sun-4 workstations.
 *
 * Earlier history unavailable.
 */

#include <stdio.h>
#include <strings.h>
#include <sys/param.h>		/* For MAXPATHLEN */
#include <sys/types.h>
#include <sys/stat.h>
#include <suntool/sunview.h>
#include <suntool/canvas.h>
#include <suntool/panel.h>
#include <suntool/icon.h>
#include "dvi.h"

/*
 * Standard definitions.
 * ====================
 */

#ifndef _TYPES_
typedef unsigned int		uint;
typedef unsigned short		ushort;
#endif _TYPES_
typedef unsigned char		uchar;

#define DEL			'\177'
#define ESC			'\033'
#define Control(c)		((c) - 0x40)

#define A_ARG			0
#define A_END			(-1)

#define forward			extern

#define Range(min, x, max)	(((x) < (min))? \
				 (min) : (((x) > (max))? (max) : (x)) \
				)

#define MAXINT			2147483647
#define NEGMAXINT		(-2147483648)

/*
 * Switches and flags.
 * ==================
 */

#define DVIPAGE_LABEL		"DVI Previewer 2.0"	/* Frame label */

/* #define USEGLOBALMAG		1	/* Use dvi global magnification ??? */

#ifndef FONTAREA
#define FONTAREA		"/usr/local/font/tex/pxl"
#endif  FONTAREA

#define PAGE_WIDTH		(8.5)	/* Inches */
#define PAGE_HEIGHT		(11)	/* Inches */

#define DEFAULT_ORIGIN_X	(1)	/* inches */
#define DEFAULT_ORIGIN_Y	(1)	/* inches */

#define DEFAULT_MONO_RES	118
#define DEFAULT_COLOUR_RES	300

#define DEFAULT_MONO_SAMPLING	1
#define DEFAULT_COLOUR_SAMPLING	4

#define DEFAULT_MAG_SIZE_X	(256)
#define DEFAULT_MAG_SIZE_Y	(128)
#define DEFAULT_MAG_BORDER	(2)

#define MAXOPEN			16	/* Max number of open font files. */
#define STACKSIZE		100	/* dvi stack max length */
#define STRSIZE			MAXPATHLEN	/* Max string length */
#define MAX_SHEETS		512	/* Pages remembered in table */

#ifndef PRINT_SPOOLER
#define PRINT_SPOOLER		"lpr -d %s >/dev/null 2>/dev/null"
#endif  PRINT_SPOOLER

#ifndef PRINT_PAGE_SPOOLER
#define PRINT_PAGE_SPOOLER \
	"texpagefilter -f %d -t %d %s | lpr -d >/dev/null 2>/dev/null"
#endif	PRINT_PAGE_SPOOLER

/*
 * Constants.
 * =========
 */

#define DVIFORMAT		2
#define READ			4	/* Code for access() */

#define SKIP			TRUE	/* Flags for process_page() */
#define RASTERISE		FALSE

#define DEBUG_FONTS		1
#define DEBUG_PRINTER		2
#define DEBUG_IMSIZE		4
#define DEBUG_SHEET		8
#define DEBUG_MAGNIFY		16

#define BAD_PAGE		NEGMAXINT
#define LAST_PAGE		MAXINT

/*
 * Forward functions.
 * =================
 */

forward int			main();

forward Notify_value		page_paint();
forward Notify_value		page_event();
forward int			page_menu();
forward void			page_magnify();
forward void			page_pan();
forward bool			goto_sheet();
forward bool			goto_page();

forward bool			init_dvi_file();
forward void			close_dvi_file();
forward bool			check_dvi_file();

forward bool			read_postamble();
forward bool			find_postamble_ptr();

forward bool			process_page();

forward void			set_char();
forward bool			load_char();
forward void			set_rule();
forward void			move_down();
forward void			move_over();
forward void			set_font_num();
forward bool			get_font_def();
forward void			close_fonts();
forward bool			read_font_def();
forward bool			open_font_file();
forward void			skip_font_def();

forward int			pix_round();
forward int			get_unsigned();
forward int			get_signed();
forward double			actual_factor();
forward int			do_convert();

forward void			sample_page();

forward struct pixrect *	pr_alloc();
forward struct pixrect *	pr_free();
forward struct pixrect *	pr_check();
forward struct pixrect *	pr_link();
forward struct pixrect *	pr_sample_4();
forward struct pixrect *	pr_sample_3();
forward struct pixrect *	pr_sample_2();

forward void			pw_cover();
forward void			pr_rect();
forward void			pw_rect();

forward char *			a_prog_name;
forward char			a_next();
forward char *			a_arg();
forward double			a_number();
forward int			a_integer();

forward void			message();

forward bool			strings_prompt();
forward bool			integers_prompt();
forward bool			doubles_prompt();

/*
 * Extern functions.
 * ================
 */

extern char	*getenv();
extern double	atof();

/*
 * Data Structures
 * ===============
 */

/*
 * Character information.
 *	Width in pixels.
 *	Height in pixels.
 *	X offset in pixels.
 *	Y offset in pixels.
 *	Location of bits
 *		Flag for bits have been loaded into memory.
 *			offset to position in file.
 *			pointer to the bits.
 *	width.
 */

struct char_entry
{
	ushort		width;
	ushort		height;
	short		xOffset;
	short		yOffset;
	struct
	{
		int		isloaded;
		union
		{
			int		fileOffset;
			struct pixrect	*pixrectptr;
		} address;
	} where;
	int		tfmw;
};

/*
 * Font information.
 */

struct font_entry
{
	int		k;
	int		c;
	int		s;
	int		d;
	int		a;
	int		l;
	char		n[STRSIZE];	/* FNT_DEF command parameters  */
	int		font_space;	/* computed from FNT_DEF s parameter */
	int		font_mag;	/* computed from s and d parameters */
	char		name[STRSIZE];	/* full name of PXL file */
	FILE		*font_file_fd;  /* file identifier (0 if none) */
	int		magnification;	/* magnification read from PXL file */
	int		designsize;	/* design size read from PXL file */
	struct char_entry ch[NPXLCHARS];/* character information */
	struct font_entry *next;
};

struct pixel_list
{
	FILE		*pixel_file_fd;	/* file identifier */
	int		use_count;	/* count of "opens" */
};

struct stack_entry		/* stack entry */
{
	int h, v, w, x, y, z;  /* what's on stack */
};

/*
 * mem_pixrect:
 *	A statically allocatable pixrect structure.
 *	Contains the pixrect data and the mpr_data.
 */

struct mem_pixrect
{
	struct pixrect		mpr_pr;
	struct mpr_data		mpr_data;
};

/*
 * Globals.
 * =======
 */

int hconv, vconv;		/* converts DVI units to pixels             */
int num;			/* numerator specified in preamble          */
int den;			/* denominator specified in preamble        */
int mag;			/* magnification specified in preamble      */
int nopen;			/* number of open PXL files                 */

struct font_entry *fontptr;     /* font_entry pointer                       */
struct font_entry *hfontptr=NULL;/* font_entry pointer                      */
struct font_entry *pfontptr = NULL; /* previous font_entry pointer          */
struct pixel_list pixel_files[MAXOPEN+1];
                                /* list of open PXL file identifiers        */

double page_w = PAGE_WIDTH;	/* page width (inches) */
double page_h = PAGE_HEIGHT;	/* page width (inches) */

int h;				/* current horizontal position              */
int hh;				/* current horizontal position in pixels    */
int v;				/* current vertical position                */
int vv;				/* current vertical position in pixels      */

bool pre_load = TRUE;		/* preload the font descriptions?	*/
bool silent = FALSE;		/* suppress messages */
bool show_page_frame = FALSE;	/* show page window */

long postambleptr;		/* Pointer to the postamble                 */

FILE *pxlfp;			/* PXL file pointer                         */
char *pxl_path;			/* PXL path name for search		    */

FILE *dvifp = NULL;		/* File pointer */

struct stat stat_buf;		/* For checking file changes. */
time_t mtime = 0;

char label[STRSIZE];

char pathname[STRSIZE] = "";	/* Complete path */
char directory[STRSIZE] = "";	/* Directory */
char filename[STRSIZE] = "";	/* File name */
char print_spooler[STRSIZE] = PRINT_SPOOLER;	/* Print commands. */
char print_page_spooler[STRSIZE] = PRINT_PAGE_SPOOLER;

int last_sheet = 0;		/* Sheet number of last page in file */
int file_sheet = 1;		/* Position of file pointer */
int disp_sheet = 0;		/* Current displayed page */
int disp_page = 0;		/* Real Page number */
int sheet_page[MAX_SHEETS];	/* Page number of each sheet. */
long sheet_table[MAX_SHEETS];	/* pointers to start of each page in file */
int last_known_sheet = 0;	/* Points to table at next unread sheet */

int resolution = 0;		/* Assumed screen resolution for rendering */
int sampling = 0;		/* Sample down sampling factor */
bool mono;			/* Monochrome screen */

Frame disp_frame;		/* Frame for display window. */
Canvas disp_canvas;		/* Canvas for display window. */
Frame mess_frame;		/* Frame for message window. */
Panel mess_panel;		/* Panel for message window. */

struct mem_pixrect page_mpr;	/* Page bitmap. */
struct pixrect *page_pr;
struct mem_pixrect sample_mpr;	/* Sampled/filtered bitmap. */
struct pixrect *sample_pr;

int origin_x;			/* Nominal origin of the dvi on the page */
int origin_y;			/* Nominal origin of the dvi on the page */

int offset_x;			/* Offsets of page in window. */
int offset_y;			/* Offsets of page in window. */

int start_x;			/* Starting position of page in the window */
int start_y;			/* Starting position of page in the window */

int verbose;			/* Flags for debugging. */

int mag_size_x = DEFAULT_MAG_SIZE_X;	/* Magnifier parameters. */
int mag_size_y = DEFAULT_MAG_SIZE_Y;
int mag_border = DEFAULT_MAG_BORDER;

uchar	cmap[] =
  { 255, 238, 221, 204, 187, 170, 153, 136, 119, 102, 85, 68, 51, 34, 17, 0 };

short icon_image[] =
{
/* Format_version=1, Width=64, Height=64, Depth=1, Valid_bits_per_item=16
 */
	0xFFFF,0xFFFF,0xFFFF,0xFFFF,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x00C0,0x0001,0x8000,0x0000,0x0338,0x0001,
	0x8000,0x0000,0x0447,0x0001,0x8000,0x0000,0x1911,0xE001,
	0x8000,0x0000,0x2888,0x9E01,0x8000,0x0000,0xC222,0x23C1,
	0x8000,0x0001,0x4444,0x4479,0x8000,0x0007,0x1111,0x1117,
	0x8000,0x0008,0x8888,0x8889,0x8000,0x0032,0x2222,0x2223,
	0x8000,0x0044,0x4444,0x4445,0x8000,0x0191,0x1111,0x1111,
	0x8000,0x0288,0x8888,0x888B,0x8000,0x0C22,0x2222,0x2225,
	0x8000,0x1444,0x4444,0x4447,0x8000,0x6111,0x1111,0x1119,
	0x8000,0xC888,0x8888,0x889B,0x8000,0xB222,0x2222,0x2235,
	0x8000,0xCE44,0x4444,0x446B,0x8000,0x91D1,0x1111,0x11D5,
	0x8000,0x88B8,0x8888,0x88AB,0x8000,0xAC2E,0x2222,0x2355,
	0x8000,0xC345,0xC444,0x46AB,0x8000,0x9CD1,0x3111,0x1555,
	0x8000,0xB338,0x8E88,0x8AA5,0x8000,0xE0C6,0x23E2,0x3559,
	0x8000,0xC039,0x8474,0x56B1,0x8000,0xC006,0x611F,0x2D41,
	0x8000,0x9001,0x9C89,0xDEA1,0x8001,0x0C00,0x6322,0xFD41,
	0x8002,0x43E0,0x1CC4,0xBE81,0x8004,0x2C18,0x0331,0xFD01,
	0x8008,0x3606,0x00C8,0xBB01,0x8031,0xC781,0x0032,0xF601,
	0x8040,0x4CC1,0x0014,0xAC01,0x818C,0x9840,0xC021,0xD801,
	0x8602,0xB208,0xB048,0xA801,0x9861,0x6788,0x4CC2,0xD001,
	0xB011,0xCCD9,0xC0C4,0xA001,0xA30D,0x1817,0x60B1,0xC001,
	0x9883,0x3E1C,0x590E,0xC001,0x8463,0x6270,0x4101,0x8001,
	0x821F,0xC1D0,0xC100,0x0001,0x813F,0xB310,0xB200,0x0001,
	0x81FC,0x5831,0x0200,0x0001,0x87F8,0x4021,0x0400,0x0001,
	0x9F8C,0x3006,0xC400,0x0001,0x9E07,0x0C18,0x0800,0x0001,
	0x8C00,0x83E8,0x1000,0x0001,0x8000,0x6106,0x2000,0x0001,
	0x8000,0x18C0,0x4000,0x0001,0x8000,0x0430,0x8000,0x0001,
	0x8000,0x0309,0x0000,0x0001,0x8000,0x0081,0x0000,0x0001,
	0x8000,0x0062,0x0000,0x0001,0x8000,0x0014,0x0000,0x0001,
	0x8000,0x0008,0x0000,0x0001,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0xFFFF,0xFFFF,0xFFFF,0xFFFF
};

DEFINE_ICON_FROM_IMAGE(icon, icon_image);

short hand[] =
{
  0x0C00,0x1200,0x1200,0x1380,0x1240,0x7270,0x9248,0x924E,
  0x9249,0x9249,0x9009,0x8001,0x4002,0x4002,0x2004,0x2004
};
mpr_static(hand_pr, 16, 16, 1, hand);
Cursor hand_cursor;

/*
 * Functions.
 * =========
 */

/*
 * main:
 *	Interpret args, open windows, loop forever.
 */

int
main(argc, argv)
int argc;
char *argv[];
{
	int f;
	char opt;
	char *slash;
	char *extension;
	Pixwin *pw;
	bool fake_mono;
	char *printer;
	double set_origin_x;
	double set_origin_y;
	double set_start_x;
	double set_start_y;

	/*
	 * local initialisations.
	 */
	fake_mono = FALSE;
	set_origin_x = 0.0;
	set_origin_y = 0.0;
	set_start_x = 0.0;
	set_start_y = 0.0;

#ifdef SPAR_HACKS

	/*
	 * Set local printer hacks.
	 */
	printer = getenv("PRINTER");
	if(printer && strncmp(printer, "lw", 2) == 0)
	{
		sprintf(print_spooler,
		  "dvips -P%s %%s >/dev/null 2>/dev/null",
		  printer);
		sprintf(print_page_spooler,
		  "dvips -P%s -f %%d -t %%d %%s >/dev/null 2>/dev/null",
		  printer);
	}
	else if(printer && strncmp(printer, "im", 2) == 0)
	{
		sprintf(print_spooler,
		  "dviimp -P%s %%s >/dev/null 2>/dev/null",
		  printer);
		sprintf(print_page_spooler,
		  "dviimp -P%s -S %%d -E %%d %%s >/dev/null 2>/dev/null",
		  printer);
	}
	else
	{
		fprintf(stderr, "PRINTER environment not recognised:\n");
		fprintf(stderr, " using `%s' to print files\n",
		  print_spooler);
		fprintf(stderr, " using `%s' to print pages\n",
		  print_page_spooler);
	}

	if(verbose & DEBUG_PRINTER)
	{
		fprintf(stderr, "Using `%s' to print files\n",
		  print_spooler);
		fprintf(stderr, "Using `%s' to print pages\n",
		  print_page_spooler);
	}

#endif SPAR_HACKS

	/*
	 * Find pixel path environment.
	 */
	if((pxl_path = getenv("BGPXLPATH")) == NULL)
		pxl_path = FONTAREA;

	/*
	 * Get cursor.
	 */
	hand_cursor = cursor_create(
	  CURSOR_IMAGE, &hand_pr,
	  CURSOR_XHOT, 5,
	  CURSOR_YHOT, 0,
	  CURSOR_OP, PIX_SRC ^ PIX_DST,
	  0);

	/*
	 * Create a disp_frame.
	 */
	disp_frame = window_create(0, FRAME,
	  WIN_X, 300,
	  WIN_Y, 50,	/* I hate the bug where the window is off the screen */
	  WIN_WIDTH,
	    (int)(page_w * DEFAULT_COLOUR_RES /
	    DEFAULT_COLOUR_SAMPLING) + 10,
	  WIN_HEIGHT,
	    (int)(page_h * DEFAULT_COLOUR_RES /
	    DEFAULT_COLOUR_SAMPLING) + 20,
	  FRAME_ARGC_PTR_ARGV, &argc, argv,
	  FRAME_LABEL, DVIPAGE_LABEL,
	  FRAME_ICON, &icon,
	  0);

	/*
	 * Create the disp_canvas.
	 */
	disp_canvas = window_create(disp_frame, CANVAS,
	  CANVAS_RETAINED, FALSE,
	  CANVAS_AUTO_CLEAR, FALSE,
	  CANVAS_FIXED_IMAGE, TRUE,
	  WIN_CURSOR, hand_cursor,
	  WIN_CONSUME_PICK_EVENTS,
	    WIN_NO_EVENTS,
	    LOC_DRAG,
	    WIN_MOUSE_BUTTONS,
	    LOC_WINENTER,	/* Otherwise misses first event */
	    LOC_WINEXIT,
	    0,
	  WIN_CONSUME_KBD_EVENTS,
	    WIN_NO_EVENTS,
	    WIN_ASCII_EVENTS,
	    WIN_LEFT_KEYS,	/* For Expose, Hide, Close etc. */
	    KBD_USE,		/* Otherwise click to type doesn't work */
	    KBD_DONE,
	    0,
	  CANVAS_REPAINT_PROC, page_paint,
	  WIN_EVENT_PROC, page_event,
	  WIN_WIDTH, WIN_EXTEND_TO_EDGE,
	  WIN_HEIGHT, WIN_EXTEND_TO_EDGE,
	  0);

	/*
	 * Interpret args.
	 */
	f = 0;
	while((opt = a_next(argc, argv)) != A_END)
	{
		switch(opt)
		{
		case 'H':
			fprintf(stderr,
  "Usage: %s \\\n", a_prog_name);
			fprintf(stderr,
  "	[-v mode]		# Verbose mode (for debugging) \\\n");
			fprintf(stderr,
  "	[-m]			# Force monochrome mode \\\n");
			fprintf(stderr,
  "	[-p]			# Don't preload font data \\\n");
			fprintf(stderr,
  "	[-q]			# Quiet: no warning messages \\\n");
			fprintf(stderr,
  "	[-f]			# Show rendering frame on page \\\n");
			fprintf(stderr,
  "	[-r res]		# Use `res' dpi fonts \\\n");
			fprintf(stderr,
  "	[-s sample]		# Reduce by factor of `sample' \\\n");
			fprintf(stderr,
  "	[-x x] [-y y]		# Initial pos of sheet in inches \\\n");
			fprintf(stderr,
  "	[-X ox] [-Y oy]		# Pos of (0, 0) on page in inches \\\n");
			fprintf(stderr,
  "	[-w width] [-h height]	# Total size of page in inches \\\n");
			fprintf(stderr,
  "	[dvifile[.dvi]]\n");
			exit(1);

		case 'v':
			verbose = a_integer(argc, argv);
			break;

		case 'm':
			fake_mono = TRUE;
			break;

		case 'p':
			pre_load = ! pre_load;
			break;

		case 'q':
			silent = ! silent;
			break;

		case 'f':
			show_page_frame = ! show_page_frame;
			break;

		case 'r':
			resolution = a_integer(argc, argv);
			break;

		case 's':
			sampling = a_integer(argc, argv);
			break;

		case 'x':
			set_start_x = a_number(argc, argv);
			break;

		case 'y':
			set_start_y = a_number(argc, argv);
			break;

		case 'X':
			set_origin_x = a_number(argc, argv);
			break;

		case 'Y':
			set_origin_y = a_number(argc, argv);
			break;

		case 'w':
			page_w = a_number(argc, argv);
			break;

		case 'h':
			page_h = a_number(argc, argv);
			break;

		case A_ARG:
			switch(f++)
			{
			case 0:
				/*
				 * Get the whole pathname.
				 */
				strcpy(pathname, a_arg(argc, argv));

				/*
				 * Get the filename and directory
				 */
				strcpy(directory, pathname);
				if((slash = rindex(directory, '/')) != NULL)
				{
					strcpy(filename, slash+1);
					*++slash = '\0';
				}
				else
				{
					directory[0] = '\0';
					strcpy(filename, pathname);
				}

				/*
				 * If the filename has no extension, or if it
				 * has an extension and it is not '.dvi' then
				 * cat .dvi onto the filename.
				 */
				if((extension = rindex(pathname, '.')) == NULL ||
				  strcmp(extension, ".dvi") != 0)
					strcat(pathname, ".dvi");

				break;

			default:
				fprintf(stderr,
  "%s: too many dvi files\n", a_prog_name);
				exit(1);
			}
			break;

		default:
			fprintf(stderr, "%s: illegal flag -%c\n",
			  a_prog_name, opt);
			exit();
		}
	}

	pw = canvas_pixwin(disp_canvas);

	/*
	 * Now that we know whether we are on a colour machine or a monochrome,
	 * we can set the defaults for the resolution and sampling, unless
	 * they have already been set from the args.
	 */
	if(fake_mono || (pw->pw_pixrect->pr_depth == 1))
	{
		/*
		 * Monochrome
		 */
		mono = TRUE;

		if(resolution == 0)
			resolution = DEFAULT_MONO_RES;
		if(sampling == 0)
			sampling = DEFAULT_MONO_SAMPLING;
	}
	else
	{
		/*
		 * Colour
		 */
		mono = FALSE;

		if(resolution == 0)
			resolution = DEFAULT_COLOUR_RES;
		if(sampling == 0)
			sampling = DEFAULT_COLOUR_SAMPLING;

		/*
		 * Set a colour map
		 */
		pw_setcmsname(pw, "16-reverse-greys");
		pw_putcolormap(pw, 0, 16, cmap, cmap, cmap);
	}

	/*
	 * Now that we know the resolution and sampling, we can set
	 * the margin and origin properly.
	 */
	if(set_origin_x != 0.0)
		origin_x = (int)(set_origin_x * resolution);
	else
		origin_x = (int)(DEFAULT_ORIGIN_X * resolution);

	if(set_origin_y != 0.0)
		origin_y = (int)(set_origin_y * resolution);
	else
		origin_y = (int)(DEFAULT_ORIGIN_Y * resolution);

	if(set_start_x != 0.0)
		start_x = (int)(set_start_x * resolution);
	else
		start_x = 0;

	if(set_start_y != 0.0)
		start_y = (int)(set_start_y * resolution);
	else
		start_y = 0;

	/*
	 * Insert the window into the heap now, so that if a message is
	 * generated by the init_dvi_file below, it is displayed after the
	 * window, and is therefore on top of it. If we display the window
	 * after doing the initialisation of the dvi file, it would obscure
	 * any error messages which might have been generated.
	 */
	window_set(disp_frame,
	  WIN_SHOW, TRUE,
	  0);

	/*
	 * If we don't run the notifier at this time, the window will
	 * not be painted, and the effect will be a gross area of the
	 * screen which is not painted, through which the previous windows
	 * are still visible.
	 */
	notify_dispatch();

	/*
	 * If there was a filename specified, then open it
	 * and prepare the first page.
	 */
	if(f >= 1)
	{
		/*
		 * Init the file.
		 */
		if(init_dvi_file())
		{
			process_page(RASTERISE);
			offset_x = start_x / sampling;
			offset_y = start_y / sampling;
			page_paint(disp_canvas, pw, 0);
		}
	}
	else
		getwd(directory);

	/*
	 * Loop forever in the notifier.
	 */
	notify_start();
}

/*
 * Window Functions.
 * ================
 */

/*
 * page_paint:
 *	Called whenever the window is to be painted.
 *	Just maps the sampled pixrect into the screen at the appropriate
 *	offset position.
 */

Notify_value
page_paint(canvas, pw, area)
Canvas canvas;
Pixwin *pw;
Rectlist *area;
{
	if(sample_pr == NULL)
	{
		pw_rop(pw, 0, 0,
		  (int)window_get(canvas, WIN_WIDTH),
		  (int)window_get(canvas, WIN_HEIGHT),
		  PIX_CLR, NULL, 0, 0);

		sprintf(label, "%s:   No File", DVIPAGE_LABEL);
		window_set(disp_frame,
		  FRAME_LABEL, label,
		  0);
	}
	else
	{
		pw_cover(pw, 0, 0,
		  (int)window_get(canvas, WIN_WIDTH),
		  (int)window_get(canvas, WIN_HEIGHT),
		  PIX_SRC, sample_pr, -offset_x, -offset_y);

		sprintf(label, "%s:   File \"%s\"   %s %d",
		  DVIPAGE_LABEL, filename,
		  (last_sheet && disp_sheet >= last_sheet-1)?
		  "Last page" : "Page",
		  disp_page);
		window_set(disp_frame,
		  FRAME_LABEL, label,
		  0);
	}
}

/*
 * page_event:
 *	Called whenever an input event arrives at the window.
 *	Controls panning of the page, turning to the next page,
 *	and reloading a new file.
 */

Notify_value
page_event(canvas, event, arg)
Window canvas;
Event *event;
caddr_t arg;
{
	Pixwin *pw;
	int e;
	int pp;
	bool page_or_sheet;
	static int num = 0;
	static bool valid_num = FALSE;
	bool keep_num;
	char *extension;
	char command[STRSIZE];
	double x, y;

	if(event_is_up(event))
		return;

	pw = canvas_pixwin(canvas);

	keep_num = FALSE;

	if(verbose & DEBUG_SHEET)
		fprintf(stderr, "page_event: num = %d @ %d\n", num, valid_num);

	/*
	 * If there is a call for a menu, then translate that into
	 * a command character.
	 */
	if((e = event_id(event)) == MS_RIGHT)
		if((e = page_menu(canvas, pw, event)) == 0)
			return;

	switch(e)
	{
	case MS_LEFT:
		page_magnify(canvas, pw, event);
		break;

	case MS_MIDDLE:
		page_pan(canvas, pw, event);
		break;

	default:
		if(e >= '0' && e <= '9')
			num = num * 10 + e - '0';
		else if(e == DEL || e == Control('H'))
			num = num / 10;
		else
			break;

		keep_num = TRUE;
		valid_num = TRUE;

		break;

	case '\r':		/* Next page */
	case 'n':
	case ' ':
	case '+':
		if(! valid_num)
			num = 1;
		if(! goto_sheet(disp_sheet + num))
			break;

		process_page(RASTERISE);
		offset_x = start_x / sampling;
		offset_y = start_y / sampling;
		page_paint(canvas, pw, 0);

		break;

	case '\n':		/* Previous page */
	case 'p':
	case '-':
		if(! valid_num)
			num = 1;
		if(! goto_sheet(disp_sheet - num))
			break;

		process_page(RASTERISE);
		offset_x = start_x / sampling;
		offset_y = start_y / sampling;
		page_paint(canvas, pw, 0);

		break;

	case '_':
		num = 1;
		valid_num = TRUE;
		/* FALLTHROUGH */

	case 'G':
		if(! valid_num)
			num = LAST_PAGE;

		if(! goto_sheet(num))
			break;

		process_page(RASTERISE);
		offset_x = start_x / sampling;
		offset_y = start_y / sampling;
		page_paint(canvas, pw, 0);

		break;

	case 'g':
		if(! valid_num)
			num = LAST_PAGE;

		if(! goto_page(num))
			break;

		process_page(RASTERISE);
		offset_x = start_x / sampling;
		offset_y = start_y / sampling;
		page_paint(canvas, pw, 0);

		break;

	case 'h':		/* Home page */
		offset_x = start_x / sampling;
		offset_y = start_y / sampling;
		page_paint(canvas, pw, 0);
		break;

	case 'l':		/* Left page */
		offset_x -= 200;
		page_paint(canvas, pw, 0);
		break;

	case 'r':		/* Right page */
		offset_x += 200;
		page_paint(canvas, pw, 0);
		break;

	case 'u':		/* Up page */
		offset_y -= 300;
		page_paint(canvas, pw, 0);
		break;

	case 'd':		/* Down page */
		offset_y += 300;
		page_paint(canvas, pw, 0);
		break;

		
	case 'm':		/* Mark margins */
		start_x = offset_x * sampling;
		start_y = offset_y * sampling;
		break;

	case 'M':		/* Set margins */
		x = ((double)start_x / resolution);
		y = ((double)start_y / resolution);
		if(! doubles_prompt(1152/2, 900/2,
		  "left margin: (inches) ", &x,
		  "top margin: (inches)  ", &y,
		  0))
			break;
		start_x = (int)(x * resolution);
		start_y = (int)(y * resolution);
		offset_x = start_x / sampling;
		offset_y = start_y / sampling;
		page_paint(canvas, pw, 0);
		break;

	case '%':
	case '!':
	case '@':
	case '#':
	case '$':
		valid_num = TRUE;
		if(e == '%')
			num = (mono ?
			  DEFAULT_MONO_SAMPLING : DEFAULT_COLOUR_SAMPLING);
		else if(e == '!')
			num = 1;
		else if(e == '@')
			num = 2;
		else if(e == '#')
			num = 3;
		else if(e == '$')
			num = 4;
		else
			valid_num = FALSE;
		/* FALLTHROUGH */

	case 's':
		if(! valid_num || (num < 1 || num > 4))
			sampling = (mono ?
			  DEFAULT_MONO_SAMPLING : DEFAULT_COLOUR_SAMPLING);
		else
			sampling = num;

		sample_page();
		offset_x = start_x / sampling;
		offset_y = start_y / sampling;
		page_paint(canvas, pw, 0);
		break;

	case 'S':
		if(! integers_prompt(1152/2, 900/2,
		  "sampling: (1, 2, 3, 4) ", &sampling,
		  0))
			break;
		if(sampling < 1 || sampling > 4)
			sampling = (mono ?
			  DEFAULT_MONO_SAMPLING : DEFAULT_COLOUR_SAMPLING);

		sample_page();
		offset_x = start_x / sampling;
		offset_y = start_y / sampling;
		page_paint(canvas, pw, 0);
		break;

	case 'x':
		if(valid_num)
			mag_size_x = num;
		else
			mag_size_x = DEFAULT_MAG_SIZE_X;
		break;

	case 'y':
		if(valid_num)
			mag_size_y = num;
		else
			mag_size_y = DEFAULT_MAG_SIZE_Y;
		break;

	case 'X':
	case 'Y':
		if(! integers_prompt(1152/2, 900/2,
		  "magnifier size (x) : ", &mag_size_x,
		  "magnifier size (y) : ", &mag_size_y,
		  0))
			break;
		break;

	case '[':
		mag_size_x = 128;
		mag_size_y = 64;
		break;

	case ']':
		mag_size_x = 128;
		mag_size_y = 128;
		break;

	case '{':
		mag_size_x = 256;
		mag_size_y = 128;
		break;

	case '}':
		mag_size_x = 256;
		mag_size_y = 256;
		break;

	case '(':
		mag_size_x = 512;
		mag_size_y = 256;
		break;

	case ')':
		mag_size_x = 512;
		mag_size_y = 512;
		break;

	case 'b':
		if(valid_num)
			mag_border = num;
		else
			mag_border = DEFAULT_MAG_BORDER;
		break;

	case 'F':
		if(! strings_prompt(1152/2, 900/2,
		  "Directory: ", directory,
		  "Filename:  ", filename,
		  0))
			break;

		/*
		 * Build the whole pathname.
		 */
		if(directory[0] != '\0')
		{
			strcpy(pathname, directory);
			if(pathname[strlen(pathname)-1] != '/')
				strcat(pathname, "/");
			strcat(pathname, filename);
		}
		else
			strcpy(pathname, filename);

		/*
		 * If the filename has no extension, or if it
		 * has an extension and it is not '.dvi' then
		 * cat .dvi onto the filename.
		 */
		if((extension = rindex(pathname, '.')) == NULL ||
		  strcmp(extension, ".dvi") != 0)
			strcat(pathname, ".dvi");

		sprintf(label, "%s:   Opening file \"%s\"",
		  DVIPAGE_LABEL, filename);
		window_set(disp_frame,
		  FRAME_LABEL, label,
		  0);
		close_dvi_file();
		if(init_dvi_file())
		{
			process_page(RASTERISE);
			offset_x = start_x / sampling;
			offset_y = start_y / sampling;
			page_paint(canvas, pw, 0);
		}
		break;

	case 'R':		/* Reopen file */
		sprintf(label, "%s:   Reopening file \"%s\"",
		  DVIPAGE_LABEL, filename);
		window_set(disp_frame,
		  FRAME_LABEL, label,
		  0);
		if(valid_num)
		{
			pp = num;
			page_or_sheet = TRUE;
		}
		else
		{
			pp = disp_sheet;
			page_or_sheet = FALSE;
		}
		close_dvi_file();
		if(init_dvi_file())
		{
			if(page_or_sheet)
			{
				if(! goto_page(pp))
					(void)goto_sheet(1);
			}
			else
			{
				if(! goto_sheet(pp))
					(void)goto_sheet(1);
			}

			process_page(RASTERISE);
			offset_x = start_x / sampling;
			offset_y = start_y / sampling;
			page_paint(canvas, pw, 0);
		}
		break;

	case 'P':
		sprintf(command, print_page_spooler,
		  disp_page, disp_page, pathname);
		if(verbose & DEBUG_PRINTER)
			fprintf(stderr, "Printer command '%s'\n", command);
		system(command);
		break;

	case Control('P'):
		sprintf(command, print_spooler, pathname);
		if(verbose & DEBUG_PRINTER)
			fprintf(stderr, "Printer command '%s'\n", command);
		system(command);
		break;

	case 'w':
		show_page_frame = ! show_page_frame;

		if(goto_sheet(disp_sheet))
		{
			process_page(RASTERISE);
			offset_x = start_x / sampling;
			offset_y = start_y / sampling;
			page_paint(canvas, pw, 0);
		}

		break;

	case 'v':
		if(valid_num)
			verbose = num;
		else
			verbose = 0;
		break;

	case 'q':
		exit();
	}

	if(! keep_num)
	{
		num = 0;
		valid_num = FALSE;
	}

	if(verbose & DEBUG_SHEET)
		fprintf(stderr, "end__event: num = %d @ %d\n", num, valid_num);
}

/*
 * page_menu:
 *	Displays a menu in response to MS_RIGHT, returns a character
 *	code to calling function to effect action.
 */

int
page_menu(canvas, pw, event)
Canvas canvas;
Pixwin *pw;
Event *event;
{
	static Menu menu = NULL;
	int action;

	if(! menu)
	{
		/*
		 * Return values from this menu are passed to the
		 * event procedure, and the codes here must match
		 * codes in that function.
		 */
		menu = menu_create(
		  MENU_ITEM,
		    MENU_STRING, "Next Page",
		    MENU_VALUE, 'n',
		    0,
		  MENU_ITEM,
		    MENU_STRING, "Previous Page",
		    MENU_VALUE, 'p',
		    0,
		  MENU_ITEM,
		    MENU_STRING, "First Page",
		    MENU_VALUE, '_',
		    0,
		  MENU_ITEM,
		    MENU_STRING, "Last Page",
		    MENU_VALUE, 'G',
		    0,
		  MENU_ITEM,
		    MENU_STRING, "Sampling",
		    MENU_PULLRIGHT, menu_create(
		      MENU_ITEM,
			MENU_STRING, "Default Sampling",
			MENU_VALUE, '%',
			0,
		      MENU_ITEM,
			MENU_STRING, "No Sampling",
			MENU_VALUE, '!',
			0,
		      MENU_ITEM,
			MENU_STRING, "2:1 Sampling",
			MENU_VALUE, '@',
			0,
		      MENU_ITEM,
			MENU_STRING, "3:1 Sampling",
			MENU_VALUE, '#',
			0,
		      MENU_ITEM,
			MENU_STRING, "4:1 Sampling",
			MENU_VALUE, '$',
			0,
		      0),
		    0,
		  MENU_ITEM,
		    MENU_STRING, "Magnifier",
		    MENU_PULLRIGHT, menu_create(
		      MENU_ITEM,
			MENU_STRING, "128 x 64",
			MENU_VALUE, '[',
			0,
		      MENU_ITEM,
			MENU_STRING, "128 x 128",
			MENU_VALUE, ']',
			0,
		      MENU_ITEM,
			MENU_STRING, "256 x 128",
			MENU_VALUE, '{',
			0,
		      MENU_ITEM,
			MENU_STRING, "256 x 256",
			MENU_VALUE, '}',
			0,
		      MENU_ITEM,
			MENU_STRING, "512 x 256",
			MENU_VALUE, '(',
			0,
		      MENU_ITEM,
			MENU_STRING, "512 x 512",
			MENU_VALUE, ')',
			0,
		      0),
		    0,
		  MENU_ITEM,
		    MENU_STRING, "Reopen DVI file",
		    MENU_VALUE, 'R',
		    0,
		  MENU_ITEM,
		    MENU_STRING, "New DVI file",
		    MENU_VALUE, 'F',
		    0,
		  MENU_ITEM,
		    MENU_STRING, "Print Page",
		    MENU_VALUE, 'P',
		    0,
		  MENU_ITEM,
		    MENU_STRING, "Print Document",
		    MENU_VALUE, Control('P'),
		    0,
		  MENU_ITEM,
		    MENU_STRING, "Quit",
		    MENU_VALUE, 'q',
		    0,
		  0);
	}

	return (int)menu_show(menu, canvas, event, 0);
}

/*
 * page_magnify:
 *	Pops up magnified region (only if sampling != 1).
 *	Currently unimplemented.
 */

void
page_magnify(canvas, pw, event)
Canvas canvas;
Pixwin *pw;
Event *event;
{
	Rect r;
	int w, h;
	double scale_x, scale_y;
	int sample_w, sample_h;
	int new_sample_x, new_sample_y;
	int delta_x, delta_y;
	int sample_x, sample_y;
	int dst_x, dst_y;
	int page_x, page_y;
	int old_cursor_op;
	bool first_time = TRUE;
	int win_x, win_y;

	if(sampling == 1 || sample_pr == NULL)
		return;

	if(mag_size_x < 4)
		mag_size_x = 4;
	if(mag_size_y < 4)
		mag_size_y = 4;
	if(mag_size_x > sample_pr->pr_width)
		mag_size_x = sample_pr->pr_width;
	if(mag_size_y > sample_pr->pr_height)
		mag_size_y = sample_pr->pr_height;
	if(mag_border < 0)
		mag_border = 0;
	if(mag_border > 8)
		mag_border = 8;

	/*
	 * Get Lock rect.
	 */
	r.r_left = 0;
	r.r_top = 0;
	r.r_width = (int)window_get(canvas, WIN_WIDTH);
	r.r_height = (int)window_get(canvas, WIN_HEIGHT);

	/*
	 * Precompute some window sizes.
	 */
	w = sample_pr->pr_width;
	h = sample_pr->pr_height;
	switch(sampling)
	{
	case 2:
		scale_x = 2.0;
		scale_y = 2.0;
		break;

	case 3:
		scale_x = 8.0 / 3.0;
		scale_y = 3.0;
		break;

	case 4:
		scale_x = 4.0;
		scale_y = 4.0;
		break;

	default:
		return;
	}
	sample_w = mag_size_x / scale_x;
	sample_h = mag_size_y / scale_y;

	if(verbose & DEBUG_MAGNIFY)
		fprintf(stderr, "page_magnify: scale %lf %lf; %d %d -> %d %d\n",
		  scale_x, scale_y, sample_w, sample_h, mag_size_x, mag_size_y);

	/*
	 * Remove the cursor
	 */
	old_cursor_op = (int)cursor_get(hand_cursor, CURSOR_OP);
	cursor_set(hand_cursor, CURSOR_OP, PIX_DST, 0);
	window_set(canvas, WIN_CURSOR, hand_cursor, 0);

	/*
	 * Grab all input
	window_set(canvas, WIN_GRAB_ALL_INPUT, TRUE, 0);
	 */

	/*
	 * Loop until up mouse.
	 */
	sample_x = MAXINT;
	sample_y = MAXINT;
	while(! event_is_up(event))
	{
		/*
		 * Compute the region which will be magnified.
		 */
		new_sample_x =
		  Range(0, event_x(event)-offset_x-sample_w/2, w-sample_w);
		new_sample_y =
		  Range(0, event_y(event)-offset_y-sample_h/2, h-sample_h);

		/*
		 * See how this differs from last magnified region.
		 */
		delta_x = new_sample_x - sample_x;
		delta_y = new_sample_y - sample_y;

		/*
		 * Lock
		 */
		pw_lock(pw, &r);

		if(! first_time)
		{
			if(verbose & DEBUG_MAGNIFY)
				fprintf(stderr, " covering with %d %d\n",
				  delta_x, delta_y);

			/*
			 * Paint those portions of the image which were
			 * covered by the last magnifier, and exposed now.
			 * We could just paint the entire patch, but this
			 * gives unpleasant flashing when moving the window.
			 */
			if(delta_x > 0)
				pw_cover(pw, win_x, win_y,
				  delta_x, mag_size_y,
				  PIX_SRC, sample_pr, dst_x, dst_y);
			else if(delta_x < 0)
				pw_cover(pw, win_x+mag_size_x+delta_x, win_y,
				  -delta_x, mag_size_y,
				  PIX_SRC, sample_pr,
				  dst_x+mag_size_x+delta_x, dst_y);
			if(delta_y > 0)
				pw_cover(pw, win_x, win_y,
				  mag_size_x, delta_y,
				  PIX_SRC, sample_pr, dst_x, dst_y);
			else if(delta_y < 0)
				pw_cover(pw, win_x, win_y+mag_size_y+delta_y,
				  mag_size_x, -delta_y,
				  PIX_SRC, sample_pr,
				  dst_x, dst_y+mag_size_y+delta_y);
		}
		else
			first_time = FALSE;

		/*
		 * Compute the new destination and window positions
		 * for the new magnified region.
		 */
		sample_x = new_sample_x;
		sample_y = new_sample_y;
		dst_x = sample_x - (mag_size_x - sample_w)/2;
		dst_y = sample_y - (mag_size_y - sample_h)/2;
		win_x = dst_x + offset_x;
		win_y = dst_y + offset_y;
		page_x = sample_x * scale_x;
		page_y = sample_y * scale_y;

		if(verbose & DEBUG_MAGNIFY)
			fprintf(stderr, " painting at %d %d from %d %d\n",
			  dst_x, dst_y, page_x, page_y);

		/*
		 * Display the magnified region.
		 */
		pw_write(pw, win_x, win_y, mag_size_x, mag_size_y,
		  PIX_SRC, page_pr, page_x, page_y);
		if(mag_border)
			pw_rect(pw, win_x, win_y, mag_size_x, mag_size_y,
			  mag_border, PIX_SRC, -1);

		/*
		 * Unlock
		 */
		pw_unlock(pw);

		/*
		 * Read another event.
		 */
		window_read_event(canvas, event);
	}

	/*
	 * Ungrab all input.
	 */
	window_set(canvas, WIN_GRAB_ALL_INPUT, FALSE, 0);

	/*
	 * Repaint
	 */
	pw_cover(pw, win_x, win_y, mag_size_x, mag_size_y,
	  PIX_SRC, sample_pr, dst_x, dst_y);

	/*
	 * Restore the cursor.
	 */
	cursor_set(hand_cursor, CURSOR_OP, old_cursor_op, 0);
	window_set(canvas, WIN_CURSOR, hand_cursor, 0);
}

/*
 * page_pan:
 *	Pans page within screen.
 */

void
page_pan(canvas, pw, event)
Canvas canvas;
Pixwin *pw;
Event *event;
{
	int x, y;
	int dx, dy;

	if(sample_pr == NULL)
		return;

	window_set(canvas,
	  WIN_GRAB_ALL_INPUT, TRUE,
	  0);

	do
	{
		x = event_x(event);
		y = event_y(event);

		window_read_event(canvas, event);

		dx = event_x(event) - x;
		dy = event_y(event) - y;

		if(dx != 0 || dy != 0)
		{
			offset_x += dx;
			offset_y += dy;

			pw_cover(pw, 0, 0,
			  (int)window_get(canvas, WIN_WIDTH),
			  (int)window_get(canvas, WIN_HEIGHT),
			  PIX_SRC, sample_pr, -offset_x, -offset_y);
		}
	}
	while(! event_is_up(event));

	window_set(canvas,
	  WIN_GRAB_ALL_INPUT, FALSE,
	  0);
}

/*
 * goto_sheet:
 *	Opens requested sheet on screen.
 */

bool
goto_sheet(new_sheet)
int new_sheet;
{
	int sheet;

	if(! check_dvi_file())
		return FALSE;

	if(verbose & DEBUG_SHEET)
		fprintf(stderr, "goto_sheet(%d)\n", new_sheet);

	/*
	 * Check against page limits.
	 */
	if(new_sheet <= 0)
	{
		message("Attempt to go to sheet %d", new_sheet);
		return FALSE;
	}

	/*
	 * Are we already at the desired page ?
	 */
	if(file_sheet == new_sheet)
		return TRUE;

	/*
	 * Do we already know where the page is ?
	 */
	if(new_sheet < MAX_SHEETS && new_sheet <= last_known_sheet)
	{
		fseek(dvifp, sheet_table[new_sheet], 0);
		file_sheet = new_sheet;
		return TRUE;
	}

	/*
	 * Can't find it directly in the table:
	 * Go to the last known sheet...
	 */
	file_sheet = last_known_sheet;
	fseek(dvifp, sheet_table[file_sheet], 0);

	/*
	 * Skip through the rest of the pages to the new page.
	 */
	while(file_sheet < new_sheet)
	{
		/*
		 * Check for last page:
		 * Last page is always returned.
		 */
		if(last_sheet && file_sheet >= last_sheet)
		{
			file_sheet = last_sheet - 1;
			fseek(dvifp, sheet_table[file_sheet], 0);
			return TRUE;
		}

		/*
		 * Otherwise, skip this page and look at the next.
		 */
		process_page(SKIP);
	}
	return TRUE;
}

/*
 * goto_page:
 *	Opens requested page on screen.
 */

bool
goto_page(new_page)
int new_page;
{
	int sheet;

	if(! check_dvi_file())
		return FALSE;

	if(verbose & DEBUG_SHEET)
		fprintf(stderr, "goto_page(%d)\n", new_page);

	/*
	 * Search for page in the table.
	 */
	for(sheet = 1; sheet < last_known_sheet; sheet++)
	{
		if(sheet_page[sheet] == new_page)
		{
			file_sheet = sheet;
			fseek(dvifp, sheet_table[file_sheet], 0);
			return TRUE;
		}
	}

	/*
	 * Can't find it directly in the table:
	 * Go to the last known sheet...
	 */
	file_sheet = last_known_sheet;
	fseek(dvifp, sheet_table[file_sheet], 0);

	/*
	 * Skip through the rest of the pages to the new page.
	 */
	for( ; ; )
	{
		/*
		 * Check for last page:
		 */
		if(last_sheet && file_sheet >= last_sheet)
		{
			if(new_page == LAST_PAGE)
			{
				file_sheet = last_sheet - 1;
				fseek(dvifp, sheet_table[file_sheet], 0);
				return TRUE;
			}
			else
				return FALSE;
		}

		/*
		 * Otherwise, examine this page.
		 */
		sheet = file_sheet;
		process_page(SKIP);

		/*
		 * If this was the page, go back,
		 * and return it.
		 */
		if(sheet_page[sheet] == new_page)
		{
			file_sheet = sheet;
			fseek(dvifp, sheet_table[file_sheet], 0);
			return TRUE;
		}
	}
}

/*
 * DVI file functions.
 * ==================
 */

/*
 * init_dvi_file:
 *	Opens the dvi file, and checks for valid codes etc.
 *	Reads the postamble (if enabled)
 *	Leaves the file pointer at the start of the first page.
 */

bool
init_dvi_file()
{
	int i;

	/*
	 * Open the file
	 */
	if((dvifp = fopen(pathname, "r")) == NULL)
	{
		message("Cant open file %s", pathname);
		return FALSE;
	}

	/*
	 * Read the magic number and version number
	 */
	if((i = get_unsigned(dvifp, 1)) != PRE)
	{
		message("%s: not a dvi file", filename);
		fclose(dvifp);
		return FALSE;
	}
	if((i = get_signed(dvifp, 1)) != DVIFORMAT)
	{
		message("%s: dvi format %d not supported", filename, i);
		fclose(dvifp);
		return FALSE;
	}

	/*
	 * Make a note of the access time.
	 */
	if(fstat(fileno(dvifp), &stat_buf) == 0)
	{
		mtime = stat_buf.st_mtime;
	}
	else
	{
		message("%s: dvifile stat failed.", filename);
		mtime = 0;
	}

	if(pre_load)
	{
		/*
		 * Load font information from postable.
		 */
		if(! read_postamble())
		{
			fclose(dvifp);
			return FALSE;
		}

		/*
		 * Return to start of first page.
		 */
		fseek(dvifp, (long)14, 0);
	}
	else
	{
		/*
		 * Read basic data from preamble.
		 */
		num = get_unsigned(dvifp, 4);
		den = get_unsigned(dvifp, 4);
		mag = get_unsigned(dvifp, 4);
		hconv = vconv = do_convert(num, den, resolution);
	}

	/*
	 * Skip i more bytes of preamble.
	 */
	i = get_unsigned(dvifp, 1);
	fseek(dvifp, (long)i, 1);

	/*
	 * Allocate buffer for the page.
	 */
	if(! (page_pr = pr_alloc(&page_mpr,
	  (int)(page_w * resolution), (int)(page_h * resolution), 1)))
	{
		message("Out of memory for image allocation");
		fclose(dvifp);
		return FALSE;
	}

	if(verbose & DEBUG_IMSIZE)
		fprintf(stderr, "Allocated buffer (%d x %d)\n",
		  page_pr->pr_width, page_pr->pr_height);

	/*
	 * Set up the page fseek pointer table.
	 * We are now at page 0.
	 */
	for(i = 0; i < MAX_SHEETS; i++)
	{
		sheet_table[i] = 0;
		sheet_page[i] = BAD_PAGE;
	}
	file_sheet = 1;
	last_sheet = 0;	/* last page == unknown */
	last_known_sheet = 1;
	sheet_table[last_known_sheet] = ftell(dvifp);

	if(verbose & DEBUG_SHEET)
		fprintf(stderr, "sheet_table[%d] = %d\n",
		  last_known_sheet, ftell(dvifp));

	return TRUE;
}

/*
 * close_dvi_file:
 *	Cleans up after reading a file.
 */

void
close_dvi_file()
{
	if(dvifp == NULL)
		return;

	/*
	 * Get rid of image memory.
	 */
	sample_pr = pr_free(&sample_mpr);
	page_pr = pr_free(&page_mpr);

	/*
	 * close the dvifile.
	 */
	fclose(dvifp);
	dvifp = NULL;
	mtime = 0;

	/*
	 * Hack the sheet numbers to prevent access to the file.
	 */
	last_sheet = -1;
	last_known_sheet = -1;

	/*
	 * Close the fonts and free up memory.
	 */
	close_fonts();
}

/*
 * check_dvi_file:
 *	Checks that this is the same file -- has not been modified since
 *	it was opened.
 */

bool
check_dvi_file()
{
	if(dvifp == NULL)
	{
		message("No dvifile open");
		return FALSE;
	}

	if(fstat(fileno(dvifp), &stat_buf) != 0)
	{
		message("%s: dvifile fstat failed.", filename);
		return FALSE;
	}

	if(stat_buf.st_mtime != mtime)
	{
		message("%s: dvifile modified", filename);
		return FALSE;
	}

	return TRUE;
}

/*
 * read_postamble:
 *	This  routine  is  used  to  read  in  the  postamble  values.	It
 *	initializes the magnification and checks  the stack height prior  to
 *	starting printing the document.
 */

bool
read_postamble()
{
	if(! check_dvi_file())
		return FALSE;

	if(! find_postamble_ptr (&postambleptr))
		return FALSE;

	if(get_unsigned(dvifp, 1) != POST)
	{
		message("%s: bad dvi file: no POST at head of postamble",
		  filename);
		return FALSE;
	}

	(void)get_unsigned(dvifp, 4); /* discard last page pointer */
	num = get_unsigned(dvifp, 4);
	den = get_unsigned(dvifp, 4);
	mag = get_unsigned(dvifp, 4);
	hconv = vconv = do_convert(num, den, resolution);

	get_unsigned(dvifp, 4);	/* height-plus-depth of tallest page */
	get_unsigned(dvifp, 4);	/* width of widest page */

	if(get_unsigned(dvifp, 2) >= STACKSIZE)
	{
		message("%s: bad dvi file: stack is too large",
		  filename);
		return FALSE;
	}

	/* last_sheet = */ get_unsigned(dvifp, 2);

	if(! get_font_def())
		return FALSE;

	return TRUE;
}

/*
 * find_post_amble_ptr
 *	Move to the end of the dvifile and find the start of the postamble.
 */

bool
find_postamble_ptr(postambleptr)
long *postambleptr;
{
	int i;

	fseek(dvifp, (long) 0, 2);
	*postambleptr = ftell(dvifp) - 4;
	fseek(dvifp, *postambleptr, 0);

	for( ; ; )
	{
		fseek(dvifp, --(*postambleptr), 0);
		if(((i = get_unsigned(dvifp, 1)) != 223) && (i != DVIFORMAT))
		{
			message(
  "%s: Bad dvi file: bad end of file", filename);
			return FALSE;
		}
		if (i == DVIFORMAT)
			break;
	}

	fseek(dvifp, (*postambleptr) - 4, 0);
	*postambleptr = get_unsigned(dvifp, 4);
	fseek(dvifp, *postambleptr, 0);

	return TRUE;
}

/*
 * process_page:
 *	Rasterises the next page in the dvifile into page_mpr.
 *	Leaves the file pointer at the start of the next page.
 *
 *	If skip mode is true, then nothing is actually drawn, the commands
 *	are interpreted only for the side effect of moving the filepointer
 *	to the next page.
 */

bool
process_page(skip_mode)
register bool skip_mode;
{
	int command;	    /* current command			     */
	register int i;	    /* command parameter; loop index	     */
	int k;		    /* temporary parameter		     */
	int val, val2;      /* temporarys to hold command information*/
	int w;		    /* current horizontal spacing	     */
	int x;		    /* current horizontal spacing	     */
	int y;		    /* current vertical spacing		     */
	int z;		    /* current vertical spacing		     */
	int counter[10];
	int sp;		    /* stack pointer			     */
	static struct stack_entry stack[STACKSIZE];   /* stack */

	if(! check_dvi_file())
		return FALSE;

	if(verbose & DEBUG_SHEET)
		fprintf(stderr, "sheet %d starts at %d\n",
		  file_sheet, ftell(dvifp));

	while((command = get_unsigned(dvifp, 1)) != EOP)
	{
		switch(command)
		{

		case SET1:
		case SET2:
		case SET3:
		case SET4:
			val = get_unsigned(dvifp, command-SET1+1);
			if(! skip_mode)
				set_char(val, command);
			break;

		case SET_RULE:
			val = get_unsigned(dvifp, 4);
			val2 = get_unsigned(dvifp, 4);
			if(! skip_mode)
				set_rule(val, val2, 1);
			break;

		case PUT1:
		case PUT2:
		case PUT3:
		case PUT4:
			val = get_unsigned(dvifp,command-PUT1+1);
			if(! skip_mode)
				set_char(val, command);
			break;

		case PUT_RULE:
			val = get_unsigned(dvifp, 4);
			val2 = get_unsigned(dvifp, 4);
			if(! skip_mode)
				set_rule(val, val2, 0);
			break;

		case NOP:
			break;

		case BOP:
			/*
			 * These are the 10 counters.
			 * Discard previous page pointer.
			 */
			for(i=0; i<10; i++)
				counter[i] = get_unsigned(dvifp, 4);
			(void)get_unsigned(dvifp, 4);

			/*
			 * The first counter is the page number.
			 */
			disp_page = counter[0];
			if(file_sheet < MAX_SHEETS)
				sheet_page[file_sheet] = disp_page;

			/*
			 * Show what is happening.
			 */
			sprintf(label, "%s:   File \"%s\"   Page %d   %s",
			  DVIPAGE_LABEL, filename,
			  disp_page, (skip_mode) ? "Skipping" : "Processing");
			window_set(disp_frame,
			  FRAME_LABEL, label,
			  0);

			if(! skip_mode)
			{
				/*
				 * Clear the page
				 */
				pr_rop(page_pr, 0, 0,
				  page_pr->pr_width, page_pr->pr_height,
				  PIX_CLR, NULL, 0, 0);

				/*
				 * Mark the edges of the page
				 */
				pr_rect(page_pr, 0, 0,
				  (int)(page_w * resolution),
				  (int)(page_h * resolution),
				  3, PIX_SET, 1);

				/*
				 * Mark the nominal page window.
				 */
				if(show_page_frame)
				{
					pr_rect(page_pr, 0+origin_x, 0+origin_y,
					  (int)(page_w*resolution) - 2*origin_x,
					  (int)(page_h*resolution) - 2*origin_y,
					  1, PIX_SET, 1);
				}
			}

			h = v = w = x = y = z = 0;
			sp = 0;
			fontptr = NULL;
			break;

		case PUSH:
			if (sp >= STACKSIZE)
			{
				message(
  "%s: Bad dvi file: stack overflow",
				  filename);
				return FALSE;
			}
			stack[sp].h = h;
			stack[sp].v = v;
			stack[sp].w = w;
			stack[sp].x = x;
			stack[sp].y = y;
			stack[sp].z = z;
			sp++;
			break;

		case POP:
			--sp;
			if (sp < 0)
			{
				message(
  "%s: Bad dvi file: stack underflow",
				  filename);
				return FALSE;
			}
			h = stack[sp].h;
			v = stack[sp].v;
			w = stack[sp].w;
			x = stack[sp].x;
			y = stack[sp].y;
			z = stack[sp].z;
			break;

		case RIGHT1:
		case RIGHT2:
		case RIGHT3:
		case RIGHT4:
			val = get_signed(dvifp,command-RIGHT1+1);
			if(! skip_mode)
				move_over(val);
			break;

		case W0:
			if(! skip_mode)
				move_over(w);
			break;

		case W1:
		case W2:
		case W3:
		case W4:
			w = get_signed(dvifp,command-W1+1);
			if(! skip_mode)
				move_over(w);
			break;

		case X0:
			if(! skip_mode)
				move_over(x);
			break;

		case X1:
		case X2:
		case X3:
		case X4:
			x = get_signed(dvifp,command-X1+1);
			if(! skip_mode)
				move_over(x);
			break;

		case DOWN1:
		case DOWN2:
		case DOWN3:
		case DOWN4:
			val = get_signed(dvifp,command-DOWN1+1);
			if(! skip_mode)
				move_down(val);
			break;

		case Y0:
			if(! skip_mode)
				move_down(y);
			break;

		case Y1:
		case Y2:
		case Y3:
		case Y4:
			y = get_signed(dvifp,command-Y1+1);
			if(! skip_mode)
				move_down(y);
			break;

		case Z0:
			if(! skip_mode)
				move_down(z);
			break;

		case Z1:
		case Z2:
		case Z3:
		case Z4:
			z = get_signed(dvifp,command-Z1+1);
			if(! skip_mode)
				move_down(z);
			break;

		case FNT1:
		case FNT2:
		case FNT3:
		case FNT4:
			if(! skip_mode)
				set_font_num(
				  get_unsigned(dvifp,command-FNT1+1));
			break;

		case XXX1:
		case XXX2:
		case XXX3:
		case XXX4:
			k = get_unsigned(dvifp,command-XXX1+1);
			while(k--)
				get_unsigned(dvifp, 1);
			break;

		case FNT_DEF1:
		case FNT_DEF2:
		case FNT_DEF3:
		case FNT_DEF4:
			if(pre_load)
				skip_font_def(
				  get_unsigned(dvifp, command-FNT_DEF1+1));
			else
				read_font_def(
				  get_unsigned(dvifp, command-FNT_DEF1+1));
			break;

		case PRE:
			message(
  "%s: Bad dvi file: preamble found within main section",
			  filename);
			return FALSE;

		case POST:
			fseek(dvifp, (long) -1, 1);
			last_sheet = file_sheet;

			/*
			 * We have done nothing, so there is no need to
			 * resample the page or increment the page counter.
			 */
			return FALSE;

		case POST_POST:
			message(
  "%s: Bad dvi file: postpostamble found within main section",
			  filename);
			return FALSE;

		default:
			if(command >= FONT_00 && command <= FONT_63)
			{
				if(! skip_mode)
					set_font_num(command - FONT_00);
			}
			else if(command >= SETC_000 && command <= SETC_127)
			{
				if(! skip_mode)
					set_char(command - SETC_000, command);
			}
			else
			{
				message(
  "%s: Bad dvi file: undefined command (%d) found",
				  filename, command);
				return FALSE;
			}
		}
	}

	/*
	 * End of page.
	 */
	if(! skip_mode)
	{
		/*
		 * Sample the page.
		 */
		sample_page();
		disp_sheet = file_sheet;
	}

	/*
	 * The file is now at the start of the next page.
	 */
	file_sheet++;
	if(file_sheet > last_known_sheet)
	{
		if(file_sheet < MAX_SHEETS)
		{
			last_known_sheet = file_sheet;
			sheet_table[file_sheet] = ftell(dvifp);
		}

		if(verbose & DEBUG_SHEET)
			fprintf(stderr, "sheet %d starts at %d\n",
			  file_sheet, ftell(dvifp));
	}

	return TRUE;
}

/*
 * Draw and Move Functions.
 * ========================
 */

/*
 * set_char:
 */

int buf[8];

void
set_char(c, command)
int c, command;
{
	register struct char_entry *ptr;  /* temporary char_entry pointer */
	int k;

	ptr = &(fontptr->ch[c]);
	hh = pix_round(h, hconv);
	vv = pix_round(v, vconv);
	if(!ptr->where.isloaded)
		if(! load_char(ptr))
			return;

	pr_rop(page_pr, hh-ptr->xOffset+origin_x, vv-ptr->yOffset+origin_y,
	  ptr->width, ptr->height, PIX_SRC | PIX_DST,
	  ptr->where.address.pixrectptr, 0, 0);

	if(command <= SET4)
		h += ptr->tfmw;

	return;
}

/*
 * load_char:
 */

bool
load_char(ptr)
register struct char_entry *ptr;
{
	register struct pixrect *pr;
	register int nshorts, i, col, nints;
	register short *dp, *sp;

	if(ptr->where.address.fileOffset == 0)
	{
		ptr->where.address.pixrectptr = NULL;
		return FALSE;
	}

	if(! open_font_file())
		return FALSE;

	fseek(pxlfp, ptr->where.address.fileOffset, 0);
	pr = mem_create(ptr->width, ptr->height, 1);
	nshorts = mpr_mdlinebytes(pr) >> 1;
	nints = (ptr->width + 31) >> 5;
	dp = mpr_d(pr)->md_image;
	for(col = 0; col < ptr->height; col++)
	{
		fread(buf, 4, nints, pxlfp);
		sp = (short *) &buf[0];
		for(i = nshorts; i > 0; i--)
			*dp++ = *sp++;
	}
	ptr->where.address.pixrectptr = pr;
	ptr->where.isloaded = TRUE;

	return TRUE;
}

/*
 * set_rule:
 *	This routine will draw a rule on the screen
 */

void
set_rule(a, b, Set)
int a, b;
bool Set;
{
	int ehh, evv;

	hh = pix_round(h, hconv);
	vv = pix_round(v-a, vconv);
	ehh = pix_round(h + b, hconv);
	evv = pix_round(v, vconv);

	if(hh == ehh)
		ehh++;
	if(vv == evv)
		vv--;
	if((a > 0) && (b > 0))
		pr_rop(page_pr, hh+origin_x, vv+origin_y,
		  ehh-hh, evv-vv, PIX_SET, NULL, 0, 0);
	if(Set)
	{
		h += b;
/*		v += a; */
	}
}

/*
 * move_down:
 */

void
move_down(a)
int a;
{
	v += a;
}


/*
 * move_over:
 */

void
move_over(b)
int b;
{
	h += b;
}

/*
 * set_font_num:/
 *	This routine is used to specify the font to be used in printing future
 *	chars.
 */

void
set_font_num(k)
int k;
{
	fontptr = hfontptr;
	while((fontptr!=NULL) && (fontptr->k!=k))
		fontptr = fontptr->next;

	if(fontptr == NULL)
		message("Font %d undefined", k);
}

/*
 * Font functions
 * ==============
 */

/*
 * get_font_def:
 *	Read the font  definitions as they  are in the  postamble of the  DVI
 *	file.  Note that the font directory  is not yet loaded.  In order  to
 *	adapt ourselves to the existing "verser" the following font paramters
 *	are  copied   onto   output   fontno  (4   bytes),  chksum,  fontmag,
 *	fontnamelength (1 byte), fontname.  At the end, a -1 is put onto  the
 *	file.
 */

bool
get_font_def()
{
	char	str[50], *calloc ();
	unsigned char   byte;
	int	 i, fnamelen;

	while(((byte = get_unsigned(dvifp, 1)) >= FNT_DEF1) &&
	  (byte <= FNT_DEF4))
	{
		switch (byte)
		{
		case FNT_DEF1:
			if(! read_font_def (get_unsigned(dvifp, 1)))
				return FALSE;
			break;

		case FNT_DEF2:
			if(! read_font_def (get_unsigned(dvifp, 2)))
				return FALSE;
			break;

		case FNT_DEF3:
			if(! read_font_def (get_unsigned(dvifp, 3)))
				return FALSE;
			break;

		case FNT_DEF4:
			if(! read_font_def (get_unsigned(dvifp, 4)))
				return FALSE;
			break;

		default:
			message(
  "%s: Bad dvi file: bad font specification", filename);
			return FALSE;
		}
	}
	if(byte != POST_POST)
	{
		message(
  "%s: Bad dvi file: no postpostamble after fontdefs", filename);
		return FALSE;
	}

	return TRUE;
}

/*
 * close_fonts:
 *	Closes all the font files, and frees up all the memory.
 */

void
close_fonts()
{
	register struct font_entry *pf, *next;
	register struct pixrect *pr;
	register int i;

	for(pf = hfontptr; pf; pf = next)
	{
		if(verbose & DEBUG_FONTS)
			fprintf(stderr, "Freeing font %s\n", pf->name);

		if(pf->font_file_fd)
			fclose(pf->font_file_fd);

		for(i = 0; i < NPXLCHARS; i++)
		{
			if(pf->ch[i].where.isloaded == TRUE)
			{
				pr = pf->ch[i].where.address.pixrectptr;
				pr_destroy(pr);
			}
		}

		free(pf);

		next = pf->next;
	}

	pfontptr = NULL;
	hfontptr = NULL;
	fontptr = NULL;

	if(verbose & DEBUG_FONTS)
		fprintf(stderr, "Closing %d font files\n", nopen);

	for(i = 0; i <= nopen; i++)
	{
		fclose(pixel_files[i].pixel_file_fd);
		pixel_files[i].pixel_file_fd = NULL;
		pixel_files[i].use_count = 0;
	}

	nopen = 0;
}

/*
 * read_font_def:
 */

bool
read_font_def(k)
int k;
{
	int t, i;
	register struct font_entry *tfontptr;	/* temporary pointer */
	register struct char_entry *tcharptr;/* temporary char_entry pointer */
	char *direct, *tcp, *tcp1;
	int found;
	char curarea[STRSIZE];

	if((tfontptr = (struct font_entry *)calloc(1,
	  sizeof(struct font_entry)))
	  == NULL)
	{
		message("Out of memory for font entries");
		return FALSE;
	}
	tfontptr->next = hfontptr;
	fontptr = hfontptr = tfontptr;

	tfontptr->k = k;
	tfontptr->c = get_unsigned(dvifp, 4); /* checksum */
	tfontptr->s = get_unsigned(dvifp, 4); /* space size */
	tfontptr->d = get_unsigned(dvifp, 4); /* design size */
	tfontptr->a = get_unsigned(dvifp, 1); /* area length for font name */
	tfontptr->l = get_unsigned(dvifp, 1); /* device length */
	fread(tfontptr->n, 1, tfontptr->a+tfontptr->l, dvifp);
	tfontptr->n[tfontptr->a+tfontptr->l] = '\0';
	tfontptr->font_space = tfontptr->s/6; /* never used */
	tfontptr->font_mag = (int)((actual_factor((int)(((float)tfontptr->s/
				(float)tfontptr->d)*1000.0 + 0.5)) * 
#ifdef USEGLOBALMAG
			actual_factor(mag) *
#endif
			(float)resolution * 5.0) + 0.5);
	if (tfontptr->a != 0)
		sprintf(tfontptr->name,
		  "%s.%dpxl", tfontptr->n, tfontptr->font_mag);
	else
	{
		direct = pxl_path;
		found = FALSE;
		do
		{ 
			tcp = index(direct, ':');
			if(tcp == NULL)
				tcp = strlen(direct) + direct;
			strncpy(curarea, direct, tcp-direct);
			tcp1 = curarea + (tcp - direct);
			*tcp1++ = '/';
			*tcp1++ = '\0';

			sprintf(tfontptr->name, "%s%s.%dpxl",
			  curarea, tfontptr->n, tfontptr->font_mag);
			found = (access(tfontptr->name, READ) == 0);
			if(*tcp)
				direct = tcp + 1;
			else
				direct = tcp;
		}
		while( !found && *direct != '\0');
	}
	if(tfontptr != pfontptr)
	{
		if(! open_font_file())
			return FALSE;
	}

	if((t = get_unsigned(pxlfp, 4)) != PXLID)
	{
		message("Bad PXL file version %d", t);
		return FALSE;
	}
	fseek(pxlfp, -20, 2);
	t = get_unsigned(pxlfp, 4);
	if((tfontptr->c != 0) && (t != 0) && (tfontptr->c != t))
		message("Bad PXL checksum, %s", tfontptr->name);

	tfontptr->magnification = get_unsigned(pxlfp, 4);
	tfontptr->designsize = get_unsigned(pxlfp, 4);

	fseek(pxlfp, get_unsigned(pxlfp, 4) * 4, 0);

	for(i = FIRSTPXLCHAR; i <= LASTPXLCHAR; i++)
	{
		tcharptr = &(tfontptr->ch[i]);
		tcharptr->width = get_unsigned(pxlfp, 2);
		tcharptr->height = get_unsigned(pxlfp, 2);
		tcharptr->xOffset= get_signed(pxlfp, 2);
		tcharptr->yOffset = get_signed(pxlfp, 2);
		tcharptr->where.isloaded = FALSE;
		tcharptr->where.address.fileOffset = get_unsigned(pxlfp, 4) * 4;
		tcharptr->tfmw =
		  ((float)get_unsigned(pxlfp, 4)*(float)tfontptr->s) /
		  (float)(1<<20);
	}

	return TRUE;
}


/*
 * open_font_file:
 *	The original version of this dvi driver reopened the font file  each
 *	time the font changed, resulting in an enormous number of relatively
 *	expensive file  openings.   This version  keeps  a cache  of  up  to
 *	MAXOPEN open files,  so that when  a font change  is made, the  file
 *	pointer, pxlfp, can  usually be  updated from the  cache.  When  the
 *	file is not found in  the cache, it must  be opened.  In this  case,
 *	the next empty slot  in the cache  is assigned, or  if the cache  is
 *	full, the least used font file is closed and its slot reassigned for
 *	the new file.  Identification of the least used file is based on the
 *	counts of the number  of times each file  has been "opened" by  this
 *	routine.  On return, the file pointer is always repositioned to  the
 *	beginning of the file.
 */

bool
open_font_file()
{
	register int i,least_used,current;

	if(pfontptr == fontptr)
		return TRUE;

	/*
	 * Search open list for file.
	 */
	for(current = 1; current <= nopen; current++)
		if(pixel_files[current].pixel_file_fd == fontptr->font_file_fd)
			break;

	if(current <= nopen)
	{
		/*
		 * If it is already open
		 * reposition to start of file
		 */
		pxlfp = pixel_files[current].pixel_file_fd;
		fseek(pxlfp,0,0);

		if(verbose & DEBUG_FONTS)
			fprintf(stderr, "Found (%x, %d) font '%s'\n",
			  pxlfp, current, fontptr->name);
	}
	else
	{
		/*
		 * If less than MAXOPEN, then just open it.
		 */
		if(nopen < MAXOPEN)
			current = ++nopen;
		else
		{
			/*
			 * Close least recently used ?
			 */
			register struct font_entry *p = hfontptr;/* {1} */

			least_used = 1;
			for(i = 2; i <= MAXOPEN; ++i)
				if(pixel_files[least_used].use_count >
				  pixel_files[i].use_count)
					least_used = i;

			while(p->font_file_fd !=
			  pixel_files[least_used].pixel_file_fd )
				p = p->next;			/* {1} */

			if(verbose & DEBUG_FONTS)
				fprintf(stderr, "Closing (%x, %d) font '%s'\n",
				  p->font_file_fd, least_used, p->name);

			p->font_file_fd = NULL;		/* {1} */
			fclose(pixel_files[least_used].pixel_file_fd);
			current = least_used;
		}

		/*
		 * Open the file.
		 */
		if((pxlfp=fopen(fontptr->name,"r")) == NULL)
		{
			message("Cant open PXL file %s", fontptr->name);
			return FALSE;
		}
		else
		{
			if(verbose & DEBUG_FONTS)
				fprintf(stderr, "Opened (%x, %d) font '%s'\n",
				  pxlfp, current, fontptr->name);
		}

		pixel_files[current].pixel_file_fd = pxlfp;
		pixel_files[current].use_count = 0;
	}
	pfontptr = fontptr;		/* make previous = current font */
	fontptr->font_file_fd = pxlfp;	/* set file identifier */
	pixel_files[current].use_count++;	/* update reference count */

	return TRUE;
}

/*
 * skip_font_def:
 *	Ignore font definition when fonts have been read from the postamble.
 */

void
skip_font_def(k)
int k;
{
	int a, l;
	char n[STRSIZE];

	get_unsigned(dvifp, 4);
	get_unsigned(dvifp, 4);
	get_unsigned(dvifp, 4);
	a = get_unsigned(dvifp, 1);
	l = get_unsigned(dvifp, 1);
	fseek(dvifp, a+l, 1);
}

/*
 * Utility Functions.
 * =================
 */

/*
 * pix_round:
 */

int
pix_round(x, conv)	/* return rounded number of pixels */
register int x;		/* in DVI units	 */
int conv;		/* conversion factor */
{
	return((int)((x + (conv >> 1)) / conv));
}

/*
 * get_unsigned:
 *
 */

int
get_unsigned(fp, n)	/* return n byte quantity from file fd */
register FILE *fp;	/* file pointer	*/
register int n;		/* number of bytes */
{
	register int x;	/* number being constructed */

	x = 0;
	while (n--)
	{
		x <<= 8;
		x |= getc(fp);
	}

	return(x);
}

/*
 * get_signed:
 *
 */

int
get_signed(fp, n)   /* return n byte quantity from file fd */
register FILE *fp;  /* file pointer	*/
register int n;	 /* number of bytes */
{
	int n1;		 /* number of bytes		*/
	register int x; /* number being constructed */

	x = getc(fp);   /* get first (high-order) byte */
	n1 = n--;
	while (n--)
	{
		x <<= 8;
		x |= getc(fp);
	}

	/* NOTE: This code assumes that the right-shift is an arithmetic, rather
	than logical, shift which will propagate the sign bit right.   According
	to Kernighan and Ritchie, this is compiler dependent! */

	x<<=32-8*n1;
	x>>=32-8*n1;  /* sign extend */

	return(x);
}

/*
 * actual_factor:
 *	compute the actual size factor given the approximation.
 */

double
actual_factor(unmodsize)
int unmodsize;  /* actually factor * 1000 */
{
	float realsize;	/* the actual magnification factor */

	realsize = (float)unmodsize / 1000.0;
	/* a real hack to correct for rounding in some cases--rkf */
	if(unmodsize==1095) realsize = 1.095445;	/*stephalf*/
	else if(unmodsize==1315) realsize=1.314534;	/*stepihalf*/
	else if(unmodsize==2074) realsize=2.0736;	/*stepiv*/
	else if(unmodsize==2488) realsize=2.48832;  /*stepv*/
	else if(unmodsize==2986) realsize=2.985984;	/*stepiv*/
	/* the remaining magnification steps are represented with sufficient
	   accuracy already */
	return(realsize);
}


/*
 * do_convert
 */

int
do_convert(num, den, convResolution)
int num;
int den;
int convResolution;
{
	register float conv;
	conv = ((float)num/(float)den) * 
#ifdef USEGLOBALMAG
/*	actual_factor(mag) * why was this in as Actual Factor?  jls */
	((float) mag/1000.0) *
#endif
	((float)convResolution/254000.0);
	return((int) (1.0 / conv + 0.5));
}

/*
 * Pixrect functions.
 * =================
 */

/*
 * sample_page:
 *	Filter and sample down page according to current sampling rate,
 *	and prepare for display.
 */

void
sample_page()
{
	switch(sampling)
	{
	default:
	case 1:
		sample_pr = pr_link(&page_mpr, &sample_mpr);
		break;

	case 2:
		if(! (sample_pr = pr_sample_2(&page_mpr, &sample_mpr)))
			message("Out of memory for resampling image");
		break;

	case 3:
		if(! (sample_pr = pr_sample_3(&page_mpr, &sample_mpr)))
			message("Out of memory for resampling image");
		break;

	case 4:
		if(! (sample_pr = pr_sample_4(&page_mpr, &sample_mpr)))
			message("Out of memory for resampling image");
		break;
	}
}

/*
 * Here follow some functions to deal with pixrects under the special case
 * assumption that they are mem_pixrects, where the mpr_data is part
 * of a parent structure.
 */

extern struct pixrectops mem_ops;

/*
 * pr_alloc:
 *	Allocate memory for a pixrect of size w, h, d.
 *	Returns a pointer to the pixrect structure.
 */

struct pixrect *
pr_alloc(mpr, w, h, d)
struct mem_pixrect *mpr;
int w, h, d;
{
	int size;
	int linebytes;
	short *image;

	/*
	 * Compute the size of memory needed, and alloc it.
	 */
	linebytes = mpr_linebytes(w, d);
	size = linebytes * h;
	if(! (image = (short *)malloc(size)))
		return (struct pixrect *)NULL;

	/*
	 * Set up the pr.
	 */
	mpr->mpr_pr.pr_ops = &mem_ops;
	mpr->mpr_pr.pr_width = w;
	mpr->mpr_pr.pr_height = h;
	mpr->mpr_pr.pr_depth = d;
	mpr->mpr_pr.pr_data = (caddr_t)&mpr->mpr_data;

	/*
	 * Set up the mpr_data
	 */
	mpr->mpr_data.md_linebytes = linebytes;
	mpr->mpr_data.md_image = image;
	mpr->mpr_data.md_offset.x = 0;
	mpr->mpr_data.md_offset.y = 0;
	mpr->mpr_data.md_primary = TRUE;
	mpr->mpr_data.md_flags = 0;

	/*
	 * Return the pr.
	 */
	return &mpr->mpr_pr;
}

/*
 * pr_free:
 *	Free the memory associated with a pixrect.
 *	Returns a pointer to no pixrect structure.
 */

struct pixrect *
pr_free(mpr)
struct mem_pixrect *mpr;
{
	short *image;

	if((image = mpr->mpr_data.md_image))
	{
		if(mpr->mpr_data.md_primary)
			free(image);
		mpr->mpr_data.md_image = (short *)NULL;
	}
	mpr->mpr_pr.pr_width = 0;
	mpr->mpr_pr.pr_height = 0;

	return (struct pixrect *)NULL;
}

/*
 * pr_check:
 *	Check that a designated pixrect has memory allocated for an image
 *	of size w, h, d. If not, free any existing memory and allocate
 *	more memory. This is equivalent to, but much faster than, a
 *	sequence of
 *		pr_destroy(mpr);
 *		mpr = mem_create(w, h, d);
 */

struct pixrect *
pr_check(mpr, w, h, d)
struct mem_pixrect *mpr;
int w, h, d;
{
	/*
	 * If there is an image, check that it is the correct size.
	 */
	if(mpr->mpr_data.md_image)
	{
		if(mpr->mpr_pr.pr_width == w &&
		  mpr->mpr_pr.pr_height == h &&
		  mpr->mpr_pr.pr_depth == d)
			return &mpr->mpr_pr;

		(void)pr_free(mpr);
	}

	return pr_alloc(mpr, w, h, d);
}

/*
 * pr_link:
 *	Link the memory of mpr1 to mpr2, making mpr2 a secondary pixrect.
 */

struct pixrect *
pr_link(mpr1, mpr2)
struct mem_pixrect *mpr1;
struct mem_pixrect *mpr2;
{
	/*
	 * Free the existing memory, if any.
	 */
	(void)pr_free(mpr2);

	/*
	 * Set up the pr.
	 */
	mpr2->mpr_pr.pr_ops = &mem_ops;
	mpr2->mpr_pr.pr_width = mpr1->mpr_pr.pr_width;
	mpr2->mpr_pr.pr_height = mpr1->mpr_pr.pr_height;
	mpr2->mpr_pr.pr_depth = mpr1->mpr_pr.pr_depth;
	mpr2->mpr_pr.pr_data = (caddr_t)&mpr2->mpr_data;

	/*
	 * Set up the mpr_data
	 */
	mpr2->mpr_data.md_linebytes = mpr1->mpr_data.md_linebytes;
	mpr2->mpr_data.md_image = mpr1->mpr_data.md_image;
	mpr2->mpr_data.md_offset.x = mpr1->mpr_data.md_offset.x;
	mpr2->mpr_data.md_offset.y = mpr1->mpr_data.md_offset.y;
	mpr2->mpr_data.md_primary = FALSE;
	mpr2->mpr_data.md_flags = 0;

	/*
	 * Return the pr.
	 */
	return &mpr2->mpr_pr;
}

/*
 * Tally table used to compute the number of bits in packed words.
 * This is used in computing the number of bits in a 4x4 region of
 * a packed image.
 *
 * The packed word is used as an index into this table;
 * The most significant 16 bits of the value obtained represent the
 * number of bits in the upper half of the index, with each bit counted
 * with a weight of 15.
 * The least significant 16 bits of the value obtained represent the
 * number of bits in the l;ower hald of the index, with each bit counted
 * with a weight of 15.
 *
 * By combining the upper and lower halves in a single word, the values
 * for the four vertically adjacent words can be combined with each other
 * in a single addition per word, adding simultaneously both the tallies
 * for the left and the right halves of the word.
 *
 * The maximum number of bits which can be on is 4x4 = 16, having
 * a maximum value of 16*15. A division by 16 gives 15 as the pixel
 * value to be used. This saves a division to normalise a scale of
 * 0..16 into 0..15.
 */

static int tally4[] =
{
	0x00000000, 0x0000000f, 0x0000000f, 0x0000001e,
	0x0000000f, 0x0000001e, 0x0000001e, 0x0000002d,
	0x0000000f, 0x0000001e, 0x0000001e, 0x0000002d,
	0x0000001e, 0x0000002d, 0x0000002d, 0x0000003c,
	0x000f0000, 0x000f000f, 0x000f000f, 0x000f001e,
	0x000f000f, 0x000f001e, 0x000f001e, 0x000f002d,
	0x000f000f, 0x000f001e, 0x000f001e, 0x000f002d,
	0x000f001e, 0x000f002d, 0x000f002d, 0x000f003c,
	0x000f0000, 0x000f000f, 0x000f000f, 0x000f001e,
	0x000f000f, 0x000f001e, 0x000f001e, 0x000f002d,
	0x000f000f, 0x000f001e, 0x000f001e, 0x000f002d,
	0x000f001e, 0x000f002d, 0x000f002d, 0x000f003c,
	0x001e0000, 0x001e000f, 0x001e000f, 0x001e001e,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e001e, 0x001e002d, 0x001e002d, 0x001e003c,
	0x000f0000, 0x000f000f, 0x000f000f, 0x000f001e,
	0x000f000f, 0x000f001e, 0x000f001e, 0x000f002d,
	0x000f000f, 0x000f001e, 0x000f001e, 0x000f002d,
	0x000f001e, 0x000f002d, 0x000f002d, 0x000f003c,
	0x001e0000, 0x001e000f, 0x001e000f, 0x001e001e,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e001e, 0x001e002d, 0x001e002d, 0x001e003c,
	0x001e0000, 0x001e000f, 0x001e000f, 0x001e001e,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e001e, 0x001e002d, 0x001e002d, 0x001e003c,
	0x002d0000, 0x002d000f, 0x002d000f, 0x002d001e,
	0x002d000f, 0x002d001e, 0x002d001e, 0x002d002d,
	0x002d000f, 0x002d001e, 0x002d001e, 0x002d002d,
	0x002d001e, 0x002d002d, 0x002d002d, 0x002d003c,
	0x000f0000, 0x000f000f, 0x000f000f, 0x000f001e,
	0x000f000f, 0x000f001e, 0x000f001e, 0x000f002d,
	0x000f000f, 0x000f001e, 0x000f001e, 0x000f002d,
	0x000f001e, 0x000f002d, 0x000f002d, 0x000f003c,
	0x001e0000, 0x001e000f, 0x001e000f, 0x001e001e,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e001e, 0x001e002d, 0x001e002d, 0x001e003c,
	0x001e0000, 0x001e000f, 0x001e000f, 0x001e001e,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e001e, 0x001e002d, 0x001e002d, 0x001e003c,
	0x002d0000, 0x002d000f, 0x002d000f, 0x002d001e,
	0x002d000f, 0x002d001e, 0x002d001e, 0x002d002d,
	0x002d000f, 0x002d001e, 0x002d001e, 0x002d002d,
	0x002d001e, 0x002d002d, 0x002d002d, 0x002d003c,
	0x001e0000, 0x001e000f, 0x001e000f, 0x001e001e,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e000f, 0x001e001e, 0x001e001e, 0x001e002d,
	0x001e001e, 0x001e002d, 0x001e002d, 0x001e003c,
	0x002d0000, 0x002d000f, 0x002d000f, 0x002d001e,
	0x002d000f, 0x002d001e, 0x002d001e, 0x002d002d,
	0x002d000f, 0x002d001e, 0x002d001e, 0x002d002d,
	0x002d001e, 0x002d002d, 0x002d002d, 0x002d003c,
	0x002d0000, 0x002d000f, 0x002d000f, 0x002d001e,
	0x002d000f, 0x002d001e, 0x002d001e, 0x002d002d,
	0x002d000f, 0x002d001e, 0x002d001e, 0x002d002d,
	0x002d001e, 0x002d002d, 0x002d002d, 0x002d003c,
	0x003c0000, 0x003c000f, 0x003c000f, 0x003c001e,
	0x003c000f, 0x003c001e, 0x003c001e, 0x003c002d,
	0x003c000f, 0x003c001e, 0x003c001e, 0x003c002d,
	0x003c001e, 0x003c002d, 0x003c002d, 0x003c003c,
};

/*
 * Tally table used to compute the number of bits in packed words.
 * This is used in computing the number of bits in a 3x3 region of
 * a packed image.
 *
 * The packed word is used as an index into this table;
 * Bits 23..16 of the value obtained represent the number of bits in
 * the upper 2.67 bits of the index, with each bit counted with a weight
 * of 15. etc. etc.
 *
 * By combining the three parts in a single word, the values
 * for the three vertically adjacent words can be combined with each other
 * in a single addition per word, adding simultaneously both the tallies
 * for the left and the right halves of the word.
 *
 * The maximum number of bits which can be on is 2.67x3 = 8, having
 * a maximum value of 8*15. A division by 8 gives 15 as the pixel
 * value to be used. This saves a division to normalise a scale of
 * 0..8 into 0..15.
 */

static int tally3[] =
{
	0x000000, 0x00000f, 0x00000f, 0x00001e,
	0x00050a, 0x000519, 0x000519, 0x000528,
	0x000f00, 0x000f0f, 0x000f0f, 0x000f1e,
	0x00140a, 0x001419, 0x001419, 0x001428,
	0x000f00, 0x000f0f, 0x000f0f, 0x000f1e,
	0x00140a, 0x001419, 0x001419, 0x001428,
	0x001e00, 0x001e0f, 0x001e0f, 0x001e1e,
	0x00230a, 0x002319, 0x002319, 0x002328,
	0x0a0500, 0x0a050f, 0x0a050f, 0x0a051e,
	0x0a0a0a, 0x0a0a19, 0x0a0a19, 0x0a0a28,
	0x0a1400, 0x0a140f, 0x0a140f, 0x0a141e,
	0x0a190a, 0x0a1919, 0x0a1919, 0x0a1928,
	0x0a1400, 0x0a140f, 0x0a140f, 0x0a141e,
	0x0a190a, 0x0a1919, 0x0a1919, 0x0a1928,
	0x0a2300, 0x0a230f, 0x0a230f, 0x0a231e,
	0x0a280a, 0x0a2819, 0x0a2819, 0x0a2828,
	0x0f0000, 0x0f000f, 0x0f000f, 0x0f001e,
	0x0f050a, 0x0f0519, 0x0f0519, 0x0f0528,
	0x0f0f00, 0x0f0f0f, 0x0f0f0f, 0x0f0f1e,
	0x0f140a, 0x0f1419, 0x0f1419, 0x0f1428,
	0x0f0f00, 0x0f0f0f, 0x0f0f0f, 0x0f0f1e,
	0x0f140a, 0x0f1419, 0x0f1419, 0x0f1428,
	0x0f1e00, 0x0f1e0f, 0x0f1e0f, 0x0f1e1e,
	0x0f230a, 0x0f2319, 0x0f2319, 0x0f2328,
	0x190500, 0x19050f, 0x19050f, 0x19051e,
	0x190a0a, 0x190a19, 0x190a19, 0x190a28,
	0x191400, 0x19140f, 0x19140f, 0x19141e,
	0x19190a, 0x191919, 0x191919, 0x191928,
	0x191400, 0x19140f, 0x19140f, 0x19141e,
	0x19190a, 0x191919, 0x191919, 0x191928,
	0x192300, 0x19230f, 0x19230f, 0x19231e,
	0x19280a, 0x192819, 0x192819, 0x192828,
	0x0f0000, 0x0f000f, 0x0f000f, 0x0f001e,
	0x0f050a, 0x0f0519, 0x0f0519, 0x0f0528,
	0x0f0f00, 0x0f0f0f, 0x0f0f0f, 0x0f0f1e,
	0x0f140a, 0x0f1419, 0x0f1419, 0x0f1428,
	0x0f0f00, 0x0f0f0f, 0x0f0f0f, 0x0f0f1e,
	0x0f140a, 0x0f1419, 0x0f1419, 0x0f1428,
	0x0f1e00, 0x0f1e0f, 0x0f1e0f, 0x0f1e1e,
	0x0f230a, 0x0f2319, 0x0f2319, 0x0f2328,
	0x190500, 0x19050f, 0x19050f, 0x19051e,
	0x190a0a, 0x190a19, 0x190a19, 0x190a28,
	0x191400, 0x19140f, 0x19140f, 0x19141e,
	0x19190a, 0x191919, 0x191919, 0x191928,
	0x191400, 0x19140f, 0x19140f, 0x19141e,
	0x19190a, 0x191919, 0x191919, 0x191928,
	0x192300, 0x19230f, 0x19230f, 0x19231e,
	0x19280a, 0x192819, 0x192819, 0x192828,
	0x1e0000, 0x1e000f, 0x1e000f, 0x1e001e,
	0x1e050a, 0x1e0519, 0x1e0519, 0x1e0528,
	0x1e0f00, 0x1e0f0f, 0x1e0f0f, 0x1e0f1e,
	0x1e140a, 0x1e1419, 0x1e1419, 0x1e1428,
	0x1e0f00, 0x1e0f0f, 0x1e0f0f, 0x1e0f1e,
	0x1e140a, 0x1e1419, 0x1e1419, 0x1e1428,
	0x1e1e00, 0x1e1e0f, 0x1e1e0f, 0x1e1e1e,
	0x1e230a, 0x1e2319, 0x1e2319, 0x1e2328,
	0x280500, 0x28050f, 0x28050f, 0x28051e,
	0x280a0a, 0x280a19, 0x280a19, 0x280a28,
	0x281400, 0x28140f, 0x28140f, 0x28141e,
	0x28190a, 0x281919, 0x281919, 0x281928,
	0x281400, 0x28140f, 0x28140f, 0x28141e,
	0x28190a, 0x281919, 0x281919, 0x281928,
	0x282300, 0x28230f, 0x28230f, 0x28231e,
	0x28280a, 0x282819, 0x282819, 0x282828,
};

/*
 * Tally table used to compute the number of bits in packed words.
 * This is used in computing the number of bits in a 2x2 region of
 * a packed image.
 *
 * The packed word is used as an index into this table;
 * The most significant 8 bits of the value obtained represent the
 * number of bits in the upper half of the index, with each bit counted
 * with a weight of 15.
 * ...
 * The least significant 8 bits of the value obtained represent the
 * number of bits in the l;ower hald of the index, with each bit counted
 * with a weight of 15.
 *
 * By combining the four pairs of tallies in a single word, the values
 * for the two vertically adjacent words can be combined with each other
 * in a single addition per word, adding simultaneously all four tallies
 * for the four pairs of the word.
 *
 * The maximum number of bits which can be on is 2x2 = 4, having
 * a maximum value of 4*15. A division by 4 gives 15 as the pixel
 * value to be used. This saves a multiplication to normalise a scale of
 * 0..4 into 0..15.
 */

static int tally2[] =
{
	0x00000000, 0x0000000f, 0x0000000f, 0x0000001e,
	0x00000f00, 0x00000f0f, 0x00000f0f, 0x00000f1e,
	0x00000f00, 0x00000f0f, 0x00000f0f, 0x00000f1e,
	0x00001e00, 0x00001e0f, 0x00001e0f, 0x00001e1e,
	0x000f0000, 0x000f000f, 0x000f000f, 0x000f001e,
	0x000f0f00, 0x000f0f0f, 0x000f0f0f, 0x000f0f1e,
	0x000f0f00, 0x000f0f0f, 0x000f0f0f, 0x000f0f1e,
	0x000f1e00, 0x000f1e0f, 0x000f1e0f, 0x000f1e1e,
	0x000f0000, 0x000f000f, 0x000f000f, 0x000f001e,
	0x000f0f00, 0x000f0f0f, 0x000f0f0f, 0x000f0f1e,
	0x000f0f00, 0x000f0f0f, 0x000f0f0f, 0x000f0f1e,
	0x000f1e00, 0x000f1e0f, 0x000f1e0f, 0x000f1e1e,
	0x001e0000, 0x001e000f, 0x001e000f, 0x001e001e,
	0x001e0f00, 0x001e0f0f, 0x001e0f0f, 0x001e0f1e,
	0x001e0f00, 0x001e0f0f, 0x001e0f0f, 0x001e0f1e,
	0x001e1e00, 0x001e1e0f, 0x001e1e0f, 0x001e1e1e,
	0x0f000000, 0x0f00000f, 0x0f00000f, 0x0f00001e,
	0x0f000f00, 0x0f000f0f, 0x0f000f0f, 0x0f000f1e,
	0x0f000f00, 0x0f000f0f, 0x0f000f0f, 0x0f000f1e,
	0x0f001e00, 0x0f001e0f, 0x0f001e0f, 0x0f001e1e,
	0x0f0f0000, 0x0f0f000f, 0x0f0f000f, 0x0f0f001e,
	0x0f0f0f00, 0x0f0f0f0f, 0x0f0f0f0f, 0x0f0f0f1e,
	0x0f0f0f00, 0x0f0f0f0f, 0x0f0f0f0f, 0x0f0f0f1e,
	0x0f0f1e00, 0x0f0f1e0f, 0x0f0f1e0f, 0x0f0f1e1e,
	0x0f0f0000, 0x0f0f000f, 0x0f0f000f, 0x0f0f001e,
	0x0f0f0f00, 0x0f0f0f0f, 0x0f0f0f0f, 0x0f0f0f1e,
	0x0f0f0f00, 0x0f0f0f0f, 0x0f0f0f0f, 0x0f0f0f1e,
	0x0f0f1e00, 0x0f0f1e0f, 0x0f0f1e0f, 0x0f0f1e1e,
	0x0f1e0000, 0x0f1e000f, 0x0f1e000f, 0x0f1e001e,
	0x0f1e0f00, 0x0f1e0f0f, 0x0f1e0f0f, 0x0f1e0f1e,
	0x0f1e0f00, 0x0f1e0f0f, 0x0f1e0f0f, 0x0f1e0f1e,
	0x0f1e1e00, 0x0f1e1e0f, 0x0f1e1e0f, 0x0f1e1e1e,
	0x0f000000, 0x0f00000f, 0x0f00000f, 0x0f00001e,
	0x0f000f00, 0x0f000f0f, 0x0f000f0f, 0x0f000f1e,
	0x0f000f00, 0x0f000f0f, 0x0f000f0f, 0x0f000f1e,
	0x0f001e00, 0x0f001e0f, 0x0f001e0f, 0x0f001e1e,
	0x0f0f0000, 0x0f0f000f, 0x0f0f000f, 0x0f0f001e,
	0x0f0f0f00, 0x0f0f0f0f, 0x0f0f0f0f, 0x0f0f0f1e,
	0x0f0f0f00, 0x0f0f0f0f, 0x0f0f0f0f, 0x0f0f0f1e,
	0x0f0f1e00, 0x0f0f1e0f, 0x0f0f1e0f, 0x0f0f1e1e,
	0x0f0f0000, 0x0f0f000f, 0x0f0f000f, 0x0f0f001e,
	0x0f0f0f00, 0x0f0f0f0f, 0x0f0f0f0f, 0x0f0f0f1e,
	0x0f0f0f00, 0x0f0f0f0f, 0x0f0f0f0f, 0x0f0f0f1e,
	0x0f0f1e00, 0x0f0f1e0f, 0x0f0f1e0f, 0x0f0f1e1e,
	0x0f1e0000, 0x0f1e000f, 0x0f1e000f, 0x0f1e001e,
	0x0f1e0f00, 0x0f1e0f0f, 0x0f1e0f0f, 0x0f1e0f1e,
	0x0f1e0f00, 0x0f1e0f0f, 0x0f1e0f0f, 0x0f1e0f1e,
	0x0f1e1e00, 0x0f1e1e0f, 0x0f1e1e0f, 0x0f1e1e1e,
	0x1e000000, 0x1e00000f, 0x1e00000f, 0x1e00001e,
	0x1e000f00, 0x1e000f0f, 0x1e000f0f, 0x1e000f1e,
	0x1e000f00, 0x1e000f0f, 0x1e000f0f, 0x1e000f1e,
	0x1e001e00, 0x1e001e0f, 0x1e001e0f, 0x1e001e1e,
	0x1e0f0000, 0x1e0f000f, 0x1e0f000f, 0x1e0f001e,
	0x1e0f0f00, 0x1e0f0f0f, 0x1e0f0f0f, 0x1e0f0f1e,
	0x1e0f0f00, 0x1e0f0f0f, 0x1e0f0f0f, 0x1e0f0f1e,
	0x1e0f1e00, 0x1e0f1e0f, 0x1e0f1e0f, 0x1e0f1e1e,
	0x1e0f0000, 0x1e0f000f, 0x1e0f000f, 0x1e0f001e,
	0x1e0f0f00, 0x1e0f0f0f, 0x1e0f0f0f, 0x1e0f0f1e,
	0x1e0f0f00, 0x1e0f0f0f, 0x1e0f0f0f, 0x1e0f0f1e,
	0x1e0f1e00, 0x1e0f1e0f, 0x1e0f1e0f, 0x1e0f1e1e,
	0x1e1e0000, 0x1e1e000f, 0x1e1e000f, 0x1e1e001e,
	0x1e1e0f00, 0x1e1e0f0f, 0x1e1e0f0f, 0x1e1e0f1e,
	0x1e1e0f00, 0x1e1e0f0f, 0x1e1e0f0f, 0x1e1e0f1e,
	0x1e1e1e00, 0x1e1e1e0f, 0x1e1e1e0f, 0x1e1e1e1e,
};

/*
 * P_row:
 *	Access macro for obtaining a pointer to the bytes of a pixrect
 *	starting at row `row'.
 */

#define P_row(mpr, row) \
	(uchar *)((int)((mpr)->mpr_data.md_image) + \
	  (int)((short)(row) * (short)((mpr)->mpr_data.md_linebytes)));

/*
 * pr_sample_4:
 *	Filter and sample mpr1 on a 4x4 basis into mpr2.
 */

struct pixrect *
pr_sample_4(mpr1, mpr2)
struct mem_pixrect *mpr1;
struct mem_pixrect *mpr2;
{
	int cols, rows;
	int j;
	register uchar *p0;		/* a5 */
	register uchar *p1;		/* a4 */
	register uchar *p2;		/* a3 */
	register uchar *p3;		/* a2 */
	register uchar *pr;
	register uchar *er;
	register int tallies;		/* d7 */
	int line_offset;

	cols = mpr1->mpr_pr.pr_width / 4;
	rows = mpr1->mpr_pr.pr_height / 4;

	if(verbose & DEBUG_IMSIZE)
		fprintf(stderr, "pr_sample_4: (%d x %d) -> (%d x %d)\n",
		  mpr1->mpr_pr.pr_width, mpr1->mpr_pr.pr_height, cols, rows);

	/*
	 * Allocate output image.
	 */
	if(! pr_check(mpr2, cols, rows, 8))
		return (struct pixrect *)NULL;

	/*
	 * Do the sampling.
	 */
	line_offset = mpr1->mpr_data.md_linebytes;
	for(j = 0; j < rows; j++)
	{
		p0 = P_row(mpr1, j*4);
		p1 = p0 + line_offset;
		p2 = p1 + line_offset;
		p3 = p2 + line_offset;

		pr = P_row(mpr2, j);

		for(er = pr + cols; pr < er; )
		{
			/*
			 * Test for all zero.
			 */
			if(*p0 == 0 && *p1 == 0 && *p2 == 0 && *p3 == 0)
			{
				*pr++ = 0;

				if(pr >= er)
					break;

				*pr++ = 0;
			}
			else
			{
				tallies =
				  tally4[*p0] +
				  tally4[*p1] +
				  tally4[*p2] +
				  tally4[*p3];

				/*
				 * High order 4 bits
				 */
				*pr++ = (tallies >> 20);

				if(pr >= er)
					break;

				/*
				 * Low order 4 bits
				 */
				*pr++ = (tallies >> 4);
			}
			p0++;
			p1++;
			p2++;
			p3++;
		}
	}

	return &mpr2->mpr_pr;
}

/*
 * pr_sample_3:
 *	Filter and sample mpr1 on a 3x3 basis into mpr2.
 *	Note that the horizontal sampling is actually 3/8 rather than 1/3.
 */

struct pixrect *
pr_sample_3(mpr1, mpr2)
struct mem_pixrect *mpr1;
struct mem_pixrect *mpr2;
{
	int cols, rows;
	int j;
	register uchar *p0;		/* a5 */
	register uchar *p1;		/* a4 */
	register uchar *p2;		/* a3 */
	register uchar *pr;		/* a2 */
	register uchar *er;
	register int tallies;		/* d7 */
	int line_offset;

	cols = mpr1->mpr_pr.pr_width * 3 / 8;
	rows = mpr1->mpr_pr.pr_height / 3;

	if(verbose & DEBUG_IMSIZE)
		fprintf(stderr, "pr_sample_3: (%d x %d) -> (%d x %d)\n",
		  mpr1->mpr_pr.pr_width, mpr1->mpr_pr.pr_height, cols, rows);

	/*
	 * Allocate output image.
	 */
	if(! pr_check(mpr2, cols, rows, 8))
		return (struct pixrect *)NULL;

	/*
	 * Do the sampling.
	 */
	line_offset = mpr1->mpr_data.md_linebytes;
	for(j = 0; j < rows; j++)
	{
		p0 = P_row(mpr1, j*3);
		p1 = p0 + line_offset;
		p2 = p1 + line_offset;

		pr = P_row(mpr2, j);

		for(er = pr + cols; pr < er; )
		{
			/*
			 * Test for all zero.
			 */
			if(*p0 == 0 && *p1 == 0 && *p2 == 0)
			{
				*pr++ = 0;

				if(pr >= er)
					break;

				*pr++ = 0;

				if(pr >= er)
					break;

				*pr++ = 0;
			}
			else
			{
				tallies =
				  tally3[*p0] +
				  tally3[*p1] +
				  tally3[*p2];

				/*
				 * High order 3 bits
				 */
				*pr++ = (tallies >> 19);

				if(pr >= er)
					break;

				*pr++ = (tallies >> 11);

				if(pr >= er)
					break;

				/*
				 * Low order 3 bits
				 */
				*pr++ = (tallies >> 3);
			}
			p0++;
			p1++;
			p2++;
		}
	}

	return &mpr2->mpr_pr;
}

/*
 * pr_sample_2:
 *	Filter and sample mpr1 on a 2x2 basis into mpr2.
 */

struct pixrect *
pr_sample_2(mpr1, mpr2)
struct mem_pixrect *mpr1;
struct mem_pixrect *mpr2;
{
	int cols, rows;
	int j;
	register uchar *p0;		/* a5 */
	register uchar *p1;		/* a4 */
	register uchar *pr;		/* a3 */
	register uchar *er;		/* a2 */
	register int tallies;		/* d7 */
	int line_offset;

	cols = mpr1->mpr_pr.pr_width / 2;
	rows = mpr1->mpr_pr.pr_height / 2;

	if(verbose & DEBUG_IMSIZE)
		fprintf(stderr, "pr_sample_2: (%d x %d) -> (%d x %d)\n",
		  mpr1->mpr_pr.pr_width, mpr1->mpr_pr.pr_height, cols, rows);

	/*
	 * Allocate output image.
	 */
	if(! pr_check(mpr2, cols, rows, 8))
		return (struct pixrect *)NULL;

	/*
	 * Do the sampling.
	 */
	line_offset = mpr1->mpr_data.md_linebytes;
	for(j = 0; j < rows; j++)
	{
		p0 = P_row(mpr1, j*2);
		p1 = p0 + line_offset;

		pr = P_row(mpr2, j);

		for(er = pr + cols; pr < er; )
		{
			/*
			 * Test for all zero.
			 */
			if(*p0 == 0 && *p1 == 0)
			{
				*pr++ = 0;

				if(pr >= er)
					break;

				*pr++ = 0;

				if(pr >= er)
					break;

				*pr++ = 0;

				if(pr >= er)
					break;

				*pr++ = 0;
			}
			else
			{
				tallies =
				  tally2[*p0] +
				  tally2[*p1];

				/*
				 * Highest two bits
				 */
				*pr++ = (tallies >> 26);

				if(pr >= er)
					break;

				*pr++ = (tallies >> 18);

				if(pr >= er)
					break;

				*pr++ = (tallies >> 10);

				if(pr >= er)
					break;

				/*
				 * Lowest two bits
				 */
				*pr++ = (tallies >> 2);
			}
			p0++;
			p1++;
		}
	}

	return &mpr2->mpr_pr;
}

/*
 * pw_cover:
 *	Function which writes a pixrect onto a pixwin;
 *	where there are no src pixels, it writes background colour.
 */

void
pw_cover(dpw, dx, dy, dw, dh, op, spr, sx, sy)
Pixwin *dpw;
int dx, dy, dw, dh;
int op;
Pixrect *spr;
int sx, sy;
{
	int aw, ah;

	/*
	 * Handle the left margin.
	 * If the left margin is less than the width to be painted,
	 * paint a margin, else paint the whole region and return.
	 */
	if(sx < 0)
	{
		if(-sx < dw)
		{
			pw_writebackground(dpw, dx, dy, -sx, dh, op);
			dx -= sx;
			sx = 0;
			dw += sx;
		}
		else
		{
			pw_writebackground(dpw, dx, dy, dw, dh, op);
			return;
		}
	}

	/*
	 * Handle the top margin.
	 * If the top margin is less thatn the width to be painted,
	 * paint a margin, else paint the whole region and return.
	 */
	if(sy < 0)
	{
		if(-sy < dh)
		{
			pw_writebackground(dpw, dx, dy, dw, -sy, op);
			dy -= sy;
			sy = 0;
			dh += sy;
		}
		else
		{
			pw_writebackground(dpw, dx, dy, dw, dh, op);
			return;
		}
	}

	/*
	 * Handle the right margin.
	 * aw = available width of source image.
	 * If available width > 0 paint a margin of dw-aw width,
	 * otherwise paint the whole region.
	 */
	aw = spr->pr_width-sx;
	if(dw > aw)
	{
		if(aw > 0)
		{
			pw_writebackground(dpw, dx+aw, dy, dw-aw, dh, op);
			dw = aw;
		}
		else
		{
			pw_writebackground(dpw, dx, dy, dw, dh, op);
			return;
		}
	}

	/*
	 * Handle the bottom margin.
	 * ah = available height of source image.
	 * If available height > 0 paint a margin of dh-ah height,
	 * otherwise paint the whole region.
	 */
	ah = spr->pr_height-sy;
	if(dh > ah)
	{
		if(ah > 0)
		{
			pw_writebackground(dpw, dx, dy+ah, dw, dh-ah, op);
			dh = ah;
		}
		else
		{
			pw_writebackground(dpw, dx, dy, dw, dh, op);
			return;
		}
	}

	/*
	 * Paint the image.
	 */
	pw_write(dpw, dx, dy, dw, dh, op, spr, sx, sy);
}

/*
 * pr_rect:
 *	Draws a box with op and colour as specified.
 */

void
pr_rect(pr, x, y, w, h, t, op, value)
struct pixrect *pr;
int x, y, w, h;
int t;
int op, value;
{
	int i;

	for(i = 0; i < t; i++)
	{
		pr_vector(pr, x, y, x+w-1, y, op, value);
		pr_vector(pr, x+w-1, y+1, x+w-1, y+h-2, op, value);
		pr_vector(pr, x, y+h-1, x+w-1, y+h-1, op, value);
		pr_vector(pr, x, y+1, x, y+h-2, op, value);

		x += 1;
		y += 1;
		w -= 2;
		h -= 2;

		if(w <= 0 || h <= 0)
			break;
	}
}

/*
 * pw_rect:
 *	Draws a box with op and colour as specified.
 */

void
pw_rect(pw, x, y, w, h, t, op, value)
Pixwin *pw;
int x, y, w, h;
int t;
int op, value;
{
	int i;
	Rect r;

	r.r_left = x;
	r.r_top = y;
	r.r_width = w;
	r.r_height = h;
	pw_lock(pw, &r);

	for(i = 0; i < t; i++)
	{
		pw_vector(pw, x, y, x+w-1, y, op, value);
		pw_vector(pw, x+w-1, y+1, x+w-1, y+h-2, op, value);
		pw_vector(pw, x, y+h-1, x+w-1, y+h-1, op, value);
		pw_vector(pw, x, y+1, x, y+h-2, op, value);

		x += 1;
		y += 1;
		w -= 2;
		h -= 2;

		if(w <= 0 || h <= 0)
			break;
	}

	pw_unlock(pw);
}

/*
 * Args stuff
 * ==========
 */

static char *	a_arg_ptr = NULL;
static int	a_arg_index = 0;
static bool	a_escape_seen;

char *		a_prog_name = "Anonymous";

/*
 * a_next:
 *	Returns the next flag in the command line,
 *	or A_ARG if it is not a flag,
 *	or A_END if there are no more args.
 */

char
a_next(argc, argv)
int argc;
char **argv;
{
	char opt;

	/*
	 * Checks.
	 */
	if(argv == NULL || argc < 1)
	{
		fprintf(stderr, "a_next: bad arguments\n");
		exit(2);
	}

	/*
	 * Get program name on first call.
	 */
	if(a_arg_index == 0)
	{
		a_prog_name = argv[0];
		a_arg_index = 1;
	}

	/*
	 * If there is part of the previous word left, then return it.
	 */
	if(a_arg_ptr && *a_arg_ptr)
		return *a_arg_ptr++;

	/*
	 * Return A_END after the end of the list.
	 */
	if(a_arg_index >= argc)
		return A_END;

	/*
	 * Look at the next word.
	 */
	a_arg_ptr = argv[a_arg_index++];

	/*
	 * If we have seen the escape "--",
	 * or if the first char of the word * is not a '-',
	 * or if this is an isolated "-",
	 * then return ARG.
	 */
	if(a_escape_seen || a_arg_ptr[0] != '-' || a_arg_ptr[1] == '\0')
		return A_ARG;

	/*
	 * Look at the next char.
	 */
	a_arg_ptr++;
	opt = *a_arg_ptr++;

	/*
	 * If the next char is '-', then this is the escape.
	 * start over...
	 */
	if(opt == '-')
	{
		a_escape_seen = TRUE;
		return a_next(argc, argv);
	}

	/*
	 * Otherwise, return this option.
	 */
	return opt;
}

/*
 * a_arg:
 *	Returns the next argument in the command line,
 *	or NULL if there are no more args.
 */

char *
a_arg(argc, argv)
int argc;
char **argv;
{
	char *arg;

	/*
	 * Checks.
	 */
	if(argv == NULL || argc < 1)
	{
		fprintf(stderr, "a_arg: bad arguments\n");
		exit(2);
	}

	/*
	 * Get program name on first call.
	 */
	if(a_arg_index == 0)
	{
		a_prog_name = argv[0];
		a_arg_index = 1;
	}

	/*
	 * If there is part of the previous word left, then return it.
	 */
	if(a_arg_ptr && *a_arg_ptr)
	{
		arg = a_arg_ptr;
		a_arg_ptr = NULL;
		return arg;
	}

	/*
	 * Return NULL after the end of the list.
	 */
	if(a_arg_index >= argc)
		return NULL;

	/*
	 * Return the next word.
	 */
	return argv[a_arg_index++];
}

/*
 * a_number:
 *	Interpret the next word or part word as a number.
 */

double
a_number(argc, argv)
int argc;
char **argv;
{
	char *arg;

	if((arg = a_arg(argc, argv)) == NULL)
		return 0.0;
	else
		return atof(arg);
}

/*
 * a_integer:
 *	Interpret the next word or part word as an integer.
 */

int
a_integer(argc, argv)
int argc;
char **argv;
{
	char *arg;

	if((arg = a_arg(argc, argv)) == NULL)
		return 0;
	else
		return atoi(arg);
}

/*
 * Error panel functions
 * =====================
 */

void		mess_done();

static int errors_displayed = 0;

/*
 * message:
 *	Pops up a message window (if one is not already displayed)
 *	Formats a message (using fmt and args) and displays it into the window.
 *	Repeated calls to this function result in multiple lines of messages.
 */

void
message(fmt, arg0, arg1, arg2, arg3)
char *fmt;
void *arg0, *arg1, *arg2, *arg3;
{
	static char string[100];

	if(silent)
		return;

	sprintf(string, fmt, arg0, arg1, arg2, arg3);

	/*
	 * If there are no errors displayed,
	 * then build a window..
	 */
	if(errors_displayed == 0)
	{
		mess_frame = window_create(disp_frame, FRAME,
		  FRAME_NO_CONFIRM, TRUE,
		  0);
		mess_panel = window_create(mess_frame, PANEL,
		  0);
		panel_create_item(mess_panel, PANEL_BUTTON,
		  PANEL_LABEL_IMAGE,
		    panel_button_image(mess_panel, "OK", 0, 0),
		  PANEL_ITEM_X, ATTR_COL(1),
		  PANEL_ITEM_Y, ATTR_ROW(0),
		  PANEL_NOTIFY_PROC, mess_done,
		  0);
	}
	else
		window_set(mess_frame,
		  WIN_SHOW, FALSE,
		  0);

	panel_create_item(mess_panel, PANEL_MESSAGE,
	  PANEL_LABEL_STRING, string,
	  PANEL_ITEM_X, ATTR_COL(1),
	  PANEL_ITEM_Y, ATTR_ROW(++errors_displayed)+5,
	  0);

	window_fit(mess_panel);
	window_fit(mess_frame);

	window_set(mess_frame,
	  WIN_SHOW, TRUE,
	  0);
}

/*
 * mess_done:
 *	This function is called when the OK button is pressed on the message
 *	window; the window is closed, and displayed messages are cleared.
 */

void
mess_done()
{
	window_destroy(mess_frame);

	errors_displayed = 0;
}

/*
 * Pop up prompts:
 * ==============
 */

static bool				prompt_return_value = TRUE;

static Pixrect				*ok_button = NULL;
static Pixrect				*abort_button = NULL;

static Panel_setting			prompt_ok_proc();
static Panel_setting			prompt_abort_proc();
static Panel_setting			prompt_notify_proc();

/*
 * strings_prompt: eg.
 *	strings_prompt(1152/2, 900/2,
 *	  "Directory: ", &dir[0],
 *	  "Filename: ", &fname[0],
 *	  0);
 */

#define MAX_STRING_ITEMS	10

bool
strings_prompt(x, y, first_prompt, first_value)
int x, y;
char *first_prompt;
char *first_value;
{
	Frame frame;
	Panel panel;
	Panel_item string_item[MAX_STRING_ITEMS], ok_item, abort_item;
	Event *event;
	int w, h;
	int i;
	char **prompt;
	char **value;

	/*
	 * Create the frame and panel.
	 */
	frame = window_create(NULL, FRAME,
	  FRAME_SHOW_LABEL, FALSE,
	  FRAME_NO_CONFIRM, TRUE,
	  0);

	panel = window_create(frame, PANEL,
	  PANEL_ITEM_X_GAP, 1000, /* Enforces vertical layout */
	  0);

	prompt = &first_prompt;
	value = &first_value;
	for(i = 0; i < MAX_STRING_ITEMS; i++)
	{
		/*
		 * Stop if no more strings.
		 */
		if(*prompt == NULL)
			break;

		/*
		 * Create a text item for the string.
		 */
		string_item[i] = panel_create_item(panel, PANEL_TEXT,
		  PANEL_LABEL_STRING, *prompt,
		  PANEL_VALUE, *value,
		  PANEL_VALUE_DISPLAY_LENGTH, 20,
		  PANEL_VALUE_STORED_LENGTH, 132,
		  PANEL_NOTIFY_LEVEL, PANEL_ALL,
		  PANEL_NOTIFY_PROC, prompt_notify_proc,
		  0);

		/*
		 * Advance the prompt and value pointers up the list.
		 */
		prompt = value + 1;
		value = prompt + 1;
	}

	/*
	 * Create an OK button.
	 */
	if(ok_button == NULL)
		ok_button = panel_button_image(panel, "OK", 0, 0);

	ok_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, ok_button,
	  PANEL_NOTIFY_PROC, prompt_ok_proc,
	  0);

	/*
	 * Create an abort button.
	 */
	if(abort_button == NULL)
		abort_button = panel_button_image(panel, "Abort", 0, 0);

	abort_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, abort_button,
	  PANEL_NOTIFY_PROC, prompt_abort_proc,
	  0);

	/*
	 * Make them windows fit.
	 */
	window_fit(panel);
	window_fit(frame);

	/*
	 * Centre the prompt
	 */
	w = (int)window_get(frame, WIN_WIDTH);
	h = (int)window_get(frame, WIN_HEIGHT);
	x -= w/2;
	y -= h/2;
	window_set(frame,
	  WIN_X, x,
	  WIN_Y, y,
	  0);

	/*
	 * Set the flag to TRUE to indicate not aborted.
	 */
	prompt_return_value = TRUE;

	/*
	 * Display and wait for done.
	 */
	(void)window_loop(frame);

	/*
	 * If OK, then retrieve the window values and store them.
	 */
	if(prompt_return_value)
	{
		/*
		 * Loop through the items copying back values.
		 */
		prompt = &first_prompt;
		value = &first_value;
		for(i = 0; i < MAX_STRING_ITEMS; i++)
		{
			/*
			 * Stop if no more strings.
			 */
			if(*prompt == NULL)
				break;

			/*
			 * Get the value of this item.
			 */
			strcpy(*value, panel_get_value(string_item[i]));

			/*
			 * Advance the prompt and value pointers up the list.
			 */
			prompt = value + 1;
			value = prompt + 1;
		}
	}

	/*
	 * Destroy the window.
	 */
	window_destroy(frame);

	return prompt_return_value;
}

/*
 * integers_prompt: eg.
 *	integers_prompt(1152/2, 900/2,
 *	  "Width: ", &w,
 *	  "Height: ", &h,
 *	  0);
 */

#define MAX_INTEGER_ITEMS	MAX_STRING_ITEMS

bool
integers_prompt(x, y, first_prompt, first_value)
int x, y;
char *first_prompt;
int *first_value;
{
	Frame frame;
	Panel panel;
	Panel_item integer_item[MAX_INTEGER_ITEMS], ok_item, abort_item;
	Event *event;
	int w, h;
	int i;
	char **prompt;
	int **value;
	char value_string[20];

	/*
	 * Create the frame and panel.
	 */
	frame = window_create(NULL, FRAME,
	  FRAME_SHOW_LABEL, FALSE,
	  FRAME_NO_CONFIRM, TRUE,
	  0);

	panel = window_create(frame, PANEL,
	  PANEL_ITEM_X_GAP, 1000, /* Enforces vertical layout */
	  0);

	prompt = &first_prompt;
	value = &first_value;
	for(i = 0; i < MAX_INTEGER_ITEMS; i++)
	{
		/*
		 * Stop if no more strings.
		 */
		if(*prompt == NULL)
			break;

		/*
		 * Create a text item for the string.
		 */
		sprintf(value_string, "%d", **value);
		integer_item[i] = panel_create_item(panel, PANEL_TEXT,
		  PANEL_LABEL_STRING, *prompt,
		  PANEL_VALUE, value_string,
		  PANEL_VALUE_DISPLAY_LENGTH, 20,
		  PANEL_VALUE_STORED_LENGTH, 20,
		  PANEL_NOTIFY_STRING, "\033",
		  PANEL_NOTIFY_LEVEL, PANEL_ALL,
		  PANEL_NOTIFY_PROC, prompt_notify_proc,
		  0);

		/*
		 * Advance the prompt and value pointers up the list.
		 */
		prompt = (char **)value + 1;
		value = (int **)prompt + 1;
	}

	/*
	 * Create an OK button.
	 */
	if(ok_button == NULL)
		ok_button = panel_button_image(panel, "OK", 0, 0);

	ok_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, ok_button,
	  PANEL_NOTIFY_PROC, prompt_ok_proc,
	  0);

	/*
	 * Create an abort button.
	 */
	if(abort_button == NULL)
		abort_button = panel_button_image(panel, "Abort", 0, 0);

	abort_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, abort_button,
	  PANEL_NOTIFY_PROC, prompt_abort_proc,
	  0);

	/*
	 * Make them windows fit.
	 */
	window_fit(panel);
	window_fit(frame);

	/*
	 * Centre the prompt
	 */
	w = (int)window_get(frame, WIN_WIDTH);
	h = (int)window_get(frame, WIN_HEIGHT);
	x -= w/2;
	y -= h/2;
	window_set(frame,
	  WIN_X, x,
	  WIN_Y, y,
	  0);

	/*
	 * Set the flag to TRUE to indicate not aborted.
	 */
	prompt_return_value = TRUE;

	/*
	 * Display and wait for done.
	 */
	(void)window_loop(frame);

	/*
	 * If OK, then retrieve the window values and store them.
	 */
	if(prompt_return_value)
	{
		/*
		 * Loop through the items copying back values.
		 */
		prompt = &first_prompt;
		value = &first_value;
		for(i = 0; i < MAX_INTEGER_ITEMS; i++)
		{
			/*
			 * Stop if no more strings.
			 */
			if(*prompt == NULL)
				break;

			/*
			 * Get the value of this item.
			 */
			**value = atoi(panel_get_value(integer_item[i]));

			/*
			 * Advance the prompt and value pointers up the list.
			 */
			prompt = (char **)value + 1;
			value = (int **)prompt + 1;
		}
	}

	/*
	 * Destroy the window.
	 */
	window_destroy(frame);

	return prompt_return_value;
}

/*
 * doubles_prompt: eg.
 *	doubles_prompt(1152/2, 900/2,
 *	  "Width: ", &w,
 *	  "Height: ", &h,
 *	  0);
 */

#define MAX_DOUBLE_ITEMS	MAX_INTEGER_ITEMS

bool
doubles_prompt(x, y, first_prompt, first_value)
int x, y;
char *first_prompt;
double *first_value;
{
	Frame frame;
	Panel panel;
	Panel_item double_item[MAX_DOUBLE_ITEMS], ok_item, abort_item;
	Event *event;
	int w, h;
	int i;
	char **prompt;
	double **value;
	char value_string[20];

	/*
	 * Create the frame and panel.
	 */
	frame = window_create(NULL, FRAME,
	  FRAME_SHOW_LABEL, FALSE,
	  FRAME_NO_CONFIRM, TRUE,
	  0);

	panel = window_create(frame, PANEL,
	  PANEL_ITEM_X_GAP, 1000, /* Enforces vertical layout */
	  0);

	prompt = &first_prompt;
	value = &first_value;
	for(i = 0; i < MAX_DOUBLE_ITEMS; i++)
	{
		/*
		 * Stop if no more strings.
		 */
		if(*prompt == NULL)
			break;

		/*
		 * Create a text item for the string.
		 */
		sprintf(value_string, "%g", **value);
		double_item[i] = panel_create_item(panel, PANEL_TEXT,
		  PANEL_LABEL_STRING, *prompt,
		  PANEL_VALUE, value_string,
		  PANEL_VALUE_DISPLAY_LENGTH, 20,
		  PANEL_VALUE_STORED_LENGTH, 40,
		  PANEL_NOTIFY_STRING, "\033",
		  PANEL_NOTIFY_LEVEL, PANEL_ALL,
		  PANEL_NOTIFY_PROC, prompt_notify_proc,
		  0);

		/*
		 * Advance the prompt and value pointers up the list.
		 */
		prompt = (char **)value + 1;
		value = (double **)prompt + 1;
	}

	/*
	 * Create an OK button.
	 */
	if(ok_button == NULL)
		ok_button = panel_button_image(panel, "OK", 0, 0);

	ok_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, ok_button,
	  PANEL_NOTIFY_PROC, prompt_ok_proc,
	  0);

	/*
	 * Create an abort button.
	 */
	if(abort_button == NULL)
		abort_button = panel_button_image(panel, "Abort", 0, 0);

	abort_item = panel_create_item(panel, PANEL_BUTTON,
	  PANEL_LABEL_IMAGE, abort_button,
	  PANEL_NOTIFY_PROC, prompt_abort_proc,
	  0);

	/*
	 * Make them windows fit.
	 */
	window_fit(panel);
	window_fit(frame);

	/*
	 * Centre the prompt
	 */
	w = (int)window_get(frame, WIN_WIDTH);
	h = (int)window_get(frame, WIN_HEIGHT);
	x -= w/2;
	y -= h/2;
	window_set(frame,
	  WIN_X, x,
	  WIN_Y, y,
	  0);

	/*
	 * Set the flag to FALSE to indicate not aborted.
	 */
	prompt_return_value = FALSE;

	/*
	 * Display and wait for done.
	 */
	(void)window_loop(frame);

	/*
	 * If OK, then retrieve the window values and store them.
	 */
	if(prompt_return_value)
	{
		/*
		 * Loop through the items copying back values.
		 */
		prompt = &first_prompt;
		value = &first_value;
		for(i = 0; i < MAX_DOUBLE_ITEMS; i++)
		{
			/*
			 * Stop if no more strings.
			 */
			if(*prompt == NULL)
				break;

			/*
			 * Get the value of this item.
			 */
			**value = atof(panel_get_value(double_item[i]));

			/*
			 * Advance the prompt and value pointers up the list.
			 */
			prompt = (char **)value + 1;
			value = (double **)prompt + 1;
		}
	}

	/*
	 * Destroy the window.
	 */
	window_destroy(frame);

	return prompt_return_value;
}

/*
 * prompt_notify_proc:
 */

static Panel_setting
prompt_notify_proc(item, event)
Panel_item item;
Event *event;
{
	if(event_id(event) == ESC)
	{
		prompt_return_value = TRUE;
		window_return(NULL);
		return PANEL_NONE;
	}
	else if(event_id(event) == Control('C'))
	{
		prompt_return_value = FALSE;
		window_return(NULL);
		return PANEL_NONE;
	}
	else
		return panel_text_notify(item, event);
}

/*
 * prompt_ok_proc:
 *	Normal return from string prompt.
 */

static Panel_setting
prompt_ok_proc(item, event) 
Panel_item item; 
Event *event;
{ 
	prompt_return_value = TRUE;

	window_return(NULL);

	return PANEL_NONE;
}

/*
 * prompt_abort_proc:
 *	Return from prompt with null string.
 */

static Panel_setting
prompt_abort_proc(item, event)
Panel_item item;
Event *event;
{
	prompt_return_value = FALSE;

	window_return(NULL);

	return PANEL_NONE;
}
